/* computer_player.cpp
 *
 * Pieter Eendebak ( pte@ddsw.nl )
 *
 */

#include "includes.h"

#include "computer_player.moc"
#include "go.h"
#include "go_board.h"
#include "go_player.h"

KGoComputer::KGoComputer( int c, KGoBoard *b,
		QWidget *parent, const char *name )
	: KGoPlayer( c, b, parent, name ),
		xSize(0), ySize(0),
		p(0), l(0), ma(0), ml(0) 

{
	boardSetup() ;
	initData( c ) ;
}

KGoComputer::~KGoComputer()
{
}

void KGoComputer::boardSetup()
{
	#ifdef DEBUG
	debug("KGoComputer::boardSetup()") ;
	//checkFields() ;
	#endif

	/* first remove old arrays */
	KGoBoard::deleteArray( p, xSize, ySize ) ;
	KGoBoard::deleteArray( l, xSize, ySize ) ;
	KGoBoard::deleteArray( ma, xSize, ySize ) ;
	KGoBoard::deleteArray( ml, xSize, ySize ) ;

	KGoBoard *b = getBoard() ;

	xSize = b->getXWidth() ;
	ySize = b->getYWidth() ;

	/* and make new ones */
	KGoBoard::createArray( p, xSize, ySize ) ;
	KGoBoard::createArray( l, xSize, ySize ) ;
	KGoBoard::createArray( ma, xSize, ySize ) ;
	KGoBoard::createArray( ml, xSize, ySize ) ;

	/* init board */
	for (int i = 0; i < xSize; i++)
        for (int j = 0; j < ySize; j++)
		p[i][j] = b->get( i, j ) ;
}

void KGoComputer::updateFromBoard( KGoBoard *b)
{
	setBoard( b ) ;
}

void KGoComputer::initData( int color )
{
	time_t tm;

	mymove = color ;
	umove = OPPONENT(color) ;

	/* init opening pattern numbers to search */
	for (int i = 0; i < 9; i++)
		opn[i] = 1;
	opn[4] = 0;


	/* init global variables */
	mk = 0;  uk = 0;
 
	/* init global variables */
	play = 1;
	pass = 0;
	mik = -1; mjk = -1;
	uik = -1; ujk = -1;

	srand((unsigned)time(&tm));	/* start random number seed */
}

int KGoComputer::load( QDataStream &d, KGoBoard *b )
{
	Q_INT32 c = 0 ;
	d >> c ;

	initData( c ) ;

	updateFromBoard( b ) ;

	return GOOD ;
} 

int KGoComputer::save( QDataStream &d )
{
	d << (Q_INT32)getColor() ;
	return GOOD ;
} 

bool KGoComputer::canBeSaved()
{
	return true ;
}

void KGoComputer::illegalMove( int r, int p )
{
	if( p==getColor() )
		printf( "error: computer made illegal move %s",
			KGoBoard::getIllegalMoveReason(r).data() ) ;
}

void KGoComputer::opponentMoved( int x, int y )
{
	if ( x!=PASS )
	{
		if ( (p[x][y] != EMPTY) || suicide(x, y) )
		{
		       debug("internal error - board made illegal move!") ;
		}
	}

	// This is sometimes done in suicide()
	if ( x!=PASS)
		p[x][y] = OPPONENT( getColor() ) ;

	examboard( getColor() );	 /* remove my dead pieces */

	makeMove() ;
}

void KGoComputer::stateChange( int )
{
	if( toMove() )
		makeMove() ;
}

void KGoComputer::setBoard( KGoBoard *b )
{
	this->KGoPlayer::setBoard( b ) ;
	boardSetup() ;
}
 
void KGoComputer::checkFields()
{
	debug("comp field - true field") ;

	KGoBoard *t = getBoard() ;
	bool tst = false ;

	for( int n=0;n<xSize;n++)
	for(int m=0;m<ySize;m++)
	{
		printf("%d,%d ", p[m][n], t->get(m,n) ) ;
		if ( m==ySize-1 ) printf("\n");  

		if( t->get(n,m)!=p[n][m])
			tst=true;
	}
	if (tst ) debug("boards unequal!");

}

void KGoComputer::makeMove()
{
	#ifdef DEBUG
	//checkFields() ;
	#endif

	int i, j ;

	genmove( &i, &j ) ;

	if ( i!=PASS )
		p[i][j]= getColor() ;
	else
	{
		//debug("pass");
	}
	examboard( OPPONENT(getColor()) );  

	emit doMove( i, j ) ;
}

void KGoComputer::genmove(int *i,
             int *j)
/* generate computer move */
{
   int ti, tj, tval;
   int val;
   int tries = 0;   /* number of try */

/* initialize move and value */
   *i = -1;  *j = -1;  val = -1;

/* re-evaluate liberty of opponent pieces */
   eval(umove);

	/* find opponent piece to capture or attack */
	if (findwinner(&ti, &tj, &tval))
		if (tval > val)
		{
			val = tval;
			*i = ti;
			*j = tj;
		}

	/* save any piece if threaten */
	if (findsaver(&ti, &tj, &tval))
		if (tval > val)
		{
			val = tval;
			*i = ti;
			*j = tj;
		}

	/* try match local play pattern for new move */
	if (findpatn(&ti, &tj, &tval))
		if (tval > val)
		{
			val = tval;
			*i = ti;
			*j = tj;
		}

	/* no urgent move then do random move */
	if (val < 0)
		do
		{
			*i = rand() % xSize;

			/* avoid low line and center region */
			if (	(*i < 2) || (*i > xSize-3) ||
				(   (*i > (xSize/2)-2) &&
				    (*i < (xSize/2)+1)
				)
			   )
			{
				*i = rand() % xSize;
				if ((*i < 2) || (*i > xSize-3))
					*i = rand()%xSize;
			}

			*j = rand() % ySize;

			/* avoid low line and center region */
			if (	(*j < 2) || (*j > ySize-3) ||
				(   (*j > (ySize/2)-2) &&
				    (*j < (ySize/2)+1)
				)
			   )
			{
        			*j = rand() % ySize;

				if ((*j < 2) || (*j > 16))
        				*j = rand() % ySize;
			}
		    	lib = 0;
		    	countlib(*i, *j, mymove);
		}
		/* avoid illegal move, liberty one or suicide,
			fill in own eye */
 		while ((++tries < MAXTRY) 
			&& ((p[*i][*j] != EMPTY) ||
			(lib < 2) || fioe(*i, *j)));

	if (tries >= MAXTRY)  /* computer pass */
	{
		pass++;
		*i = -1;
	}
	else   /* find valid move */
	{
		pass = 0;      
	}
/* end genmove */
}

void KGoComputer::count(int i,     /* row number 0 to xSize-1 */
           int j,     /* column number 0 to ySize-1 */
           int color) /* BLACK or WHITE */
/* count liberty of color piece at location i, j
   and return value in lib */
{
	/* set current piece as marked */
	ml[i][j] = MARKED;

	/* check North neighbor */
	if (i != 0)
	{
		if ((p[i - 1][j] == EMPTY ) && ml[i - 1][j]==UN_MARKED )
		{
			++lib;
			ml[i - 1][j] = MARKED;
		}
		else
		if ((p[i - 1][j] == color) && ml[i - 1][j]==UN_MARKED)
			count(i - 1, j, color);
	}

	/* check South neighbor */
	if (i != xSize-1)
	{
		if ((p[i + 1][j] == EMPTY) && ml[i + 1][j]==UN_MARKED)
		{
			++lib;
			ml[i + 1][j] = MARKED;
		}
		else
		if ((p[i + 1][j] == color) && ml[i + 1][j]==UN_MARKED)
			count(i + 1, j, color);
	}

	/* check West neighbor */
 if (j != 0 )
   {
    if ((p[i][j - 1] == EMPTY) && ml[i][j - 1]==UN_MARKED)
      {
       ++lib;
       ml[i][j - 1] = MARKED;
     }
    else
       if ((p[i][j - 1] == color) && ml[i][j - 1]==UN_MARKED)
	  count(i, j - 1, color);
  }
/* check East neighbor */
 if (j != ySize-1)
   {
    if ((p[i][j + 1]==EMPTY ) && ml[i][j + 1]==UN_MARKED)
      {
       ++lib;
       ml[i][j + 1] = MARKED;
     }
    else
       if ((p[i][j + 1] == color) && ml[i][j + 1]==UN_MARKED)
	  count(i, j + 1, color);
  }
}  /* end count */


void KGoComputer::examboard(int color) /* BLACK or WHITE */
/* examine pieces */
{
	int i, j, n;

	/* find liberty of each piece */
	eval(color);

	/* initialize piece captured */
	if (color == mymove)
	{
		mik = -1;
		mjk = -1;
	}
	else
	{
		uik = -1;
		ujk = -1;
	}
	n = 0; /* The number of captures this move for Ko purposes */

	/* remove all piece of zero liberty */
	for (i = 0; i < xSize; i++)
	for (j = 0; j < ySize; j++)
		if ((p[i][j] == color) && (l[i][j] == 0))
		{
			p[i][j] = EMPTY;
			/* record piece captured */
			if (color == mymove)
			{
				mik = i;
				mjk = j;
				++mk;
			}
			else
			{
				uik = i;
				ujk = j;
				++uk;
			}
			++n;  /* increment number of captures on this move */
		}
		/* reset to -1 if more than one stone captured
			since  no Ko possible */
		if (color == mymove && n > 1)
		{
			mik = -1;   
			mjk = -1;
		}
		else if ( n > 1 )
		{
			uik = -1;
			ujk = -1;
		}
}  /* end examboard */


void KGoComputer::eval(int color)  /* BLACK or WHITE */
/* evaluate liberty of color pieces */
{
	int i, j;

	/* find liberty of each piece */
	for (i = 0; i < xSize; i++)
	for (j = 0; j < ySize; j++)
		if (p[i][j] == color)
		{
			lib = 0;
			countlib(i, j, color);
			l[i][j] = lib;
		}
}  /* end eval */

void KGoComputer::countlib(int m,     /* row number 0 to xSize-1 */
              int n,     /* column number 0 to ySize-1 */
              int color) /* BLACK or WHITE */
/* count liberty of color piece at m, n */
{
	//printf("countlib: %d, %d\n", m, n ) ;
	int i, j;

	/* set all piece as unmarked */
	for (i = 0; i < xSize; i++)
	for (j = 0; j < ySize; j++)
		ml[i][j] = UN_MARKED ;

	/* count liberty of current piece */
	count(m, n, color);
}  /* end countlib */

unsigned int KGoComputer::findcolor(int i,   /* row number 0 to xSize-1 */
                       int j)   /* column number 0 to ySize-1 */
/* find color for empty piece */
{
	int k, result=0, color[4];

	/*
	 * return color if all four neighbors are the same or empty
	 */

	if (p[i][j] != EMPTY) return p[i][j];


	/* check West neighbor */
	if (i>0) {  /* The if prevents reading off edge of board. */
		k = i;
		do --k;
		while ((p[k][j] == EMPTY) && (k > 0));
	color[0] = p[k][j];
	}
	else color[0] = p[i][j];

/* check East neighbor */
   
   if (i<xSize-1) {
     k = i;
     do ++k;
     while ((p[k][j] == EMPTY) && (k < xSize-1));
     color[1] = p[k][j];
   }
   else color[1] = p[i][j];

/* check North neighbor */
   if (j>0) {
     k = j;
     do --k;
     while ((p[i][k] == EMPTY) && (k > 0));
     color[2] = p[i][k];
   }
   else color[2] = p[i][j];

/* check South neighbor */
   if (j<ySize-1) {
     k = j;
     do ++k;
     while ((p[i][k] == EMPTY) && (k < ySize-1));
     color[3] = p[i][k];
   }
   else color[3] = p[i][j];

/* Any nonEMPTY color is what we want */

   for (k=0;k<4;k++) {
     if (color[k] == EMPTY) continue;
        else {
          result = color[k];
          break;
        }
   }

/* We know the right color.  Now cross check it.*/
/* If we find an error then all the dead pieces were not taken 
   from the board and we need to prompt the players to fix this. */

  for (k=0;k<4;k++) {
/* If this next test fails, results are inconsistent.  Problem. */
     if ((color[k] != EMPTY) && (color[k] != result)) return 0;
  }

/* If we get to this point everything checks out OK.  Report results. */

   return result;
}  /* end findcolor */



int KGoComputer::findnextmove(int m,       /* current stone row number */
                 int n,       /* current stone column number */
                 int *i,      /* next move row number */
                 int *j,      /* next move column number */
                 int *val,    /* next move value */
                 int minlib)  /* current stone liberty */
/* find new move i, j from group containing m, n */
 {
  int ti, tj, tval;
  int found = 0;

  *i = -1;   *j = -1;	*val = -1;
/* mark current position */
  ma[m][n] = MARKED ;

/* check West neighbor */
  if (m != 0)
     if (p[m - 1][n] == EMPTY)
      {
       ti = m - 1;
       tj = n;
       lib = 0;
       countlib(ti, tj, mymove);
       tval = fval(lib, minlib);
       found = 1;
      }
     else
       if ((p[m - 1][n] == mymove) && ma[m - 1][n]!=MARKED )
	 if (findnextmove(m - 1, n, &ti, &tj, &tval, minlib))
	    found = 1;

  if (found)
    {
     found = 0;
     if (tval > *val && fioe(ti, tj) != 1)
       {
	*val = tval;
	*i = ti;
	*j = tj;
      }
   }

/* check East neighbor */
  if (m != xSize-1)
     if (p[m + 1][n] == EMPTY)
      {
       ti = m + 1;
       tj = n;
       lib = 0;
         countlib(ti, tj, mymove);
       tval = fval(lib, minlib);
       found = 1;
      }
     else
       if ((p[m + 1][n] == mymove) && ma[m + 1][n]!=MARKED)
	  if (findnextmove(m + 1, n, &ti, &tj, &tval, minlib))
	     found = 1;

  if (found)
    {
     found = 0;
     if (tval > *val && fioe(ti, tj) != 1)
       {
	*val = tval;
	*i = ti;
	*j = tj;
      }
   }

/* check North neighbor */
  if (n != 0)
     if (p[m][n - 1] == EMPTY)
      {
       ti = m;
       tj = n - 1;
       lib = 0;
        countlib(ti, tj, mymove);
       tval = fval(lib, minlib);
       found = 1;
      }
     else
       if ((p[m][n - 1] == mymove) && ma[m][n - 1]!=MARKED )
	  if (findnextmove(m, n - 1, &ti, &tj, &tval, minlib))
	      found = 1;

  if (found)
    {
     found = 0;
     if (tval > *val && fioe(ti, tj) != 1)
       {
	*val = tval;
	*i = ti;
	*j = tj;
      }
   }

/* check South neighbor */
  if (n != ySize-1 )
     if (p[m][n + 1] == EMPTY)
      {
       ti = m;
       tj = n + 1;
       lib = 0;
         countlib(ti, tj, mymove);
       tval = fval(lib, minlib);
       found = 1;
      }
     else
       if ((p[m][n + 1] == mymove) && ma[m][n + 1]!=MARKED )
	  if (findnextmove(m, n + 1, &ti, &tj, &tval, minlib))
	      found = 1;

  if (found)
    {
     found = 0;
     if (tval > *val && fioe(ti, tj) != 1)
       {
	*val = tval;
	*i = ti;
	*j = tj;
      }
   }

 if (*val > 0)	/* found next move */
    return 1;
 else	/* next move failed */
    return 0;
}  /* end findnextmove */

int KGoComputer::fval(int newlib,   /* new liberty */
         int minlib)   /* current liberty */
/* evaluate function for new move */
{
 int k, val;

 if (newlib <= minlib)
    val = -1;
 else
   {
    k = newlib - minlib;
    val = 40 + (k - 1) * 50 / (minlib * minlib * minlib);
  }
 return val;
}  /* end fval */


int KGoComputer::findopen(int m,      /* current row number 0 to xSize-1 */
             int n,      /* current column number 0 to ySize-1 */
             int i[],    /* row array for possible moves */
             int j[],    /* column array for possible moves */
             int color,  /* BLACK or WHITE */
             int minlib, /* current liberty */
             int *ct)    /* number of possible moves */
/* find all open spaces i, j from m, n */
{
	/* mark this one */
	ma[m][n] = MARKED ;

	/* check West neighbor */
	if (m != 0)
	{
		if ((p[m - 1][n] == EMPTY) &&
			(((m - 1) != mik) || (n !=mjk)))
		{
			i[*ct] = m - 1;
			j[*ct] = n;
			++*ct;
			if (*ct == minlib) return 1;
		}
		else
		if ((p[m - 1][n] == color) && ma[m - 1][n]!=MARKED )
		if (findopen(m - 1, n, i, j, color, minlib, ct) &&
			(*ct == minlib))
				return 1;
	}

	/* check East neighbor */
	if (m != xSize-1)
	{
	if ((p[m + 1][n] == EMPTY) && (((m + 1) != mik) || (n != mjk)))
	{
		i[*ct] = m + 1;
		j[*ct] = n;
		++*ct;
		if (*ct == minlib) return 1;
	}
	else
	if ((p[m + 1][n] == color) && ma[m + 1][n]!=MARKED)
	 if (findopen(m + 1, n, i, j, color, minlib, ct) && (*ct == minlib))
	    return 1;
  }

/* check North neighbor */
 if (n != 0)
   {
    if ((p[m][n - 1] == EMPTY) && ((m != mik) || ((n - 1) != mjk)))
      {
       i[*ct] = m;
       j[*ct] = n - 1;
       ++*ct;
       if (*ct == minlib) return 1;
     }
    else
      if ((p[m][n - 1] == color) && ma[m][n - 1]!=MARKED)
	 if (findopen(m, n - 1, i, j, color, minlib, ct) && (*ct == minlib))
	    return 1;
  }

	/* check South neighbor */
	if (n != ySize-1)
   {
    if ((p[m][n + 1] == EMPTY) && ((m != mik) || ((n + 1) != mjk)))
      {
       i[*ct] = m;
       j[*ct] = n + 1;
       ++*ct;
       if (*ct == minlib) return 1;
     }
    else
      if ((p[m][n + 1] == color) && !ma[m][n + 1])
	 if (findopen(m, n + 1, i, j, color, minlib, ct) && (*ct == minlib))
	    return 1;
  }

/* fail to find open space */
 return 0;
}  /* end findopen */


int KGoComputer::findpatn(int *i,    /* row number of next move */
             int *j,    /* column number of next move */
             int *val)  /* value of next move */
/* find pattern to match for next move */
{
 int m, n;
 int ti, tj, tval;
 static int cnd, mtype;  /* game tree node number, move type */
/* mtype = 0, basic; 1, inverted; 2, reflected; 3, inverted & reflected */

/* open game then occupy corners */
 if (opn[4])   /* continue last move */
   {
    opn[4] = 0;  /* clear flag */
    if (opening(i, j, &cnd, mtype)) opn[4] = 1; /* more move then reset flag */
    if (p[*i][*j] == EMPTY)  /* valid move */
      {
       *val = 80;
       return 1;
     }
    else
      opn[4] = 0;
  }

 if (opn[0])   /* Northwest corner */
   {
    opn[0] = 0;  /* clear flag */
    if (openregion(0, 0, 5, 5))
      {
       cnd = 0;
       mtype = 0;
       opening(i, j, &cnd, mtype);  /* get new node for next move */
       if (opening(i, j, &cnd, mtype)) opn[4] = 1;
       *val = 80;
       return 1;
     }
 }

 if (opn[1])   /* Southwest corner */
   {
    opn[1] = 0;
    if (openregion(13, 0, 18, 5))
      {
       cnd = 0;
       mtype = 1;
       opening(i, j, &cnd, mtype);  /* get new node for next move */
       if (opening(i, j, &cnd, mtype)) opn[4] = 1;
       *val = 80;
       return 1;
     }
  }

 if (opn[2])   /* Northeast corner */
   {
    opn[2] = 0;
    if (openregion(0, 13, 5, 18))
      {
       cnd = 0;
       mtype = 2;
       opening(i, j, &cnd, mtype);  /* get new node for next move */
       if (opening(i, j, &cnd, mtype)) opn[4] = 1;
       *val = 80;
       return 1;
     }
  }

 if (opn[3])   /* Northeast corner */
   {
    opn[3] = 0;
    if (openregion(13, 13, 18, 18))
      {
       cnd = 0;
       mtype = 3;
       opening(i, j, &cnd, mtype);  /* get new node for next move */
       if (opening(i, j, &cnd, mtype)) opn[4] = 1;
       *val = 80;
       return 1;
     }
  }

/* occupy edges */
 if (opn[5])   /* North edge */
   {
    opn[5] = 0;
    if (openregion(0, 6, 4, 11))
      {
       *i = 3;
       *j = 9;
       *val = 80;
       return 1;
     }
  }

 if (opn[6])   /* South edge */
   {
    opn[6] = 0;
    if (openregion(18, 6, 14, 11))
      {
       *i = 15;
       *j = 9;
       *val = 80;
       return 1;
     }
  }

 if (opn[7])   /* West edge */
   {
    opn[7] = 0;
    if (openregion(6, 0, 11, 4))
      {
       *i = 9;
       *j = 3;
       *val = 80;
       return 1;
     }
  }

 if (opn[8])   /* East edge */
   {
    opn[8] = 0;
    if (openregion(6, 18, 11, 14))
      {
       *i = 9;
       *j = 15;
       *val = 80;
       return 1;
     }
  }

 *i = -1;
 *j = -1;
 *val = -1;

/* find local pattern */
 for (m = 0; m < xSize; m++)
   for (n = 0; n < ySize; n++)
     if ((p[m][n] == mymove) &&
         (matchpat(m, n, &ti, &tj, &tval) && (tval > *val)))
       {
        *val = tval;
        *i = ti;
        *j = tj;
      }
 if (*val > 0)  /* pattern found */
    return 1;
 else  /* no match found */
    return 0;
}  /* end findpatn */


int KGoComputer::findsaver(int *i,    /* row number of next move */
              int *j,    /* column number of next move */
              int *val)  /* value of next move */
/* find move if any pieces is threaten */
{
   int m, n, minlib;
   int ti, tj, tval;

   *i = -1;   *j = -1;	 *val = -1;
   for (minlib = 1; minlib < 4; minlib++)
      {
/* count piece with minimum liberty */
       for (m = 0; m < xSize; m++)
	 for (n = 0; n < ySize; n++)
	   if ((p[m][n] == mymove) && (l[m][n] == minlib))
/* find move to save pieces */
	     {
	      initmark();
	      if (findnextmove(m, n, &ti, &tj, &tval, minlib) && (tval > *val))
		{
		 *val = tval;
		 *i = ti;
		 *j = tj;
	       }
	     }
     }
    if (*val > 0)   /* find move */
       return 1;
    else	    /* move not found */
       return 0;
 }  /* findsaver */

int KGoComputer::findwinner(int *i,    /* row number of next move */
               int *j,    /* column number of next move */
               int *val)  /* value of next move */
/* find opponent piece to capture or attack */
{
 int m, n, ti[3], tj[3], tval, ct, u, v, lib1;

 *i = -1;   *j = -1;   *val = -1;

/* find opponent with liberty less than four */
 for (m = 0; m < xSize; m++)
   for (n = 0; n < ySize; n++)
     if ((p[m][n] == umove) && (l[m][n] < 4))
       {
	ct = 0;
	initmark();
	if (findopen(m, n, ti, tj, umove, l[m][n], &ct))
	  {
	   if (l[m][n] == 1)
	     {
	      if (*val < 120)
		{
		 *val = 120;
		 *i = ti[0];
		 *j = tj[0];
	       }
	    }
	   else
	     for (u = 0; u < (int)l[m][n]; u++)
	       for (v = 0; v < (int)l[m][n]; v++)
		  if (u != v)
		    {
		     lib = 0;
  		     countlib(ti[u], tj[u], mymove);
		     if (lib > 0) /* valid move */
		       {
                        lib1 = lib;
			p[ti[u]][tj[u]] = mymove;
		    /* look ahead opponent move */
			lib = 0;
			countlib(ti[v], tj[v], umove);
			if ((lib1 == 1) && (lib > 0))
                          tval = 0;
                        else
                          tval = 120 - 20 * lib;
			if (*val < tval)
			  {
			   *val = tval;
			   *i = ti[u];
			   *j = tj[u];
			 }
			p[ti[u]][tj[u]] = EMPTY;
		      }
		   }
	 }
      }
 if (*val > 0)	/* find move */
    return 1;
 else  /* fail to find winner */
    return 0;
}  /* end findwinner */


int KGoComputer::fioe(int i,   /* stone row number 0 to xSize-1 */
         int j)   /* stone column number 0 to ySize-1 */
{
	/* check west edge */
	if (i == 0)
	{
		if ((j == 0) && ((p[1][0] == mymove) &&
			(p[0][1] == mymove)))
				return 1;
 		if ((j == ySize-1) && ((p[1][ySize-1] == mymove) &&
			(p[0][ySize-2] == mymove)))
				return 1;
		if ((p[1][j] == mymove) && ((p[0][j - 1] == mymove) &&
			(p[0][j + 1] == mymove)))
				return 1;
		else return 0;
	}

	/* check east edge */
	if (i == xSize-1)
	{
		if ((j == 0) && ((p[xSize-2][0] == mymove) &&
			(p[xSize-1][1] == mymove)))
				return 1;
		if ((j == ySize-1) && ((p[xSize-2][ySize-1] == mymove) &&
			(p[xSize-1][ySize-2] == mymove)))
				return 1;
     		if ((p[xSize-2][j] == mymove) &&
			((p[xSize-1][j - 1] == mymove) &&
			(p[xSize-1][j + 1] == mymove)))
				return 1;
		else return 0;
	}

	/* check top edge */
	if (j == 0)
		if ((p[i][1] == mymove) && ((p[i - 1] [0] == mymove) &&
			(p[i +1][0] == mymove)))
				return 1;
	else return 0;

	/* check bottom edge */
	if (j == xSize-1)
		if ((p[i][ySize-2] == mymove) &&
			((p[i - 1][ySize-1] == mymove) &&
			(p[i+ 1][ySize-1] == mymove)))
				return 1;
		else return 0;

	/* check center pieces */
	if (((p[i][j - 1] == mymove) && (p[i][j + 1] == mymove)) &&
	     ((p[i - 1][j] == mymove) && (p[i + 1][j] == mymove)))
		    return 1;
	else	return 0;
	/* end of fioe */
}

void KGoComputer::initmark(void)
/* initialize all marking with zero */
{
	int i, j;

	for (i = 0; i < xSize; i++)
	for (j = 0; j < ySize; j++)
		ma[i][j] = UN_MARKED ;
}  /* end initmark */


int KGoComputer::matchpat(int m,     /* row origin */
             int n,     /* column origin */
             int *i,    /* row number of next move */
             int *j,    /* column number of next move */
             int *val)  /* next move value */
/* match pattern and get next move */
{
/* transformation matrice */
 static int trf [8][2][2] = {
   {{1, 0}, {0, 1}},   /* linear transfomation matrix */
   {{1, 0}, {0, -1}},  /* invert */
   {{0, 1}, {-1, 0}},  /* rotate 90 */
   {{0, -1}, {-1, 0}}, /* rotate 90 and invert */
   {{-1, 0}, {0, 1}},  /* flip left */
   {{-1, 0}, {0, -1}}, /* flip left and invert */
   {{0, 1}, {1, 0}},   /* rotate 90 and flip left */
   {{0, -1}, {1, 0}}   /* rotate 90, flip left and invert */
 };
 int k, my, nx, ll, r, cont;
 int ti=0, tj=0, tval;

 *i = -1;   *j = -1;   *val = -1;
 for (r = 0; r < PATNO; r++)
/* try each pattern */
    for (ll = 0; ll < pat[r].trfno; ll++)
/* try each orientation transformation */
      {
       k = 0;  cont = 1;
       while ((k != pat[r].patlen) && cont)
/* match each point */
	 {
/* transform pattern real coordinate */
	  nx = n + trf[ll][0][0] * pat[r].patn[k].x
		 + trf[ll][0][1] * pat[r].patn[k].y;
	  my = m + trf[ll][1][0] * pat[r].patn[k].x
		 + trf[ll][1][1] * pat[r].patn[k].y;

/* outside the board */
	  if ((my < 0) || ( my > xSize-1) || (nx < 0) || (nx > ySize-1))
	    {
	     cont = 0;
	     break;
	   }
	  switch (pat[r].patn[k].att) {
	  case 0 : if (p[my][nx] == EMPTY)	/* open */
		      break;
		   else
		     {
		      cont = 0;
		      break;
		    }
	  case 1 : if (p[my][nx] == umove)  /* your piece */
		      break;
		   else
		     {
		      cont = 0;
		      break;
		    }
	  case 2 : if (p[my][nx] == mymove)  /* my piece */
		      break;
		   else
		     {
		      cont = 0;
		      break;
		    }
	  case 3 : if (p[my][nx] == EMPTY)	/* open for new move */
		    {
		     lib = 0;
		     countlib(my, nx, mymove);	/* check liberty */
		     if (lib > 1)  /* move o.k. */
		       {
			ti = my;
			tj = nx;
			break;
		       }
		     else
		       {
			cont = 0;
			break;
		      }
		     }
		   else
		     {
		      cont = 0;
		      break;
		    }
	  case 4 : if ((p[my][nx] == EMPTY)  /* open on edge */
		       && ((my == 0) || (my == xSize) || (nx == 0) ||
				(nx == ySize)))
 		      break;
		   else
		     {
		      cont = 0;
		      break;
		    }
	  case 5 : if ((p[my][nx] == umove)  /* your piece on edge */
		       && ((my == 0) || (my == xSize) || (nx == 0) ||
				(nx == ySize)))
	 		      break;
		   else
		     {
		      cont = 0;
		      break;
		    }
	  case 6 : if ((p[my][nx] == mymove)  /* my piece on edge */
		       && ((my == 0) || (my == xSize) || (nx == 0) ||
				(nx ==ySize)))
	 		      break;
		   else
		     {
		      cont = 0;
		      break;
		    }
		 }
	  ++k;
	 }
	 if (cont)   /* match pattern */
	   {
	    tval = pat[r].patwt;
	    if ((r >= 8) && (r <= 13))	/* patterns for expand region */
	      {
	       if (line(ti) > 7)  /* penalty on line 1, 2 */
		  tval--;
	       else
		  if ((line(ti) == 6) || (line(ti) == 7))
		     tval++;	/* reward on line 3, 4 */

	       if (line(tj) > 7)  /* penalty on line 1, 2 */
		  tval--;
	       else
		  if ((line(tj) == 6) || (line(tj) == 7))
		     tval++;	/* reward on line 3, 4 */
	     }
	    if (tval > *val)
	      {
	       *val = tval;
	       *i = ti;
	       *j = tj;
	     }
	  }
      }
 if (*val > 0)	/* pattern matched */
    return 1;
 else  /* match failed */
    return 0;
}  /* end matchpat */


int KGoComputer::opening(int *i,
            int *j,
            int *cnd,
            int type)
/* get move for opening from game tree */
{
 struct tnode {
   int i, j, ndct, next[8];
  };

 static struct tnode tree[] = {
  {-1, -1, 8, { 1, 2, 3, 4, 5, 6, 7, 20}},	/* 0 */
  {2, 3, 2, { 8, 9}},
  {2, 4, 1, {10}},
  {3, 2, 2, {11, 12}},
  {3, 3, 6, {14, 15, 16, 17, 18, 19}},
  {3, 4, 1, {10}},  /* 5 */
  {4, 2, 1, {13}},
  {4, 3, 1, {13}},
  {4, 2, 0},
  {4, 3, 0},
  {3, 2, 0},  /* 10 */
  {2, 4, 0},
  {3, 4, 0},
  {2, 3, 0},
  {2, 5, 1, {10}},
  {2, 6, 1, {10}},  /* 15 */
  {3, 5, 1, {10}},
  {5, 2, 1, {13}},
  {5, 3, 1, {13}},
  {6, 2, 1, {13}},
  {2, 2, 0}  /* 20 */
};
int m;

/* get i, j */
 if ((type == 1) || (type == 3))
    *i = 18 - tree[*cnd].i;   /* inverted */
 else
    *i = tree[*cnd].i;
 if ((type == 2) || (type == 3))
    *j = 18 - tree[*cnd].j;   /* reflected */
 else
    *j = tree[*cnd].j;
 if (tree[*cnd].ndct)  /* more move */
   {
    m = rand() % tree[*cnd].ndct;  /* select move */
    *cnd = tree[*cnd].next[m];	/* new	current node */
    return 1;
  }
 else
    return 0;
}  /* end opening */

int KGoComputer::suicide(int i,
            int j)
/* check for suicide move of opponent at p[i][j] */
{
	int m, n, k;

	/* check liberty of new move */
	lib = 0;
	countlib(i, j, umove);
	if (lib == 0)
	/* new move is suicide then check
		if kill my pieces and Ko possibility */
	{
		/* assume alive */
		p[i][j] = umove;

		/* check my pieces */
		eval(mymove);
		k = 0;
	
		for (m = 0; m < xSize; m++)
		for (n = 0; n < ySize; n++)
		/* count pieces will be killed */
			if ((p[m][n] == mymove) && !l[m][n]) ++k;

		if ((k == 0) || (k == 1 && ((i == uik) && (j == ujk))))
		/* either no effect on my pieces or an illegal Ko take back */
		{
			p[i][j] = EMPTY;   /* restore to open */
			return 1;
		}
		else
		/* good move */
			return 0;
	}
	else
	/* valid move */
		return 0;
}  /* end suicide */

int KGoComputer::openregion(int i1,
               int j1,
               int i2,
               int j2)
/* check if region from i1, j1 to i2, j2 is open */
{
	int minx, maxx, miny, maxy, x, y;

	/* exchange upper and lower limits */

	if (i1 < i2)
	{
		minx = i1; maxx = i2;
	}
	else
	{
		minx = i2; maxx = i1;
	}

	if (j1 < j2)
	{
	miny = j1; maxy = j2;
	}
	else
	{	
		miny = j2; maxy = j1;
	}

	if ( (minx<0) || (miny<0) || (maxx>xSize-1) || (maxy>ySize-1) )
	{
		printf("outside board!\n") ;
		return 0 ;
	}

	/* check for empty region */
	for (x = minx; x <= maxx; x++)
	for (y = miny; y <= maxy; y++)
		if (p[x][y] != EMPTY) return 0;
	return 1;
}  /* end openregion */

