/* -*- C++ -*- */

#ifndef _OBSTACLE_H_
#define _OBSTACLE_H_

#include "Figure.h"

/** Contains the cubes of the obstacle and draws the border and a figure. */
class Obstacle : public CubeContainer
{
public:
	Obstacle (const int _sizeX,
		  const int _sizeY,
		  const int _sizeZ);
	~Obstacle();

	/** Return true if _fig is inside border. */
	bool isInsideBorder (const Figure& _fig) const;

	/** Removes possible levels and accumulate points.
	 *
	 * @return points accumulated on removing
	 */
	int removeLevelsWherePossible();

	void clear();


	/** Checks if any cube of this is on the same absolute position as any of
	 * fig and returns true in this case.
	 */
	bool areCollisions (const Figure& fig);

	/** Removes the cubes from fig and inserts them in this at same
	 * absolute position.
	 *
	 * No check for collisions is done - use checkForCollision before.
	 */
	void insertFigure (Figure& fig1) { insertFigure (fig1, cubes, true); };

	/** Draws border and figure on Draw3D::globalInstance using a z-buffer
	 * for planes.
	 *
	 * <b>Drawing of obstacle and figures are combined because the figure can't
	 * be drawed after the obstacle because by moving it can be lower
	 * than obstacle cubes. </b>
	 */
	void draw (Figure* fig);

protected:

	/** Color of the border. */
	Color borderColor;

	void addCube (const Position& pos, const Color* color, const Color** buffer);

	/** Removes all cubes on specified level and drops all higher level. */
	void removeCubesOnLevel (int level);

	/** Counts all cubes on specified level. */
	int countCubesOnLevel (int level);

	/** Draws cubes of obstacle and figure in mergedCubes. */
	void drawCubes();
	void drawBorder();

	/** Calculates the z-Buffer from the mergedCubes buffer. */
	void calculateZBuffer();

	/** Inserts the cubes of fig into buffer and if remove==true,
	 * removes them from fig.
	 */
	void insertFigure (Figure& fig, const Color ** buffer, bool remove);

	/** Contains not deletable references of the cubes of the obstacle and the
	 * figure.
	 *
	 * Has same size as cubes.
	 */
	const Color **mergedCubes;
	
	/** Center of view. */
	double viewX, viewY;

	
	/** Z-Buffer for sides of a cube and not only points.
	 *
	 * If a variable is set, the side of the cube is visible and must be
	 * displayed.
	 */

	const int Left=1, Right=2, Front=4, Back=8, Top=16;
	
	/** Stores the visibility of the planes of the cubes of the figure.
	 *
	 * planeZBuffer=[z][y][x] with 0<=x<sizeX, 0<=y<sizeY, 0<=z<sizeZ.
	 */
	unsigned char *planeZBuffer;

	/** True in case of new cubes in figure or movement. */
	bool zBufferNeedsUpdate;
};

#endif _OBSTACLE_H_
