/***************************************************************************
                                 ServerWork.cpp
                             -------------------
    begin                : 01.09.1999
    copyright            : (C) 1999 by Ullrich Wagner
    email                : Ullrich.Wagner@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include "ksqlreplace.h"
#include <qmessagebox.h>
#include "serverwork.h"
#include <qlistbox.h>
#include <ctype.h>
#include <qstrlist.h>
#include <iostream.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef OS_LINUX
#	include <resolv.h>
#endif // OS_LINUX

#define BUFFER_LEN	1048576

int	sessionSocket = 0;
char	sessionBuffer[BUFFER_LEN];


int ServerWork::connectKOraServer(QString& logonMessage)
{
	QString	hostname;
	int	port;
	
	ServerWork::getKOraSConnData(hostname, &port);

	return ServerWork::connectToKOraS(hostname, port, logonMessage);
}

int ServerWork::reconnectKOraServer(QString& logonMessage)
{
	ServerWork::releaseFromKOraS();
	close(sessionSocket);
	return ServerWork::connectKOraServer(logonMessage);
}

int ServerWork::connectToKOraS(const QString& server, int port, QString& logonMessage)
{
	int			numbytes;
	struct hostent*		he;
	struct sockaddr_in	their_addr;
	char			portString[21];
	
	// Get the IP adress
	if ((he = gethostbyname((const char*) server)) == NULL)
	{
		// We couldn't get the host name
		logonMessage = i18n("Unable to resolve server name for KOra Server");
		return false;
	}
		
	// ok, we got the address
	// create a socket
	if ((sessionSocket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		// We couldn'T create the socket
		logonMessage = i18n("Unable to create the communication socket.");
		return false;
	}

	// set the connection data
	their_addr.sin_family = AF_INET;
	their_addr.sin_port = htons(port);
	their_addr.sin_addr = *((struct in_addr*) he->h_addr);
	bzero(&(their_addr.sin_zero), 8);

	// connect to the KOra Server
	if (connect(sessionSocket, (struct sockaddr*) &their_addr, sizeof(struct sockaddr)) < 0)
	{
		// We couldn't connect to the KOra Server
		logonMessage = i18n("Unable to connect to the KOra Server at host '");
		logonMessage += server;
		logonMessage += "', Port ";
		sprintf(portString, "%d", port);
		logonMessage += portString;
		logonMessage += "\n";
		logonMessage += i18n("Maybe the server (koras) isn't running.If not, start it now and reconnect");
		logonMessage += "\n";
		return false;
	}

	// ok, we got the connection. The server will send an identification. Receive this identification
	if ((numbytes = recv(sessionSocket, sessionBuffer, BUFFER_LEN, 0)) < 0)
	{
		logonMessage = i18n("Unable to read data from KOra server");
		close(sessionSocket);
		sessionSocket = 0;
		return false;
	}

	// ok, we connected to the server and got data from the server
	// terminate this data correctly
	sessionBuffer[numbytes] = '\0';

	// deliver the text
	logonMessage = sessionBuffer;

	// everything is fine
	return true;
}

void ServerWork::releaseFromKOraS()
{
	// tell the server to quit the connection
	if (send(sessionSocket, "QUIT", 4, 0) < 0)
	{
		printf("%s\n", i18n("Unable to tell server to quit connection."));
	}

	// close the connection
	close(sessionSocket);
	sessionSocket = 0;
}

int ServerWork::executeKOraS(const QString& command, QString& result)
{
	int numbytes;
	int numbytes2;
	char	localBuffer[BUFFER_LEN];
	char	strLength[21];
	int	length;
	bool	debug_mode = false;
	char*	koraDebug;

	koraDebug = getenv("KORA_DEBUG");

	if (koraDebug != NULL)
	{
		if (!strcmp(koraDebug, "true"))
		{
			debug_mode = true;
		}
	}

	if (send(sessionSocket, (const char*) command, command.length(), 0) < 0)
	{
		result = i18n("Unable to send command ");
		result += command;
		return false;
	}

	if ((numbytes = recv(sessionSocket, sessionBuffer, BUFFER_LEN, 0)) < 0)
	{
		result = i18n("Unable to get confirmation for command ");
		result += command;
		return false;
	}

	if (debug_mode)
		printf("Received data len: %d bytes\n", numbytes);

	sessionBuffer[numbytes] = '\0';

	if (strncmp(sessionBuffer, "DATA LEN: ", 10))
	{
		QString	logonMessage;
		result = i18n("Communication mismatch. Try again");
		ServerWork::reconnectKOraServer(logonMessage);
		return false;
	}

	// ok, we have the data info first. Get the expected data len
	strncpy(strLength, &(sessionBuffer[10]), 7);
	length = atoi(strLength);

	if (debug_mode)
		printf("I WANT TO READ %d BYTES\n", length);

	strcpy(localBuffer, &(sessionBuffer[18]));

	// is the length equal to the needed length ?
	for(;;)
	{
		if (debug_mode)
			printf("UNTIL NOW I'VE READ %d bytes\n", numbytes);
		// is the length equal to the needed length ?
		if (length == numbytes)
		{
			// alles ist fein
			if (debug_mode)
				printf("EVERYTHING IS FINE. I HAVE MY DATA\n");
			break;
		}

		if (numbytes > length)
		{
			QString	logonMessage;
			// we got more data than we expected.
			result = i18n("Communication mismatch. Try again");
			ServerWork::reconnectKOraServer(logonMessage);
			return false;
		}

		// ok, we have to read another bunch of data
		if ((numbytes2 = recv(sessionSocket, sessionBuffer, BUFFER_LEN, 0)) < 0)
		{
			result = i18n("Unable to read secondary data.");
			return false;
		}

		// terminate the read string
		sessionBuffer[numbytes2] = '\0';

		// append the data to the local buffer
		strcat (localBuffer, sessionBuffer);
		// add the new length to the length before
		numbytes += numbytes2;
	}
	

	if (!strncmp(localBuffer, "ERROR\n", 6))
	{
		result = &(localBuffer[6]);
		return false;
	}

	// ok, we got an real result. Get it
	result = &(localBuffer[3]);

	if (debug_mode)
	{
		printf("Read data:\n%s\n", localBuffer);
		printf("The length of the read data is: %d\n", (int) strlen(localBuffer));
	}

	return true;
}


int ServerWork::connectToOracle(const QString& userid, const QString& password, const QString& connect, QString& oraMessage)
{
	// 0 = ok
	QString	command;
	QString result;

	command = "LOGON ";
	command += userid;
	command += " ";
	command += password;
	command += " ";
	command += connect;

	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return -1;
	}

	return 0;
}

void ServerWork::releaseFromOracle()
{
	QString	command;
	QString	result;

	command = "LOGOFF";

	if (!ServerWork::executeKOraS(command, result))
	{
		fprintf(stderr, "%s\n", (const char*) result);
		return;
	}
}

int ServerWork::executeSql(const QString& sqlCommand, QString& oraMessage, int* processedRows, QString& sqlResult, int isSingleCommand)
{
	// 0 = ok
	QString	command;
	QString	result;

	if (isSingleCommand)
	{
		command = "EXECUTE SINGLE ";
	}
	else
	{
		command = "EXECUTE MULTI ";
	}

	command += sqlCommand;
	*processedRows = 0; // not supported right now
	
	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return -1;
	}

	// we got the result
	sqlResult = result;

	return 0;
}




int ServerWork::descTable(const QString& tableName, QString& oraMessage, QString& sqlResult)
{
	// 0 = ok
	QString	command;
	QString	result;

	command = "DESC ";
	command += tableName;

	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return -1;
	}
	else
	{
		sqlResult = result;
		return 0;
	}
}


int ServerWork::fillSequenceList(QListBox* listbox, QString& oraMessage)
{
	// 0 = ok
	QString	command;
	QString	result;

	command = "LIST SEQUENCES";

	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return -1;
	}
	else
	{
		ServerWork::fillListbox(listbox, result);
		return 0;
	}
}

int ServerWork::getSequenceData(int index, QString& maxValue, QString& minValue, long* increment, QString& cycleFlag)
{
	// true = ok
	char	strIndex[21];
	QString	command;
	QString	result;

	sprintf(strIndex, "%d", index);
	command = "DETAIL SEQUENCE ";
	command += strIndex;

	if (!ServerWork::executeKOraS(command, result))
	{
		fprintf(stderr, "%s\n", (const char*) result);
		return false;
	}
	else
	{
		maxValue = ServerWork::getKOraSItem(result, 0);
		minValue = ServerWork::getKOraSItem(result, 1);
		*increment = atol((const char*) ServerWork::getKOraSItem(result, 2));
		cycleFlag = ServerWork::getKOraSItem(result, 3);
		return true;
	}
}

int ServerWork::fillSynonymList(QListBox* listbox, QString& oraMessage)
{
	// 0 = ok
	QString	command;
	QString	result;

	command = "LIST SYNONYMS";

	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return -1;
	}
	else
	{
		ServerWork::fillListbox(listbox, result);
		return 0;
	}
}

int ServerWork::getSynonymData(int index, QString& tableOwner, QString& tableName, QString& dbLink)
{
	// true = ok
	char	strIndex[21];
	QString	command;
	QString	result;

	sprintf(strIndex, "%d", index);
	command = "DETAIL SYNONYM ";
	command += strIndex;

	if (!ServerWork::executeKOraS(command, result))
	{
		fprintf(stderr, "%s\n", (const char*) result);
		return false;
	}
	else
	{
		tableOwner = ServerWork::getKOraSItem(result, 0);
		tableName = ServerWork::getKOraSItem(result, 1);
		dbLink = ServerWork::getKOraSItem(result, 2);
		return true;
	}
}

int ServerWork::fillTableList(QListBox* listbox, QString& oraMessage)
{
	// 0 = ok
	QString	command;
	QString	result;

	command = "LIST TABLES";

	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return -1;
	}
	else
	{
		ServerWork::fillListbox(listbox, result);
		return 0;
	}
}


int ServerWork::getTableData(int index, QString& oraMessage, QString& tableData)
{
	// true = ok
	char	strIndex[21];
	QString	command;
	QString	result;

	sprintf(strIndex, "%d", index);
	command = "DETAIL TABLE ";
	command += strIndex;

	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return false;
	}
	else
	{
		tableData = result;
		return true;
	}
}

int ServerWork::fillPackageList(QListBox* listbox, QString& oraMessage)
{
	// 0 = ok
	QString	command;
	QString	result;

	command = "LIST PACKAGES";

	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return -1;
	}
	else
	{
		ServerWork::fillListbox(listbox, result);
		return 0;
	}
}

int ServerWork::getPackageData(int index, QString& oraMessage, QString& packageName, QString& packageDeclData, QString& packageDeclDataRaw,
			   QString& packageBodyData, QString& packageBodyDataRaw)
{
	// true = ok
	char	strIndex[21];
	QString	command;
	QString	result;
	int	count;
	int	base;


	sprintf(strIndex, "%d", index);
	command = "DETAIL PACKAGE ";
	command += strIndex;

	if (!ServerWork::executeKOraS(command, result))
	{
		oraMessage = result;
		return false;
	}
	else
	{
		// first get the package name from the result
		base = 0;
		for(count = base; count < (int) result.length(); count++)
		{
			if (result.mid(count, 6) == "\nDECL\n")
			{
				packageName = result.mid(base,(count-base));
				base = count + 6;
				break;
			}
		}
		for(count = base; count < (int) result.length(); count++)
		{
			if (result.mid(count, 10) == "\nDECL RAW\n")
			{
				packageDeclData = result.mid(base, (count-base));
				base = count + 10;
				break;
			}
		}
		for(count = base; count < (int) result.length(); count++)
		{
			if (result.mid(count, 6) == "\nBODY\n")
			{
				packageDeclDataRaw = result.mid(base, (count-base));
				base = count + 6;
				break;
			}
		}
		for(count = base; count < (int) result.length(); count++)
		{
			if (result.mid(count, 10) == "\nBODY RAW\n")
			{
				packageBodyData = result.mid(base, (count-base));
				base = count + 10;
				break;
			}
		}
		packageBodyDataRaw = result.mid(base, result.length());
			
		return true;
	}
}


QString ServerWork::getPlSqlCompileError(int index, const QString& type)
{
	// return result string
	QString	command;
	QString	result;
	char	strIndex[21];

	sprintf(strIndex, "%d", index);
	command = "ERROR PACKAGE ";
	command += type;
	command += " ";
	command += strIndex;

	ServerWork::executeKOraS(command, result);

	return result;
}

void ServerWork::fillListbox(QListBox* listbox, const QString& data)
{
	QString	line;
	int	count;

	// init the line var
	line = "";

	// init the listbox
	listbox->clear();

	// loop through the data
	for(count = 0; count < (int) data.length(); count++)
	{
		// is the actual character a newline ?
		if (data.mid(count, 1) == "\n")
		{
			// append the data we have inside the line var to the listbox
			listbox->insertItem(line);
			// reset the line var
			line = "";
		}
		else
		{
			// ok, the found charcter is to append to the line var
			line += data.mid(count,1);
		}
		// go to the next character
	}

	// is there still some data inside the line var ?
	if (!line.isEmpty())
	{
		// append this data to the listbox
		listbox->insertItem(line);
	}
}

QString ServerWork::getKOraSItem(const QString& data, int position)
{
	int	actPosition = 0;
	QString	line;
	int	count;

	// init the line var
	line = "";

	// loop through the data
	for (count = 0; count < (int) data.length(); count++)
	{
		// is the actual charcter a newline ?
		if (data.mid(count, 1) == "\n")
		{
			// ok, is the actual position the wanted position ?
			if (actPosition == position)
			{
				// Ok, we deliver the data
				return line;
			}
			else
			{
				// ok, possibly next time.
				// increment the position
				actPosition++;
				// clear the line
				line = "";
			}
		}
		else
		{
			// save the actual charcater
			line += data.mid(count, 1);
		}

		// go to the next character
	}

	// ok we've left the loop. Our last chance to deliver the line if position is correct
	if (actPosition == position)
	{
		return line;
	}
	else
	{
		// ok, for that position we don't have data. Deliver an empty string
		line = "";
		return line;
	}
}

void ServerWork::getKOraSConnData(QString& hostname, int* port)
{
	FILE*	handle;
        char*	homeDir;
	char	setFile[256];
	char	dataLine[256];

	// get the name of the home directory
	homeDir = getenv("HOME");

	// is the home dir not found ?
	if (homeDir == NULL)
	{
		// so we must use the global settings from file /etc/koras
		handle = fopen("/etc/kora", "r");

		// is the file not opened ?
		if (handle == NULL)
		{
			// we don't have any file to open
			fprintf(stderr, "%s\n", i18n("Unable to open settings file $HOME/.kora or /etc/kora. Aborting."));
			exit(-1);
		}

		// ok, the file is opened
	}
	else
	{
		// ok, we found the homedirectory. Build the name of the settings file
		sprintf(setFile, "%s/.kora", homeDir);

		// open the local settings file
		handle = fopen(setFile, "r");

		// couldn't the file be opened ?
		if (handle == NULL)
		{
			// We wasn't able to open a local file. So try to open the global file
			handle = fopen("/etc/kora", "r");
			
			// is the file not opened ?
			if (handle == NULL)
			{
				// we don't have any file to open
				fprintf(stderr, "%s\n", i18n("Unable to open settings file $HOME/.kora or /etc/kora. Aborting."));
				exit(-1);
			}
		}
	}

	// if we are still alive we have opened a settings file. If possible the local .kora file. Otherwise the global
	// file /etc/kora

	// read the first line from file
	if (fgets(dataLine, 256, handle) == NULL)
	{
		// we wasn't able to read parameter from file
		fprintf(stderr, "%s\n", i18n("Unable to get connection info from settings file."));
		exit(-1);
	}

	// correct the terminator of line if needed
	if (dataLine[strlen(dataLine)-1] == '\n')
	{
		dataLine[strlen(dataLine)-1] = '\0';
	}

	// is the leader of the found string the info for the port ?
	if (!strncmp(dataLine, "HOST=", 5))
	{
		// ok, we got the host line.
		// set the hostname
		hostname = &(dataLine[5]);
	}
	else
	{
		fprintf(stderr, "%s\n", i18n("The first line of the settings file must hold the server name of KOraS"));
		exit(-1);
	}


	// read the first line from file
	if (fgets(dataLine, 256, handle) == NULL)
	{
		// we wasn't able to read parameter from file
		fprintf(stderr, "%s\n", i18n("Unable to get connection info from settings file."));
		exit(-1);
	}

	// correct the terminator of line if needed
	if (dataLine[strlen(dataLine)-1] == '\n')
	{
		dataLine[strlen(dataLine)-1] = '\0';
	}

	// is the leader of the found string the info for the port ?
	if (!strncmp(dataLine, "PORT=", 5))
	{
		// ok, we got the port line
		// set the port
		*port = atoi(&(dataLine[5]));
	}
	else
	{
		fprintf(stderr, "%s\n", i18n("The second line of the settings file must hold the port number to KOraS"));
		exit(-1);
	}

	// close the settings file
	fclose(handle);
}
