/* This may look like C, but it's really -*- C++ -*- */

/*
    kdb - A Simple but fast database editor for the KDE
    Copyright (C) 1998  Thorsten Niedzwetzki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "charsepdb.h"


/*
 * Read data list from a file.
 */
bool
TCharSepDB::loadFromFile(const char *fileName)
{
  QStrList *record;
  char *buf;  // input buffer
  char c;  // to check if line too long
  
  ifstream input(fileName);
  if (!input) return false;
  buf = new char[bufSize];
  
  while (input && !input.eof()) {
    
    input.get(buf, bufSize);
    if (input.get(c) && c != '\n') {
      // buffer was too small, try again!
      delete[] buf;
      clear();  // remove all fields and records
      return false;
    }

    if ((*buf != 0) && (*buf != '#') && (*buf != '\n')) {

      /* not empty, not newline, not a comment */

      char *bufp = buf;  // scan through input line buffer
      char *token = new char[bufSize];
      char *tokenp = token;  // scan through output token buffer
      record = new QStrIList();

      /*
       * Scan through buf. On every occurence of the separator char,
       * store the already read characters to a new record item.
       * This additinally recognizes if the separator character or
       * escape character is escaped.
       */
      while (*bufp) {
	c = *bufp++;
	if ( (c == escapeChar) && (*bufp) ) {

	  /* New character, escaped */
	  *tokenp++ = *bufp++;
	  
	} else {
	  if (c == separator) {

	    /* Separator */
	    *tokenp = 0;
	    record->append(token);
	    tokenp = token;

	  } else {

	    /* New character in c, not separator */
	  *tokenp++ = c;

	  }
	}
      }
      /* Finish last item */
      *tokenp = 0;
      record->append(token);

      // append record or set fields if no previous record
      if (containsHeader) {
	if (getFieldCount() == 0) {
	  setFieldNames(record);
	} else {
	  appendRecordAsText(record);
	}
      } else {
	if (getFieldCount() == 0) {
	  QStrList *header = new QStrList();
	  for (unsigned int i = 0; i < record->count(); i++) {
	    ostrstream format;
	    format << "#" << i + 1 << ends;
	    header->append(format.str());
	  }
	  setFieldNames(header);
	}
	appendRecordAsText(record);
      }
    }
  }

  delete[] buf;
  return getFieldCount() > 0;
}


/*
 * Reset separator and escapeChar characters.
 */
void
TCharSepDB::setSeparator(char newSeparator, char newEscapeChar)
{
  separator = newSeparator;
  if (newEscapeChar != 0)
    escapeChar = newEscapeChar;
}


/*
 * Put escapeChar everywhere a escapeChar or separator occurs.
 */
void
TCharSepDB::saveEscaped(ostream out, char* s)
{
  char *p = s;
  while (*p) {
    if ( (*p == escapeChar) || (*p == separator) )
      out << escapeChar;
    out << *p++;
  }
}


/*
 * Save data list to a file.
 */
bool
TCharSepDB::writeToFile(const char *fileName)
{
  ofstream output(fileName);
  if (!output) return false;

  // print header
  if (containsHeader) {
    saveEscaped(output, getFieldName(0));
    for (unsigned int i = 1; i < getFieldCount(); i++) {
      output << separator;
      saveEscaped(output, getFieldName(i));
    }
    output << endl;
  }

  // print records
  for (unsigned int rec = 0; rec < getRecordCount(); rec++) {
    saveEscaped(output, getItemAsText(rec, 0));
    for (unsigned int fld = 1; fld < getFieldCount(); fld++) {
      output << separator;
      saveEscaped(output, getItemAsText(rec, fld));
    }
    output << endl;
  }
  return output.good();
}
