/*

    Copyright (C) 1997 Frithjof Brestrich
                       brestrich@kde.org


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xmd.h>


#include "xf86vidmode.h"

/****************************************************************************/

/* Minimum extension version required */
#define MINMAJOR 0
#define MINMINOR 5

/* Mode flags -- ignore flags not in V_FLAG_MASK */
//#define V_FLAG_MASK     0x1FF;
//#define V_PHSYNC        0x001
//#define V_NHSYNC        0x002
//#define V_PVSYNC        0x004
//#define V_NVSYNC        0x008
//#define V_INTERLACE     0x010
//#define V_DBLSCAN       0x020
//#define V_CSYNC         0x040
//#define V_PCSYNC        0x080
//#define V_NCSYNC        0x100
 
/****************************************************************************/


XF86VidModeLine::XF86VidModeLine() {
  ClearData();
}

XF86VidModeLine::XF86VidModeLine(XF86VidModeLine &templ) {
  ClearData();
  Set(*templ.Get(),templ.GetDotClock());
};

XF86VidModeLine::~XF86VidModeLine()
{
}

void XF86VidModeLine::ClearData()
{
  memset(&ModeLine,0,sizeof(ModeLine)); 
  DotClock = 0;
  Name     = strdup("unknown");
}

void XF86VidModeLine::Set(XF86VidModeModeLine& ml, int dotclk, const char* name)
{
  if (ModeLine.privsize && ModeLine.c_private) 
    free(ModeLine.c_private);
  
  memcpy(&ModeLine,&ml,sizeof(ModeLine));
  DotClock = dotclk;

  if (ModeLine.privsize && ModeLine.c_private) {
    ModeLine.c_private = (INT32*) malloc (ModeLine.privsize);
    // ModeLine.privsize != 0 && ModeLine.c_private == NULL inicated error
    if (ModeLine.c_private) 
      memcpy(ModeLine.c_private,ml.c_private,ModeLine.privsize);
  }

  SetName(name);
}

void XF86VidModeLine::Set(XF86VidModeModeInfo& mi, const char* name)
{
  // very little hack to save 4 bytes of code
  Set(*((XF86VidModeModeLine*)&(mi.hdisplay)),mi.dotclock,name);
}

XF86VidModeModeLine* XF86VidModeLine::Get()
{
  return &ModeLine;
}

 bool XF86VidModeLine::SameValues(const XF86VidModeLine &ml)
 {
     return ( DotClock            == ml.DotClock            )
         && ( ModeLine.hdisplay   == ml.ModeLine.hdisplay   )
         && ( ModeLine.hsyncstart == ml.ModeLine.hsyncstart )
         && ( ModeLine.hsyncend   == ml.ModeLine.hsyncend   )
         && ( ModeLine.htotal     == ml.ModeLine.htotal     )
         && ( ModeLine.vdisplay   == ml.ModeLine.vdisplay   )
         && ( ModeLine.vsyncstart == ml.ModeLine.vsyncstart )
         && ( ModeLine.vsyncend   == ml.ModeLine.vsyncend   )
         && ( ModeLine.vtotal     == ml.ModeLine.vtotal     )
         && ( ModeLine.flags      == ml.ModeLine.flags      );
 }

double XF86VidModeLine::GetVerticalSyncClock()
{ 
  double f;
  
  f = GetHorizontalSyncClock()/ModeLine.vtotal;
  if (ModeLine.flags & VM_FLAG_INTERLACE) f*=2;
  if (ModeLine.flags & VM_FLAG_DBLSCAN) f/=2;
  
  return f;
};
			  

bool XF86VidModeLine::Move(int dx, int dy)
{
  bool b;

  b = TRUE;

  if ( (ModeLine.hdisplay <= ModeLine.hsyncstart - dx) && (ModeLine.hsyncend - dx < ModeLine.htotal) ) {
    ModeLine.hsyncstart -= dx;
    ModeLine.hsyncend   -= dx;
  } else {
    b = b && (dx == 0);
  }

  if ( (ModeLine.vdisplay <= ModeLine.vsyncstart - dy) && (ModeLine.vsyncend - dy < ModeLine.vtotal) ) {
    ModeLine.vsyncstart -= dy;
    ModeLine.vsyncend   -= dy;
  } else {
    b = b && (dy == 0);
  }

  return b;
}

bool XF86VidModeLine::Resize(int dx, int dy)
{
  bool b;

  b = TRUE;

  if ( ModeLine.htotal - dx > ModeLine.hsyncend ) {
    ModeLine.htotal -= dx;
  } else {
    b = b && (dx == 0);
  }

  if ( ModeLine.vtotal - dy > ModeLine.vsyncend ) {
    ModeLine.vtotal -= dy;
  } else {
    b = b && (dy == 0);
  }

  return b;
}

char*  XF86VidModeLine::GetModeline()
{
  char buf[256];

//  snprintf(buf,sizeof(buf)-1,"\"%dx%d\""
//	   ModeLine.hdisplay,ModeLine.vdisplay,
  snprintf(buf,sizeof(buf)-1,"\"%s\""
           "  %6.2f  %4d %4d %4d %4d  %4d %4d %4d %4d"
	   "%s%s%s%s%s%s%s%s%s",
	   Name,
	   (float)DotClock/1000.0,
	   ModeLine.hdisplay,
	   ModeLine.hsyncstart,
	   ModeLine.hsyncend,
	   ModeLine.htotal,
	   ModeLine.vdisplay,
	   ModeLine.vsyncstart,
	   ModeLine.vsyncend,
	   ModeLine.vtotal,
	   ModeLine.flags & VM_FLAG_PHSYNC    ?" +hsync":"",
	   ModeLine.flags & VM_FLAG_NHSYNC    ?" -hsync":"",
	   ModeLine.flags & VM_FLAG_PVSYNC    ?" +vsync":"",
	   ModeLine.flags & VM_FLAG_NVSYNC    ?" -vsync":"",
	   ModeLine.flags & VM_FLAG_INTERLACE ?" interlace":"",
	   ModeLine.flags & VM_FLAG_CSYNC     ?" composite":"",
	   ModeLine.flags & VM_FLAG_PCSYNC    ?" +csync":"",
	   ModeLine.flags & VM_FLAG_NCSYNC    ?" -csync":"",
	   ModeLine.flags & VM_FLAG_DBLSCAN   ?" doublescan":"");

  buf[sizeof(buf)-1] = 0;
  return strdup(buf);
}

void XF86VidModeLine::SetName(const char* name)
{
  if ( Name != NULL ) free(Name);
  if ( name != NULL ) {
      Name = strdup(name);
  } else {
      char buf[100];
      snprintf(buf,sizeof(buf),"%ux%u", ModeLine.hdisplay, ModeLine.vdisplay);
      buf[sizeof(buf)-1] = 0; // is this and snprintf paranoid ?
      Name = strdup(buf);
  }
}

bool XF86VidModeLine::IsInterlace()  { return (ModeLine.flags & VM_FLAG_INTERLACE) != 0; };
bool XF86VidModeLine::IsDoublescan() { return (ModeLine.flags & VM_FLAG_DBLSCAN)   != 0; };
bool XF86VidModeLine::IsComposite()  { return (ModeLine.flags & VM_FLAG_CSYNC)     != 0; };
bool XF86VidModeLine::IsPHSync()     { return (ModeLine.flags & VM_FLAG_PHSYNC)    != 0; };
bool XF86VidModeLine::IsNHSync()     { return (ModeLine.flags & VM_FLAG_NHSYNC)    != 0; };
bool XF86VidModeLine::IsPVSync()     { return (ModeLine.flags & VM_FLAG_PVSYNC)    != 0; };
bool XF86VidModeLine::IsNVSync()     { return (ModeLine.flags & VM_FLAG_NVSYNC)    != 0; };
bool XF86VidModeLine::IsPCSync()     { return (ModeLine.flags & VM_FLAG_PCSYNC)    != 0; };
bool XF86VidModeLine::IsNCSync()     { return (ModeLine.flags & VM_FLAG_NCSYNC)    != 0; };
				     
bool XF86VidModeLine::Interlace(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_INTERLACE;
  else ModeLine.flags &= ~VM_FLAG_INTERLACE;
  return TRUE;
}

bool XF86VidModeLine::Doublescan(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_DBLSCAN;
  else ModeLine.flags &= ~VM_FLAG_DBLSCAN;
  return TRUE;
}

bool XF86VidModeLine::Composite(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_CSYNC;
  else ModeLine.flags &= ~VM_FLAG_CSYNC;
  return TRUE;
}

bool XF86VidModeLine::PHSync(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_PHSYNC;
  else ModeLine.flags &= ~VM_FLAG_PHSYNC;
  return TRUE;
}

bool XF86VidModeLine::NHSync(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_NHSYNC;
  else ModeLine.flags &= ~VM_FLAG_NHSYNC;
  return TRUE;
}

bool XF86VidModeLine::PVSync(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_PVSYNC;
  else ModeLine.flags &= ~VM_FLAG_PVSYNC;
  return TRUE;
}

bool XF86VidModeLine::NVSync(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_NVSYNC;
  else ModeLine.flags &= ~VM_FLAG_NVSYNC;
  return TRUE;
}

bool XF86VidModeLine::PCSync(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_PCSYNC;
  else ModeLine.flags &= ~VM_FLAG_PCSYNC;
  return TRUE;
}

bool XF86VidModeLine::NCSync(bool on)
{
  if (on) ModeLine.flags |= VM_FLAG_NCSYNC;
  else ModeLine.flags &= ~VM_FLAG_NCSYNC;
  return TRUE;
}


/****************************************************************************/

int XF86VidMode::X11VMErrorBase = -1;
int (*XF86VidMode::XErrorFunc)(Display *, XErrorEvent *) = NULL;
int XF86VidMode::XErrorOccured = 0;
int XF86VidMode::XErrorCode = 0;

XF86VidMode::XF86VidMode()
{
  ErrorText    = NULL;

  pDisplay     = NULL;
  Screen       = 0;

  MajorVersion = 0;
  MinorVersion = 0;
  EventBase    = 0;
  ErrorBase    = 0;

  memset(&Monitor,0,sizeof(Monitor)); 
  ModeCount    = 0;
  //ModesInfo    = NULL;
  ModeLines    = NULL;
  AktMode      = 0;
}

XF86VidMode::~XF86VidMode()
{
  Disconnect();
  if (ErrorText) 
    free(ErrorText);
}

void XF86VidMode::SetErrorText(char *txt)
{
  if (ErrorText) 
    free(ErrorText);

  if (XErrorOccured) {
    char buf[256];
    XGetErrorText(pDisplay,XErrorCode,buf,sizeof(buf));
    XErrorOccured = 0;
    ErrorText = strdup(buf);
  } else {
    // I remember some libs doesnt work well with NULLs
    ErrorText = txt ? strdup(txt) : NULL;
  }

}

char *XF86VidMode::LastErrorText()
{
  char *txt;
  
  txt = ErrorText;
  ErrorText = NULL;
  
  return txt;
}

int XF86VidMode::X11ErrorHandler( Display *dis, XErrorEvent *err)
{
  if ((err->error_code >= X11VMErrorBase &&
       err->error_code < X11VMErrorBase + XF86VidModeNumberErrors) ||
       err->error_code == BadValue)
  {
      XErrorOccured++;
      XErrorCode = err->error_code;
  } else {
      if (XErrorFunc) return (*XErrorFunc)(dis,err);
  }
  return 0;
}

char* XF86VidMode::X11ProtocolVersion()
{
   char buf[1024];

   snprintf(buf,sizeof(buf),"%u.%u",ProtocolVersion(pDisplay),ProtocolRevision(pDisplay));

   return strdup(buf);
}

char* XF86VidMode::X11Vendor()
{
   return strdup(ServerVendor(pDisplay));
}

char* XF86VidMode::X11ReleaseVersion()
{
   char buf[1024];

   snprintf(buf,sizeof(buf),"%u",VendorRelease(pDisplay));

   return strdup(buf);
}

char* XF86VidMode::XF86VidModeVersion()
{
   char buf[1024];

   snprintf(buf,sizeof(buf),"%u.%u",MajorVersion,MinorVersion);

   return strdup(buf);
}
	     
bool XF86VidMode::ConnectDisplay(Display* display, int screen)
{
  if (!XF86VidModeQueryVersion(display, &MajorVersion, &MinorVersion)) {
    SetErrorText("Unable to query video extension version");
    return FALSE;
  }
  
  if (MajorVersion < MINMAJOR
  || (MajorVersion == MINMAJOR && MinorVersion < MINMINOR) ) {
    char buf[256];
    snprintf(buf,sizeof(buf),
             "Xserver is running an old XFree86-VidModeExtension version (%d.%d)\n"
	     "Minimum required version is %d.%d",
	     MajorVersion, MinorVersion, MINMAJOR, MINMINOR);
    buf[sizeof(buf)-1]=0;
    SetErrorText(buf);
    return FALSE;
  } 
  
  if (!XF86VidModeQueryExtension(display, &EventBase, &ErrorBase)) {
    SetErrorText("Unable to query video extension information");
    return FALSE;
  }
  X11VMErrorBase = ErrorBase;

  pDisplay = display;
  Screen = screen;

  XErrorFunc = XSetErrorHandler(X11ErrorHandler);
  
  LockModeSwitch(TRUE);

  return TRUE;
}

bool XF86VidMode::Disconnect()
{
  bool status = TRUE;

  /* Make sure mode switching is not locked out at exit */
  if (pDisplay) { 
    status=LockModeSwitch(FALSE);
    XFlush(pDisplay); 
  }

  if (Monitor.vendor) free(Monitor.vendor); 
  if (Monitor.vendor) free(Monitor.model); 
  if (Monitor.vendor) free(Monitor.hsync); 
  if (Monitor.vendor) free(Monitor.vsync); 
  memset(&Monitor,0,sizeof(Monitor)); 

/*
  if (ModesInfo) {
    for (int i=0;i<ModeCount; i++)
      if (ModesInfo[i]->privsize && ModesInfo[i]->c_private)
        free(ModesInfo[i]->c_private); 
    free(ModesInfo); 
  }
  ModeCount = 0;
  ModesInfo = NULL;
*/
  ModeCount = 0;
  ModeLines = NULL;
  AktMode = 0;
  
  pDisplay = NULL;
  Screen = 0;

  MajorVersion = 0;
  MinorVersion = 0;
  EventBase = 0;
  ErrorBase = 0;

  return status;
}

bool XF86VidMode::RequestData() 
{
  return GetMonitorData() && GetVideoModesData();
}

char* XF86VidMode::MonitorBandwidth()
{ 
/* // OBSOLETE //
  char buf[32]; 
  
  snprintf(buf,sizeof(buf),"%0.2f",Monitor.bandwidth);
  buf[sizeof(buf)-1]=0;

  return strdup(buf);
*/
  return strdup("--");
}

char* XF86VidMode::MonitorSyncRatesH()
{
   int i;
   char buf[1024], val[32];

   if (Monitor.nhsync == 0 ) return strdup("--");

   snprintf(val,sizeof(val),"%0.2f - %0.2f",Monitor.hsync[0].lo, Monitor.hsync[0].hi);
   strncpy(buf,val,sizeof(buf)); buf[sizeof(buf)-1]=0; // sure Iam paranoid
    
   for (i = 1; i < Monitor.nhsync; i++) {
     snprintf(val,sizeof(val),"%0.2f - %0.2f",Monitor.hsync[i].lo, Monitor.hsync[i].hi);
     if ((strlen(buf)+3)<sizeof(buf)) {
       strcat(buf,", ");
       strcat(buf,val);
     }
   }

   return strdup(buf);
}

char* XF86VidMode::MonitorSyncRatesV()
{
   int i;
   char buf[1024], val[32];

   if (Monitor.nvsync == 0 ) return strdup("--");

   snprintf(val,sizeof(val),"%0.2f - %0.2f",Monitor.vsync[0].lo, Monitor.vsync[0].hi);
   strncpy(buf,val,sizeof(buf)); buf[sizeof(buf)-1]=0; // sure Iam paranoid
    
   for (i = 1; i < Monitor.nvsync; i++) {
     snprintf(val,sizeof(val),"%0.2f - %0.2f",Monitor.vsync[i].lo, Monitor.vsync[i].hi);
     if ((strlen(buf)+3)<sizeof(buf)) {
       strcat(buf,", ");
       strcat(buf,val);
     }
   }

   return strdup(buf);
}

bool XF86VidMode::LockModeSwitch(bool Locked)
{
  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::LockModeSwitch()"); 
    return FALSE;
  }

  if (!XF86VidModeLockModeSwitch(pDisplay,Screen,Locked)||XErrorOccured) {    
    SetErrorText(Locked?"Failed to disable mode-switch hot-keys"
                       :"Failed to enable mode-switch hot-keys");
    return FALSE;
  }
  return TRUE;
}

bool XF86VidMode::GetMonitorData()
{
  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::GetMonitorData()"); 
    return FALSE;
  }

  /* free all data if GetMonitorData called a second time */
  if (Monitor.vendor) free(Monitor.vendor); 
  if (Monitor.vendor) free(Monitor.model); 
  if (Monitor.vendor) free(Monitor.hsync); 
  if (Monitor.vendor) free(Monitor.vsync); 
  memset(&Monitor,0,sizeof(Monitor)); 
  
  if (!XF86VidModeGetMonitor (pDisplay, Screen, &Monitor)||XErrorOccured) {
    SetErrorText("Unable to query monitor information");
    return FALSE;
  }
  return TRUE;
}

int XF86VidMode::GetColorDepth()
{
  unsigned cold;

  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::GetColorDepth()"); 
    return 0;
  }

  if (!(cold=DefaultDepth(pDisplay, Screen))||XErrorOccured) {
    SetErrorText("Unable to get color depth information"); 
    return 0;
  }

  return cold;
}

bool XF86VidMode::GetVideoModesData()
{
    if (ModeLines) 
        delete [] ModeLines;
  
    ModeLines = NULL;
    ModeCount = 0;
    AktMode   = 0;
  
    if (!GetAllModeLines(ModeLines,ModeCount)) {
        ModeLines = NULL;
        ModeCount = 0;
        return FALSE;
    }
  
    // 0 is current 
    XF86VidModeLine current = ModeLines[0];
  
    SortModeLineList(ModeLines,ModeCount);
  
    // yes and a ugl search alg.
    for ( unsigned n = 0 ; n < ModeCount ; n++ ) {
        if ( ModeLines[n].SameValues(current) ) {
            AktMode = n;
            break;
         }
    }
  
    return TRUE;
}

bool XF86VidMode::GetModeLineList(XF86VidModeLine *&mlsi, unsigned &count)
{
  if (!GetAllModeLines(mlsi,count)) {
    mlsi  = NULL;
    count = 0;
    return FALSE;
  }

  SortModeLineList(mlsi,count);

  return TRUE;
}

void XF86VidMode::SortModeLineList(XF86VidModeLine *&mls, unsigned count)
{
  // You found it: a ugly sort alg. ( qsort() and c++ !? )
  bool            b;
  unsigned        n;
  XF86VidModeLine ml;
  unsigned        res1, res2;
  double          ref1, ref2;

  if ( mls == NULL ) return;
  
  do {
    b = FALSE;
    for ( n = 0 ; n < count - 1 ; n++ ) {
      res1 = mls[n].GetHorizontalResolution() * mls[n].GetVerticalResolution();
      res2 = mls[n+1].GetHorizontalResolution() * mls[n+1].GetVerticalResolution();
      ref1 = mls[n].GetVerticalSyncClock();
      ref2 = mls[n+1].GetVerticalSyncClock();
      if ( (res1 > res2) || ( (res1 == res2) && (ref1 > ref2) ) ) {
        ml = mls[n]; 
	mls[n] = mls[n+1]; 
	mls[n+1] = ml;
	b = TRUE;
      }
    }
  } while ( b );
};

#if 0
char* XF86VidMode::GetModeName(unsigned mode)
{
  char buf[100];
  
  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::GetModeName()");
    return NULL;
  }

  mode = (mode == 0) ? AktMode : mode - 1;
  if ( mode >= ModeCount /*|| mode < 0*/ ) {
    SetErrorText("Requested mode is out of range");
    return NULL;
  }

  snprintf(buf,sizeof(buf),"%u x %u", ModeLines[mode].GetHorizontalResolution(), 
                                      ModeLines[mode].GetVerticalResolution());
  buf[sizeof(buf)-1] = 0; // is this and snprintf paranoid ?
  
  return strdup(buf);
}
#endif

bool XF86VidMode::SetMode(unsigned mode)
{
  bool            retval;
  XF86VidModeLine *ml; 
  unsigned        cnt;
  unsigned        n;

  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::SetMode()"); 
    return FALSE;
  }

  mode = (mode == 0) ? AktMode : mode - 1;
  if ( mode >= ModeCount /*|| mode < 0*/) {
    SetErrorText("Requested mode is out of range");
    return FALSE;
  } 

  if (!GetAllModeLines(ml,cnt)) return FALSE;
  
  // find modeline and count steps (n) to that modeline
  for ( n = 0 ; n < cnt ; n++ ) {
      if ( ModeLines[mode].SameValues(ml[n]) )
      break;
  }

  if ( n == cnt ) {
      SetErrorText("Requested mode does not exist any more");
      return FALSE;
  }

  if ( n == 0 ) { // if we alredy there ( other program has set video mode )
    AktMode = mode;
     LockModeSwitch(TRUE);
    return TRUE;
  }

  retval = TRUE;
  if (!LockModeSwitch(FALSE)) {
    // ErrorText set by LockModeSwitch
    return FALSE;
  }

  AktMode = mode;

  // find shortest way
  if ( n < ModeCount - n ) {
    while ( n ) {
      if (!XF86VidModeSwitchMode(pDisplay, Screen, +1)||XErrorOccured) {
        SetErrorText("Unable to switch to new video mode slot");
        retval = FALSE;
        break;
      }
      n--;
    }
  } else {
    n = ModeCount - n;
    while ( n ) {
      if (!XF86VidModeSwitchMode(pDisplay, Screen, -1)||XErrorOccured) {
        SetErrorText("Unable to switch to new video mode slot");
        retval = FALSE;
        break;
      }
      n--;
    }
  }

  LockModeSwitch(TRUE); // exit code doesn't matter
  
  return retval;
}

bool XF86VidMode::GetModeLine(XF86VidModeLine &ml)
{
  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::SetMode()"); 
    return FALSE;
  }

  int                 dotclk;
  XF86VidModeModeLine vmml;

  memset(&vmml,0,sizeof(vmml)); 

  if (!XF86VidModeGetModeLine(pDisplay, Screen, &dotclk, &vmml)||XErrorOccured) {
    SetErrorText("Unable to get video mode data"); 
    return FALSE;
  }

  ml.Set(vmml,dotclk);

  if (vmml.privsize && vmml.c_private)
    free(vmml.c_private); 

  return TRUE;
}

bool XF86VidMode::SetModeLine(XF86VidModeLine &ml)
{
  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::SetMode()"); 
    return FALSE;
  }

  bool b;

  b = XF86VidModeModModeLine(pDisplay, Screen,  ml.Get());
  XSync(pDisplay , False);  // wait until all work is done - for error handling routine

  if ( !b || XErrorOccured ) {
    SetErrorText("Unable to set video mode data"); 
    return FALSE;
  }

  ModeLines[AktMode] = ml;
  
  return TRUE;
}

bool XF86VidMode::GetAllModeLines(XF86VidModeLine *&mls, unsigned &count)
{
  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::GetAllModeLines()"); 
    return FALSE;
  }

  XF86VidModeModeInfo **minfo = NULL;
  
  if (!XF86VidModeGetAllModeLines(pDisplay, Screen, (int*)&count, &minfo)||XErrorOccured) {
    SetErrorText("Unable to query all video modelines"); 
    return FALSE;
  }

  if ( minfo == NULL || count == 0 ) {
    SetErrorText("Mode list is empty"); 
    return FALSE;
  }

  mls = new XF86VidModeLine[count];

  if ( mls != NULL ) 
    for (unsigned i=0;i<count; i++) 
      mls[i].Set(*(minfo[i]));

  for (unsigned i=0;i<count; i++) {
    if (minfo[i]->privsize && minfo[i]->c_private)
      free(minfo[i]->c_private); 
  }
  free(minfo); 

  if ( mls == NULL ) {
    count = 0;
    SetErrorText("out of memory"); 
    return FALSE;
  }

  return TRUE;
}
/*
bool XF86VidMode::GetAllModeLineList(char **&list, unsigned &count)
{
  if (!pDisplay) {
    SetErrorText("no context available for XF86VidMode::GetAllModeLineList()"); 
    return FALSE;
  }

  char     *s;
  unsigned size;
  unsigned i;

  size = (ModeCount+1) * 20 ;

  if (! (list = (char**) malloc (size) )) {
      SetErrorText("out of memory in XF86VidMode::GetAllModeLineList()"); 
      return FALSE;
  }
  memset ( list , 0, size );

  for ( i = 0 ; i < ModeCount ; i++ ) 
      list[i] = (char*) (((unsigned)(void*)list) + sizeof (void*) * (ModeCount+1) + 15 * i) ;
  list[ModeCount] = NULL;
  
  for ( i = 1 ; i <= ModeCount ; i++ ) {
      s = GetModeName(i);
      strncpy ( list[i-1], s, 15 ) ; 
      list[i-1][14] = 0;
      free ( s );
  }

  count = ModeCount;
	 
  return TRUE;
}


void XF86VidMode::DumpSetting()
{
  printf("[dotclock  ] %u\n",ModesInfo[AktMode]->dotclock);
  printf("[hdisplay  ] %u\n",ModesInfo[AktMode]->hdisplay);
  printf("[hsyncstart] %u\n",ModesInfo[AktMode]->hsyncstart);
  printf("[hsyncend  ] %u\n",ModesInfo[AktMode]->hsyncend);
  printf("[htotal    ] %u\n",ModesInfo[AktMode]->htotal);
  printf("[vdisplay  ] %u\n",ModesInfo[AktMode]->vdisplay);
  printf("[vsyncstart] %u\n",ModesInfo[AktMode]->vsyncstart);
  printf("[vsyncend  ] %u\n",ModesInfo[AktMode]->vsyncend);
  printf("[vtotal    ] %u\n",ModesInfo[AktMode]->vtotal);
  printf("[flags     ] %u\n",ModesInfo[AktMode]->flags);
  printf("[privsize  ] %u\n",ModesInfo[AktMode]->privsize);
  puts(ModeLines[AktMode].GetModeline());
}
*/
