/* -*- C++ -*-
 * This source file is part of the KInetEd program.
 *
 * Copyright (C) 1998 by Jrgen Sigvardsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA
 *
 */                          
#include <qfiledlg.h>
#include "KInetEd.h"
#include "KRequestDlg.h"
#include <kapp.h> // For i18n
#include <fstream.h>
#include <iostream.h>
#include <kmenubar.h>
#include <ktablistbox.h>
#include <inetd.h>
#include <kpopmenu.h>
#include <qcursor.h>
#include "KSheetDlg.h"
#include "KReqSheet.h"
#include "KMenuSheet.h"
#include "KServerSheet.h"
#include <ktoolbar.h>
#include "version.h"

#include "xpm/filenew.xpm"
#include "xpm/filenews.xpm"
#include "xpm/fileopen.xpm"
#include "xpm/filefloppy.xpm"
#include "xpm/filedelete.xpm"

#include "KInetEd.moc"

/* Menu ID's */
static const int 
IDSOCK_STREAM = 1, IDSOCK_DGRAM = 2;
static const int
IDPROTO_UDP = 1, IDPROTO_TCP = 2, IDPROTO_RPCUDP = 3, IDPROTO_RPCTCP = 4;
static const int
IDFLAG_WAIT = 1, IDFLAG_NOWAIT = 2;
static const int
IDUSER_ROOT = 1, IDUSER_NOBODY = 2, IDUSER_EDIT = 3;
static const int
IDPATH_TCPW = 1, IDPATH_INTERNAL = 2, IDPATH_EDIT = 4;
static const int
IDARG_BLANK = 1, IDARG_EDIT = 2;
static const int
IDST_OFF = 0, IDST_ON = 1;
static const int 
TB_NEWSERV = 1, TB_NEWCONF = 2, TB_OPEN = 3, TB_SAVE = 4, TB_DELSERV = 5;

/* Column numbers */
static const int 
COL_ONOFF    = 0,
COL_NAME     = 1,
COL_SOCKTYPE = 2,
COL_PROTO    = 3,
COL_FLAGS    = 4,
COL_USER     = 5,
COL_PATH     = 6,
COL_ARGS     = 7;

static const int NUM_COLS = 8;

KInetEd::KInetEd() : m_pMenuBar(0), m_pListBox(0) {
  m_pMenuBar = new KMenuBar(this, i18n("Menu"));
  m_pListBox = new KTabListBox(this, i18n("LisBox"), NUM_COLS);
  m_pListBox->setSeparator('\t');

  // Create popup menus
  m_pNamePopup = new QPopupMenu;
  m_pNamePopup->insertItem(i18n("Edit..."));
  connect(m_pNamePopup, SIGNAL(activated(int)), SLOT(onNamePopup(int)));
  
  m_pSockPopup = new QPopupMenu;
  m_pSockPopup->insertItem(i18n("stream"), IDSOCK_STREAM);
  m_pSockPopup->insertItem(i18n("dgram"), IDSOCK_DGRAM);
  connect(m_pSockPopup, SIGNAL(activated(int)), SLOT(onSockPopup(int)));
  
  m_pProtoPopup = new QPopupMenu;
  m_pProtoPopup->insertItem(i18n("udp"), IDPROTO_UDP);
  m_pProtoPopup->insertItem(i18n("tcp"), IDPROTO_TCP);
  m_pProtoPopup->insertItem(i18n("rpc/udp"), IDPROTO_RPCUDP);
  m_pProtoPopup->insertItem(i18n("rpc/tcp"), IDPROTO_RPCTCP);
  connect(m_pProtoPopup, SIGNAL(activated(int)), SLOT(onProtoPopup(int)));
  
  m_pFlagPopup = new QPopupMenu;
  m_pFlagPopup->insertItem(i18n("wait"), IDFLAG_WAIT);
  m_pFlagPopup->insertItem(i18n("nowait"), IDFLAG_NOWAIT);
  connect(m_pFlagPopup, SIGNAL(activated(int)), SLOT(onFlagPopup(int)));

  m_pUserPopup = new QPopupMenu;
  m_pUserPopup->insertItem(i18n("root"), IDUSER_ROOT);
  m_pUserPopup->insertItem(i18n("nobody"), IDUSER_NOBODY);
  m_pUserPopup->insertItem(i18n("other..."), IDUSER_EDIT);
  connect(m_pUserPopup, SIGNAL(activated(int)), SLOT(onUserPopup(int)));

  m_pPathPopup = new QPopupMenu;
  m_pPathPopup->insertItem(i18n("/usr/sbin/tcpd"), IDPATH_TCPW);
  m_pPathPopup->insertItem(i18n("internal"), IDPATH_INTERNAL);
  m_pPathPopup->insertItem(i18n("edit..."), IDPATH_EDIT);
  connect(m_pPathPopup, SIGNAL(activated(int)), SLOT(onPathPopup(int)));

  m_pArgPopup = new QPopupMenu;
  m_pArgPopup->insertItem(i18n("blank"), IDARG_BLANK);
  m_pArgPopup->insertItem(i18n("edit..."), IDARG_EDIT);
  connect(m_pArgPopup, SIGNAL(activated(int)), SLOT(onArgPopup(int)));

  m_pStatusPopup = new QPopupMenu;
  m_pStatusPopup->insertItem(i18n("on"), IDST_ON);
  m_pStatusPopup->insertItem(i18n("off"), IDST_OFF);
  connect(m_pStatusPopup, SIGNAL(activated(int)), SLOT(onStatusPopup(int)));

  // Create menus
  QPopupMenu* filemenu = new QPopupMenu;
  filemenu->insertItem(i18n("&New"), this, SLOT(onNewConfig()));
  filemenu->insertItem(i18n("N&ew service"), this, SLOT(onNewService()));
  filemenu->insertItem(i18n("&Open"), this, SLOT(openConfigFile()));
  filemenu->insertItem(i18n("Save &As"), this, SLOT(onSaveConfigFileAs()));
  filemenu->insertSeparator();
  filemenu->insertItem(i18n("&Quit"), this, SLOT(quit()));

  m_pMenuBar->insertItem(i18n("&File"), filemenu);

  QPopupMenu* helpmenu = KApplication::getKApplication()->getHelpMenu
    (TRUE, QString(i18n("KInetEd")) + " " + VERSION + 
     i18n("\n\nby Jorgen Sigvardsson\t<jorgen@linux.nu>"));

  m_pMenuBar->insertItem("Help", helpmenu);
  m_pMenuBar->show();
  setMenu(m_pMenuBar);

  // Create listbox tabs
  int dx = width() / 8;
  m_pListBox->setColumn(COL_ONOFF,    i18n("Status"), dx);
  m_pListBox->setColumn(COL_NAME,     i18n("Name"), dx);
  m_pListBox->setColumn(COL_SOCKTYPE, i18n("Socket"), dx);
  m_pListBox->setColumn(COL_PROTO,    i18n("Protocol"), dx);
  m_pListBox->setColumn(COL_FLAGS,    i18n("Flags"), dx);
  m_pListBox->setColumn(COL_USER,     i18n("User"), dx);
  m_pListBox->setColumn(COL_PATH,     i18n("Path"), dx);
  m_pListBox->setColumn(COL_ARGS,     i18n("Arguments"), dx);
  
  m_pListBox->setGeometry(0, m_pMenuBar->height(), 
			  width(), height() - m_pMenuBar->height());

  connect(m_pListBox, SIGNAL(popupMenu(int,int)), SLOT(itemSelected(int,int)));
  setView(m_pListBox);
  m_pListBox->show();

  // Toolbar
  KToolBar* m_pToolBar = new KToolBar(this);
  QPixmap pm((const char**)filenew);
  QPixmap pms((const char**)filenews);
  QPixmap pmo((const char**)fileopen);
  QPixmap pms2((const char**)filefloppy);
  QPixmap del((const char**)filedelete);
  m_pToolBar->insertButton(pm, TB_NEWCONF, SIGNAL(pressed()), this,
			   SLOT(onNewConfig()), TRUE, 
			   i18n("Create new configuration"));
  m_pToolBar->insertButton(pms, TB_NEWSERV, SIGNAL(pressed()), this,
			   SLOT(onNewService()), TRUE, 
			   i18n("Create new service"));
  m_pToolBar->insertButton(del, TB_DELSERV, SIGNAL(pressed()), this,
			   SLOT(onDeleteService()), TRUE,
			   i18n("Delete service"));
  m_pToolBar->insertButton(pmo, TB_OPEN, SIGNAL(pressed()), this,
			   SLOT(openConfigFile()), TRUE, 
			   i18n("Open /etc/inetd.conf"));
  m_pToolBar->insertButton(pms2, TB_SAVE, SIGNAL(pressed()), this,
			   SLOT(onSaveConfigFileAs()), TRUE, 
			   i18n("Save to file"));
  m_pToolBar->setBarPos(KToolBar::Top);
  enableToolBar(KToolBar::Toggle, addToolBar(m_pToolBar));
  m_pToolBar->show();
  m_pToolBar->updateRects(TRUE);
}

void KInetEd::openConfigFile() {
  char buf[2048], argbuf[1024];
  inetd_h handle;
  inetd_Service s;

  // Reset the toolbarbutton
  toolBar()->setButton(TB_OPEN, FALSE);

  // Clean up "document"
  m_pListBox->clear();
  
  handle = inetd_ReadConf();
  
  m_pListBox->setAutoUpdate(FALSE);
  for(inetd_BeginIter(handle);
      inetd_CurrentService(handle) != 0;
      inetd_Iterate(handle)) {
    s = inetd_CurrentService(handle);
    strcpy(argbuf, "");
    for(int i = 0; i < serv_GetNumArgs(s); i++) {
      strcat(argbuf, serv_GetArg(s, i));
      if(i < serv_GetNumArgs(s) - 1)
	strcat(argbuf, " ");
    }
    sprintf(buf, "%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s",
	    inetd_OnOff(serv_IsOff(s)),
	    serv_GetName(s),
	    inetd_SockTypeName(serv_GetSockType(s)),
	    inetd_ProtoName(serv_GetProto(s)),
	    inetd_FlagName(serv_GetFlags(s)),
	    serv_GetUser(s),
	    serv_GetPath(s),
	    argbuf);
    m_pListBox->appendItem(buf);
  }
  inetd_Free(handle);
  m_pListBox->setAutoUpdate(TRUE);
  m_pListBox->repaint();
}

void KInetEd::quit(void) {
  KApplication::getKApplication()->quit();
}

void KInetEd::onSaveConfigFileAs() {
  toolBar()->setButton(TB_SAVE, FALSE);

  QString str(QFileDialog::getSaveFileName());
  
  if(str != "")
    doSaveConfigFile(str);
}

void KInetEd::doSaveConfigFile(const char* fname) {
  char buf[1024];
  unsigned int i;
  ofstream fout(fname);

  // Separator character is \t which makes it perfect to
  // just print each entry as a line...
  fout << "# inetd configuration file." << endl <<
    "# This file was generated with KInetEd "VERSION"\n" << endl <<
    "# KInetEd was written by Jorgen Sigvafdsson <jorgen@linux.nu>" << endl <<
    "# Send eventual bug reports to me" << endl;
  for(i = 0; i < m_pListBox->count(); i++) {
    strcpy(buf, "");
    if(!strncmp(m_pListBox->text(i), "off", 3))
      sprintf(buf, "%s", OFFMARKER);
    sprintf(buf, "%s%s\n", buf, strchr((const char*)m_pListBox->text(i), '\t') + 1);
    fout << buf;
  }  
}

void KInetEd::itemSelected(int index, int column) {
  m_pListBox->unmarkAll();
  m_pListBox->markItem(index);
  m_pListBox->setCurrentItem(index);
  switch(column) {
  case COL_ONOFF:
    m_pStatusPopup->popup(QCursor::pos());
    break;
  case COL_NAME: /* Service name */
    m_pNamePopup->popup(QCursor::pos());
    break;
  case COL_SOCKTYPE: /* socket type */
    m_pSockPopup->popup(QCursor::pos());
    break;
  case COL_PROTO: /* protocol type */
    m_pProtoPopup->popup(QCursor::pos());
    break;
  case COL_FLAGS: /* flags */
    m_pFlagPopup->popup(QCursor::pos());
    break;
  case COL_USER: /* user */
    m_pUserPopup->popup(QCursor::pos());
    break;
  case COL_PATH: /* server path */
    m_pPathPopup->popup(QCursor::pos());
    break;
  case COL_ARGS: /* arguments */
    m_pArgPopup->popup(QCursor::pos());
    break;
  default:
    cerr << "Unknown column" << endl;
  }
}

// Popup handlers..
void KInetEd::onSockPopup(int id) {
  int curItem = m_pListBox->currentItem();

  switch(id) {
  case IDSOCK_STREAM:
    m_pListBox->changeItemPart(inetd_SockTypeName(SOCKTYPE_STREAM), 
				curItem, COL_SOCKTYPE);
    break;
  case IDSOCK_DGRAM:
    m_pListBox->changeItemPart(inetd_SockTypeName(SOCKTYPE_DGRAM),
				curItem, COL_SOCKTYPE);
    break;
  default:
    cerr << "Unknown menuid (onSockPopup()): " << id << endl;
  }
}

void KInetEd::onNamePopup(int id) {
  KRequestDlg dlg("Enter a name", this);
  if(dlg.exec() && dlg.getInput() != "") {
    m_pListBox->changeItemPart(dlg.getInput(), m_pListBox->currentItem(),
			       COL_NAME);
  }
}

void KInetEd::onProtoPopup(int id) {
  int curItem = m_pListBox->currentItem();
  
  switch(id) {
  case IDPROTO_UDP:
    m_pListBox->changeItemPart(inetd_ProtoName(PROTO_UDP), 
			       curItem, COL_PROTO);
    break;
  case IDPROTO_TCP:
    m_pListBox->changeItemPart(inetd_ProtoName(PROTO_TCP), 
			       curItem, COL_PROTO);
    break;
  case IDPROTO_RPCUDP:
    m_pListBox->changeItemPart(inetd_ProtoName(PROTO_RPCUDP), 
			       curItem, COL_PROTO);
    break;
  case IDPROTO_RPCTCP:
    m_pListBox->changeItemPart(inetd_ProtoName(PROTO_RPCTCP), 
			       curItem, COL_PROTO);
    break;
  default:
    cerr << "Unknown menuid (onProtoPopup()): " << id << endl;
  }
}

void KInetEd::onFlagPopup(int id) {
  int curItem = m_pListBox->currentItem();

  switch(id) {
  case IDFLAG_WAIT:
    m_pListBox->changeItemPart(inetd_FlagName(FLAG_WAIT), 
				curItem, COL_FLAGS);
    break;
  case IDFLAG_NOWAIT:
    m_pListBox->changeItemPart(inetd_FlagName(FLAG_NOWAIT),
				curItem, COL_FLAGS);
    break;
  default:
    cerr << "Unknown menuid (onFlagPopup()): " << id << endl;
  }
}

void KInetEd::onUserPopup(int id) {
  int curItem = m_pListBox->currentItem();

  switch(id) {
  case IDUSER_ROOT:
    m_pListBox->changeItemPart("root", curItem, COL_USER);
    break;
  case IDUSER_NOBODY:
    m_pListBox->changeItemPart("nobody", curItem, COL_USER);
    break;
  case IDUSER_EDIT:
    {
      KRequestDlg dlg("Enter a user name", this);
      if(dlg.exec() && dlg.getInput() != "") {
	m_pListBox->changeItemPart(dlg.getInput(), m_pListBox->currentItem(),
				   COL_USER);
      }
    }
    break;
  default:
    cerr << "Unknown menuid (onUserPopup()): " << id << endl;
  }
}

void KInetEd::onPathPopup(int id) {
  int curItem = m_pListBox->currentItem();

  switch(id) {
  case IDPATH_INTERNAL:
    m_pListBox->changeItemPart("internal", curItem, COL_PATH);
    m_pListBox->changeItemPart("", curItem, COL_ARGS);
    break;
  case IDPATH_TCPW:
    m_pListBox->changeItemPart("/usr/sbin/tcpd", curItem, COL_PATH);
    break;
  case IDPATH_EDIT:
    {
      KRequestDlg dlg("Enter a server path", this);
      if(dlg.exec() && dlg.getInput() != "") {
	m_pListBox->changeItemPart(dlg.getInput(), m_pListBox->currentItem(),
				   COL_PATH);
      }
    }
    break;
  default:
    cerr << "Uknown menuid (onPathPopup()): " << id << endl;
  }
}

void KInetEd::onArgPopup(int id) {
  int curItem = m_pListBox->currentItem();
  switch(id) {
  case IDARG_BLANK:
    m_pListBox->changeItemPart("", curItem, COL_ARGS);
    break;
  case IDARG_EDIT:
    {
      KRequestDlg dlg("Enter arguments", this);
      if(dlg.exec() && dlg.getInput() != "") {
	m_pListBox->changeItemPart(dlg.getInput(), m_pListBox->currentItem(),
				   COL_ARGS);
      }
    }
    break;
  default:
    cerr << "Uknown menuid (onPathPopup()): " << id << endl;
  }
}

void KInetEd::onStatusPopup(int id) {
  int curItem = m_pListBox->currentItem();
  switch(id) {
  case IDST_OFF:
    m_pListBox->changeItemPart("off", curItem, COL_ONOFF);
    break;
  case IDST_ON:
    m_pListBox->changeItemPart("on", curItem, COL_ONOFF);
    break;
  default:
    cerr << "Unknown menuid (onStatusPopup()): " << id << endl;
  }
}

// End of popup handlers

// Menu and toolbar handlers

void KInetEd::onDeleteService() {
  toolBar()->setButton(TB_DELSERV, FALSE);

  int curItem = m_pListBox->currentItem();

  if(curItem != -1)
    m_pListBox->removeItem(curItem);
}

void KInetEd::onNewService() {
  toolBar()->setButton(TB_NEWSERV, FALSE);
  
  KSheetDlg dlg("Service creation wizard");
  KReqSheet* name = new KReqSheet(i18n("Enter service name"));
  dlg.addSheet(name);
  
  KMenuSheet* sock = new KMenuSheet(i18n("Choose a socket type"));
  sock->addItem(inetd_SockTypeName(SOCKTYPE_STREAM));
  sock->addItem(inetd_SockTypeName(SOCKTYPE_DGRAM));
  dlg.addSheet(sock);
  
  KMenuSheet* proto = new KMenuSheet(i18n("Choose a protocol"));
  proto->addItem(inetd_ProtoName(PROTO_UDP));
  proto->addItem(inetd_ProtoName(PROTO_TCP));
  proto->addItem(inetd_ProtoName(PROTO_RPCUDP));
  proto->addItem(inetd_ProtoName(PROTO_RPCTCP));
  dlg.addSheet(proto);
  
  KMenuSheet* flags = new KMenuSheet(i18n("Choose flags"));
  flags->addItem(inetd_FlagName(FLAG_WAIT));
  flags->addItem(inetd_FlagName(FLAG_NOWAIT));
  
  KReqSheet* user = new KReqSheet(i18n("Enter a user name"));
  dlg.addSheet(user);
  
  KServerSheet* spath = new KServerSheet;
  dlg.addSheet(spath);
  
  KReqSheet* args = new KReqSheet(i18n("Enter arguments"));
  dlg.addSheet(args);
  
  if(dlg.exec()) {
    QString instxt;
    
    instxt = name->getAnswer() + "\t" + sock->getAnswer() + "\t" +
      proto->getAnswer() + "\t" + flags->getAnswer() + "\t" + 
      user->getAnswer() + "\t" +  spath->getAnswer() + "\t" + 
      args->getAnswer();
    m_pListBox->insertItem(instxt);
  }
}

void KInetEd::onNewConfig() {
  toolBar()->setButton(TB_NEWCONF, FALSE);
  m_pListBox->clear();
  m_pListBox->repaint();
}
