/***************************************************************************
                          KHDisplayWidget.cpp  -  description
                             -------------------                                         
    begin                : Sun Jul 18 1999                                           
    copyright            : (C) 1999 by Ruediger.Anlauf                         
    email                : Ruediger.Anlauf@gmx.net                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <kapp.h>
#include <kiconloader.h>
#include <qspinbox.h>
#include <qlcdnumber.h>
#include "khdisplayitem.h"
#include "khchip.h"
#include "khconfigdlg.h"
#include "khglobals.h"
#include "khcountingmessage.h"
#include "khdisplaywidget.h"
#include "khfeatureprofile.h"

KHDisplayWidget::KHDisplayWidget( KHChip* ioObj, KHFeatureProfile *profile ) : KHDisplayBase( profile ) {

	setIcon( ICON( "khealthcare.xpm" ) );
	config = kapp->getConfig();
	this->ioObj = ioObj;
	this->profile = profile;

	for ( SensorId sid = Fan1; sid < SensorIdSize; sid++ ) {
		mbox[sid] = 0;
		display[sid] = 0;
	}

	initDialog();
	initAlarmSettings(); // sets the alarm selectors ( SpinBoxes ) to values from config file
	//debug("alarm settings done");
	setUpdatesEnabled( TRUE );

	// connect the SpinBoxes to the slots of the Winbond object
	// => writes changes of alarm values to the chip
	
	for ( SensorId sid = Fan1; sid < VCore; sid++ ) {
		if ( display[sid] ) {
			connect( display[sid]->lobox(),SIGNAL( valueChanged	( int ) ),
							 this,	     					 SLOT  ( writeAlarm( int ) ) );
			connect( display[sid]->hibox(),SIGNAL( valueChanged 	( int ) ),
							 this,     						 SLOT  ( writeAlarm( int ) ) );
		}
	}


	for ( SensorId sid = VCore; sid < SensorIdSize; sid++ ) {
		if ( display[sid] ) {
			connect( display[sid]->lobox(),SIGNAL( valueChanged	( int ) ),
							 this,	     					 SLOT  ( writeLoLim( int ) ) );
			connect( display[sid]->hibox(),SIGNAL( valueChanged 	( int ) ),
							 this,     						 SLOT  ( writeHiLim( int ) ) );
		}
	}

	connect( ioObj,SIGNAL( newDataAcquired ( int, SensorId ) ),
					 this, SLOT  ( showData		     ( int, SensorId ) ) );	
	connect( ioObj,SIGNAL( newDataAcquired ( float, SensorId ) ),
					 this, SLOT  ( showData				 ( float, SensorId ) ) );
	
	connect( ioObj,SIGNAL	( reportCond			 ( SensorId, SensorStatus ) ),
					 this, SLOT  	( processSensorCond( SensorId, SensorStatus ) ) );	

	connect( kapp, SIGNAL( saveYourself() ),
					 this, SLOT	 ( saveMyself() ) );

	connect( this, SIGNAL( alarmSettingChanged( int, SensorId ) ),
					 ioObj, SLOT ( setAlarm( int, SensorId ) ) );

	connect( this, SIGNAL( loLimSettingChanged( int, SensorId ) ),
					 ioObj, SLOT ( setLoLim( int, SensorId ) ) );

	connect( this, SIGNAL( hiLimSettingChanged( int, SensorId ) ),
					 ioObj, SLOT ( setHiLim( int, SensorId ) ) );

	ioObj->start();

}

KHDisplayWidget::~KHDisplayWidget() {
	//debug( "destroying KHDisplayWidget" );
	config->sync();
}

void KHDisplayWidget::showData( int value, SensorId sid ){

	if ( !display[sid] ) return;

	config->setGroup( "Monitoring" );
	if ( !config->readBoolEntry( EnableKeys[sid] ) ) value = 0;

	if ( value != 0 ) display[sid]->lcd()->display( value );
	else display[sid]->lcd()->display( "----" );
}

void KHDisplayWidget::showData( float value, SensorId sid ){

	if ( !display[sid] ) return;	

	config->setGroup( "Monitoring" );
	if ( !config->readBoolEntry( EnableKeys[sid] ) ) value = 0.0;

	if ( value == 0.0 ) display[sid]->lcd()->display( "------" );
	else display[sid]->lcd()->display( QString().sprintf( "%2.2f", value ) );
}

void KHDisplayWidget::initOptionsDialog(){

	KHConfigDlg * optionsDialog = new KHConfigDlg( ioObj, this, "Configuration" );
	optionsDialog->exec();
	delete optionsDialog;
}

void KHDisplayWidget::initAlarmSettings(){

	config->setGroup( "Alarms" );
	for ( SensorId sid = Fan1; sid < Temp1; sid++ ) {
		if ( display[sid] )
			display[sid]->lobox()->setValue( config->readNumEntry( AlarmKeysLo[sid] ) );
	}

	for ( SensorId sid = Temp1; sid < VCore; sid++ ) {
		if ( display[sid] )
			display[sid]->hibox()->setValue( config->readNumEntry( AlarmKeysHi[sid] ) );
	}

	for ( SensorId sid = VCore; sid < SensorIdSize; sid++ )
		if ( display[sid] ) {
			display[sid]->lobox()->setValue( config->readNumEntry( AlarmKeysLo[sid] ) );
			display[sid]->hibox()->setValue( config->readNumEntry( AlarmKeysHi[sid] ) );
		}

}

void KHDisplayWidget::writeAlarm( int value ){
	SensorId sid = End; // a usually unused value of the enum
	QSpinBox *box = (QSpinBox*) sender();
	
	for ( SensorId i = Fan1; i < Temp1; i++ ) {
		if ( ( display[i] ) && ( box == display[i]->lobox() ) ) sid = i;
 	}
 	
 	for ( SensorId i = Temp1; i < VCore; i++ ) {
 		if ( ( display[i] ) && ( box == display[i]->hibox() ) ) sid = i;
 	}

 	if ( sid == End ) { // no assignment took place in above "for"
 		debug("KHDisplayWidget::writeAlarm(): Couldn't determine sender of message");
 		return;
 	}
  	
 	config->setGroup( "Alarms" );
  switch( sid ) {
		case Fan1:
		case Fan2:
		case Fan3:
			config->writeEntry( AlarmKeysLo[sid], value );
		break;
		case Temp1:
		case Temp2:
		case Temp3:
			config->writeEntry( AlarmKeysHi[sid], value );
		break;
		default:
			// suppress others
		break;
	}
 	//debug("%i detected", value );
 	emit alarmSettingChanged( value, sid );
 }


void KHDisplayWidget::writeLoLim( int value ){	
	SensorId sid = End;
	QSpinBox *box = (QSpinBox*) sender();
	
	for ( SensorId i = VCore; i < SensorIdSize; i++ ) {
		if ( ( display[i] ) && ( box == display[i]->lobox() ) ) sid = i;
	}

	if ( sid == End ) {
		debug("KHDisplayWidget::writeLoLim(): Couldn't determine sender of message");
		return;
	}

	config->setGroup( "Alarms" );
	config->writeEntry( AlarmKeysLo[sid], value );    	
	emit loLimSettingChanged( value, sid );
}

void KHDisplayWidget::writeHiLim( int value ){
	SensorId sid = End;
	QSpinBox *box = (QSpinBox*) sender();
	
	for ( SensorId i = VCore; i < SensorIdSize; i++ ) {
		if ( ( display[i] ) && ( box == display[i]->hibox() ) ) sid = i;
	}

	if ( sid == End ) {
		debug("KHDisplayWidget::writeHiLim(): Couldn't determine sender of message");
		return;
	}
	
	config->setGroup( "Alarms" );
	config->writeEntry( AlarmKeysHi[sid], value );
 	emit hiLimSettingChanged( value, sid );
}


void KHDisplayWidget::processSensorCond( SensorId sid, SensorStatus status ) {
	if ( display[sid] )
		if ( status == SensorRed ) showAlertMessage( sid );
		else cancelAlert( sid );
}


void KHDisplayWidget::cancelAlert( SensorId sid ) {
	
	display[sid]->lcd()->setBackgroundMode( PaletteBackground );

}

void KHDisplayWidget::showAlertMessage( SensorId sid ){

	config->setGroup( "Monitoring" );

	// disabled sensor always show status green
	if ( !config->readBoolEntry( EnableKeys[sid] ) ) {
		cancelAlert( sid );
		return;
	}

	QString message;
	QString pre = i18n( "The sensor monitoring " );
	QString post = i18n( " is \nreporting a critical condition\naccording to your settings !\nPlease check !" );

	display[sid]->lcd()->setBackgroundMode( PaletteMid );

	debug( "%i, %s", sid, (const char*)display[sid]->name() );	
	message = pre.append( display[sid]->name() ).append( post );
	// display message box if its not already there
	if ( !mbox[sid] ) mbox[sid] = new KHCountingMessage( message, sid );
	else mbox[sid]->show();

}

void KHDisplayWidget::saveMyself(){
	KConfig *sessionconf = kapp->getSessionConfig();
	sessionconf->setGroup( "Default" );
	if ( isVisible() ) sessionconf->writeEntry( "DisplayWidgetVisible", true );
	else sessionconf->writeEntry( "DisplayWidgetVisible", false );
}






































