# -*- coding: iso-8859-1 -*-

# GNU Solfege - ear training for GNOME
# Copyright (C) 2000, 2001, 2002, 2003, 2004, 2005  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

copyright = "Copyright  1999-2006 Tom Cato Amundsen <tca@gnu.org>"

import sys
import os
import time
import urllib

import runtime

import configureoutput

import optparse

opt_parser = optparse.OptionParser(usage='GNU Solfege %s\nType "man solfege" for more details.' % configureoutput.VERSION_STRING )
opt_parser.add_option('-v', '--version', action='store_true', dest='version')
opt_parser.add_option('-w', '--warranty', action='store_true', dest='warranty',
    help='show warranty and copyright')
opt_parser.add_option('--no-splash', action='store_false', dest='no_splash',
    default=False)
opt_parser.add_option('--verbose-sound-init', action='store_true',
    default=False,
    dest='verbose_sound_init', help='Display more info about the sound setup')
opt_parser.add_option('--no-sound', action='store_true', dest='no_sound',
    default=False,
    help='Do not play any sounds. Useful when porting.')
opt_parser.add_option('--no-random', action='store_true', dest='no_random',
    help='For debugging only: Select questions from lesson files in sequential order.')
opt_parser.add_option('--without-gtkhtml', action='store_true', 
    dest='without_gtkhtml',
    help='Run without gtkhtml2 even if it is installed')
opt_parser.add_option('--without-gnome', action='store_true', 
    dest='without_gnome',
    help='Run without the gnome module even if it is installed')
opt_parser.add_option('--show-gtk-warnings', action='store_true',
    dest='show_gtk_warnings',
    help='Show GtkWarnings in the traceback window')


options, args = opt_parser.parse_args()

if options.version:
    print """GNU Solfege %s
This is free software. It is covered by the GNU General Public License,
and you are welcome to change it and/or distribute copies of it under
certain conditions. Invoke as `solfege --warranty` for more information.

%s
        """ % (configureoutput.VERSION_STRING, copyright)
    sys.exit()

if options.warranty:
    print """GNU Solfege %s
%s
    This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License version 2
as published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

    You should have received a copy (refer to the file COPYING) of the
GNU General Public License along with this program; if not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
USA.
    """ % (configureoutput.VERSION_STRING, copyright)
    sys.exit()

# silent, GNOME, be silent!
sys.argv.append('--disable-sound')
runtime.init(options)

import gtk

if not runtime.has_gnome():
    sys.path.insert(0, "gnomeemu")

class SplashWin(gtk.Window):
    def __init__(self):
        gtk.Window.__init__(self, gtk.WINDOW_POPUP)
        self.set_position(gtk.WIN_POS_CENTER)
        self.set_resizable(True)
        frame = gtk.Frame()
        frame.set_shadow_type(gtk.SHADOW_OUT)
        self.add(frame)
        vbox = gtk.VBox()
        vbox.set_border_width(20)
        frame.add(vbox)
        l = gtk.Label(_("Starting GNU Solfege %s") % configureoutput.VERSION_STRING)
        l.set_name("Heading1")
        vbox.pack_start(l)
        l = gtk.Label("http://www.solfege.org")
        vbox.pack_start(l)
        self.g_infolabel = gtk.Label('')
        vbox.pack_start(self.g_infolabel)
        self.show_all()
    def show_progress(self, txt):
        self.g_infolabel.set_text(txt)
        while gtk.events_pending():
            gtk.main_iteration(0)

if not options.no_splash:
    splash_win = SplashWin()
    time.sleep(0.1)
    gtk.gdk.flush()
    while gtk.events_pending():
        gtk.main_iteration(0)
else:
    splash_win = None

if splash_win:
    splash_win.show_progress("importing GNOME modules")

import gnome, gnome.ui
# FIXME PAD AND PAD_SMALL are missing from some gnome-python releases.
try:
    gnome.ui.PAD
except:
    gnome.ui.PAD = 8
    gnome.ui.PAD_SMALL = 4

if splash_win:
    splash_win.show_progress("importing application modules")

import tracebackwindow
# redirect error messages to a window that will popup if
# something bad happens.

sys.stderr = tracebackwindow.TracebackWindow(options.show_gtk_warnings)

import const
import app
from htmlwidget import HtmlWidget
import utils
import history
from configwindow import ConfigWindow
import soundcard
import mpd
import mpd.musicdisplayer
import gu
import cfg
import stock
import lessonfile
import abstract

import chord
import chordvoicing
import harmonicinterval
import melodicinterval
import singinterval

import idbyname
import dictation
import twelvetone
import idtone
import compareintervals
import singchord
import rhythm
import identifybpm
import harmonicprogressiondictation
import singanswer
#import example
import learning_tree_editor

class DocViewer(gtk.VBox):
    def __init__(self, activate_cb):
        gtk.VBox.__init__(self)
        self.m_htmlwidget = HtmlWidget(activate_cb, self.on_anchor_track)
        self.m_htmlwidget.show_all()
        self.pack_start(self.m_htmlwidget)
        self.set_size_request(500, 300)
        self.get_vadjustment = self.m_htmlwidget.get_vadjustment
        self.on_key_press_event = self.m_htmlwidget.on_key_press_event
        self.m_statusbar = gtk.Statusbar()
        self.m_statusbar.show()
        self.pack_start(self.m_statusbar, False)
        self.source = self.m_htmlwidget.source
        self.grab_focus = self.m_htmlwidget.grab_focus
    def read_docfile(self, fn, anchor):
        """
        We reread the docfile even if we are alreado displaying this file,
        just in case it has changed on disk.
        """
        if fn:
            self.m_htmlwidget.read_docfile(fn)
        if anchor and 'g_view' in dir(self.m_htmlwidget):
            self.m_htmlwidget.g_view.jump_to_anchor(anchor)
    def on_anchor_track(self, url):
        if url:
            # remove newlines in url because it make the window resize
            s = url.replace("\n", "")
            self.m_statusbar.pop(1)
            self.m_statusbar.push(1, s)
        else:
            self.m_statusbar.pop(1)

class ResolveLessonIdCrashDlg(gtk.Dialog):
    def __init__(self, lessonfile_manager, lesson_id):
        gtk.Dialog.__init__(self, _("Resolve lesson_id crash"), None,
            buttons=(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                     gtk.STOCK_OK, gtk.RESPONSE_OK))
        vbox = gtk.VBox()
        vbox.set_border_width(12)
        self.vbox.pack_start(vbox)
        msg1 = gtk.Label("<b>%s</b>" % _("Lesson_id crash"))
        msg1.set_alignment(0.0, 0.5)
        msg1.set_use_markup(True)
        vbox.pack_start(msg1, False)
        msg2 = gtk.Label(_("The lesson_id has to be unique for each lesson file. Please select which lesson file shall keep the current lesson_id. The preselected file is Solfege's educated guess. Press 'Cancel' to postpone the decision."))
        msg2.set_alignment(0.0, 0.5)
        msg2.set_line_wrap(True)
        vbox.pack_start(msg2)
        self.radio_dict = {}
        for info in lessonfile_manager.get_lesson_file_info(lesson_id):
            if self.radio_dict.keys():
                g = self.radio_dict[self.radio_dict.keys()[0]]
            else:
                g = None
            ss = "%s %s" % (info['filename'], info['timestr'])
            if not isinstance(ss, unicode):
                ss = ss.decode(locale.getpreferredencoding(), 'replace')
            radiobutton = gtk.RadioButton(g, ss)
            radiobutton.show()
            radiobutton.set_data('fn', info['filename'])
            radiobutton.set_data('lessonfile', info['lessonfile'])
            radiobutton.set_data('lessoncollection', info['lessoncollection'])
            self.radio_dict[info['mtime']] = radiobutton
            vbox.pack_start(radiobutton)
        self.radio_dict[min(self.radio_dict.keys())].set_active(True)
        self.show_all()

class MusicViewerWindow(gtk.Dialog):
    def __init__(self, app):
        gtk.Dialog.__init__(self)
        self.m_app = app
        self.set_default_size(500, 300)
        self.g_music_displayer = mpd.musicdisplayer.MusicDisplayer(utils.play_tone)
        self.vbox.pack_start(self.g_music_displayer)
        b = gu.bButton(self.action_area, _("Close"), self.m_app.close_musicviewer)
        b.grab_focus()
        self.connect('destroy', self.m_app.close_musicviewer)
        self.show_all()
    def display_music(self, music):
        fontsize = cfg.get_int('config/feta_font_size=20')
        self.g_music_displayer.display(music, fontsize)

class MainWin(gnome.ui.App, cfg.ConfigUtils):
    default_learning_tree = ('solfege', 'learningtree.txt')
    def __init__(self, options, datadir, lessonfile_manager):
        gnome.ui.App.__init__(self, 'solfege', 'GNU Solfege')
        self.icons = stock.SolfegeIconFactory(self, datadir)
        self.set_default_size(400, 400)
        pixbuf = self.render_icon('solfege-icon', gtk.ICON_SIZE_DIALOG)
        self.set_icon(pixbuf)
        cfg.ConfigUtils.__dict__['__init__'](self, 'mainwin')
        self.set_resizable(self.get_bool('gui/mainwin_user_resizeable'))
        self.add_watch('gui/mainwin_user_resizeable', lambda s: self.set_resizable(self.get_bool('gui/mainwin_user_resizeable')))
        self.connect('destroy', self.quit_program)
        self.connect('key_press_event', self.on_key_press_event)
        self.g_about_window = None
        self.g_learning_tree_editor = None
        self.main_box = gtk.VBox()
        self.main_box.show()
        self.set_contents(self.main_box)
        self.m_exercise = None
        self.m_viewer = None
        self.box_dict = {}
        self.g_config_window = None
        self.g_musicviewer_window = None
        self.m_history = history.History()
        self.m_key_bindings = {'history_back_ak': self.history_back,
                               'history_forward_ak': self.history_forward,
                               'history_reload_ak': self.history_reload}
        self.m_app = app.SolfegeApp(options, self, lessonfile_manager)
        self.g_ui_manager = gtk.UIManager()
        self.m_action_groups = {
            'Practise': gtk.ActionGroup('Practise'),
            'Exit': gtk.ActionGroup('Exit'),
            'NotExit': gtk.ActionGroup('NotExit'),
        }
        for a in self.m_action_groups.values():
            self.g_ui_manager.insert_action_group(a, 1)
        self.setup_menu()
        self.m_app.handle_href('welcome.html')
    def change_learning_tree(self, tree_fn):
        """
        Change to a different learning tree.
        """
        self.set_list('app/learningtree', tree_fn)
        self.on_learning_tree_changed()
    def on_learning_tree_changed(self, *v):
        """
        We call this when we have changed the current tree or
        switched tree.
        """
        self.m_tree = learning_tree_editor.LearningTree(self.m_app.lessonfile_manager)
        try:
            loc, filename = self.get_list("app/learningtree")
        except ValueError:
            loc, filename = self.default_learning_tree
        try:
            if loc == 'solfege':
                self.m_tree.load(filename)
            else:
                assert loc == 'user'
                self.m_tree.load(os.path.join(os.path.expanduser('~'), '.solfege', 'learningtrees', filename))
        except IOError:
            self.m_tree.load(self.default_learning_tree[1])
            self.set_list('app/learningtree', self.default_learning_tree)
            self.dialog_ok(_("Learning tree '%s' not found. Using default tree." % filename))
        self.create_practise_and_test_menu()
    def setup_menu(self):
        self.m_action_groups['Exit'].add_actions([
          ('FileMenu', None, _('_File')),
          ('AppQuit', 'gtk-quit', None, None, None, self.quit_program),
        ])
        self.m_action_groups['NotExit'].add_actions([
          ('MiscMenu', None, _('_Misc exercises')),
          ('TheoryMenu', None, _('_Theory')),
          ('TheoryIntervals', None, _('_Intervals'), None, None,
            lambda o: self.m_app.handle_href('theory-intervals.html')),
          ('EditMenu', None, _("_Edit")),
          ('TreeEditor', None, _('Learning tree'), None, None,
            self.do_tree_editor),
          ('PractiseMenu', None, _("_Practise")),
          ('TestsMenu', None, _("_Tests")),
          ('ViewMenu', None, _('_View')),
          ('OpenPreferencesWindow', 'gtk-preferences', None, '<ctrl>F12', None,
            self.open_preferences_window),
          ('HelpMenu', None, _('_Help')),
          ('HelpHelp', 'gtk-help', _('_Help on current exercise'), 'F1', None,
            lambda o: self.m_app.please_help_me()),
          ('HelpIndex', None, _('_All help files'), None, None,
            lambda o: self.m_app.handle_href('toc.html')),
          ('HelpAllLessonFiles', None, _('All installed _lesson files'), None,
            None, lambda o: self.m_app.handle_href('solfege:all-lessonfiles')),
          ('HelpCopyright', None, _('_Copyright notice'), None, None,
            lambda o: self.m_app.handle_href('copyright.html')),
          ('HelpOnline', None, _('_Mailinglists, web page etc.'), None, None,
            lambda o: self.m_app.handle_href('online-resources.html')),
          ('HelpReportingBugs', None, _('Reporting _bugs'), None, None,
            lambda o: self.m_app.handle_href('bug-reporting.html')),
          ('HelpAbout', 'gtk-about', None, None, None, self.show_about_window),
          ('ViewZoomIn', 'gtk-zoom-in', _('Help viewer zoom in'), '<ctrl>plus', None,
           lambda w: self.box_dict['docviewer'].m_htmlwidget.g_view.zoom_in()),
          ('ViewZoomOut', 'gtk-zoom-out', _('Help viewer zoom out'), '<ctrl>minus', None,
           lambda w: self.box_dict['docviewer'].m_htmlwidget.g_view.zoom_out()),
          ('ViewZoom100', 'gtk-zoom-100', _('Help viewer zoom reset'), '<ctrl>equal', None,
           lambda w: self.box_dict['docviewer'].m_htmlwidget.g_view.zoom_reset()),
          ('BrowserGoBack', 'gtk-go-back', None, None, None, self.history_back),
          ('BrowserGoForward', 'gtk-go-forward', None, None, None, self.history_forward),
          ('ShowBugReports', None, _('See your bug reports'), None, None,
            self.show_bug_reports),
        ])

        self.g_ui_manager.add_ui_from_file("ui.xml")

        if runtime.has_gtkhtml2():
            self.g_ui_manager.add_ui_from_string("""\
<menubar name='Menubar'>
 <menu action='ViewMenu'>
      <menuitem action='ViewZoomIn'/>
      <menuitem action='ViewZoomOut'/>
      <menuitem action='ViewZoom100'/>
 </menu>
</menubar>""")

        self.add_accel_group(self.g_ui_manager.get_accel_group())
        self.g_ui_manager.get_accel_group().connect_group(gtk.keysyms.KP_Equal,
           gtk.gdk.CONTROL_MASK, 0,
           lambda w, a, b, c: self.box_dict['docviewer'].m_htmlwidget.g_view.zoom_reset())
        self.g_ui_manager.get_accel_group().connect_group(gtk.keysyms.KP_Subtract,
           gtk.gdk.CONTROL_MASK, 0,
           lambda w, a, b, c: self.box_dict['docviewer'].m_htmlwidget.g_view.zoom_out())
        self.g_ui_manager.get_accel_group().connect_group(gtk.keysyms.KP_Add,
           gtk.gdk.CONTROL_MASK, 0,
           lambda w, a, b, c: self.box_dict['docviewer'].m_htmlwidget.g_view.zoom_in())
        self.set_menus(self.g_ui_manager.get_widget('/Menubar'))
        self.m_help_on_current_merge_id = None
    def create_learning_trees_menu(self):
        if self.m_learning_trees_merge_id:
            self.g_ui_manager.remove_ui(self.m_learning_trees_merge_id)
        try:
            v = os.listdir(os.path.expanduser("~/.solfege/learningtrees"))
        except OSError:
            v = []
        if not v:
            return
        actions = [('LearningTreesMenu', None, _('Learning tree')),
            ('DefaultTree', None, _("Default learning tree"),
            None, None,
           lambda o: self.change_learning_tree(self.default_learning_tree)
            )]
        s = """
<menubar name='Menubar'>
 <menu action='FileMenu'>
  <menu action='LearningTreesMenu' position='top'>
    <menuitem action='DefaultTree'/>"""
        for tree_fn in v:
            s += "<menuitem action='LTree-%s'/>" % tree_fn
            actions.append(('LTree-%s' % tree_fn, None,
                    tree_fn, None, None,
                    lambda o, t=tree_fn: self.change_learning_tree(('user', t))))
        s += "</menu></menu></menubar>"
        self.m_action_groups['NotExit'].add_actions(actions)
        self.m_learning_trees_merge_id = self.g_ui_manager.add_ui_from_string(s)
    def create_practise_and_test_menu(self):
        """
        Create the Practise menu.
        This function has to be called after we have cleared out any
        lesson_id crashes.
        """
        actions = []
        if self.m_practise_and_test_merge_id:
            self.g_ui_manager.remove_ui(self.m_practise_and_test_merge_id)
        s = """
<menubar name='Menubar'>
 <menu action='PractiseMenu'>"""
        tests = "<menu action='TestsMenu'>"
        for topic in self.m_tree.m_topics:
            s += "<menu action='%s'>" % topic['name']
            tests += "<menu action='TEST_%s'>" % topic['name']
            actions.append((topic['name'], None, _(topic['name'])))
            actions.append(('TEST_%s' % topic['name'], None, _(topic['name'])))
            for lesson_id in topic['lessons']:
                if self.m_tree.m_visibilities[lesson_id] > self.m_tree.m_visibility:
                    continue
                menutitle = self.m_app.lessonfile_manager.m_uiddb[lesson_id]['header']['title']
                actions.append((
                    lesson_id, None, menutitle, None, None,
                    lambda o, e=lesson_id: self.m_app.practise_lesson_id(e)))
                if self.m_app.lessonfile_manager.is_test_passed(lesson_id):
                    menutitle = _("%s (passed)") % menutitle
                if self.m_app.lessonfile_manager.get(lesson_id, 'test'):
                    actions.append((
                        'TEST_%s' % lesson_id, None,
                        menutitle,
                        None, None,
                        lambda o, e=lesson_id: self.m_app.test_lesson_id(e)))
                    tests += "<menuitem action='TEST_%s' />" % lesson_id
                s += "<menuitem action='%s' />" % lesson_id
            s += "</menu>"
            tests += "</menu>"
        self.m_action_groups['NotExit'].add_actions(actions)
        s += "</menu>"
        tests += "</menu>"
        s += tests
        s += "</menubar>"
        self.m_practise_and_test_merge_id = self.g_ui_manager.add_ui_from_string(s)
    def show_help_on_current(self):
        """
        Show the "Help on current exercise F1" menu entry on the
        Help menu.
        """
        if self.m_help_on_current_merge_id:
            return
        self.m_help_on_current_merge_id = self.g_ui_manager.add_ui_from_string("""
<menubar name='Menubar'>
  <menu action='HelpMenu'>
    <menuitem position='top' action='HelpHelp' />
  </menu>
</menubar>""")
    def hide_help_on_current(self):
        """
        Hide the "Help on current exercise F1" menu entry on
        the Help menu.
        """
        if not self.m_help_on_current_merge_id:
            return
        self.g_ui_manager.remove_ui(self.m_help_on_current_merge_id)
        self.m_help_on_current_merge_id = None
    def dialog_ok(self, text):
        """"
        Return the gtk.RESPONSE_XXXX returned by .run()
        """
        m = gtk.MessageDialog(self, gtk.DIALOG_MODAL, gtk.MESSAGE_INFO,
                gtk.BUTTONS_OK, text)
        ret = m.run()
        m.destroy()
        return ret
    def dialog_yesno(self, text):
        """Return True if the answer is yes, False if the answer is no.
        """
        m = gtk.MessageDialog(self, gtk.DIALOG_MODAL, gtk.MESSAGE_QUESTION,
                gtk.BUTTONS_YES_NO, text)
        ret = m.run()
        m.destroy()
        return ret == gtk.RESPONSE_YES
    def show_bug_reports(self, *v):
        m = gtk.Dialog(_("Question"), self, 0)
        m.add_button(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
        m.add_button(gtk.STOCK_OK, gtk.RESPONSE_OK)
        vbox = gtk.VBox()
        m.vbox.pack_start(vbox, False)
        vbox.set_spacing(18)
        vbox.set_border_width(12)
        l = gtk.Label(_("Please enter the email used when you submitted the bugs:"))
        vbox.pack_start(l, False)
        self.g_email = gtk.Entry()
        m.action_area.get_children()[0].grab_default()
        self.g_email.set_activates_default(True)
        vbox.pack_start(self.g_email, False)
        m.show_all()
        ret = m.run()
        m.destroy()
        if ret == gtk.RESPONSE_OK:
            params = urllib.urlencode({
                    'pagename': 'SITS-Incoming/SearchBugs',
                    'q': 'SITS-Incoming/"Submitter: %s"' % utils.mangle_email(self.g_email.get_text()),
                })
            import webbrowser
            webbrowser.open_new("http://www.solfege.org?%s" % params)
    def display_exception_message(self, exception, err_file=None):
        """Call this function only inside an except clause."""
        if err_file:
            msg = _("Exception catched in %s:%i\n") % (err_file, sys.exc_info()[2].tb_lineno) + "\n"
        else:
            msg = ""
        exception_str = unicode(exception)
        if not isinstance(exception_str, unicode):
            exception_str = exception_str.decode(locale.getpreferredencoding(), 'replace')
        msg = "%s %s" %(msg, exception_str)
        m = gtk.MessageDialog(self, gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR,
                              gtk.BUTTONS_CLOSE, msg)
        m.set_title(sys.exc_info()[0].__name__)
        m.run()
        m.destroy()
    def display_error_message(self, msg):
        if not isinstance(msg, unicode):
            msg = msg.decode(locale.getpreferredencoding(), 'replace')
        m = gtk.MessageDialog(self, gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR,
                              gtk.BUTTONS_CLOSE, None)
        m.set_markup(msg)
        m.run()
        m.destroy()
    def display_question_music_error_message(self, idx, lessonfile, exception, err_file=None):
        """Call this function only inside an except clause."""
        msg = _("Failed to parse the music for question number %(idx)i in '%(lf)s':\n%(ex)s") % {'idx': idx, 'lf': lessonfile, 'ex': exception}
        self.display_exception_message(msg, err_file)
    def show_about_window(self, widget):
        #if not self.g_about_window:
        if 1:
            trans = _("SOLFEGETRANSLATORS")
            if trans == 'SOLFEGETRANSLATORS':
                trans = ""
            pixbuf = self.render_icon('solfege-icon', gtk.ICON_SIZE_DIALOG)
            self.g_about_window = gnome.ui.About("GNU Solfege",
                  configureoutput.VERSION_STRING,
                  copyright.decode("iso-8859-1"),
                  _("GPL'ed ear training.") + "\n" + "http://www.solfege.org",
                  ["Tom Cato Amundsen",
                  'Oliver Vollmer %s' % _("(toolbar icons)"),
                  'Giovanni Chierico %s' % _("(some lessonfiles)"),
                  'Michael Becker %s' % _("(some lessonfiles)"),
                  'Joe Lee %s' % _("(sound code for the MS Windows port)"),
                  'Steve Lee %s' % _("(ported winmidi.c to gcc)"),
                  'Thibaus Cousin %s' % _("(spec file for SuSE 8.2)"),
                  'David Coe %s' %_("(spec file cleanup)"),
                  'David Petrou %s' % _("(testing and portability fixes for FreeBSD)"),
                  'Han-Wen Nienhuys %s' % _("(the music font from Lilypond)"),
                  'Jan Nieuwenhuizen %s' % _("(the music font from Lilypond)"),
                  ],
                  ["Tom Cato Amundsen", "Tom Eykens"],
                  trans, pixbuf)
            #self.g_about_window.connect('destroy', self.hide_about_window)
            #self.g_about_window.show()
            #FIXME this sucks.
            i = self.g_about_window.run()
            #self.g_about_window.destroy()
    def hide_about_window(self, widget):
        self.g_about_window.destroy()
        self.g_about_window = None
    def do_tree_editor(self, *v):
        # Raise it if it already exist.
        if self.g_learning_tree_editor:
            self.g_learning_tree_editor.present()
            return
        # And if it don't exist, then we have to create it.
        self.g_learning_tree_editor = win = learning_tree_editor.Window(self.m_app)
        try:
            loc, filename = self.get_list("app/learningtree")
        except ValueError:
            loc, filename = self.default_learning_tree
        if loc == 'solfege':
            win.load_file(filename)
        else:
            assert loc == 'user'
            win.load_file(os.path.join(os.path.expanduser('~'), '.solfege', 'learningtrees', filename))
        win.show()
        x, y = win.g_topic_scrollwin.size_request()
        win.g_lesson_scrollwin.set_size_request(x, int(y*1.6))
        win.g_deps_scrollwin.set_size_request(x, int(y*1.6))
    def post_constructor(self):
        global splash_win
        self.m_practise_and_test_merge_id = None
        self.m_learning_trees_merge_id = None
        self.on_learning_tree_changed()
        self.create_learning_trees_menu()
        self.box_dict['docviewer'].grab_focus()
        if self.m_app.m_sound_init_exception is not None:
            if splash_win:
                splash_win.destroy()
                splash_win = None
            self.m_app.display_sound_init_error_message(self.m_app.m_sound_init_exception)
    def activate_exercise(self, module, urlobj=None):
        if self.m_viewer:
            self.box_dict[self.m_viewer].hide()
        self.m_viewer = module
        self.box_dict[module].show()
        # We need this test because not all exercises use a notebook.
        if self.box_dict[self.m_viewer].g_notebook:
            if urlobj and urlobj.action in ['practise', 'config', 'statistics']:
                self.box_dict[self.m_viewer].g_notebook.set_current_page(
                   ['practise', 'config', 'statistics'].index(urlobj.action))
            else:
                self.box_dict[self.m_viewer].g_notebook.set_current_page(0)
        if self.box_dict[module].m_t.m_P:
            if isinstance(self.box_dict[module].m_t, abstract.LessonbasedTeacher):
                self.set_title("Solfege - " + self.box_dict[module].m_t.m_P.header.title)
            else:
                self.set_title("Solfege - " + self.box_dict[module].m_t.exercise_data['name'].replace("_", ""))
        else:
            self.set_title("Solfege")
        #self.m_history.add(urlobj.href)
        #self.show_help_on_current()
    def _show_docviewer(self):
        if 'docviewer' not in self.box_dict:
            self.box_dict['docviewer'] = DocViewer(self.m_app.handle_href)
            self.main_box.pack_start(self.box_dict['docviewer'])
        if self.m_viewer and (self.m_viewer != 'docviewer'):
            self.box_dict[self.m_viewer].hide()
        self.m_viewer = 'docviewer'
        self.box_dict['docviewer'].show()
    def display_docfile(self, fn, anchor):
        self._show_docviewer()
        self.box_dict['docviewer'].read_docfile(fn, anchor)
        self.set_title("Solfege - %s" % fn)
        self.m_history.add(fn)
    def display_html(self, html):
        self._show_docviewer()
        self.box_dict['docviewer'].source(html)
    def history_back(self, *v):
        #if self.m_viewer == 'docviewer':
        #    self.m_history.set_adj_of_current(
        #      self.box_dict['docviewer'].get_vadjustment().get_value())
        self.m_history.back()
        self.m_history.lock()
        url, adj = self.m_history.get_current()
        self.m_app.handle_href(url, display_docfile_set_adj=0)
        #if self.m_viewer == 'docviewer':
        #    self.box_dict['docviewer'].get_vadjustment().set_value(adj)
        self.m_history.unlock()
    def history_forward(self, *v):
        #if self.m_viewer == 'docviewer':
        #    self.m_history.set_adj_of_current(
        #      self.box_dict['docviewer'].get_vadjustment().get_value())
        self.m_history.forward()
        self.m_history.lock()
        url, adj = self.m_history.get_current()
        self.m_app.handle_href(url, display_docfile_set_adj=0)
        #if self.m_viewer == 'docviewer':
        #    self.box_dict['docviewer'].get_vadjustment().set_value(adj)
        #self.m_history.unlock()
    def history_reload(self, *v):
        self.m_history.lock()
        self.m_app.handle_href(self.m_history.get_current()[0])
        self.m_history.unlock()
    def initialise_exercise(self, teacher):
        """
        Create a Gui object for the exercise and add it to
        the box_dict dict.
        """
        assert teacher.m_exname not in self.box_dict
        n = utils.exercise_name_to_module_name(teacher.m_exname)
        self.box_dict[teacher.m_exname] = globals()[n].Gui(teacher, self)
        self.main_box.pack_start(self.box_dict[teacher.m_exname])
        if self.m_viewer:
            self.box_dict[self.m_viewer].hide()
        self.box_dict[n].show()
        self.m_viewer = n
    def on_key_press_event(self, widget, event):
        if event.type == gtk.gdk.KEY_PRESS:
            for s in self.m_key_bindings:
                if (event.state & (gtk.gdk.CONTROL_MASK|gtk.gdk.SHIFT_MASK|gtk.gdk.MOD1_MASK), event.keyval) \
                         == utils.parse_key_string(self.get_string(s)):
                    self.m_key_bindings[s]()
                    return 1
        self.box_dict[self.m_viewer].on_key_press_event(widget, event)
    def open_preferences_window(self, widget=None):
        if not self.g_config_window:
            self.g_config_window = ConfigWindow(self.m_app)
            self.g_config_window.show()
        else:
            self.g_config_window.show()
    def on_zoom_in(self, *v):
        self.box_dict['docviewer'].m_htmlwidget.g_view.zoom_in()
    def quit_program(self, w=None):
        if self.g_learning_tree_editor:
            self.g_learning_tree_editor.close_window()
        self.m_app.quit_program()
        gtk.main_quit()
    def display_in_musicviewer(self, music):
        if not self.g_musicviewer_window:
            self.g_musicviewer_window = MusicViewerWindow(self)
            self.g_musicviewer_window.show()
        self.g_musicviewer_window.display_music(music)
    def close_musicviewer(self, widget=None):
        self.g_musicviewer_window.destroy()
        self.g_musicviewer_window = None
    def enter_test_mode(self, *v):
        self.m_action_groups['NotExit'].set_sensitive(False)
        self.g = self.box_dict[self.m_viewer].g_notebook.get_nth_page(0)
        self.box_dict[self.m_viewer].g_notebook.get_nth_page(0).reparent(self.main_box)
        self.box_dict[self.m_viewer].g_notebook.hide()
        self.box_dict[self.m_viewer].enter_test_mode()
    def exit_test_mode(self, *v):
        self.m_app.m_test_mode = False
        self.m_action_groups['NotExit'].set_sensitive(True)
        box = gtk.VBox()
        self.box_dict[self.m_viewer].g_notebook.insert_page(box, gtk.Label(_("Practise")), 0)
        self.g.reparent(box)
        self.box_dict[self.m_viewer].g_notebook.show()
        self.box_dict[self.m_viewer].g_notebook.get_nth_page(0).show()
        self.box_dict[self.m_viewer].g_notebook.set_current_page(0)
        self.box_dict[self.m_viewer].exit_test_mode()
        # rebuild the menus in case we hav passed a test.
        self.create_practise_and_test_menu()

def handle_lesson_id_crash(lessonfile_manager):
    for lesson_id in lessonfile_manager.iterate_duplicated_lesson_id():
        if splash_win:
            splash_win.hide()
        dlg = ResolveLessonIdCrashDlg(lessonfile_manager, lesson_id)
        ret = dlg.run()
        if ret == gtk.RESPONSE_OK:
            for b in dlg.radio_dict.values():
                # Create a new lesson_id and parse them into m_uiddb
                if not b.get_active():
                    f = lessonfile.LessonIdParser(b.get_data('fn'))
                    f.new_lesson_id()
                    lessonfile_manager.parse_into_uiddb(
                        b.get_data('lessoncollection'),
                        b.get_data('lessonfile'))
                # and remove the duplicate entries from the entry we
                # want to keep.
                else:
                    lessonfile_manager.delete_not_fn(lesson_id, b.get_data('fn'))
        elif ret == gtk.RESPONSE_CANCEL:
            lessonfile_manager.ignore_duplicates_with_lesson_id(lesson_id)
        dlg.destroy()
    if splash_win:
        splash_win.show()

if splash_win:
    splash_win.show_progress("running gnome.program_init")

gnome.program_init('solfege', configureoutput.VERSION_STRING)
prog = gnome.program_get()
prog.parse_args()

import locale
locale.setlocale(locale.LC_NUMERIC, "C")

# check_rcfile has to be called before and
# functions that use the cfg module.
app.check_rcfile()

cfg.set_bool('config/no_random', bool(options.no_random))

if splash_win:
    splash_win.show_progress("creating LessonfileManager")
lessonfile_manager = lessonfile.LessonFileManager()

handle_lesson_id_crash(lessonfile_manager)
lessonfile_manager.create_lessonfile_index()

if splash_win:
    splash_win.show_progress("creating MainWin")

def start_app(prefix, datadir):
    w = MainWin(options, datadir, lessonfile_manager)
    w.show()
    w.post_constructor()
    if splash_win:
        splash_win.destroy()

    import gtk
    gtk.main()

