/* -*- Mode: c++ -*-
*******************************************************************************
*
* File:         fm_demod1.cc
*
*******************************************************************************
*/
/*
 * Copyright 2001,2002,2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

//
// This is a demo of a complete FM reception path using the microtune 4937
// cable modem tuner module as the front end.
//

#include <make_GrMC4020Source.h>
#include <GrFFTSink.h>
#include <VrFixOffset.h>
#include <GrFreqXlatingFIRfilterSCF.h>
#include <VrQuadratureDemod.h>
#include <GrFIRfilterFSF.h>
#include <VrAudioSink.h>
#include <VrConnect.h>
#include <VrMultiTask.h>
#include <VrGUI.h>
#include <gr_firdes.h>
#include <gr_fir_builderF.h>
#include <VrNullSink.h>
#include <VrFileSource.h>
#include <getopt.h>

const int inputRate = 20000000;		// input sample rate from PCI-DAS4020/12
const float FM_IF_Freq = 5.75e6;	// output freq of tuner module

const float IF_Freq = FM_IF_Freq;

// const int chanTaps = 150;
const int chanTaps = 75;
const int CFIRdecimate = 125;
const float chanGain = 2.0;


//const float fm_demodGain = 8800;
//const float fm_demodGain = 4400;	// works well
const float fm_demodGain = 2200;
//const float fm_demodGain = 1100;

const int RFIRdecimate = 5;
const int audioTaps = 50;
const float audioGain = 1.0;

const int quadRate = inputRate / CFIRdecimate;
const int audioRate = quadRate / RFIRdecimate;

static void 
usage (const char *name)
{
  cerr << "usage: " << name << " [-g] [-N]\n";
  cerr << "  -g : no gui\n";
  cerr << "  -N : use null sink instead of audio sink\n";
  exit (1);
}

int main (int argc, char **argv)
{
  try {
    
    bool use_gui_p = true;
    bool use_null_sink_p = false;
    char * use_filename = NULL;

    int c;
    while ((c = getopt (argc, argv, "gNf:")) != EOF){
      switch (c){
      case 'g':	use_gui_p = false;    break;
      case 'N': use_null_sink_p = true; break;
      case 'f': use_filename = optarg; break;
      default:
	usage (argv[0]);
      }
    }
    if (optind != argc)
      usage (argv[0]);


    float volume = 1.0;

    VrGUI *guimain = 0;
    VrGUILayout *horiz = 0;
    VrGUILayout *vert = 0;


    if (use_gui_p){
      guimain = new VrGUI(argc, argv);
      horiz = guimain->top->horizontal();
      vert = horiz->vertical();
    }

    cerr << "Input Sampling Rate: " << inputRate << endl;
    cerr << "Complex FIR decimation factor: " << CFIRdecimate << endl;
    cerr << "QuadDemod Sampling Rate: " << quadRate << endl;
    cerr << "Real FIR decimation factor: " << RFIRdecimate << endl;
    cerr << "Audio Sampling Rate: " << audioRate << endl;

    VrSource<short> *source;
    if (use_filename) {
    // --> short
    source = 
      new VrFileSource<short>(inputRate, use_filename, false);
    } else {
    source = 
      make_GrMC4020SourceS(inputRate, MCC_CH3_EN | MCC_ALL_1V);
    }

    // short --> short 
    VrFixOffset<short,short> *offset_fixer =
      new VrFixOffset<short,short>();

    // build channel filter
    //
    // note that the totally bogus transition width is because
    // we don't have enough mips right now to really do the right thing.
    // This results in a filter with 83 taps, which is just a few
    // more than the original 75 in microtune_fm_demo.
    
    vector<float> channel_coeffs =
      gr_firdes::low_pass (1.0,				// gain
			   inputRate,			// sampling rate
			   250e3,			// low-pass cutoff freq
			   8*100e3,			// width of transition band
			   gr_firdes::WIN_HAMMING);

    cerr << "Number of channel_coeffs: " << channel_coeffs.size () << endl;
  
    // short --> VrComplex
    GrFreqXlatingFIRfilterSCF *chan_filter =
      new GrFreqXlatingFIRfilterSCF (CFIRdecimate, channel_coeffs, IF_Freq);

    // VrComplex --> float
    VrQuadratureDemod<float> *fm_demod =
      new VrQuadratureDemod<float>(volume * fm_demodGain);

    // float --> short
    double width_of_transition_band = audioRate / 32;
    vector<float> audio_coeffs =
      gr_firdes::low_pass (1.0,				// gain
			   quadRate,			// sampling rate
			   audioRate/2 - width_of_transition_band, // low-pass cutoff freq
			   width_of_transition_band,
			   gr_firdes::WIN_HAMMING);

  
    cerr << "Number of audio_coeffs: " << audio_coeffs.size () << endl;
  
    GrFIRfilterFSF *audio_filter = 
      new GrFIRfilterFSF (RFIRdecimate, audio_coeffs);

    VrSink<VrComplex> *fft_sink1 = 0;
    VrSink<float> *fft_sink2 = 0;
    VrSink<short> *fft_sink3 = 0;

    if (use_gui_p){
      // sink1 is channel filter output
      fft_sink1 = new GrFFTSink<VrComplex>(vert, 50, 130, 512);

      // sink2 is fm demod output
      fft_sink2 = new GrFFTSink<float>(vert, 40, 140, 512);

      // sink3 is audio output
      fft_sink3 = new GrFFTSink<short>(horiz, 40, 140, 512);
    }

    VrSink<short> *final_sink;
    if (use_null_sink_p)
      final_sink = new VrNullSink<short>();
    else
      final_sink = new VrAudioSink<short>();

    //connect the modules together

    NWO_CONNECT (source, offset_fixer);
    NWO_CONNECT (offset_fixer, chan_filter);
    NWO_CONNECT (chan_filter, fm_demod);

    if (use_gui_p)
      NWO_CONNECT (chan_filter, fft_sink1);

    NWO_CONNECT (fm_demod, audio_filter);

    if (use_gui_p)
      NWO_CONNECT (fm_demod, fft_sink2);

    NWO_CONNECT (audio_filter, final_sink);

    if (use_gui_p)
      NWO_CONNECT (audio_filter, fft_sink3);

    VrMultiTask *m = new VrMultiTask ();
    if (use_gui_p){
      m->add (fft_sink1);
      m->add (fft_sink3);
      m->add (fft_sink2);
    }

    m->add (final_sink);

    m->start ();
    if (use_gui_p)
      guimain->start ();

    // *((int *) 0) = 1;	// cause segv
    
    while (1){
      if (use_gui_p)
	guimain->processEvents(10 /*ms*/);
      m->process();
    }	

  } // end try

  catch (std::exception &e){
    cerr << "std library exception: " << e.what () << endl;
    exit (1);
  }
  catch (...) {
    cerr << "unknown exception thrown" << endl;
    exit (1);
  }
}
