/* dap.h -- dap functions */

#include <stdio.h>
#include <math.h>

/*  Copyright (C) 2001  Susan Bassein
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/********************************************************
 * The following parameters control the operation of Dap.
 * To set their values, define a function:
 * void dap_param()
 * {
 * <set parameter(s) here>
 * }
 * This function will be called before your main is executed.
 ************************************************************
 ** IT IS SAFER TO CHANGE THESE VALUES USING dap_param().  **
 ** CHANGE THESE DEFAULT VALUES AT YOUR OWN RISK!!         **
 ************************************************************/

/* Parameters for variables */
int dap_maxvar = 256;		/* max number of variables in a dataset */
				/* if changed to >= 10,000, change dimstr in dap0.c */
int dap_namelen = 15;		/* max length of variable names (+1 for null) */
int dap_intlen = 20;		/* max number of char in char representation of int */
int dap_listlen = (256 * (16 + 6));
				/* max length of list of variables: dap_maxvar *
				 * (max var name length + room for bracketed index)
				 * This may not be entirely safe! (but most likely is)
				 */
int dap_strlen = 63;		/* max length of some string values */

/* Parameters for tables */
int dap_maxrows = 1024;		/* max rows for table() */
int dap_maxcols = 64;		/* max columns for table() */
int dap_maxclab = 128;		/* max number of column labels */
int dap_maxrowv = 8;
int dap_maxcolv = 8;
int dap_lablen = 63;		/* max number of non-null char in column label */

/* parameters for datasets */
char *dap_setdir = "dap_sets";	/* where datasets are stored */
int dap_maxval = 32768;		/* max number of values for some stat functions*/
int dap_maxcell = 512;          /* max number of cells in some internal tables */
int dap_maxtreat = 9;           /* max number of factors for ANOVA */

/* parameters for grouping */
int dap_maxbars = 128;          /* max number of bars for histograms, grouping */
int dap_maxlev = 96;            /* max number of levels of a variable */

/* Parameters for I/O */
int dap_linelen = 2047;         /* max number of char for input line (+1 for null) */

/* Parameters for graphics */
int dap_maxpts = 16384;         /* max number of points in a pict */
int dap_maxchar = 65536;        /* max number of text chars in a pict */
int dap_maxntxt = 128;
int dap_maxtxt = 127;
int dap_maxfont = 63;

/* Parameters for numerical algorithms */
double dap_redtol = 1e-9;
double dap_orthtol = 1e-9;
double dap_zerotol = 1e-6;
double dap_tol = 1e-8;
double dap_ctol = 1e-6;
double dap_ktol = 1e-6;
double dap_prtol = 1e-6;
double dap_addtozero = 1e-8;    /* for contingency tables */
int dap_maxiter = 25;           /* max number of iterations */
int dap_maxex1 = 20;          /* max number of values for exact test */
int dap_maxex2 = 20;          /* max number of values for exact test */


/* Parameters for memory files */
int dap_nrfiles = 128;          /* number of files stored in memory */
int dap_rfilesize = 16384;      /* max number of bytes in a memory file */
int dap_maxlines = 2048;        /* max number of lines in memory file:
                                 * keep at dap_rfilesize / 8
                                 */

/* Memory allocation tracing */
char *dap_memtrace = NULL;      /* if non-NULL, print trace of malloc and free
                                 * and if address = dap_memtrace, then...
                                 */
int dap_mabort = 0;          /* abort on malloc */
int dap_fabort = 0;          /* abort on free */

/* First, definitions for picture structure and functions */
#include <ps.h>

extern FILE *dap_lst;
extern FILE *dap_log;

int dap_newpart(int partv[], int npartv);
int dap_list(char *list, int listv[], int maxvars);
void dap_swap();
void dap_save();
void dap_rest();
void dap_backup();
void dap_rewind();
void dap_mark();
void dap_head(int markv[], int nmark);
void input(char varlist[]);
void infile(char fname[], char delim[]);
/* old */
int dap_ismark(int markv[], int nmark);
int dap_vardef(char varspec[], int invar);
int *dap_i(char vname[]);
int *dap_pi(char vname[]);
double *dap_d(char vname[]);
double *dap_pd(char *vname);
char *dap_s(char *vname);
/* end old */
int dap_vd(char varspec[], int invar);
void dap_dl(char varname[], double *dbl);
void dap_il(char varname[], int *i);
void dap_sl(char varname[], char *s);
int step();
int skip(int nlines);
void inset(char fname[]);
void outset(char fname[], char varlist[]);
void output();
void title(char *text);
void merge(char fname1[], char vars1[], char fname2[], char vars2[],
		char marks[], char outname[]);
void dataset(char oldname[], char newname[], char *action);
void sort(char fname[], char varlist[], char modifiers[]);
void print(char fname[], char *varlist);
void means(char fname[], char varlist[], char statlist[], char marks[]);
pict *plotmeans(char *dataset, char *meanvar, char *varlist, char *errbar,
					char *partvars, int noverlay);
void table(char fname[], char rowvars[], char colvars[], char format[], char marks[]);
pict *plot(char fname[], char xyvar[], char marks[],
		char style[], double (*xfunct)(), double (*yfunct)(), int nplots);
pict *normal(char fname[], char variable[], char marks[], int nplots);
pict *histogram(char fname[], char variable[], char marks[], int nbars,
		char style[], double (*xfunct)(), int nplots);

void pctiles(char fname[], char varlist[], char statlist[], char marks[]);
void group(char fname[], char varspec[], char marks[]);
void freq(char fname[], char varlist[], char stats[], char marks[]);
void trim(char fname[], char trimspec[], char marks[]);
void corr(char fname[], char varlist[], char marks[]);
void ftest(char fname[], char response[], char numerator[], char denominator[],
									char marks[]);
void effects(char fname[], char varlist[], char model[], char marks[]);
void lsmeans(char fname[], char method[], double alpha, char varlist[], char treat[],
                char marks[], char format[]);
void linreg(char fname[], char ylist[], char x0list[], char x1list[],
				char marks[], char xname[], double level);
pict *plotlinreg(char *fname, char *ylist, char *x1list, double level);
void logreg(char fname[], char yspec[], char x0list[], char x1list[],
				char marks[], char xname[], double level);
pict *plotlogreg(char *fname, char *yspec, char *x1list, int ngroups, double level);
void nonparam(char fname[], char variables[], char marks[]);
void categ(char *dataset, char *varlist, double (*prob)(),
			double param[], char *select, char *partvars);

double varnorm();
double varunif();
double probt(double t1, int di);
double probn(double z);
double invprobchisq(double p, int df);
double invprobf(double p, int numdf, int dendf);
double invprobn(double p);
double invprobt(double p, int in);
double probf(double f2, int id1, int id2);
double probchisq(double c, int df);
int dap_numdate(char date[]);
void dap_datenum(int n, char *d);
double dap_bincoeff(double n, double r);
double dap_maximize(double (*f)(double x[]), int nx, double x[],
					double step, double tol, char *trace);
