/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.Optional;
import org.apache.kafka.common.metadata.AbortTransactionRecord;
import org.apache.kafka.common.metadata.BeginTransactionRecord;
import org.apache.kafka.common.metadata.EndTransactionRecord;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.metrics.QuorumControllerMetrics;
import org.apache.kafka.raft.Batch;
import org.apache.kafka.raft.OffsetAndEpoch;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.snapshot.Snapshots;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.slf4j.Logger;

class OffsetControlManager {
    private final Logger log;
    private final SnapshotRegistry snapshotRegistry;
    private final QuorumControllerMetrics metrics;
    private final Time time;
    private OffsetAndEpoch currentSnapshotId;
    private String currentSnapshotName;
    private long lastCommittedOffset;
    private int lastCommittedEpoch;
    private long lastStableOffset;
    private long transactionStartOffset;
    private long nextWriteOffset;

    private OffsetControlManager(LogContext logContext, SnapshotRegistry snapshotRegistry, QuorumControllerMetrics metrics, Time time) {
        this.log = logContext.logger(OffsetControlManager.class);
        this.snapshotRegistry = snapshotRegistry;
        this.metrics = metrics;
        this.time = time;
        this.currentSnapshotId = null;
        this.currentSnapshotName = null;
        this.lastCommittedOffset = -1L;
        this.lastCommittedEpoch = -1;
        this.lastStableOffset = -1L;
        this.transactionStartOffset = -1L;
        this.nextWriteOffset = -1L;
        snapshotRegistry.getOrCreateSnapshot(-1L);
        metrics.setActive(false);
        metrics.setLastCommittedRecordOffset(-1L);
        metrics.setLastAppliedRecordOffset(-1L);
        metrics.setLastAppliedRecordTimestamp(-1L);
    }

    SnapshotRegistry snapshotRegistry() {
        return this.snapshotRegistry;
    }

    QuorumControllerMetrics metrics() {
        return this.metrics;
    }

    OffsetAndEpoch currentSnapshotId() {
        return this.currentSnapshotId;
    }

    String currentSnapshotName() {
        return this.currentSnapshotName;
    }

    long lastCommittedOffset() {
        return this.lastCommittedOffset;
    }

    int lastCommittedEpoch() {
        return this.lastCommittedEpoch;
    }

    long lastStableOffset() {
        return this.lastStableOffset;
    }

    long transactionStartOffset() {
        return this.transactionStartOffset;
    }

    long nextWriteOffset() {
        return this.nextWriteOffset;
    }

    boolean active() {
        return this.nextWriteOffset != -1L;
    }

    void activate(long newNextWriteOffset) {
        if (this.active()) {
            throw new RuntimeException("Can't activate already active OffsetControlManager.");
        }
        if (newNextWriteOffset < 0L) {
            throw new RuntimeException("Invalid negative newNextWriteOffset " + newNextWriteOffset + ".");
        }
        this.snapshotRegistry.getOrCreateSnapshot(this.lastStableOffset);
        this.nextWriteOffset = newNextWriteOffset;
        this.metrics.setActive(true);
    }

    void deactivate() {
        if (!this.active()) {
            throw new RuntimeException("Can't deactivate inactive OffsetControlManager.");
        }
        this.metrics.setActive(false);
        this.metrics.setLastAppliedRecordOffset(this.lastStableOffset);
        this.nextWriteOffset = -1L;
        if (!this.snapshotRegistry.hasSnapshot(this.lastStableOffset)) {
            throw new RuntimeException("Unable to reset to last stable offset " + this.lastStableOffset + ". No in-memory snapshot found for this offset.");
        }
        this.snapshotRegistry.revertToSnapshot(this.lastStableOffset);
    }

    void handleCommitBatch(Batch<ApiMessageAndVersion> batch) {
        this.lastCommittedOffset = batch.lastOffset();
        this.lastCommittedEpoch = batch.epoch();
        this.maybeAdvanceLastStableOffset();
        this.metrics.setLastCommittedRecordOffset(batch.lastOffset());
        if (!this.active()) {
            this.metrics.setLastAppliedRecordOffset(batch.lastOffset());
            this.metrics.setLastAppliedRecordTimestamp(batch.appendTimestamp());
        }
    }

    void handleScheduleAtomicAppend(long endOffset) {
        this.nextWriteOffset = endOffset + 1L;
        this.snapshotRegistry.getOrCreateSnapshot(endOffset);
        this.metrics.setLastAppliedRecordOffset(endOffset);
        this.metrics.setLastAppliedRecordTimestamp(this.time.milliseconds());
    }

    void maybeAdvanceLastStableOffset() {
        long newLastStableOffset = this.transactionStartOffset == -1L ? this.lastCommittedOffset : Math.min(this.transactionStartOffset - 1L, this.lastCommittedOffset);
        if (this.lastStableOffset < newLastStableOffset) {
            this.lastStableOffset = newLastStableOffset;
            this.snapshotRegistry.deleteSnapshotsUpTo(this.lastStableOffset);
            if (!this.active()) {
                this.snapshotRegistry.getOrCreateSnapshot(this.lastStableOffset);
            }
        }
    }

    void beginLoadSnapshot(OffsetAndEpoch snapshotId) {
        if (this.currentSnapshotId != null) {
            throw new RuntimeException("Can't begin reading snapshot for " + snapshotId + ", because we are already reading " + this.currentSnapshotId);
        }
        this.currentSnapshotId = snapshotId;
        this.currentSnapshotName = Snapshots.filenameFromSnapshotId((OffsetAndEpoch)snapshotId);
        this.log.info("Starting to load snapshot {}. Previous lastCommittedOffset was {}. Previous transactionStartOffset was {}.", new Object[]{this.currentSnapshotName, this.lastCommittedOffset, this.transactionStartOffset});
        this.snapshotRegistry.reset();
        this.lastCommittedOffset = -1L;
        this.lastCommittedEpoch = -1;
        this.lastStableOffset = -1L;
        this.transactionStartOffset = -1L;
        this.nextWriteOffset = -1L;
    }

    void endLoadSnapshot(long timestamp) {
        if (this.currentSnapshotId == null) {
            throw new RuntimeException("Can't end loading snapshot, because there is no current snapshot.");
        }
        this.log.info("Successfully loaded snapshot {}.", (Object)this.currentSnapshotName);
        this.snapshotRegistry.getOrCreateSnapshot(this.currentSnapshotId.offset());
        this.lastCommittedOffset = this.currentSnapshotId.offset();
        this.lastCommittedEpoch = this.currentSnapshotId.epoch();
        this.lastStableOffset = this.currentSnapshotId.offset();
        this.transactionStartOffset = -1L;
        this.nextWriteOffset = -1L;
        this.metrics.setLastCommittedRecordOffset(this.currentSnapshotId.offset());
        this.metrics.setLastAppliedRecordOffset(this.currentSnapshotId.offset());
        this.metrics.setLastAppliedRecordTimestamp(timestamp);
        this.currentSnapshotId = null;
        this.currentSnapshotName = null;
    }

    public void replay(BeginTransactionRecord message, long offset) {
        if (this.currentSnapshotId != null) {
            throw new RuntimeException("BeginTransactionRecord cannot appear within a snapshot.");
        }
        if (this.transactionStartOffset != -1L) {
            throw new RuntimeException("Can't replay a BeginTransactionRecord at " + offset + " because the transaction at " + this.transactionStartOffset + " was never closed.");
        }
        this.snapshotRegistry.getOrCreateSnapshot(offset - 1L);
        this.transactionStartOffset = offset;
        this.log.info("Replayed {} at offset {}.", (Object)message, (Object)offset);
    }

    public void replay(EndTransactionRecord message, long offset) {
        if (this.currentSnapshotId != null) {
            throw new RuntimeException("EndTransactionRecord cannot appear within a snapshot.");
        }
        if (this.transactionStartOffset == -1L) {
            throw new RuntimeException("Can't replay an EndTransactionRecord at " + offset + " because there is no open transaction.");
        }
        this.transactionStartOffset = -1L;
        this.log.info("Replayed {} at offset {}.", (Object)message, (Object)offset);
    }

    public void replay(AbortTransactionRecord message, long offset) {
        if (this.currentSnapshotId != null) {
            throw new RuntimeException("AbortTransactionRecord cannot appear within a snapshot.");
        }
        if (this.transactionStartOffset == -1L) {
            throw new RuntimeException("Can't replay an AbortTransactionRecord at " + offset + " because there is no open transaction.");
        }
        long preTransactionOffset = this.transactionStartOffset - 1L;
        this.snapshotRegistry.revertToSnapshot(preTransactionOffset);
        this.transactionStartOffset = -1L;
        this.log.info("Replayed {} at offset {}. Reverted to offset {}.", new Object[]{message, offset, preTransactionOffset});
    }

    void setNextWriteOffset(long newNextWriteOffset) {
        this.nextWriteOffset = newNextWriteOffset;
    }

    public static class Builder {
        private LogContext logContext = null;
        private SnapshotRegistry snapshotRegistry = null;
        private QuorumControllerMetrics metrics = null;
        private Time time = Time.SYSTEM;

        Builder setLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        Builder setSnapshotRegistry(SnapshotRegistry snapshotRegistry) {
            this.snapshotRegistry = snapshotRegistry;
            return this;
        }

        Builder setMetrics(QuorumControllerMetrics metrics) {
            this.metrics = metrics;
            return this;
        }

        Builder setTime(Time time) {
            this.time = time;
            return this;
        }

        public OffsetControlManager build() {
            if (this.logContext == null) {
                this.logContext = new LogContext();
            }
            if (this.snapshotRegistry == null) {
                this.snapshotRegistry = new SnapshotRegistry(this.logContext);
            }
            if (this.metrics == null) {
                this.metrics = new QuorumControllerMetrics(Optional.empty(), this.time, false);
            }
            return new OffsetControlManager(this.logContext, this.snapshotRegistry, this.metrics, this.time);
        }
    }
}

