// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/*
 * This file contains information about particular algorithm suites used
 * within the encryption SDK.  In most cases, end-users don't need to
 * manipulate this structure, but it can occasionally be needed for more
 * advanced use cases, such as writing keyrings.
 *
 * These are the Node.js specific values the AWS Encryption SDK for JavaScript
 * Algorithm Suites.
 */
import { AlgorithmSuite, AlgorithmSuiteIdentifier, MessageFormat, } from './algorithm_suites';
/* References to https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/algorithms-reference.html
 * These are the composed parameters for each algorithm suite specification for
 * for the WebCrypto environment.
 */
const nodeAlgAes128GcmIv12Tag16 = {
    id: AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-128-gcm',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const nodeAlgAes192GcmIv12Tag16 = {
    id: AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-192-gcm',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const nodeAlgAes256GcmIv12Tag16 = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const nodeAlgAes128GcmIv12Tag16HkdfSha256 = {
    id: AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-128-gcm',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha256',
    cacheSafe: true,
    commitment: 'NONE',
};
const nodeAlgAes192GcmIv12Tag16HkdfSha256 = {
    id: AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-192-gcm',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha256',
    cacheSafe: true,
    commitment: 'NONE',
};
const nodeAlgAes256GcmIv12Tag16HkdfSha256 = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha256',
    cacheSafe: true,
    commitment: 'NONE',
};
const nodeAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256 = {
    id: AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-128-gcm',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha256',
    cacheSafe: true,
    signatureCurve: 'prime256v1',
    signatureHash: 'sha256',
    commitment: 'NONE',
};
const nodeAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384 = {
    id: AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-192-gcm',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha384',
    cacheSafe: true,
    signatureCurve: 'secp384r1',
    signatureHash: 'sha384',
    commitment: 'NONE',
};
const nodeAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384 = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    messageFormat: MessageFormat.V1,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha384',
    cacheSafe: true,
    signatureCurve: 'secp384r1',
    signatureHash: 'sha384',
    commitment: 'NONE',
};
const nodeAlgAes256GcmHkdfSha512Committing = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY,
    messageFormat: MessageFormat.V2,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha512',
    cacheSafe: true,
    commitment: 'KEY',
    commitmentHash: 'sha512',
    suiteDataLength: 32,
    commitmentLength: 256,
    saltLengthBytes: 32,
};
const nodeAlgAes256GcmHkdfSha512CommittingEcdsaP384 = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    messageFormat: MessageFormat.V2,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha512',
    cacheSafe: true,
    signatureCurve: 'secp384r1',
    signatureHash: 'sha384',
    commitment: 'KEY',
    commitmentHash: 'sha512',
    suiteDataLength: 32,
    commitmentLength: 256,
    saltLengthBytes: 32,
};
const nodeAlgorithms = Object.freeze({
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16]: Object.freeze(nodeAlgAes128GcmIv12Tag16),
    [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16]: Object.freeze(nodeAlgAes192GcmIv12Tag16),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16]: Object.freeze(nodeAlgAes256GcmIv12Tag16),
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(nodeAlgAes128GcmIv12Tag16HkdfSha256),
    [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(nodeAlgAes192GcmIv12Tag16HkdfSha256),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(nodeAlgAes256GcmIv12Tag16HkdfSha256),
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256]: Object.freeze(nodeAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256),
    [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(nodeAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(nodeAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY]: Object.freeze(nodeAlgAes256GcmHkdfSha512Committing),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384]: Object.freeze(nodeAlgAes256GcmHkdfSha512CommittingEcdsaP384),
});
export class NodeAlgorithmSuite extends AlgorithmSuite {
    type = 'node';
    constructor(id) {
        super(nodeAlgorithms[id]);
        Object.setPrototypeOf(this, NodeAlgorithmSuite.prototype);
        Object.freeze(this);
    }
}
Object.freeze(NodeAlgorithmSuite.prototype);
Object.freeze(NodeAlgorithmSuite);
//# sourceMappingURL=data:application/json;base64,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