// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import BN from 'bn.js';
import { needs } from './needs';
const prime256v1 = eccDecodeCompressedPoint(new BN('FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF', 16), new BN('FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFC', 16), new BN('5AC635D8AA3A93E7B3EBBD55769886BC651D06B0CC53B0F63BCE3C3E27D2604B', 16)
// new BN('FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632551', 16)
);
const secp384r1 = eccDecodeCompressedPoint(new BN('FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFF', 16), new BN('FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFC', 16), new BN('B3312FA7E23EE7E4988E056BE3F82D19181D9C6EFE8141120314088F5013875AC656398D8A2ED19D2A85C8EDD3EC2AEF', 16)
// new BN('FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC7634D81F4372DDF581A0DB248B0A77AECEC196ACCC52973', 16)
);
export const decodeNamedCurves = Object.freeze({
    // NodeJS/OpenSSL names
    prime256v1,
    secp384r1,
    // WebCrypto/Browser names
    'P-256': prime256v1,
    'P-384': secp384r1,
});
/*
 * 1. This only works for prime curves
 * 2. This will not handle the point at infinity
 */
function eccDecodeCompressedPoint(p, a, b /*, order: BN */) {
    const zero = new BN(0);
    const one = new BN(1);
    const two = new BN(2);
    const three = new BN(3);
    const four = new BN(4);
    // # Only works for p % 4 == 3 at this time.
    // # This is the case for all currently supported algorithms.
    // # This will need to be expanded if curves which do not match this are added.
    // #  Python-ecdsa has these algorithms implemented.  Copy or reference?
    // #  https://en.wikipedia.org/wiki/Tonelli%E2%80%93Shanks_algorithm
    // #  Handbook of Applied Cryptography, algorithms 3.34 - 3.39
    needs(p.mod(four).eq(three), 'Curve not supported at this time');
    const montP = BN.mont(p);
    const redPow = p.add(one).div(four);
    const yOrderMap = {
        2: zero,
        3: one,
    };
    const compressedLength = 1 + p.bitLength() / 8;
    return function decode(compressedPoint) {
        /* Precondition: compressedPoint must be the correct length. */
        needs(compressedPoint.byteLength === compressedLength, 'Compressed point length is not correct.');
        const xBuff = compressedPoint.slice(1);
        const keyLength = xBuff.byteLength;
        const x = new BN([...xBuff]);
        const yOrder = yOrderMap[compressedPoint[0]];
        const x3 = x.pow(three).mod(p);
        const ax = a.mul(x).mod(p);
        const alpha = x3.add(ax).add(b).mod(p);
        const beta = alpha.toRed(montP).redPow(redPow).fromRed();
        if (beta.mod(two).eq(yOrder)) {
            const y = beta;
            return returnBuffer(x, y, keyLength);
        }
        else {
            const y = p.sub(beta);
            return returnBuffer(x, y, keyLength);
        }
    };
}
function returnBuffer(x, y, keyLength) {
    return new Uint8Array([
        4,
        ...x.toArray('be', keyLength),
        ...y.toArray('be', keyLength),
    ]);
}
//# sourceMappingURL=data:application/json;base64,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