module ActiveSupport
  module Testing
    module Assertions
      # Test numeric difference between the return value of an expression as a result of what is evaluated
      # in the yielded block.
      #
      #   assert_difference 'Article.count' do
      #     post :create, :article => {...}
      #   end
      #
      # An arbitrary expression is passed in and evaluated.
      #
      #   assert_difference 'assigns(:article).comments(:reload).size' do
      #     post :create, :comment => {...}
      #   end
      #
      # An arbitrary positive or negative difference can be specified. The default is +1.
      #
      #   assert_difference 'Article.count', -1 do
      #     post :delete, :id => ...
      #   end
      #
      # An array of expressions can also be passed in and evaluated.
      #
      #   assert_difference [ 'Article.count', 'Post.count' ], +2 do
      #     post :create, :article => {...}
      #   end
      #
      # A error message can be specified.
      #
      #   assert_difference 'Article.count', -1, "An Article should be destroyed" do
      #     post :delete, :id => ...
      #   end
      def assert_difference(expression, difference = 1, message = nil, &block)
        b = block.send(:binding)
        exps = Array.wrap(expression)
        before = exps.map { |e| eval(e, b) }

        yield

        exps.each_with_index do |e, i|
          error = "#{e.inspect} didn't change by #{difference}"
          error = "#{message}.\n#{error}" if message
          assert_equal(before[i] + difference, eval(e, b), error)
        end
      end

      # Assertion that the numeric result of evaluating an expression is not changed before and after
      # invoking the passed in block.
      #
      #   assert_no_difference 'Article.count' do
      #     post :create, :article => invalid_attributes
      #   end
      #
      # A error message can be specified.
      #
      #   assert_no_difference 'Article.count', "An Article should not be destroyed" do
      #     post :create, :article => invalid_attributes
      #   end
      def assert_no_difference(expression, message = nil, &block)
        assert_difference expression, 0, message, &block
      end
      
      # Test if an expression is blank. Passes if object.blank? is true.
      #
      #   assert_blank [] # => true
      def assert_blank(object)
        assert object.blank?, "#{object.inspect} is not blank"
      end
      
      # Test if an expression is not blank. Passes if object.present? is true.
      #
      #   assert_present {:data => 'x' } # => true
      def assert_present(object)
        assert object.present?, "#{object.inspect} is blank"
      end
    end
  end
end
