# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.

from twisted.trial.unittest import TestCase

import tempfile
import os
from elisa.core import config

from elisa.extern.configobj import Section

class TestConfig(TestCase):

    test_conf = "test.conf"

    def tearDown(self):
        try:
            os.unlink(self.test_conf)
        except OSError:
            pass

    def setUp(self):
        self.config = config.Config(self.test_conf)

        self.config.set_section('tests')
        self.config.set_option('test_list', ['some','list'], section='tests')
        self.config.set_option('test_int', 0, section='tests')
        self.config.set_option('test_float', 0.1, section='tests')
        self.config.set_option('test_bool_t', True, section='tests')
        self.config.set_option('test_bool_f', False, section='tests')
        self.config.set_option('test_str', 'foobar', section='tests')

    def test_malformed_config(self):
        """ Create a dumb config file containing formatting errors and
        check errors are reported.
        """
        data="""\
[general]

foo=['bar'
"""
        f = open('malformed.conf','w')
        f.write(data)
        f.close()

        self.assertRaises(Exception, config.Config, 'malformed.conf')
        os.unlink('malformed.conf')

    def test_list_option(self):
        """ Test tuple and list option types
        """
        self.assertEqual(self.config.get_option('resolution'), None)

        self.assertEqual(self.config.get_option('test_list', section='tests'),
                         ['some','list'])
        self.config.write()

    def test_int_option(self):
        """ Test integer option type
        """
        self.assertEqual(self.config.get_option('test_int', section='tests'), 0)

    def test_float_option(self):
        """ Test float option type
        """
        self.assertEqual(self.config.get_option('test_float', section='tests'), 0.1)

    def test_boolean_option(self):
        self.assertEqual(self.config.get_option('test_bool_t', section='tests'), True)
        self.assertEqual(self.config.get_option('test_bool_f', section='tests'), False)


    def test_string_option(self):
        """ Test string option type
        """
        self.assertEqual(self.config.get_option('test_str', section='tests'), 'foobar')

    def test_get_config_dir(self):
        self.assertEqual(self.config.get_config_dir(), os.getcwd())

        fd, filename = tempfile.mkstemp()
        tmp_config = config.Config(self.test_conf)
        tmp_config.write(filename)

        cfg = config.Config(filename)
        self.assertEqual(cfg.get_config_dir(), tempfile.gettempdir())

        os.close(fd)
        os.unlink(filename)

    def test_sub_section_option_saved(self):
        test_section = self.config.get_section('tests')
        self.assertTrue(isinstance(test_section, Section)) 

        self.assertFalse(test_section.has_key('foo'))
        test_section['foo'] = 'bar'

        self.assertEquals(self.config.get_option('foo', section='tests'), 'bar')

        fd, filename = tempfile.mkstemp()
        self.config.write(filename)

        new_config = config.Config(filename)
        section = new_config.get_section('tests')
        self.assertTrue(isinstance(test_section, Section))
        self.assertEquals(section['foo'], 'bar')
        self.assertEquals(new_config.get_option('foo', section='tests'), 'bar')

        os.close(fd)
        os.unlink(filename)

    def test_get_false_section(self):
        # ugly. we should not go into the internals
        self.failIf(self.config._config.get('false_section', None))

        self.assertEquals(self.config.get_option('foo',
                section='false_section', default='Yeah'), 'Yeah')

        false_section = self.config.get_section('false_section')
        self.assertEquals(false_section, None)

        self.assertEquals(self.config.get_option('foo',
                section='false_section'), None)

    def test_save_new_section(self):
        # ugly. we should not go into the internals
        self.failIf(self.config._config.get('save_section', None))

        my_new_section = {'yes' : True, 'pocahontas': 'disney'}

        self.assertEquals(self.config.get_section('save_section'), None)

        new_section = self.config.set_section('save_section', my_new_section)
        self.failUnless(isinstance(new_section, Section))

        fd, filename = tempfile.mkstemp()
        self.config.write(filename)

        new_config = config.Config(filename)
        self.assertEquals(dict(new_config.get_section('save_section')),
                            my_new_section)
        os.close(fd)
        os.unlink(filename)

    def test_set_option_for_not_existing_section(self):
        # ugly. we should not go into the internals
        self.failIf(self.config._config.get('not_exists', None))
        self.assertRaises(config.SectionNotFound,
                self.config.set_option,'yay', 'p', 'not_exists')

