/*-
 * Copyright (c) 2010 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Yorick Hardy
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _TUSB3410_H_
#define _TUSB3410_H_

#include <stdint.h>

/*
 * Note that sdcc seems to lack a method to request
 * a packed struct (it might actually be packed already).
 * So I think it is better to avoid using structs when
 * directly accessing registers in the xdata area. Using
 * structs would probably improve the code readability though.
 * (This comment refers to the endpoint registers and
 *  USB setup data area.)
 */

static volatile __xdata at 0xFF00 uint8_t	TI3410_setup_bmRequestType;
static volatile __xdata at 0xFF01 uint8_t	TI3410_setup_bRequest;
static volatile __xdata at 0xFF02 uint8_t	TI3410_setup_wValueLo;
static volatile __xdata at 0xFF03 uint8_t	TI3410_setup_wValueHi;
static volatile __xdata at 0xFF04 uint8_t	TI3410_setup_wIndexLo;
static volatile __xdata at 0xFF05 uint8_t	TI3410_setup_wIndexHi;
static volatile __xdata at 0xFF06 uint8_t	TI3410_setup_wLengthLo;
static volatile __xdata at 0xFF07 uint8_t	TI3410_setup_wLengthHi;

/* Memory configuration */

static volatile __xdata at 0xFF90 uint8_t TI3410_ROMS;
#define		TI3410_ROMS_SDW		0x01
#define		TI3410_ROMS_RSVD	0x1E
#define		TI3410_ROMS_S		0x60
#define		TI3410_ROMS_ROA		0x80

/* Watchdog timer */

static volatile __xdata at 0xFF93 uint8_t TI3410_WDCSR;
#define	TI3410_WDCSR_WDT		0x01
#define	TI3410_WDCSR_WDD		0x2E
#define	TI3410_WDCSR_WDR		0x40
#define	TI3410_WDCSR_WDD0		0x80
/* #define	TI3410_WDCSR_OFF		0x6A */
#define	TI3410_WDCSR_OFF		0x2B

/* Endpoint configuration */

/*
 * The output endpoints are host output i.e.
 * the firmware receives data
 */

static volatile __xdata at 0xFF08 uint8_t TI3410_OEPCNF_1;
static volatile __xdata at 0xFF09 uint8_t TI3410_OEPBBAX_1;
static volatile __xdata at 0xFF0A uint8_t TI3410_OEPBCTX_1;
static volatile __xdata at 0xFF0D uint8_t TI3410_OEPBBAY_1;
static volatile __xdata at 0xFF0E uint8_t TI3410_OEPBCTY_1;
static volatile __xdata at 0xFF0F uint8_t TI3410_OEPSIZXY_1;

static volatile __xdata at 0xFF10 uint8_t TI3410_OEPCNF_2;
static volatile __xdata at 0xFF11 uint8_t TI3410_OEPBBAX_2;
static volatile __xdata at 0xFF12 uint8_t TI3410_OEPBCTX_2;
static volatile __xdata at 0xFF15 uint8_t TI3410_OEPBBAY_2;
static volatile __xdata at 0xFF16 uint8_t TI3410_OEPBCTY_2;
static volatile __xdata at 0xFF17 uint8_t TI3410_OEPSIZXY_2;

static volatile __xdata at 0xFF18 uint8_t TI3410_OEPCNF_3;
static volatile __xdata at 0xFF19 uint8_t TI3410_OEPBBAX_3;
static volatile __xdata at 0xFF1A uint8_t TI3410_OEPBCTX_3;
static volatile __xdata at 0xFF1D uint8_t TI3410_OEPBBAY_3;
static volatile __xdata at 0xFF1E uint8_t TI3410_OEPBCTY_3;
static volatile __xdata at 0xFF1F uint8_t TI3410_OEPSIZXY_3;

/*
 * The input endpoints are host input i.e.
 * the firmware sends data
 */

static volatile __xdata at 0xFF48 uint8_t TI3410_IEPCNF_1;
static volatile __xdata at 0xFF49 uint8_t TI3410_IEPBBAX_1;
static volatile __xdata at 0xFF4A uint8_t TI3410_IEPBCTX_1;
static volatile __xdata at 0xFF4D uint8_t TI3410_IEPBBAY_1;
static volatile __xdata at 0xFF4E uint8_t TI3410_IEPBCTY_1;
static volatile __xdata at 0xFF4F uint8_t TI3410_IEPSIZXY_1;

static volatile __xdata at 0xFF50 uint8_t TI3410_IEPCNF_2;
static volatile __xdata at 0xFF51 uint8_t TI3410_IEPBBAX_2;
static volatile __xdata at 0xFF52 uint8_t TI3410_IEPBCTX_2;
static volatile __xdata at 0xFF55 uint8_t TI3410_IEPBBAY_2;
static volatile __xdata at 0xFF56 uint8_t TI3410_IEPBCTY_2;
static volatile __xdata at 0xFF57 uint8_t TI3410_IEPSIZXY_2;

static volatile __xdata at 0xFF58 uint8_t TI3410_IEPCNF_3;
static volatile __xdata at 0xFF59 uint8_t TI3410_IEPBBAX_3;
static volatile __xdata at 0xFF5A uint8_t TI3410_IEPBCTX_3;
static volatile __xdata at 0xFF5D uint8_t TI3410_IEPBBAY_3;
static volatile __xdata at 0xFF5E uint8_t TI3410_IEPBCTY_3;
static volatile __xdata at 0xFF5F uint8_t TI3410_IEPBSIZXY_3;

static volatile __xdata at 0xFF80 uint8_t TI3410_IEPCNFG_0;
static volatile __xdata at 0xFF81 uint8_t TI3410_IEPBCNT_0;
static volatile __xdata at 0xFF82 uint8_t TI3410_OEPCNFG_0;
static volatile __xdata at 0xFF83 uint8_t TI3410_OEPBCNT_0;

/* endpoint 0 buffers */
static volatile __xdata at 0xFEF0 uint8_t TI3410_OEP_0_BUF[8];
static volatile __xdata at 0xFEF8 uint8_t TI3410_IEP_0_BUF[8];

/* for all the endpoint registers */
#define	TI3410_EPCNF_RSV		0x03
#define	TI3410_EPCNF_USBIE		0x04
#define	TI3410_EPCNF_STALL		0x08
#define	TI3410_EPCNF_DBUF		0x10
#define	TI3410_EPCNF_TOGLE		0x20
#define	TI3410_EPCNF_ISO		0x40
#define	TI3410_EPCNF_UBME		0x80
#define	TI3410_EPBCT_C			0x7f
#define	TI3410_EPBCT_NAK		0x80

/* USB configuration */

static volatile __xdata at 0xFFFF uint8_t TI3410_FUNADR;

static volatile __xdata at 0xFFFE uint8_t TI3410_USBSTA;
#define	TI3410_USBSTA_STPOW		0x01
#define	TI3410_USBSTA_WAKEUP		0x02
#define	TI3410_USBSTA_SETUP		0x04
#define	TI3410_USBSTA_URRI		0x08
#define	TI3410_USBSTA_RSV		0x10
#define	TI3410_USBSTA_RESR		0x20
#define	TI3410_USBSTA_SUSR		0x40
#define	TI3410_USBSTA_RSTR		0x80

static volatile __xdata at 0xFFFD uint8_t TI3410_USBMSK;
#define	TI3410_USBMSK_STPOW		0x01
#define	TI3410_USBMSK_WAKEUP		0x02
#define	TI3410_USBMSK_SETUP		0x04
#define	TI3410_USBMSK_URRI		0x08
#define	TI3410_USBMSK_RSV		0x10
#define	TI3410_USBMSK_RESR		0x20
#define	TI3410_USBMSK_SUSR		0x40
#define	TI3410_USBMSK_RSTR		0x80

static volatile __xdata at 0xFFFC uint8_t TI3410_USBCTL;
#define	TI3410_USBCTL_DIR		0x01
#define	TI3410_USBCTL_SIR		0x02
#define	TI3410_USBCTL_RSV		0x0C
#define	TI3410_USBCTL_FRSTE		0x10
#define	TI3410_USBCTL_RWUP		0x20
#define	TI3410_USBCTL_IREN		0x40
#define	TI3410_USBCTL_CONT		0x80

static volatile __xdata at 0xFFFB uint8_t TI3410_MODECNFG;
#define	TI3410_MODECNFG_TXCNTL		0x01
#define	TI3410_MODECNFG_SOFTSW		0x02
#define	TI3410_MODECNFG_CLKOUTEN	0x04
#define	TI3410_MODECNFG_CLKSLCT		0x08

/* Serial number */

static volatile __xdata at 0xFFEF uint8_t TI3410_SERNUM7;
static volatile __xdata at 0xFFEE uint8_t TI3410_SERNUM6;
static volatile __xdata at 0xFFED uint8_t TI3410_SERNUM5;
static volatile __xdata at 0xFFEC uint8_t TI3410_SERNUM4;
static volatile __xdata at 0xFFEB uint8_t TI3410_SERNUM3;
static volatile __xdata at 0xFFEA uint8_t TI3410_SERNUM2;
static volatile __xdata at 0xFFE9 uint8_t TI3410_SERNUM1;
static volatile __xdata at 0xFFE8 uint8_t TI3410_SERNUM0;

/* DMA registers */

static volatile __xdata at 0xFFE0 uint8_t TI3410_DMACDR1;
#define	TI3410_DMACDR1_E		0x07
#define	TI3410_DMACDR1_TR		0x08
#define	TI3410_DMACDR1_XY		0x10
#define	TI3410_DMACDR1_CNT		0x20
#define	TI3410_DMACDR1_INE		0x40
#define	TI3410_DMACDR1_EN		0x80

static volatile __xdata at 0xFFE1 uint8_t TI3410_DMACSR1;
#define	TI3410_DMACSR1_PPKT		0x01

static volatile __xdata at 0xFFE4 uint8_t TI3410_DMACDR3;
#define	TI3410_DMACDR3_E		0x07
#define	TI3410_DMACDR3_TR		0x08
#define	TI3410_DMACDR3_XY		0x10
#define	TI3410_DMACDR3_CNT		0x20
#define	TI3410_DMACDR3_INE		0x40
#define	TI3410_DMACDR3_EN		0x80

static volatile __xdata at 0xFFE5 uint8_t TI3410_DMACSR3;
#define	TI3410_DMACSR3_OVRUN		0x01
#define	TI3410_DMACSR3_TXFT		0x02
#define	TI3410_DMACSR3_C		0x7C
#define	TI3410_DMACSR3_TEN		0x80

/* UART registers */

static volatile __xdata at 0xFFA0 uint8_t TI3410_UART_RDR;
static volatile __xdata at 0xFFA1 uint8_t TI3410_UART_TDR;

static volatile __xdata at 0xFFA2 uint8_t TI3410_UART_LCR;
#define	TI3410_UART_LCR_WLMASK		0x03
#define	TI3410_UART_LCR_WL5		0x00
#define	TI3410_UART_LCR_WL6		0x01
#define	TI3410_UART_LCR_WL7		0x02
#define	TI3410_UART_LCR_WL8		0x03
#define	TI3410_UART_LCR_STP		0x04
#define	TI3410_UART_LCR_PRTY		0x08
#define	TI3410_UART_LCR_EPRTY		0x10
#define	TI3410_UART_LCR_FPTY		0x20
#define	TI3410_UART_LCR_BRK		0x40
#define	TI3410_UART_LCR_FEN		0x80

static volatile __xdata at 0xFFA3 uint8_t TI3410_UART_FCRL;
#define	TI3410_UART_FCRL_TXOF		0x01
#define	TI3410_UART_FCRL_TXOA		0x02
#define	TI3410_UART_FCRL_CTS		0x04
#define	TI3410_UART_FCRL_DSR		0x08
#define	TI3410_UART_FCRL_RXOF		0x10
#define	TI3410_UART_FCRL_RTS		0x20
#define	TI3410_UART_FCRL_DTR		0x40
#define	TI3410_UART_FCRL_485E		0x80
#define	TI3410_UART_FCRL_RTSCTS		\
	(TI3410_UART_FCRL_RTS | TI3410_UART_FCRL_CTS)
#define	TI3410_UART_FCRL_DTRCTS		\
	(TI3410_UART_FCRL_DTR | TI3410_UART_FCRL_CTS)

static volatile __xdata at 0xFFA4 uint8_t TI3410_UART_MCR;
#define	TI3410_UART_MCR_URST		0x01
#define	TI3410_UART_MCR_RCVE		0x02
#define	TI3410_UART_MCR_LOOP		0x04
#define	TI3410_UART_MCR_DTR		0x10
#define	TI3410_UART_MCR_RTS		0x20
#define	TI3410_UART_MCR_LRI		0x40
#define	TI3410_UART_MCR_LCD		0x80

static volatile __xdata at 0xFFA5 uint8_t TI3410_UART_LSR;
#define	TI3410_UART_LSR_OVR		0x01
#define	TI3410_UART_LSR_PTE		0x02
#define	TI3410_UART_LSR_FRE		0x04
#define	TI3410_UART_LSR_BRK		0x08
#define	TI3410_UART_LSR_RxF		0x10
#define	TI3410_UART_LSR_TxE		0x20
#define	TI3410_UART_LSR_TEMT		0x40

static volatile __xdata at 0xFFA6 uint8_t TI3410_UART_MSR;
#define	TI3410_UART_MSR_DCTS		0x01
#define	TI3410_UART_MSR_DDSR		0x02
#define	TI3410_UART_MSR_TRI		0x04
#define	TI3410_UART_MSR_DCD		0x08
/*
 *  See figure 7-1 on pg 45 of tusb3410.pdf
 *  When not in loop-back mode, these bits are the modem status.
 */
#define	TI3410_UART_MSR_LCTS		0x10
#define	TI3410_UART_MSR_LDSR		0x20
#define	TI3410_UART_MSR_LRI		0x40
#define	TI3410_UART_MSR_LCD		0x80

static volatile __xdata at 0xFFA7 uint8_t TI3410_UART_DLL;
static volatile __xdata at 0xFFA8 uint8_t TI3410_UART_DLH;
static volatile __xdata at 0xFFA9 uint8_t TI3410_UART_XON;
static volatile __xdata at 0xFFAA uint8_t TI3410_UART_XOFF;

static volatile __xdata at 0xFFAB uint8_t TI3410_UART_MASK;
#define	TI3410_UART_MASK_MIE		0x01
#define	TI3410_UART_MASK_SIE		0x02
#define	TI3410_UART_MASK_TRI		0x04

/* GPIO */

static volatile __xdata at 0xFF9E uint8_t TI3410_PUR_3;
#define	TI3410_PUR_3_PIN0		0x01
#define	TI3410_PUR_3_PIN1		0x02
#define	TI3410_PUR_3_PIN3		0x08
#define	TI3410_PUR_3_PIN4		0x10

/* Interrupts */

static volatile __xdata at 0xFF92 uint8_t TI3410_VECINT;
#define	TI3410_VECINT_OE1		0x12
#define	TI3410_VECINT_OE2		0x14
#define	TI3410_VECINT_OE3		0x16
#define	TI3410_VECINT_IE1		0x22
#define	TI3410_VECINT_IE2		0x24
#define	TI3410_VECINT_IE3		0x26
#define	TI3410_VECINT_STPOW		0x30
#define	TI3410_VECINT_SETUP		0x32
#define	TI3410_VECINT_RESR		0x38
#define	TI3410_VECINT_SUSR		0x3A
#define	TI3410_VECINT_RSTR		0x3C
#define	TI3410_VECINT_I2CTXE		0x40
#define	TI3410_VECINT_I2CRXF		0x42
#define	TI3410_VECINT_IE0		0x44
#define	TI3410_VECINT_OE0		0x46
#define	TI3410_VECINT_UART_STATUS	0x50
#define	TI3410_VECINT_UART_MODEM	0x52
#define	TI3410_VECINT_UART_RXF		0x60
#define	TI3410_VECINT_UART_TXE		0x62
#define	TI3410_VECINT_DMA1		0x80
#define	TI3410_VECINT_DMA3		0x84

/* I2C */

static volatile __xdata at 0xFFF0 uint8_t TI3410_I2CSTA;
#define	TI3410_I2CSTA_SWR		0x01
#define	TI3410_I2CSTA_SRD		0x02
#define	TI3410_I2CSTA_TIE		0x04
#define	TI3410_I2CSTA_TXE		0x08
#define	TI3410_I2CSTA_QTR		0x10
#define	TI3410_I2CSTA_ERR		0x20
#define	TI3410_I2CSTA_RIE		0x40
#define	TI3410_I2CSTA_RXF		0x80

static volatile __xdata at 0xFFF3 uint8_t TI3410_I2CADR;
#define	TI3410_I2CADR_RW		0x01
#define	TI3410_I2CADR_A			0xfe

static volatile __xdata at 0xFFF2 uint8_t TI3410_I2CDAI;
static volatile __xdata at 0xFFF1 uint8_t TI3410_I2CDAO;

/* firmware data buffers from 0xF800 - 0xFEEF */

#define BULKINLOC_X	0xF800
#define BULKINLOC_X_C	0x00	/* Bits 10-3 for endpoint configuration */
#define BULKINLOC_Y	0xF840
#define BULKINLOC_Y_C	0x08	/* Bits 10-3 for endpoint configuration */
#define BULKOUTLOC_X	0xF880
#define BULKOUTLOC_X_C	0x10	/* Bits 10-3 for endpoint configuration */
#define BULKOUTLOC_Y	0xF8C0
#define BULKOUTLOC_Y_C	0x18	/* Bits 10-3 for endpoint configuration */

static volatile __xdata at BULKINLOC_X	uint8_t bulkinX[64];
static volatile __xdata at BULKINLOC_Y	uint8_t bulkinY[64];
static volatile __xdata at BULKOUTLOC_X	uint8_t bulkoutX[64];
static volatile __xdata at BULKOUTLOC_Y	uint8_t bulkoutY[64];

/* ring buffer for interrupt notification */
#define INTR_BUFFERS 128
#define INTR_BUFSIZE 8
#define INTR_BUF_LOC(i) (0x20 + (i))
static volatile __xdata at 0xF900 uint8_t intrdata[INTR_BUFFERS][INTR_BUFSIZE];

#endif /* _TUSB3410_H_ */
