/*-
 * Copyright (c) 2010 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Yorick Hardy
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>

#include "interrupts.h"
#include "requests.h"
#include "tusb3410.h"

/* indexes for ring buffer, current buffer to fill and send respectively */
static volatile uint8_t intr_buf = 0, intr_next = 0, intr_dropped = 0;

static void do_usb_interrupt();
static void restart_dma1();
static void restart_dma3();

static void
do_usb_interrupt(void)
{
	/* check for empty ring buffer */
	if (intr_next == intr_buf) return;
	/* wait until transmit buffer is available */
	if ((TI3410_IEPBCTX_2 & TI3410_EPBCT_NAK) == 0) return;
	/* set the location in XDATA for the transmission */
	TI3410_IEPBBAX_2 = INTR_BUF_LOC(intr_next);
	/* set number of dropped interrupts */
	intrdata[intr_next][1] = intr_dropped;
	intr_dropped = 0;
	/* set transmission length and ack */
	TI3410_IEPBCTX_2 = intrdata[intr_next][7];
	/* next buffer to send */
	++intr_next;
	if (intr_next > INTR_BUFFERS)
		intr_next = 0;
}

void
usb_interrupt(uint8_t type, uint8_t *info, uint8_t len)
{
	uint8_t i;

	/* don't send an interrupt if the interrupt endpoint is disabled */
	if ((TI3410_IEPCNF_2 & TI3410_EPCNF_UBME) == 0)
		return;

	/* setup (and queue) the interrupt information */
	intrdata[intr_buf][0] = type;
	intrdata[intr_buf][1] = intr_buf;
	intrdata[intr_buf][7] = len + 2;
	for(i=0; i<len; ++i)
		 intrdata[intr_buf][i+2] = info[i];

	/* move on to the next buffer to fill */
	++intr_buf;
	if (intr_buf > INTR_BUFFERS)
		intr_buf = 0;

	/* ran out of buffers: drop the oldest one */
	if (intr_buf == intr_next) {
		++intr_next;
		++intr_dropped;
	}
	if (intr_next > INTR_BUFFERS)
		intr_next = 0;

	/* interrupt buffer available so send immediately */
	if (TI3410_IEPBCTX_2 & TI3410_EPBCT_NAK)
		do_usb_interrupt();
}

static void
restart_dma1(void)
{
	/*
	 * only restart dma1 if the bulk output endpoint is enabled
	 * and if dma1 is not already enabled
	 */
	if ((TI3410_OEPCNF_1 & TI3410_EPCNF_UBME) &&
	    (TI3410_DMACDR1 & TI3410_DMACDR1_EN) == 0) {
		/* clear DMA status */
		TI3410_DMACSR1 = TI3410_DMACSR1;
		TI3410_DMACDR1 |= TI3410_DMACDR1_EN;
	}
}

static void
restart_dma3(void)
{
	/*
	 * only restart dma3 if the bulk input endpoint is enabled
	 * and if dma3 is not already enabled
	 */
	if ((TI3410_IEPCNF_1 & TI3410_EPCNF_UBME) &&
	    (TI3410_DMACDR3 & TI3410_DMACDR3_EN) == 0) {
		/* clear DMA status */
		TI3410_DMACSR3 = TI3410_DMACSR3;
		TI3410_DMACDR3 |= TI3410_DMACDR3_EN;
	}
}

void
external0_isr(void) __interrupt (0)
{
	while (TI3410_VECINT != 0) {
		/* dispatch on vecint */
		switch (TI3410_VECINT) {
		case TI3410_VECINT_IE0:
			/* continue sending setup data */
			do_control_read_data();
			break;
		case TI3410_VECINT_OE0:
			/* ack for the status stage after sending setup data */
			TI3410_OEPBCNT_0 = 0;
			break;
		case TI3410_VECINT_IE1:
			usb_interrupt(TI3410_INTR_VEC, &TI3410_VECINT, 1);
			restart_dma3();
			break;
		case TI3410_VECINT_OE1:
			usb_interrupt(TI3410_INTR_VEC, &TI3410_VECINT, 1);
			restart_dma1();
			break;
		case TI3410_VECINT_IE2:
			do_usb_interrupt();
			break;
/* These endpoints are currently unused */
#if 0
		case TI3410_VECINT_OE2:
			break;
		case TI3410_VECINT_IE3:
			break;
		case TI3410_VECINT_OE3:
			break;
#endif
/* STPOW is currently checked in handle_setup_intr */
#if 0
		case TI3410_VECINT_STPOW:
			break;
#endif
		case TI3410_VECINT_SETUP:
			handle_setup_intr();
			break;
#if 0
		case TI3410_VECINT_RESR:
			break;
		case TI3410_VECINT_SUSR:
			break;
		case TI3410_VECINT_RSTR:
			break;
		case TI3410_VECINT_I2CTXE:
			break;
		case TI3410_VECINT_I2CRXF:
			break;
#endif
		case TI3410_VECINT_UART_STATUS:
#ifdef INTERRUPT_ON_LSR
			usb_interrupt(TI3410_INTR_LSR, &TI3410_UART_LSR, 1);
#endif
			if (TI3410_UART_LSR & TI3410_UART_LSR_OVR);
				usb_interrupt(TI3410_INTR_OVERRUN, NULL, 0);
			/* clear the line status */
			TI3410_UART_LSR = TI3410_UART_LSR;
			/* restart dma3 after error */
			TI3410_DMACDR3 &= ~TI3410_DMACDR3_EN;
			restart_dma3();
			break;
		case TI3410_VECINT_UART_MODEM:
			usb_interrupt(TI3410_INTR_MSR, &TI3410_UART_MSR, 1);
			/* clear the status */
			TI3410_UART_MSR = TI3410_UART_MSR;
			break;
#ifdef INTERRUPT_ON_LSR
		case TI3410_VECINT_UART_RXF:
		case TI3410_VECINT_UART_TXE:
			if (TI3410_UART_LSR != 0)
				usb_interrupt(TI3410_INTR_LSR, &TI3410_UART_LSR,
					 1);
			break;
#endif
		case TI3410_VECINT_DMA1:
#ifdef INTERRUPT_ON_DMA
			usb_interrupt(TI3410_INTR_DMA1, &TI3410_DMACSR1, 1);
#endif
			break;
		case TI3410_VECINT_DMA3:
#ifdef INTERRUPT_ON_DMA
			usb_interrupt(TI3410_INTR_DMA3, &TI3410_DMACSR3, 1);
#endif
			/* buffer overrun: notify and discard data */
			if ((TI3410_DMACDR3 & TI3410_DMACSR3_TXFT) &&
			    (TI3410_DMACDR3 & TI3410_DMACSR3_OVRUN)) {
				usb_interrupt(TI3410_INTR_OVERRUN, NULL, 0);
				if (TI3410_DMACDR3 & TI3410_DMACDR3_XY)
					TI3410_IEPBCTY_1 = TI3410_EPBCT_NAK;
				else
					TI3410_IEPBCTX_1 = TI3410_EPBCT_NAK;
			}
			if (TI3410_DMACDR3 & TI3410_DMACDR3_XY) {
				if (TI3410_IEPBCTY_1 & TI3410_EPBCT_NAK)
					restart_dma3();
			} else {
				if (TI3410_IEPBCTX_1 & TI3410_EPBCT_NAK)
					restart_dma3();
			}
			break;
		default:
			break;
 		}

#ifdef DEBUG_INTERRUPTS
		/* sending IE2 each time results in perpetual interrupts */
		if (TI3410_VECINT != TI3410_VECINT_IE2)
			usb_interrupt(TI3410_INTR_VEC, &TI3410_VECINT, 1);
#endif

		/* check for the next interrupt event */
 		TI3410_VECINT = TI3410_VECINT;
	}
} 

void timer0_isr(void)		__interrupt (1) { } 
void external1_isr(void)	__interrupt (2) { } 
void timer1_isr(void)		__interrupt (3) { } 
void serial_isr(void)		__interrupt (4) { } 
