/*
 * qrng gets true random numbers via Quantis
 * Copyright (c) 2004, 2005, id Quantique SA, Switzerland
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of id Quantique nor the names of its contributors may be
 * used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WIN32  
#include "getopt.h"
#else
#include <unistd.h>
#endif

#include "quantis.h"


/*
* Print usage
*/
static void usage(void) {
	fprintf(stderr, "qrng - Copyright (C) 2004 id Quantique SA\n");
	fprintf(stderr, "Usage: ./qrng -n bytes [-b|-d max] [-o output]"
		"[-c] [-p cardnumber]\n");
	fprintf(stderr, "-n : amount of bytes\n");
	fprintf(stderr, "-b : dump in binary (default in ASCII hex) \n");
	fprintf(stderr, "-d : dump in decimal with max the max number"
		" (must be below 65535)\n");
	fprintf(stderr, "-o : output file (default to stdout)\n");
	fprintf(stderr, "-c : order data in one column\n");
	fprintf(stderr, "-p : PCI card number (default 0)\n");
}


/*
* Get random numbers using libquantis
*/
unsigned char *getrandomnumbers(int cardnumber, unsigned bytesnum)
{
	int cnt;

	unsigned char *buffer = (unsigned char *) malloc(bytesnum);
	if (buffer == NULL) {
		fprintf(stderr, "bytes number too big!\n");
		exit(EXIT_FAILURE);
	}

	cnt = quantisCount();

	if (!cnt || (cnt < cardnumber+1)) {
		fprintf(stderr, "card %d not found\n",cardnumber);
		exit(EXIT_FAILURE);
	}

	if (quantisBoardReset(cardnumber)){
		fprintf(stderr, "cannot reset card %d\n",cardnumber);
		exit(EXIT_FAILURE);
	}

	if (bytesnum != (unsigned) quantisRead(cardnumber, (char *) buffer, bytesnum)){
		fprintf(stderr, "an error occured when reading card %d\n", cardnumber);
		exit(EXIT_FAILURE);
	}

	return buffer; 
}


/*
* Dump data to stdout
*/
void dump(char *buffer, unsigned int bytesnum, int binary)
{
	int i;
	if (!binary) {
		printf("%s",buffer);
	}
	else {
		for (i=0; i<(int)bytesnum; i++){
			printf("%c", buffer[i]);
		}
	} 
}


/*
* Transform to unsigned decimal mode
*/
char * decmode(unsigned int bytesnum, unsigned char *buffer, int declength, int decimal,
			   char col, int *nbytes)
{
	unsigned int ch;
	int amnt, i, index;

	char *asciibuffer;
	*nbytes = (declength+1)*bytesnum+1;
	asciibuffer  = (char *) malloc((*nbytes)+1);

	if (asciibuffer == NULL) {
		fprintf(stderr, "bytes number too big!\n");
		exit(EXIT_FAILURE);
	}

	index = 0;

	for (i = 0; i < 2*(int)bytesnum; i += 2){
		ch = ((unsigned int) (buffer[i]<<8))|((unsigned int)buffer[i+1]);
		ch = (unsigned int) ((1+decimal)*(ch/(65535+1.0)));
		amnt = sprintf(&asciibuffer[index], "%u%c", (unsigned) ch, col);
		index+=amnt;
	}

	asciibuffer[index] = '\0';    
	*nbytes=index;

	return asciibuffer;   
}


/*
* Transform to ASCII hex mode
*/
char * asciimode(unsigned int bytesnum, unsigned char *buffer, char col,
				 int *nbytes)
{
	int i;
	char *asciibuffer;

	*nbytes = 3*bytesnum;

	asciibuffer = (char *) malloc(*nbytes+2);	/* + ' ' and \n */

	if (asciibuffer == NULL) {
		fprintf(stderr, "bytes number too big!\n");
		exit(EXIT_FAILURE);
	}

	for (i = 0; i < (int) bytesnum; i++){
		sprintf(&asciibuffer[3*i], "%.2x%c", (unsigned char) buffer[i], col);
	}

	if (col==' ') {
		asciibuffer[*nbytes] = '\n';
		asciibuffer[*nbytes+1] = '\0';
		(*nbytes)++;
	}
	else{
		asciibuffer[*nbytes] = '\0';
	}

	return asciibuffer;
}


/*
* Write data to a file
*/
void writetofile(char *buffer, unsigned int bytesnum, char *output)
{
	int nbytes;
	FILE *fd;
	fd = fopen(output, "w+b");
	if (fd == NULL) {
		fprintf(stderr, "unable to open output file\n");
		exit(EXIT_FAILURE);
	}

	nbytes = (int) fwrite((char *) buffer, 1, bytesnum, fd);

	if (nbytes != (int) bytesnum){ 
		printf("%d %d\n", bytesnum, nbytes);
		fprintf(stderr, "unable to write to output file\n");
		exit(EXIT_FAILURE);
	}

	fclose(fd);
}

/*
* This is an exemple tool to get random numbers via the
* Quantis PCI card using the quantis library
*/
int main(int argc, char *argv[])
{
	unsigned int bytesnum;
	int c;
	int binary, cardnumber, decimal, declength, nbytes;
	int bytesrequired;
	char col;
	char *output;
	char *buffer;

	char *printbuff;

	declength = 0;
	decimal = 0;
	cardnumber = 0;
	bytesnum = 0;
	binary= 0 ;
	col = ' ';
	output = NULL;
	buffer = NULL;

	/* Parse parameters */
	while ((c = getopt(argc, argv, "n:bo:p:d:c")) >= 0) {
		switch (c) {
	case 'n':
		bytesnum =(unsigned int) atoi(optarg);
		break;
	case 'b':
		binary = 1;
		break;
	case 'o':
 	        if (output) free(output);
		output = strdup(optarg);
		break;
	case 'p':
		cardnumber = atoi(optarg);
		break;
	case 'd':
		decimal = atoi(optarg);
		declength = (int) strlen(optarg);
		break;
	case 'c':
		col = '\n';
		break;
	default:
		usage();
		exit(EXIT_FAILURE);
		}
	}

	if ((argc == 1) || (!bytesnum) || (decimal>65535)){
		usage();
		exit(EXIT_FAILURE);
	}

	/* Get data */
	if (decimal) {bytesrequired = 2*bytesnum;}
	else {bytesrequired = bytesnum;}
	buffer = (char *) getrandomnumbers(cardnumber, bytesrequired);

	nbytes = bytesnum;

	/* Transform data */
	if (decimal) {		/* decimal mode */
		printbuff = decmode(bytesnum, (unsigned char *) buffer, declength, decimal, col, &nbytes);
	}
	else if (binary) {		/* binary mode */
		printbuff = buffer;
	}
	else {			/* ASCII hex mode */
		printbuff = asciimode(bytesnum, (unsigned char *) buffer, col, &nbytes);    
	}

	/* Write data */
	if (output){
		writetofile(printbuff, nbytes, output);
	}
	else {
		dump(printbuff, nbytes, binary);
	}

	/* Clean and exit */
	if (output!=NULL) {	// avoid the double free() if !=
		free(output);	
	}
	free(printbuff);
	if (buffer != printbuff)
	        free(buffer);

	return 0;
}
