/* quantis_drv.c */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/event.h>
#include <sys/kernel.h>
#include <sys/systm.h>

#include <dev/pci/pcivar.h>

#include "../include/quantisio.h"

unsigned int card_count = 0;
struct quantis_softc;

#define CDEV_MAJOR		144
#define QUANTIS_BAR		0x14
#define QUANTIS_SOFT_T		struct quantis_softc
#define QUANTIS_VENDOR_ID	0x179a

#if DEBUG
#define QUANTIS_DEBUG(fmt, args...) uprintf(fmt "\n", ## args)
#else
#define QUANTIS_DEBUG(fmt, args...)
#endif

#define QUANTIS_DEBUG0 QUANTIS_DEBUG
#define QUANTIS_DEBUG2 QUANTIS_DEBUG

#define QUANTIS_REG(scp, reg)                                                \
 bus_space_read_4((scp)->bt, (scp)->bh, (reg))
#define QUANTIS_SET_REG(scp, reg, val)                                       \
 bus_space_write_4((scp)->bt, (scp)->bh, (reg), (val))

#include "quantis-common.h"

struct quantis_softc {
	struct device sc_dev;
	int mapped;
	bus_space_tag_t bt;
	bus_space_handle_t bh;
	bus_size_t bs;
	unsigned char buffer[QUANTIS_FIFO_SIZE * 4];
};

#include "quantis-common.c"

static int quantis_match(struct device *, struct cfdata *, void *);
static void quantis_attach(struct device *, struct device *, void *);
static int quantis_detach(struct device*, int);

CFATTACH_DECL(quantis, sizeof(struct quantis_softc),
              quantis_match, quantis_attach, quantis_detach, NULL);
CFDRIVER_DECL(quantis, DV_TTY, NULL);

static dev_type_open	(quantisopen);
static dev_type_close	(quantisclose);
static dev_type_read	(quantisread);
static dev_type_ioctl	(quantisioctl);

static struct cdevsw quantis_cdevsw = {
	quantisopen,
	quantisclose,
	quantisread,
	nowrite,
	quantisioctl,
	nostop,
	notty,
	nopoll,
	nommap,
	nokqfilter,
	D_TTY
};

/* Match the quantis board according to the vendor id */
int
quantis_match(struct device *parent, struct cfdata *match, void *aux)
{
	struct pci_attach_args *pa = (struct pci_attach_args *)aux;

	if((PCI_VENDOR(pa->pa_id) == QUANTIS_VENDOR_ID) ||
           (PCI_VENDOR(pa->pa_id) == (QUANTIS_VENDOR_ID | 0x10000)))
		return 1;

	return 0;
}

/* The start address and the length of the register page are held in
 * the second entry (QUANTIS_BAR) of the Base Address Registers configuration
 * page. The driver maps this register for use with bus_space(9).
 * If mapping is unsuccessful the device is not made available.
 */
static void
quantis_attach(struct device *parent, struct device *self, void *aux)
{
	int bmajor = -1;
	int cmajor = CDEV_MAJOR;
	pcireg_t type;
	bus_addr_t ba;
	struct pci_attach_args *pa = aux;
	struct quantis_softc *scp = (struct quantis_softc *)self;

	type = pci_mapreg_type(pa->pa_pc, pa->pa_tag, QUANTIS_BAR);
	scp->mapped = pci_mapreg_map(pa, QUANTIS_BAR, type, 0,
                                     &(scp->bt), &(scp->bh), NULL, &(scp->bs));

	if(scp->mapped)
	{
		scp->mapped = 0;
		uprintf("quantis failed to map unit %d, "
		        "the unit will not be available.\n",
		        (int)(scp->sc_dev.dv_unit));
		return;
	}
	else scp->mapped = 1;

	quantis_rng_reset(scp);
	quantis_rng_enable_modules(scp, quantis_rng_modules_mask(scp));

	devsw_attach("quantis", NULL, &bmajor, &quantis_cdevsw, &cmajor);
	card_count++;
}

/* Detach the unit. If the device was not successfully mapped
 * on attach then nothing needs to be done.
 */
static int
quantis_detach(struct device* self, int flags)
{
	struct quantis_softc *scp = (struct quantis_softc *)self;

	if(scp->mapped)
	{
		devsw_detach(NULL, &quantis_cdevsw);
		bus_space_unmap(scp->bt, scp->bh, scp->bs);
		card_count--;
	}
	return 0;
}

/* We implement the same IOCTL calls as the linux driver, except for
 * the `QUANTIS_IOCTL_SET_DEBUG_LEVEL' call.
 */
static int
quantisioctl (dev_t dev, u_long cmd, caddr_t data, int flag, struct proc *p)
{
	struct quantis_softc *scp;

	scp = device_lookup(&quantis_cd, minor(dev));
	if(scp == NULL) return ENXIO;

	QUANTIS_DEBUG("ioctl %ld\n", cmd);

	switch (cmd) {
	case QUANTIS_IOCTL_GET_DRIVER_VERSION:
	  {
	    u_int32_t qversion = QUANTIS_DRIVER_VERSION;
	    memcpy(data, &qversion, sizeof(qversion));
	    return 0;
	  }

	case QUANTIS_IOCTL_GET_CARD_COUNT:
	  memcpy(data, &card_count, sizeof(card_count));
	  return 0;

	case QUANTIS_IOCTL_GET_MODULES_MASK:
	  {
	    u_int32_t mask = quantis_rng_modules_mask(scp);
	    memcpy(data, &mask, sizeof(mask));
	    return 0;
	  }

	case QUANTIS_IOCTL_GET_BOARD_VERSION:
	  {
	    u_int32_t qversion = quantis_rng_version(scp);
	    QUANTIS_DEBUG("version: %04x", qversion);
	    memcpy(data, &qversion, sizeof(qversion));
	    return 0;
	  }

	case QUANTIS_IOCTL_RESET_BOARD:
	  quantis_rng_reset(scp);
	  return 0;

	case QUANTIS_IOCTL_ENABLE_MODULE:
	  {
	    u_int32_t modules;
	    memcpy(&modules, data, sizeof(modules));
	    quantis_rng_enable_modules(scp, modules);
	    return 0;
	  }

	case QUANTIS_IOCTL_DISABLE_MODULE:
	  {
	    u_int32_t modules;
	    memcpy(&modules, data, sizeof(modules));
	    quantis_rng_disable_modules(scp, modules);
	    return 0;
	  }

	case QUANTIS_IOCTL_SET_TEST_MODE:
	  {
	    u_int32_t modules;
	    memcpy(&modules, data, sizeof(modules));
	    quantis_rng_set_test_mode(scp, modules);
	    return 0;
	  }

	case QUANTIS_IOCTL_GET_MODULES_STATUS:
	  {
	    u_int32_t status = quantis_rng_modules_status(scp);
	    memcpy(data, &status, sizeof(status));
	    return 0;
	  }

	case QUANTIS_IOCTL_SET_DEBUG_LEVEL:
	default:
		return ENXIO;
	}
	return 0;
}

/* When an open is attempted on the device we use device lookup
 * to determine if a device (unit) is available corresponding to
 * the unit number (character device minor number). Note it is
 * possible to have card_count == 2 with units 0 and 2 present
 * and 1 absent.
 */
static int
quantisopen(dev_t dev, int oflags, int devtype, struct proc *p)
{
	struct	quantis_softc *scp;

	scp = device_lookup(&quantis_cd, minor(dev));
	if(scp == NULL || !scp->mapped)
		return ENXIO;

	return 0;
}

static int
quantisclose(dev_t dev, int fflag, int devtype, struct proc *p)
{

	return 0;
}

/* Fill the buffer with random data and copy it to userland. */
static int
quantisread(dev_t dev, struct uio *uio, int ioflag)
{
	int	toread, len;
	struct	quantis_softc *scp;

	scp = device_lookup(&quantis_cd, minor(dev));
	if(scp == NULL) return ENXIO;

	toread = (min(uio->uio_resid, sizeof(scp->buffer)));
	QUANTIS_DEBUG("got to read %d bytes", toread);
	len = quantis_rng_read(scp, scp->buffer, toread);
	  QUANTIS_DEBUG("Wanted to read %d bytes, read %d bytes\n", toread, len);
	if (len < toread) {
	  QUANTIS_DEBUG("Could not read enough...");
	  return ENXIO;
	} else {
	  int ret = uiomove(scp->buffer, len, uio);
	  if (ret != 0)
	    return ret;
	  else
	    return 0;
	}
}
