/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/

#define FILECODE "F(303)"

/*
 * Partly derived from:
 */

/*
 * freebsd/hostif.c
 * 
 * Copyright by Vladimir N. Silyaev 1999-2000
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions 
 * are met: 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE 
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * $vmFreeBSD: vmware/vmmon-only/freebsd/hostif.c,v 1.13 2000/01/23 22:29:19 vsi
lyaev Exp $
 * 
 *
 * NetBSD modifications by Frank van der Linden (fvdl@wasabisystems.com)
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/lock.h>
#include <sys/malloc.h>
#include <sys/proc.h>

#include <machine/pio.h>

#if __NetBSD_Version__ > 105009900
#include <uvm/uvm_extern.h>
#include <uvm/uvm_param.h>
#else
#include <vm/vm.h>
#include <uvm/uvm_page.h>
extern vm_map_t kernel_map;
#endif

/* use curproc for pre-nathanw-sa world, curlwp post */
#if __NetBSD_Version__ >= 106130000
#define	CURLWP		curlwp		/* new world order */
#else
#define	CURLWP		curproc		/* old world order */
#endif

#include "x86.h"
#include "vm_types.h"
#include "vm_assert.h"
#include "vm_asm.h"
#include "modulecall.h"
#include "hostif.h"
#include "memtrack.h"
#include "phystrack.h"

#define HOST_LOCK_PFN(_vm,_pfn) {		\
	host_lock_ppn(_pfn);			\
	PhysTrack_Add(_vm->physTracker,_pfn);	\
}

#define HOST_UNLOCK_PFN(_vm,_pfn) {		\
	host_unlock_ppn(_pfn);				\
	PhysTrack_Remove(_vm->physTracker,_pfn);	\
}

#define HOST_ISTRACKED_PFN(_vm, _pfn)		\
	(PhysTrack_Test(_vm->physTracker, _pfn))

#ifdef SMP_GLOBAL_VMLOCK
static struct simplelock vmmon_lock;
static proc *vmmon_lock_task;
static int vmmon_lock_holder;
#endif

static int host_lock_ppn(PPN);
static int host_unlock_ppn(PPN);
static void UnlockEntry(void *, MemTrackEntry *);
static void * FindFunc(void *, MemTrackEntry *);
static INLINE MPN HostifVa2Mpn(VA);


static
int host_lock_ppn(PPN ppn)
{
	paddr_t paddr = (paddr_t)PPN_2_PA(ppn);
	struct vm_page *pg;

	pg = PHYS_TO_VM_PAGE(paddr);
	uvm_pagewire(pg);
	return 0;
}     

static
int host_unlock_ppn(PPN ppn)
{
	paddr_t paddr = (paddr_t)PPN_2_PA(ppn);
	struct vm_page *pg;

	pg = PHYS_TO_VM_PAGE(paddr);
	uvm_pageunwire(pg);
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostifVa2Mpn --
 *
 *    Walks through the hardware page tables of the current process to try to
 *    find the the Machine Page Number (i.e. page number in the host physical
 *    address space) associated to a virtual address.
 *
 * Results:
 *    The MPN if the page is found in the tables
 *    0 if the page is not found in the tables
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */    
static INLINE MPN
HostifVa2Mpn(VA addr)
{
	pt_entry_t *pteptr = (pt_entry_t *)vtopte((vaddr_t)addr);
	PTE pte;
   
	pte = *pteptr;
	if (pte & PTE_P)
		return PTE_2_PFN(pte);
	else
		return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_InitSpinLock --
 *
 *      Initialize the vm spin lock
 *
 * Results:
 *
 *     none
 *
 * Side effects:
 *     None
 *      
 *----------------------------------------------------------------------
 */     
void
HostIF_InitSpinLock(void)
{
#ifdef SMP_GLOBAL_VMLOCK
	simple_lock_init(&vmmon_lock);
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_Init --
 *      
 *      Initialize the host depandent part of the driver.
 * 
 * Results: 
 *
 *     zero on success, non-zero on error.
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
int
HostIF_Init(VMDriver *vm)
{  
	vm->memtracker = MemTrack_Init();
	vm->physTracker = PhysTrack_Init();
	return (vm->memtracker == NULL || vm->physTracker==NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_LookupUserMPN --
 *
 *      Lookup the MPN of a locked user page.
 *
 * Results:
 *      
 *      Returned page is a valid MPN, zero on error. 
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
MPN 
HostIF_LookupUserMPN(VMDriver *vm, char * addr)
{
	MPN mpn;
	mpn = HostifVa2Mpn((VA)addr);
	if (mpn == 0) {
		NOT_TESTED();
	}
	return mpn;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_InitFP --
 *
 *      masks IRQ13 if not previously the case.
 *
 * Results:
 *      
 *      prevents INTR #0x2d (IRQ 13) from being generated --
 *      assume that Int16 works for interrupt reporting
 *      
 *
 * Side effects:
 *      PIC
 *
 *----------------------------------------------------------------------
 */
void
HostIF_InitFP(VMDriver *vm)
{
	/*
	 * Since a working int16 means that IRQ 13 will not be used
	 * on NetBSD, this should not be needed.
	 */
#if 1
	int mask = (1<<(0xd-0x8));
	uint8 val = inb(0xA1);

	if (!(val & mask)) {
		val = val | mask;
		outb(val,0xA1);
	}
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_LockPage --
 *
 *      Lockup the MPN of an pinned user-level address space 
 *
 * Results:
 *      
 *      The MPN or zero on an error. 
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
MPN
HostIF_LockPage(VMDriver *vm, void *addr)
{
	MPN mpn;
	VPN vpn;
	MemTrackEntry *entryPtr;
	volatile int c;

	vpn = PTR_2_VPN(addr);
	entryPtr = MemTrack_LookupVPN(vm->memtracker, vpn);

	if ((entryPtr == NULL) || (entryPtr->mpn == 0)) {
		/*
		 * If the page is not in the memory tracker or it has been
		 * unlocked. 
		 */

		/*
		 * fubyte forces the page in core
		 * XXX Have we validated the pointer before? If not, we should check the
		 *     return value --hpreg
		 */
		c = fubyte(addr);

		/*
		 * now locate it and write the results back to the pmap
		 *
		 * Under extreme memory pressure, the page may be gone again.
		 * Just fail the lock in that case. (It was ASSERT_BUG(6339, mpn).)
		 * -- edward
		 */
		mpn = HostifVa2Mpn((VA)addr);
		if (mpn == 0) {
			return 0;
		}
     	 
		/*
		 * XXX SMP.
		 * XXX critical region
		 */
		if (HOST_ISTRACKED_PFN(vm, mpn)) { 
			Warning("HostIF_LockPage vpn=0x%06x mpn=%06x" 
				 "already tracked\n", vpn, mpn);
	
			return 0;
		}
      
		HOST_LOCK_PFN(vm, mpn);

		if (HostifVa2Mpn((VA)addr) != mpn) {
			/*
			 * Our page was swapped out (and possibly swapped in again at another
			 * location) between the previous HostifVa2Mpn() and HOST_LOCK_PFN().
			 * So we have just locked the wrong page. Report failure and let
			 * userland retry --hpreg
			 */

			HOST_UNLOCK_PFN(vm, mpn);

			return 0;
		}
		/*
		 * If the entry doesn't exist, add it to the memtracker
		 * otherwise we just update the mpn.
		 */
		if (entryPtr == NULL) {
			entryPtr = MemTrack_Add(vm->memtracker, vpn, mpn);
			if (entryPtr == NULL) {
				HOST_UNLOCK_PFN(vm, mpn);
				return 0;
			}
		} else {
			entryPtr->mpn = mpn;  
		}
	} else {
		/*
		 * Already locked. 
		 */

		mpn = HostifVa2Mpn((VA)addr);
		ASSERT(mpn);
		ASSERT(entryPtr->mpn == mpn);
		ASSERT(HOST_ISTRACKED_PFN(vm, mpn));
	}
	return mpn;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_UnlockPage --
 *
 *      Unlock an pinned user-level page.
 *
 * Results:
 *      0 if successful, otherwise non-zero
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

int
HostIF_UnlockPage(VMDriver *vm, void *addr)
{
	VPN vpn;
	MemTrackEntry *entryPtr;

	vpn = PTR_2_VPN(addr);
	entryPtr = MemTrack_LookupVPN(vm->memtracker, vpn);

	if ((entryPtr == NULL) || (entryPtr->mpn == 0)) {
		Warning("HostIF_UnlockPage vpn=0x%06x not tracked!\n", vpn);
		/* 
		 * dangerous, we potentially leak memory here since
		 * the use count remains high.
		 * This only happens after the watchdog runs and unwedges a 
		 * process anyway
		 */
		return 1;
	}

	HOST_UNLOCK_PFN(vm, entryPtr->mpn);
	entryPtr->mpn = 0;
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_FreeAllResources --
 *
 *      Unlock all the pages pinned for a vm. 
 *
 * Results:
 *      0 if successful, otherwise non-zero
 *      
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

static void 
UnlockEntry(void *clientData, MemTrackEntry *entryPtr)
{
	VMDriver *vm = (VMDriver *)clientData;

	if (entryPtr->mpn) {
		if (HOST_ISTRACKED_PFN(vm, entryPtr->mpn)) {
			vm->releaseCount++;
			HOST_UNLOCK_PFN(vm, entryPtr->mpn);
		} else { 
			Warning("UnlockEntry vpn=0x%06x mpn=0x%06x not owned\n",
			    entryPtr->vpn, entryPtr->mpn);
		}
		entryPtr->mpn = 0;
	}
}

int
HostIF_FreeAllResources(VMDriver *vm)
{
	vm->releaseCount = 0;
#if 0
	/* XXXX causes a panic? */
	if (vm->crossvaddr != NULL) {
		/* XXX first vsunlock? */
		uvm_km_free_wakeup(kernel_map, (vaddr_t)vm->crossvaddr,
		    PAGE_SIZE);
		uvm_vsunlock(curproc, vm->crossuaddr, PAGE_SIZE);
		vm->crossvaddr = NULL;
	}
#endif

	if (vm->memtracker) {
		MemTrack_Cleanup(vm->memtracker, UnlockEntry,vm);
		vm->memtracker = 0;
	}

	if (vm->physTracker) {
		PhysTrack_Cleanup(vm->physTracker);
		vm->physTracker = 0;
	}

	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_AllocKernelMem
 *      
 *      Allocate some kernel memory for the driver.
 *    
 * Results:
 *      The address allocated or NULL on error.
 *     
 * 
 * Side effects:
 *      memory is malloced
 *----------------------------------------------------------------------
 */

void *      
HostIF_AllocKernelMem(int size, int wired)
{
	void *ptr = malloc(size, M_DEVBUF, M_WAITOK);
   
	if (ptr==NULL) {
		Warning("HostIF_AllocKernelMem failed (size=0%x)\n",size);
	}     
      
	return ptr;
}      
      
void *
HostIF_AllocPage(int wired)
{
	void *addr;

	addr = malloc(PAGE_SIZE, M_DEVBUF, M_WAITOK);
	if (addr==0) {
		Warning("HostIF_AllocPage failed\n");
	}
	memset(addr, 0, PAGE_SIZE);
	return addr;
}

/* 
 *----------------------------------------------------------------------
 * 
 * HostIF_FreeKernelMem
 * 
 *      Free kernel memory allocated for the driver.
 * 
 * Results:
 *      None.
 * 
 * Side effects:
 *      memory is freed.
 *----------------------------------------------------------------------
 */
   
void
HostIF_FreeKernelMem(void *ptr)
{  
	free(ptr, M_DEVBUF);
}
  
 
  
void
HostIF_FreePage(void *ptr)
{
	free(ptr, M_DEVBUF);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_RealTime
 * 
 *      Read the systems real time clock.
 * 
 * Results:
 *      Current real time.
 *    
 * Side effects:
 *----------------------------------------------------------------------
 */   
      
VmTimeRealClock
HostIF_ReadTime(void)
{
	struct timeval curtime;

	microtime(&curtime);
	return (((VmTimeRealClock)curtime.tv_sec) * 1000000 + curtime.tv_usec);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_CopyFromUser
 * 
 *      Copy memory from the user application into a kernel buffer.
 * 
 * Results:
 *      0 if sucessful and non-zero or error.
 * 
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */
 
int     
HostIF_CopyFromUser(void *dst, // OUT
		    void *src, // IN
		    int len)   // IN
{  
	return copyin(src, dst, len);
}

/* 
 *----------------------------------------------------------------------
 *
 * HostIF_CopytoUser
 *
 *      Copy memory to the user application from a  kernel buffer.
 *  
 * Results:
 *      0 if sucessful and non-zero or error.
 * 
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */   
   
int
HostIF_CopyToUser(void *dst, // OUT
		  void *src, // IN
		  int len)   // IN
{
	return copyout(src, dst, len);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_UserToDriverPtr
 *
 *    Convert a user virtual address into a kernel virtual address to access
 *    the same physical memory address --hpreg
 *
 * Results:
 *    The kernel virtual address on success
 *    NULL on failure
 *
 * Side effects:
 *    XXX Store the struct page of the crosspage
 *
 *-----------------------------------------------------------------------------
 */

#ifdef DEBUG
char userpage[NBPG];
#endif

void *
HostIF_UserToDriverPtr(VMDriver *vm, // IN
		       void *addr)   // IN
{
	paddr_t paddr;
	vaddr_t kvaddr, uaddr;
#ifdef DEBUG
	int error;
#endif

	if (vm->crossvaddr != NULL)
		Warning("KernelAddr already allocated\n");

	PHOLD(CURLWP);
	uvm_vslock(curproc, addr, PAGE_SIZE,
	    VM_PROT_READ|VM_PROT_WRITE|VM_PROT_EXECUTE);

	uaddr = (vaddr_t)addr;
	kvaddr = uvm_km_valloc_wait(kernel_map, PAGE_SIZE);
	pmap_extract(vm_map_pmap(&curproc->p_vmspace->vm_map), uaddr, &paddr);
	pmap_kenter_pa(kvaddr, paddr,
	    VM_PROT_READ | VM_PROT_WRITE | VM_PROT_EXECUTE);
	PRELE(CURLWP);
	vm->crossvaddr = (void *)kvaddr;
	vm->crossuaddr = addr;
#ifdef DEBUG
	printf("cross user vaddr %p kernel vaddr %p\n", addr, (void *)kvaddr);
	printf("cross user paddr %p kernel paddr %p\n", (void *)paddr,
	    (void *)vtophys(kvaddr));
	error = copyin(addr, userpage, PAGE_SIZE);
	if (error != 0)
		panic("user cross page inaccessible\n");
	if (memcmp(userpage, (void *)kvaddr, PAGE_SIZE) != 0)
		panic("cross page doesn't compare");
#endif
	return (void *)kvaddr;
}	     

/*
 *----------------------------------------------------------------------
 *
 * HostIF_IsMPNLocked
 *
 *      Return entryPtr if MPN is locked into memory
 *
 * Results:
 *      0 if sucessful and non-zero or error.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static void *
FindFunc(void *arg, MemTrackEntry *entryPtr)
{
	MPN mpn = (MPN) arg;
   
	if (entryPtr->mpn == mpn)
		return entryPtr;
	return NULL;
}

int
HostIF_IsMPNLocked(VMDriver *vm, MPN mpn)
{
	MemTrackEntry *entry;

	entry = MemTrack_Scan(vm->memtracker,vm, FindFunc);
	return (entry != NULL);
}
 
static void *
CheckFunc(void *arg,               // IN
	  MemTrackEntry *entryPtr) // IN
{ 
	VMDriver *vm = (VMDriver *)arg;
	MPN mpn;
 
	if (entryPtr->mpn == 0)
		return NULL;
 
	mpn = HostifVa2Mpn(VPN_2_VA(entryPtr->vpn));
	if (mpn == 0) {
		/* unmapped */
		return entryPtr; 
	}
 
	if (entryPtr->mpn != mpn) {
		Warning("CheckFunc vpn 0x%x mpn 0x%x and not 0x%x \n",
		    entryPtr->vpn, mpn, entryPtr->mpn);
		vm->checkFuncFailed = TRUE;
	}
 
	return NULL;
}

/*
 *-----------------------------------------------------------------------------
 *
 * HostIF_CheckMemory
 *
 *      Make sure memory locking is OK.
 *
 *      This function runs as part of the application process, as such
 *      HostifVa2Mpn is fine
 *
 * Results:
 *      None
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
VA 
HostIF_CheckMemory(VMDriver *vm)
{  
	MemTrackEntry *entryPtr;
 
	vm->checkFuncFailed = FALSE;
	entryPtr = MemTrack_Scan(vm->memtracker,vm, CheckFunc);
   
	if (vm->checkFuncFailed)
		return 1;
	else if (entryPtr)
		return (VPN_2_VA(entryPtr->vpn));
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_LockKernel --
 *
 *      grabs the global kernel lock
 *      For the UP driver, do nothing
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      lock held
 *
 *----------------------------------------------------------------------
 */
void
HostIF_LockKernel(void)
{
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_UnLockKernel --
 *
 *      releases the global kernel lock
 *      For the UP driver, do nothing
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      lock released
 *
 *----------------------------------------------------------------------
 */
void
HostIF_UnLockKernel(void)
{
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_GlobalVMLock --
 *
 *      grabs the global data structure lock.
 *      For the UP driver, assert that the lock counter is zero and increment
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      Should be a very low contention lock.
 *
 *----------------------------------------------------------------------
 */
void
HostIF_GlobalVMLock(int callerID)
{
#ifdef SMP_GLOBAL_VMLOCK
	if (!simple_lock_try(&vmmon_lock)) {
		/*
		 * Don't rely too much on this information: imagine the following
		 * scenario:
		 *
		 * process 1               process 2
		 * ---------               --------
		 * Acquire vm_lock
		 * Set vm_lock_holder to 1
		 * Release vm_lock
		 * Acquire vm_lock
		 *                         Try to acquire vm_lock and fail, so report
		 *                         a bogus vm_lock of 1 instead of the correct 2
		 * Set vm_lock_holder to 2
		 * Release vm_lock
		 *                         Acquire vm_lock
		 *                         Set vm_lock_holder to 3
		 *                         Release vm_lock
		 *
		 *  --hpreg
		 */
		printf("vmmon: %d wait for global VM lock %d\n",
		    callerID, vm_lock_holder);
		simple_lock(&vmmon_lock);
	}
	vmmon_lock_task = curproc;
	vmmon_lock_holder = callerID;
#endif
}

/*
 *----------------------------------------------------------------------
 * 
 * HostIF_GlobalVMUnlock --
 *      
 *      releases the global data structure lock.
 *      For the UP driver, assert that the lock counter is 1 and decrement
 *
 * Results:
 *
 *      void
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */ 
void
HostIF_GlobalVMUnLock(int callerID)
{
#ifdef SMP_GLOBAL_VMLOCK
	if (vmmon_lock_holder != callerID) {
		printf("/dev/vmmon: %d releasing global VM lock %d\n",
		    callerID, vmmon_lock_holder);
	}
	vmmon_lock_task = NULL;
	vmmon_lock_holder = -1;
	simple_unlock(&vmmon_lock); 
#endif
}

MA
HostIF_APICBase(VMDriver *vm, Bool setVMPtr, Bool probe)
{
	return 0;
}

/*    
 *----------------------------------------------------------------------
 *  
 * HostIF_IOAPICBase --
 *
 *      Lookup the IO-APIC physical address
 *    
 * Results:
 * 
 *      Returned value is IO-APIC physical address
 * 
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
 
MA      
HostIF_IOAPICBase(VMDriver *vm) 
{
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * GetMSR --
 *
 *      Wrapper for the macro that calls the rdmsr instruction.
 *
 * Results:
 *      Returns the value of the MSR.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE uint64
GetMSR(int index)
{
   uint64 msr;
   msr = __GET_MSR (index);
   return msr;
}

uint64
HostIF_RDMSR(int index)
{
   return GetMSR(index);
}

void
HostIF_WRMSR(int index, uint64 val)
{
   uint32 hi = val >> 32;
   uint32 low = val & (-1UL);
   asm("wrmsr" : : "c" (index), "a" (low), "d" (hi));
}

/*
 *----------------------------------------------------------------------
 *
 * GetCPUID --
 *
 *      Wrapper for the macro that calls the cpuid instruction.
 *
 * Results:
 *      Returns the value of cpuid in regs.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE void
GetCPUID(int index, uint32 *regs)
{
   __GET_CPUID (index, regs);
}

void
HostIF_GetCPUID(int index, uint32 *regs)
{
   GetCPUID(index, regs);
}


#ifdef notyet

/*
 * Utility routines for accessing and enabling the APIC
 */

#if defined(KERNEL_2_1) || defined(USE_PERFCOUNTERS)

/*
 * Defines for accessing the APIC.  We use readl/writel to access the APIC
 * which is how Linux wants you to access I/O memory (though on the x86
 * just dereferencing a pointer works just fine).
 */
#define APICR_TO_ADDR(apic, reg)      (apic + (reg << 4))
#define GET_APIC_REG(apic, reg)       (readl(APICR_TO_ADDR(apic, reg)))
#define SET_APIC_REG(apic, reg, val)  (writel(val, APICR_TO_ADDR(apic, reg)))

#define APIC_MAXLVT(apic)             ((GET_APIC_REG(apic, APICR_VERSION) >> 16) & 0xff)
#define APIC_VERSIONREG(apic)         (GET_APIC_REG(apic, APICR_VERSION) & 0xff)

/*
 *----------------------------------------------------------------------
 *
 * SetVMAPICPtr --
 *
 *      Sets a pointer to the APIC's virtual address in the VMDriver
 *      structure.  
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The VMDriver structure is updated.
 *
 *----------------------------------------------------------------------
 */

static void
SetVMAPICPtr(VMDriver *vm, MPN mpn)
{
   volatile void *hostapic;

   hostapic = (volatile void *) ioremap_nocache(MPN_2_MA(mpn), PAGE_SIZE);
   if (hostapic) {
      if ((APIC_VERSIONREG(hostapic) & 0xF0) == 0x10) {
	 vm->hostAPIC = (volatile uint32 (*)[4]) hostapic;
      }
   }
}


/*
 *----------------------------------------------------------------------
 *
 * ProbeAPIC --
 *
 *      Find the base physical address of the APIC.  On P6 family
 *      processors, this is done by reading the address from an MSR.
 *
 * Results:
 *      Returns the physical address of the APIC if found, 0 otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

MA
ProbeAPIC(VMDriver *vm,       // IN/OUT: driver state
	  Bool setVMPtr,      // IN: set a pointer to the APIC's virtual address
	  Bool forceEnable)   // IN: APIC is being enabled here in the driver 
{
   uint64 msr;
   MPN mpn;
   CPUOem cpuOem = CPUID_GetOEM();

   if (cpuOem == CPUID_OEM_INTEL) {
      uint32 version = CPUID_GetVersion();
      uint32 features = CPUID_GetFeatures();

      if ((features & CPUID_FEATURE_COMMON_MSR) &&
	  (forceEnable || (features & CPUID_FEATURE_COMMON_APIC))) {

	 /* APIC is present and enabled */
         if (CPUID_FAMILY_IS_P6(version) || CPUID_FAMILY_IS_PENTIUM4(version)) {
	    /* P6 allows APIC relocation */
	    msr = GetMSR(APIC_BASE_MSR);
	    mpn = (msr >> 12) & 0xFFFFFF;
	    if (setVMPtr) {
	       /* obtain a system address the APIC, only for P6.
		  Not recommended for P5, per Intel Book 3, page 7-16 */
	       SetVMAPICPtr(vm, mpn);
	    } else {
	       vm->hostAPIC = NULL;
	    }
	    return MPN_2_MA(mpn);
	 } else if (!forceEnable) {
	    return APIC_DEFAULT_ADDRESS;
	 }
      }
   }
   return 0;
}
#endif


/*
 *----------------------------------------------------------------------
 *
 * HostIF_APICBase --
 *
 *      Lookup the APIC physical address
 *
 * Results:
 *      
 *      Returned value is APIC physical address
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

MA
HostIF_APICBase(VMDriver *vm, Bool setVMPtr, Bool probe)
{
#if (defined(KERNEL_2_1) && defined(__SMP__)) || defined(CONFIG_X86_UP_IOAPIC) || defined(CONFIG_X86_UP_APIC)
   VA kAddr;
   MA ma;

   if (probe) {
      ma = ProbeAPIC(vm, setVMPtr, FALSE);
      if (ma != 0) {
	 return ma;
      }
   }

   kAddr = __fix_to_virt(FIX_APIC_BASE);
   if (setVMPtr) {
      vm->hostAPIC = (void *)kAddr;
   } else {
      vm->hostAPIC = NULL;
   }

   return MPN_2_MA(HostifVa2Mpn(kAddr));
#else
   return 0;
#endif
}


#ifdef USE_PERFCOUNTERS

/* 
 * Defines for masking/unmasking PIC interrupts. 
 */
#define PIC_PRIMARY_MASK_PORT         0x21
#define PIC_SECONDARY_MASK_PORT       0xA1

#define MaskPICIRQs(mask1, mask2)        \
   mask1 = INB(PIC_PRIMARY_MASK_PORT);   \
   OUTB(PIC_PRIMARY_MASK_PORT, 0xff);    \
   mask2 = INB(PIC_SECONDARY_MASK_PORT); \
   OUTB(PIC_SECONDARY_MASK_PORT, 0xff);     

#define UnmaskPICIRQs(mask1, mask2)      \
   OUTB(PIC_PRIMARY_MASK_PORT, mask1);   \
   OUTB(PIC_SECONDARY_MASK_PORT, mask2); 


/*
 *----------------------------------------------------------------------
 *
 * HostIF_APICEnable --
 *
 *      Enable the APIC, if not already enabled.  The APIC is put into
 *      through local mode.  In this mode the APIC passes interrupts 
 *      along to the processor just as if they had come directly from the PIC.
 *
 * Results:
 *      Returned value is APIC physical address.
 *      
 * Side effects:
 *      APIC gets enabled.      
 *
 *----------------------------------------------------------------------
 */

MA 
HostIF_APICEnable(VMDriver *vm)
{
   MA ma;
   volatile void *va;
   volatile uint8 picMask1, picMask2;
   uint32 apicMSR, dummy;
   uint32 tempMSR;
   uint32 flags;

   ASSERT(vm);
   ASSERT(vm->hostAPIC == 0);

   /*
    *  Find the APIC's physical address. 
    */
   ma = ProbeAPIC(vm, FALSE, TRUE);
   if (ma == 0) {
      printk(KERN_WARNING "/dev/vmmon: Unable to find APIC's physaddr\n");
      return 0;
   }

   /*
    * Map APIC's physical address into a virtual address.
    */
   va = (volatile void *) ioremap_nocache(ma, PAGE_SIZE);
   if (va == 0) {
      printk(KERN_WARNING "/dev/vmmon: Unable to map APIC at physaddr 0x%x\n", ma);
      return 0;
   } 

   /*
    * Mask interrupts and enable the APIC.  Intel's Software Developer's Guide
    * Guide Vol III, Appendix E says that we need to mask PIC interrupts before
    * programming the local interrupt pins.
    */
   SAVE_FLAGS(flags); 
   CLEAR_INTERRUPTS();
   MaskPICIRQs(picMask1, picMask2);

   RDMSR(APIC_BASE_MSR, apicMSR, dummy);
   tempMSR = apicMSR;

   if (!(apicMSR & APIC_MSR_ENABLED)) {
      apicMSR |= APIC_MSR_ENABLED;

      /*
       * The enable bit of the APIC_BASE_MSR must be set before
       * we can even read the APIC.
       */
      WRMSR(APIC_BASE_MSR, apicMSR, dummy);
   }

   /*
    * Set up the local vector table entries.
    */
   SET_APIC_REG(va, APICR_SVR, APIC_SVR_ONES|APIC_SVR_VECTOR|APIC_SVR_APICENABLE);
   SET_APIC_REG(va, APICR_LVT0, APIC_VTE_MODE_EXTINT);
   SET_APIC_REG(va, APICR_LVT1, APIC_VTE_MODE_NMI);
   SET_APIC_REG(va, APICR_ERRLVT, APIC_VTE_MASK);
   SET_APIC_REG(va, APICR_TIMERLVT, APIC_VTE_MASK);

   UnmaskPICIRQs(picMask1, picMask2);
   RESTORE_FLAGS(flags); 

   vm->hostAPIC =  (volatile uint32 (*)[4]) va;
   return ma;
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_APICSetPerfCtr --
 *
 *      Set the performance counter entry in the APIC. 
 *
 * Results:
 *      TRUE if successful, FALSE otherwise.
 *      
 * Side effects:
 *      APIC's PCLVT register is written.
 *
 *----------------------------------------------------------------------
 */

Bool
HostIF_APICSetPerfCtr(VMDriver *vm,   // IN: driver state
		      int gate)       // IN: interrupt gate for perfctr interrupts
{
   volatile void *apic;
   uint32 maxlvt;


   if (vm->hostAPIC == 0) {
      printk(KERN_WARNING "PerfCtr: Could not set PCLVT, APIC not available\n");
      return FALSE;
   }

   apic = (volatile void *) vm->hostAPIC;
   maxlvt = APIC_MAXLVT(apic);

   /*
    * We need maximum LVT entries to be >= 4 in order for there to be
    * an entry for the performance counters.
    */
   if (maxlvt >= 4) {
      SET_APIC_REG(apic, APICR_PCLVT, APIC_VTE_MODE_FIXED | gate);
   } else {
      printk(KERN_WARNING "PerfCtr: Could not set PCLVT, APIC maxlvt %d < 4", maxlvt);
      return FALSE;
   }

   return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_APICAck --
 *
 *      Send an ack to the APIC
 *
 * Results:
 *      None.
 *      
 * Side effects:
 *      APIC's EOI register is written.
 *
 *----------------------------------------------------------------------
 */

void 
HostIF_APICAck(VMDriver *vm)
{
   volatile void *apic;
   volatile uint32 dummy;

   apic = (volatile void *) vm->hostAPIC;

   /*
    * Send the APIC an EOI.  Dummy read is copied from Linux 
    * ack_APIC_irq() routine. 
    */
   if (apic) {
      dummy = GET_APIC_REG(apic, APICR_SVR);
      SET_APIC_REG(apic, APICR_EOI, 0);
   }
}

#endif // ifdef USE_PERFCOUNTERS


/*
 *----------------------------------------------------------------------
 *
 * HostIF_IOAPICBase --
 *
 *      Lookup the IO-APIC physical address
 *
 * Results:
 *      
 *      Returned value is IO-APIC physical address
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
#define PCI_CONF_ADDR_REG_PORT   0xcf8
#define PCI_CONF_DATA_REG_PORT   0xcfc
#define VMWARE__PCI_DEVICE_ID_INTEL_82371SB 0x70008086
#define VMWARE__PCI_DEVICE_ID_INTEL_82371AB 0x71108086

MA
HostIF_IOAPICBase(VMDriver *vm)
{
#if (defined(KERNEL_2_1) && defined(__SMP__)) || defined(CONFIG_X86_UP_IOAPIC)
   int bus, slot, confword;
   uint32 id, apicbase;

   bus = 0;
   for (slot = 0; slot < 32; slot++) {
      confword = 0x80000000 | (bus << 16) |  (slot << 11);
      OUT32(PCI_CONF_ADDR_REG_PORT, confword);
      id = IN32(PCI_CONF_DATA_REG_PORT);
      if ((id == VMWARE__PCI_DEVICE_ID_INTEL_82371SB) ||
	  (id == VMWARE__PCI_DEVICE_ID_INTEL_82371AB)) {
#ifdef VMX86_DEVEL
	 Log("IOAPIC: detected %s on PCI bus %d, slot %d\n",
	     (id == VMWARE__PCI_DEVICE_ID_INTEL_82371SB) ? "PIIX3" : "PIIX4",
	     bus, slot);
#endif
	 confword |= 20; /* register 80 */
	 OUT32(PCI_CONF_ADDR_REG_PORT, confword);
	 apicbase = IN32(PCI_CONF_DATA_REG_PORT);
	 apicbase = apicbase & 0x3f;
	 return (IOAPIC_DEFAULT_ADDRESS | (apicbase << 10));
      }
   }
   
   return MPN_2_MA(HostifVa2Mpn(__fix_to_virt(VMWARE__FIX_IO_APIC_BASE)));
#else
   return 0;
#endif
}

#endif /* notyet */
