// -*- mode: c++; indent-tabs-mode: nil; -*-
#include "qscreenwsfb_qws.h"

#ifndef QT_NO_QWS_WSFB

#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#include <dev/wscons/wsconsio.h>

#include <QtCore/qstringlist.h>
#include "qwindowsystem_qws.h"

QT_BEGIN_NAMESPACE


class QWSFbScreenPrivate : public QObject
{
public:
    QWSFbScreenPrivate();
    ~QWSFbScreenPrivate();

    int fd;

    QString ttyDevice;
    QString displaySpec;
};


QWSFbScreenPrivate::QWSFbScreenPrivate()
  : fd(-1)
{
}

QWSFbScreenPrivate::~QWSFbScreenPrivate()
{
}



/*!
    \fn QWSFbScreen::QWSFbScreen(int displayId)

    Constructs a QWSFbScreen object. The \a displayId argument
    identifies the \l{Qt for Embedded Linux} server to connect to.
*/

QWSFbScreen::QWSFbScreen(int display_id)
  : QScreen(display_id, CustomClass),
    d_ptr(new QWSFbScreenPrivate)
{
}


/*!
    Destroys this QWSFbScreen object.
*/

QWSFbScreen::~QWSFbScreen()
{
    delete d_ptr;
}

/*!
    \reimp

    This is called by \l{Qt for Embedded Linux} clients to map in the
    framebuffer.  It should be reimplemented by accelerated drivers to
    map in graphics card registers; those drivers should then call
    this function in order to set up offscreen memory management. The
    device is specified in \a displaySpec; e.g. "/dev/ttyE4".

    \sa disconnect()
*/

bool QWSFbScreen::connect(const QString &displaySpec)
{
    d_ptr->displaySpec = displaySpec;

    const QStringList args = displaySpec.split(QLatin1Char(':'));

    QString dev(QLatin1String("/dev/ttyE1")); // XXX: hardcode for now

    d_ptr->fd = open(dev.toLatin1().constData(), O_RDWR);
    if (d_ptr->fd < 0) {
        perror("QWSFbScreen::connect");
        qCritical("Error opening framebuffer device %s", qPrintable(dev));
        return false;
    }

    struct wsdisplay_fbinfo fbi;
    if (ioctl(d_ptr->fd, WSDISPLAYIO_GINFO, &fbi) < 0) {
        perror("QWSFbScreen::connect");
        qWarning("Error reading framebuffer information");
        return false;
    }

    unsigned int mode = WSDISPLAYIO_MODE_MAPPED;
    if (ioctl(d_ptr->fd, WSDISPLAYIO_SMODE, &mode) < 0) {
        perror("QWSFbScreen::connect");
        qCritical("Error switching to mapped mode");
        return false;
    }


    // Fill this->QScreen::foo protected memebers

    grayscale = false;          // XXX

    dw = fbi.width;
    dh = fbi.height;
    d = fbi.depth;

    physWidth = (int)(dw * 25.4 / 90.0); // XXX: hardcode 90 DPI
    physHeight = (int)(dh * 25.4 / 90.0);

    w = dw;
    h = dh;

    unsigned int stride;
    if (ioctl(d_ptr->fd, WSDISPLAYIO_LINEBYTES, &stride) < 0) {
        perror("QWSFbScreen::connect");
        qWarning("Error reading stride size");
        return false;
    }
    lstep = stride;

    if (fbi.depth <= 8) {       // XXX: not tested
        struct wsdisplay_cmap cm;
        u_char r[256], g[256], b[256];

        memset(r, 0, sizeof(r));
        memset(g, 0, sizeof(g));
        memset(b, 0, sizeof(b));

        cm.red = r;
        cm.green = g;
        cm.blue = b;

        cm.index = 0;
        cm.count = fbi.cmsize;

        if (ioctl(d_ptr->fd, WSDISPLAYIO_GETCMAP, &cm) < 0) {
            perror("QWSFbScreen::connect");
            qWarning("Error reading colormap information");
            return false;
        }

        screencols = cm.count;
        for (unsigned int i = 0; i < 256; ++i) {
            if (i < cm.count) {
                screenclut[i] = qRgb(cm.red[i], cm.green[i], cm.blue[i]);
            }
            else {
                screenclut[i] = qRgb(0, 0, 0);
            }
        }
    }

    size = stride * dh;         // visible part
    mapsize = size;             // we don't map any extra memory in dumb mode
    data = (uchar *)mmap(NULL, mapsize,
                         PROT_READ | PROT_WRITE,
                         MAP_FILE | MAP_SHARED,
                         d_ptr->fd, 0);
    if (data == MAP_FAILED) {
        perror("QWSFbScreen::connect");
        qWarning("Error: failed to map framebuffer device to memory.");
        return false;
    }

    return true;
}

/*!
    \reimp

    This unmaps the framebuffer.

    \sa connect()
*/

void QWSFbScreen::disconnect()
{
    if (data != NULL) {
        munmap(data, mapsize);
    }

    if (d_ptr->fd != -1) {
        unsigned int mode = WSDISPLAYIO_MODE_EMUL; // XXX
        if (ioctl(d_ptr->fd, WSDISPLAYIO_SMODE, &mode) < 0) {
            perror("QWSFbScreen::disconnect");
            qWarning("Error restoring mode");
            return;
        }

        close(d_ptr->fd);
        d_ptr->fd = -1;
    }

    return;
}

/*!
    \reimp

    This is called by the \l{Qt for Embedded Linux} server at startup time.
    It turns off console blinking, sets up the color palette, enables write
    combining on the framebuffer and initialises the off-screen memory
    manager.
*/

bool QWSFbScreen::initDevice()
{
#ifndef QT_NO_QWS_KBD_WSKBD
    qwsServer->setDefaultKeyboard("WSKbd");
#endif

#ifndef QT_NO_QWS_MOUSE_WSMOUSE
    qwsServer->setDefaultMouse("WSMouse");
#endif

#ifndef QT_NO_QWS_CURSOR
    // XXX: TODO: use wscons cursor if available?
    // qt_screencursor = QWSFbCursor;
    QScreenCursor::initSoftwareCursor();
#endif

    return true;
}

/*!
    \reimp

    This is called by the \l{Qt for Embedded Linux} server when it shuts
    down, and should be inherited if you need to do any card-specific cleanup.
    The default version hides the screen cursor and reenables the blinking
    cursor and screen blanking.
*/

void QWSFbScreen::shutdownDevice()
{
    return;
}

/*!
    \reimp

    Sets the framebuffer to a new resolution and bit depth. The width is
    in \a nw, the height is in \a nh, and the depth is in \a nd. After
    doing this any currently-existing paint engines will be invalid and the
    screen should be completely redrawn. In a multiple-process
    Embedded Qt situation you must signal all other applications to
    call setMode() to the same mode and redraw.
*/

void QWSFbScreen::setMode(int nw, int nh, int nd)
{
    Q_UNUSED(nw);
    Q_UNUSED(nh);
    Q_UNUSED(nd);
}

QT_END_NAMESPACE

#endif // QT_NO_QWS_WSFB
