/*	$NetBSD: $ */

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/kernel.h>

#include <dev/ofw/openfirm.h>

#include <machine/autoconf.h>
#include <sys/bus.h>
#include <machine/bswap.h>

#include <dev/ata/atareg.h>
#include <dev/ata/atavar.h>
#include <dev/ic/wdcvar.h>

#include <arch/powerpc/mpc5200/mpc5200reg.h>
#include <arch/powerpc/mpc5200/mpc5200var.h>

static int ide_match(device_t, cfdata_t, void *);
static void ide_attach(device_t, device_t, void *);
static int ide_dma_init(void *, int channel, int drive,
			void *databuf, size_t datalen, int flags);
static void ide_piobm_start(void *, int channel, int drive, int skip,
			    int xferlen, int flags);
static int ide_dma_finish(void *, int channel, int drive, int force);
static void ide_piobm_done(void *, int channel, int drive);
static void ide_dataout(struct ata_channel *, int, void *, size_t);
static void ide_datain(struct ata_channel *, int, void *, size_t);

struct ide_softc {
	struct wdc_softc	sc_wdcdev;	/* common wdc definitions */
	struct ata_channel	*sc_chanarray[1]; /* channels definition */
	struct ata_channel	sc_channel; /* generic part */
	struct ata_queue	sc_chqueue; /* channel queue */
	struct wdc_regs		sc_wdc_regs;

	void			*sc_ih;	/* irq handler */
	int			sc_irq;
};

CFATTACH_DECL_NEW(mpcide, sizeof(struct ide_softc),
	ide_match, ide_attach, NULL, NULL);

static int
ide_match(device_t parent, cfdata_t cf, void *aux)
{
	struct confargs *ca = aux;
	char name[32];

	if (strcmp(ca->ca_name, "ata") != 0)
		return 0;

	OF_getprop(ca->ca_node, "compatible", name, sizeof(name));
	if (strcmp(name, "mpc5200-ata") != 0)
		return 0;

	return 1;
}

static void
ide_attach(device_t parent, device_t self, void *aux)
{
	struct ide_softc *sc = device_private(self);
	struct confargs *ca = aux;
	struct ata_channel *cp;
	struct wdc_regs *wdr;
	struct reg {
		uint32_t addr;
		uint32_t size;
	} range, *rp = &range;
	u_int32_t interrupts[3];
	int i;

	aprint_normal("\n");

	sc->sc_wdcdev.sc_atac.atac_dev = self;
	sc->sc_wdcdev.regs = &sc->sc_wdc_regs;

	if (OF_getprop(ca->ca_node, "reg", rp, sizeof(range))
	    != sizeof(range))
		return;

	/* Fill in wdc and channel infos */
	sc->sc_wdcdev.sc_atac.atac_cap = ATAC_CAP_DATA16 /*| ATAC_CAP_PIOBM*/;
	sc->sc_wdcdev.sc_atac.atac_pio_cap = 4;
	sc->sc_wdcdev.sc_atac.atac_channels = sc->sc_chanarray;
	sc->sc_wdcdev.sc_atac.atac_nchannels = 1;
	
	/* DMA control functions */
	sc->sc_wdcdev.dma_arg = sc;
	sc->sc_wdcdev.dma_init = ide_dma_init;
	sc->sc_wdcdev.piobm_start = ide_piobm_start;
	sc->sc_wdcdev.dma_finish = ide_dma_finish;
	sc->sc_wdcdev.piobm_done = ide_piobm_done;
	sc->sc_wdcdev.datain_pio = ide_datain;
	sc->sc_wdcdev.dataout_pio = ide_dataout;

	sc->sc_chanarray[0] = &sc->sc_channel;
	cp = &sc->sc_channel;
	wdr = &sc->sc_wdc_regs;

	cp->ch_channel = 0;
	cp->ch_atac = &sc->sc_wdcdev.sc_atac;
	cp->ch_queue = &sc->sc_chqueue;
	cp->ch_ndrive = 1;
	wdr->cmd_iot = wdr->ctl_iot = ca->ca_tag;
	if (bus_space_map(wdr->cmd_iot, rp->addr, rp->size, 0, &wdr->cmd_baseioh) != 0) {
		aprint_normal_dev(self, "map failed\n");
		return;
	}

	for (i = 0; i < WDC_NREG; i++) {
		if (bus_space_subregion(wdr->cmd_iot, wdr->cmd_baseioh,
				(i * 4) + 0x60, 4, &wdr->cmd_iohs[i]) != 0) {
			bus_space_unmap(wdr->cmd_iot, wdr->cmd_baseioh, rp->size);
			aprint_normal_dev(self, "subregion failed\n");
			return;
		}
	}
	wdc_init_shadow_regs(cp);

	if (bus_space_subregion(wdr->ctl_iot, wdr->cmd_baseioh, 0x5c, 4,
				&wdr->ctl_ioh) != 0) {
		bus_space_unmap(wdr->cmd_iot, wdr->cmd_baseioh, rp->size);
		aprint_normal_dev(self, "subregion failed\n");
		return;
	}

	if (OF_getprop(ca->ca_node, "interrupts", interrupts, sizeof(interrupts))
	    != sizeof(interrupts))
		return;

	sc->sc_irq = (interrupts[0] * 32) + interrupts[1];
	sc->sc_ih = intr_establish(sc->sc_irq, IST_LEVEL, IPL_BIO, wdcintr, cp);

	aprint_normal_dev(self, "PIO1 %08x PIO2 %08x DMA1 %08x DMA2 %08x\n",
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_PIO1),
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_PIO2),
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_DMA1),
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_DMA2));
	aprint_normal_dev(self, "UDMA1 %08x UDMA2 %08x UDMA3 %08x UDMA4 %08x UDMA5 %08x\n",
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_UDMA1),
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_UDMA2),
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_UDMA3),
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_UDMA4),
			  bus_space_read_4(wdr->cmd_iot, wdr->cmd_baseioh,
					   ATA_UDMA5));

	bus_space_write_4(wdr->cmd_iot, wdr->cmd_baseioh, ATA_CONFIG,
			  ATA_CONFIG_SMR | ATA_CONFIG_FR |
			  ATA_CONFIG_IE | ATA_CONFIG_IORDY);
	delay(10);
	bus_space_write_4(wdr->cmd_iot, wdr->cmd_baseioh, ATA_CONFIG,
			  ATA_CONFIG_IE | ATA_CONFIG_IORDY);
	wdcattach(cp);
}

/*
 * map DMA buffer
 */
int
ide_dma_init(void *v, int channel, int drive, void *databuf,
	     size_t datalen, int flags)
{
	struct ide_softc *sc = v;
#if 0
	int error;
#endif

	aprint_normal_dev(sc->sc_wdcdev.sc_atac.atac_dev,
			  "ide_dma_init ch%d dv%d len %d\n", channel, drive,
			  datalen);
	KASSERT(channel == 0);
	KASSERT(flags & (WDC_DMA_PIOBM_ATA | WDC_DMA_PIOBM_ATAPI));
#if 0
	/* use PIO for short transfer */
	if (datalen < 64 /* needs tune */) {
		DPRINTF(("%s: njata32_dma_init: short transfer (%u)\n",
		    NJATA32NAME(sc), (unsigned)datalen));
		bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc),
		    NJATA32_REG_TIMING, sc->sc_timing_pio);
		return EINVAL;
	}

	/* use PIO for unaligned transfer (word alignment seems OK) */
	if (((uintptr_t)databuf & 1) || (datalen & 1)) {
		DPRINTF(("%s: njata32_dma_init: unaligned: buf %p, len %u\n",
		    NJATA32NAME(sc), databuf, (unsigned)datalen));
		return EINVAL;
	}

	DPRINTF(("%s: njata32_dma_init: %s: databuf %p, datalen %u\n",
	    NJATA32NAME(sc), (flags & WDC_DMA_READ) ? "read" : "write",
	    databuf, (unsigned)datalen));

	error = bus_dmamap_load(sc->sc_dmat, dev->d_dmamap_xfer,
	    databuf, datalen, NULL, BUS_DMA_NOWAIT | BUS_DMA_STREAMING |
	    ((flags & WDC_DMA_READ) ? BUS_DMA_READ : BUS_DMA_WRITE));
	if (error) {
		aprint_error_dev(self, "load xfer failed, error %d\n", error);
		return error;
	}

	bus_dmamap_sync(sc->sc_dmat, dev->d_dmamap_xfer, 0,
	    dev->d_dmamap_xfer->dm_mapsize,
	    (flags & WDC_DMA_READ) ?
		BUS_DMASYNC_PREREAD : BUS_DMASYNC_PREWRITE);

	dev->d_flags =
	    ((flags & WDC_DMA_READ) ? NJATA32_DEV_DMA_READ : 0) |
	    ((flags & WDC_DMA_PIOBM_ATAPI) ? NJATA32_DEV_DMA_ATAPI : 0) |
	    NJATA32_DEV_DMA_MAPPED;
#endif
	return 0;
}

/*
 * start DMA
 *
 * top:  databuf + skip
 * size: xferlen
 */
void
ide_piobm_start(void *v, int channel, int drive, int skip, int xferlen,
		int flags)
{
	struct ide_softc *sc = v;
#if 0
	int i, nsegs, seglen;
	uint8_t bmreg;
#endif
	aprint_normal_dev(sc->sc_wdcdev.sc_atac.atac_dev,
			  "ide_piobm_start: ch%d, dv%d, skip %d, xferlen %d\n",
			  channel, drive, skip, xferlen);

	KASSERT(channel == 0);

#if 0
	/*
	 * create scatter/gather table
	 * XXX this code may be slow
	 */
	for (i = nsegs = 0;
	    i < dev->d_dmamap_xfer->dm_nsegs && xferlen > 0; i++) {
		if (dev->d_dmamap_xfer->dm_segs[i].ds_len <= skip) {
			skip -= dev->d_dmamap_xfer->dm_segs[i].ds_len;
			continue;
		}

		seglen = dev->d_dmamap_xfer->dm_segs[i].ds_len - skip;
		if (seglen > xferlen)
			seglen = xferlen;

		dev->d_sgt[nsegs].sg_addr =
		    htole32(dev->d_dmamap_xfer->dm_segs[i].ds_addr + skip);
		dev->d_sgt[nsegs].sg_len = htole32(seglen);

		xferlen -= seglen;
		nsegs++;
		skip = 0;
	}
	sc->sc_piobm_nsegs = nsegs;
	/* end mark */
	dev->d_sgt[nsegs - 1].sg_len |= htole32(NJATA32_SGT_ENDMARK);

#ifdef DIAGNOSTIC
	if (xferlen)
		panic("%s: njata32_piobm_start: xferlen residue %d\n",
		    NJATA32NAME(sc), xferlen);
#endif

	bus_dmamap_sync(sc->sc_dmat, sc->sc_dmamap_sgt,
	    (char *)dev->d_sgt - (char *)sc->sc_sgtpg,
	    sizeof(struct njata32_sgtable) * nsegs,
	    BUS_DMASYNC_PREWRITE);

	/* set timing for PIO */
	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc),
	    NJATA32_REG_TIMING, sc->sc_timing_pio);

	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc), NJATA32_REG_IOBM,
	    NJATA32_IOBM_DEFAULT);
	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc), NJATA32_REG_AS,
	    NJATA32_AS_WAIT0);

	/*
	 * interrupt configuration
	 */
	if ((dev->d_flags & (NJATA32_DEV_DMA_READ | NJATA32_DEV_DMA_ATAPI)) ==
	    NJATA32_DEV_DMA_READ) {
		/*
		 * ATA piobm read is executed while device interrupt is active,
		 * so disable device interrupt here
		 */
		bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc),
		    NJATA32_REG_IRQ_SELECT, NJATA32_IRQ_XFER);
	}

	/* enable scatter/gather busmaster transfer */
	bmreg = NJATA32_BM_EN | NJATA32_BM_SG | NJATA32_BM_WAIT0 |
	    ((dev->d_flags & NJATA32_DEV_DMA_READ) ? NJATA32_BM_RD : 0);
	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc), NJATA32_REG_BM,
	    bmreg);

	/* load scatter/gather table */
	bus_space_write_4(NJATA32_REGT(sc), NJATA32_REGH(sc),
	    NJATA32_REG_DMAADDR, dev->d_sgt_dma);
	bus_space_write_4(NJATA32_REGT(sc), NJATA32_REGH(sc),
	    NJATA32_REG_DMALENGTH, sizeof(struct njata32_sgtable) * nsegs);

	/* start transfer */
	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc), NJATA32_REG_BM,
	    (bus_space_read_1(NJATA32_REGT(sc), NJATA32_REGH(sc),
		NJATA32_REG_BM)
	     & ~(NJATA32_BM_RD|NJATA32_BM_SG|NJATA32_BM_WAIT_MASK)) |
	    bmreg | NJATA32_BM_GO);

	sc->sc_devflags = dev->d_flags;
	if (flags & WDC_PIOBM_XFER_IRQ)
		sc->sc_devflags |= NJATA32_DEV_XFER_INTR;
#ifdef DIAGNOSTIC
	dev->d_flags |= NJATA32_DEV_DMA_STARTED;
#endif
#endif
}

/*
 * end of DMA
 */
int
ide_dma_finish(void *v, int channel, int drive, int force)
{

	struct ide_softc *sc = v;
	int error = 0;

	aprint_normal_dev(sc->sc_wdcdev.sc_atac.atac_dev, 
			  "ide_dma_finish: ch%d dv%d\n", channel, drive);

#if 0
	KASSERT(channel == 0);
	KASSERT(dev->d_flags & NJATA32_DEV_DMA_MAPPED);
	KASSERT(dev->d_flags & NJATA32_DEV_DMA_STARTED);

	bm = bus_space_read_1(NJATA32_REGT(sc), NJATA32_REGH(sc),
	    NJATA32_REG_BM);

#ifdef NJATA32_DEBUG
	printf("%s: irq %#x, bm %#x, 18 %#x, 1c %#x\n", NJATA32NAME(sc),
	    bus_space_read_1(NJATA32_REGT(sc), NJATA32_REGH(sc),
		NJATA32_REG_IRQ_STAT),
	    bm,
	    bus_space_read_4(NJATA32_REGT(sc), NJATA32_REGH(sc), 0x18),
	    bus_space_read_1(NJATA32_REGT(sc), NJATA32_REGH(sc), 0x1c));
#endif

	bus_dmamap_sync(sc->sc_dmat, sc->sc_dmamap_sgt,
	    (char *)dev->d_sgt - (char *)sc->sc_sgtpg,
	    sizeof(struct njata32_sgtable) * sc->sc_piobm_nsegs,
	    BUS_DMASYNC_POSTWRITE);

	/* check if DMA is active */
	if (bm & NJATA32_BM_GO) {
		error = WDC_DMAST_NOIRQ;

		switch (force) {
		case WDC_DMAEND_END:
			return error;

		case WDC_DMAEND_ABRT:
			printf("%s: aborting DMA\n", NJATA32NAME(sc));
			break;
		}
	}

	/*
	 * ???
	 * For unknown reason, PIOBM transfer sometimes fails in the middle,
	 * in which case the bit #7 of BM register becomes 0.
	 * Increasing the wait value seems to improve the situation.
	 *
	 * XXX
	 * PIO transfer may also fail, but it seems it can't be detected.
	 */
	if ((bm & NJATA32_BM_DONE) == 0) {
		error |= WDC_DMAST_ERR;
		printf("%s: busmaster error", NJATA32NAME(sc));
		if (sc->sc_atawait < 0x11) {
			if ((sc->sc_atawait & 0xf) == 0)
				sc->sc_atawait++;
			else
				sc->sc_atawait += 0x10;
			printf(", new ATA wait = %#x", sc->sc_atawait);
			njata32_setup_channel(&sc->sc_ch[0].ch_ata_channel);
		}
		printf("\n");
	}

	/* stop command */
	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc), NJATA32_REG_AS,
	    NJATA32_AS_WAIT0);
	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc), NJATA32_REG_BM,
	    NJATA32_BM_WAIT0);

	/* set timing for PIO */
	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc),
	    NJATA32_REG_TIMING, sc->sc_timing_pio);

	/*
	 * reenable device interrupt in case it was disabled for
	 * this transfer
	 */
	bus_space_write_1(NJATA32_REGT(sc), NJATA32_REGH(sc),
	    NJATA32_REG_IRQ_SELECT, NJATA32_IRQ_XFER | NJATA32_IRQ_DEV);

#if 1	/* should be? */
	if ((sc->sc_devflags & NJATA32_DEV_GOT_XFER_INTR) == 0)
		error |= WDC_DMAST_ERR;
#endif
	sc->sc_devflags = 0;

#ifdef DIAGNOSTIC
	dev->d_flags &= ~NJATA32_DEV_DMA_STARTED;
#endif
#endif
	return error;
}

/*
 * unmap DMA buffer
 */
void
ide_piobm_done(void *v, int channel, int drive)
{
	struct ide_softc *sc = v;

	aprint_normal_dev(sc->sc_wdcdev.sc_atac.atac_dev, 
			  "ide_piobm_done: ch%d dv%d\n",
			  channel, drive);

	KASSERT(channel == 0);
#if 0
	KASSERT(dev->d_flags & NJATA32_DEV_DMA_MAPPED);
	KASSERT((dev->d_flags & NJATA32_DEV_DMA_STARTED) == 0);

	/* unload dma map */
	bus_dmamap_sync(sc->sc_dmat, dev->d_dmamap_xfer,
	    0, dev->d_dmamap_xfer->dm_mapsize,
	    (dev->d_flags & NJATA32_DEV_DMA_READ) ?
		BUS_DMASYNC_POSTREAD : BUS_DMASYNC_POSTWRITE);

	bus_dmamap_unload(sc->sc_dmat, dev->d_dmamap_xfer);
	dev->d_flags &= ~NJATA32_DEV_DMA_MAPPED;
#endif
}

static void
ide_dataout(struct ata_channel *chp, int flags, void *bf, size_t len)
{
	struct wdc_regs *wdr = CHAN_TO_WDC_REGS(chp);
	uint16_t data, *bfp;

	bfp = bf;
	while (len) {
		data = bswap16(*bfp++);
		bus_space_write_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data],
				       0, data);
		len -= sizeof(data);
	}
}

static void
ide_datain(struct ata_channel *chp, int flags, void *bf, size_t len)
{
	struct wdc_regs *wdr = CHAN_TO_WDC_REGS(chp);
	uint16_t data, *bfp;
	
	bfp = bf;
	while (len) {
		data = bus_space_read_2(wdr->cmd_iot, wdr->cmd_iohs[wd_data],
				      0);
		*bfp++ = bswap16(data);
		len -= sizeof(data);
	}
}
