/*	$NetBSD: $	*/

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include "bpfilter.h"
#include "rnd.h"

#include <sys/param.h>
#include <sys/types.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/socket.h>

#include <uvm/uvm_extern.h>

#if NRND > 0
#include <sys/rnd.h>
#endif

#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_arp.h>
#include <net/if_media.h>
#include <net/if_ether.h>

#include <machine/bus.h>

#include <dev/mii/mii.h>
#include <dev/mii/miivar.h>

#include <powerpc/mpc5200/fecvar.h>

static void fec_start(struct ifnet *);
static int fec_ioctl(struct ifnet *, u_long, void *);
static int fec_init(struct ifnet *);
static void fec_stop(struct ifnet *, int);
static void fec_watchdog(struct ifnet *);

static int fec_ifmedia_upd(struct ifnet *);
static void fec_ifmedia_sts(struct ifnet *, struct ifmediareq *);
static int fec_miibus_readreg(device_t, int, int);
static void fec_miibus_writereg(device_t, int, int, int);
static void fec_miibus_statchg(device_t);

void
fec_init(struct fec_softc *sc, uint8_t *myaddr)
{
	struct ifnet *ifp;

	sc->sc_enaddr[0] = myaddr[0];
	sc->sc_enaddr[1] = myaddr[1];
	sc->sc_enaddr[2] = myaddr[2];
	sc->sc_enaddr[3] = myaddr[3];
	sc->sc_enaddr[4] = myaddr[4];
	sc->sc_enaddr[5] = myaddr[5];
	aprint_normal_dev(self, "Ethernet address %s\n",
	    ether_sprintf(sc->sc_enaddr));

	
	bus_space_read_4(sc->sc_regt, sc->sc_regh, 
	fec_reset_rxdma(sc);
	fec_reset_txdma(sc);

	ifp = &sc->sc_ethercom.ec_if;
	ifp->if_softc = sc;
	ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
	ifp->if_start = fec_start;
	ifp->if_ioctl = fec_ioctl;
	ifp->if_init = fec_init;
	ifp->if_stop = fec_stop;
	ifp->if_watchdog = fec_watchdog;
	IFQ_SET_MAXLEN(&ifp->if_snd, max(FEC_FIFO_SIZE - 1, IFQ_MAXLEN));
	IFQ_SET_READY(&ifp->if_snd);

	fec_stop(ifp, 0);

	/*
	 * Do MII setup.
	 */
	sc->sc_mii.mii_ifp = ifp;
	sc->sc_mii.mii_readreg = fec_miibus_readreg;
	sc->sc_mii.mii_writereg = fec_miibus_writereg;
	sc->sc_mii.mii_statchg = fec_miibus_statchg;

	ifmedia_init(&sc->sc_mii.mii_media, 0,
	    fec_ifmedia_upd, fec_ifmedia_sts);
	mii_attach(self, &sc->sc_mii, 0xffffffff,
	    MII_PHY_ANY, MII_OFFSET_ANY, 0);
	if (LIST_FIRST(&sc->sc_mii.mii_phys) == NULL) {
		aprint_error_dev(self, "no PHY found!\n");
		ifmedia_add(&sc->sc_mii.mii_media,
		    IFM_ETHER|IFM_MANUAL, 0, NULL);
		ifmedia_set(&sc->sc_mii.mii_media, IFM_ETHER|IFM_MANUAL);
	} else
		ifmedia_set(&sc->sc_mii.mii_media, IFM_ETHER|IFM_AUTO);

	/*
	 * Call MI attach routines.
	 */
	if_attach(ifp);

	ether_ifattach(ifp, sc->sc_enaddr);

#if NRND > 0
	rnd_attach_source(&sc->rnd_source, device_xname(sc->sc_dev),
	    RND_TYPE_NET, 0);
#endif

	return;
}

static void
fec_start(struct ifnet *ifp)
{
	struct fec_softc *sc = ifp->if_softc;
	struct mbuf *m_head = NULL;
	uint32_t idx = sc->sc_cdata.oriongbe_tx_prod;
	int pkts = 0;

	if ((ifp->if_flags & (IFF_RUNNING|IFF_OACTIVE)) != IFF_RUNNING)
		return;
	/* If Link is DOWN, can't start TX */
	if (!(ORIONGBE_READ_4(sc, ORIONGBE_PS0) & ORIONGBE_PS0_LINKUP))
		return;

	while (sc->sc_cdata.oriongbe_tx_chain[idx].oriongbe_mbuf == NULL) {
		IFQ_POLL(&ifp->if_snd, m_head);
		if (m_head == NULL)
			break;

		/*
		 * Pack the data into the transmit ring. If we
		 * don't have room, set the OACTIVE flag and wait
		 * for the NIC to drain the ring.
		 */
		if (oriongbe_encap(sc, m_head, &idx)) {
			ifp->if_flags |= IFF_OACTIVE;
			break;
		}

		/* now we are committed to transmit the packet */
		IFQ_DEQUEUE(&ifp->if_snd, m_head);
		pkts++;

#if NBPFILTER > 0
		/*
		 * If there's a BPF listener, bounce a copy of this frame
		 * to him.
		 */
		if (ifp->if_bpf)
			bpf_mtap(ifp->if_bpf, m_head);
#endif
	}
	if (pkts == 0)
		return;

	/* Transmit at Queue 0 */
	if (idx != sc->sc_cdata.oriongbe_tx_prod) {
		sc->sc_cdata.oriongbe_tx_prod = idx;
		ORIONGBE_WRITE_4(sc, ORIONGBE_TQC, ORIONGBE_TQC_ENQ);

		/*
		 * Set a timeout in case the chip goes out to lunch.
		 */
		ifp->if_timer = 5;
	}
}

static int
fec_ioctl(struct ifnet *ifp, u_long command, void *data)
{
	struct fec_softc *sc = ifp->if_softc;
	struct ifreq *ifr = data;
	struct mii_data *mii;
	int s, error = 0;

	s = splnet();

	switch (command) {
	case SIOCSIFFLAGS:
		if (ifp->if_flags & IFF_UP)
			fec_init(ifp);
		else
			if (ifp->if_flags & IFF_RUNNING)
				fec_stop(ifp, 0);
		sc->sc_if_flags = ifp->if_flags;
		error = 0;
		break;

	case SIOCGIFMEDIA:
	case SIOCSIFMEDIA:
		mii = &sc->sc_mii;
		error = ifmedia_ioctl(ifp, ifr, &mii->mii_media, command);
		break;

	default:
		error = ether_ioctl(ifp, command, data);
		if (error == ENETRESET) {
			if (ifp->if_flags & IFF_RUNNING) {
				fec_setmulti(sc);
			}
			error = 0;
		}
		break;
	}

	splx(s);

	return error;
}

static int
oriongbe_init(struct ifnet *ifp)
{
	struct oriongbe_softc *sc = ifp->if_softc;
	struct mii_data *mii = &sc->sc_mii;
	uint32_t reg, val;
	int i, s;

	DPRINTFN(2, ("oriongbe_init\n"));

	s = splnet();

	if (ifp->if_flags & IFF_RUNNING) {
		splx(s);
		return 0;
	}

	/* Cancel pending I/O and free all RX/TX buffers. */
	oriongbe_stop(ifp, 0);

	/* clear all ethernet port interrupts */
	ORIONGBE_WRITE_4(sc, ORIONGBE_IC, 0);
	ORIONGBE_WRITE_4(sc, ORIONGBE_ICE, 0);

	/* Init TX/RX descriptors */
	if (oriongbe_init_tx_ring(sc) == ENOBUFS) {
		aprint_error_ifnet(ifp,
		    "initialization failed: no memory for tx buffers\n");
		splx(s);
		return ENOBUFS;
	}
	if (oriongbe_init_rx_ring(sc) == ENOBUFS) {
		aprint_error_ifnet(ifp,
		    "initialization failed: no memory for rx buffers\n");
		splx(s);
		return ENOBUFS;
	}

	if (sc->sc_version == 2) {
		reg = ORIONGBE_READ_4(sc, ORIONGBE_PSC1);
		aprint_normal_ifnet(ifp, "PSC1 %x\n", reg);
		ORIONGBE_WRITE_4(sc, ORIONGBE_PSC1,
				 reg | ORIONGBE_PSC1_RGMIIEN);
	}

	ORIONGBE_WRITE_4(sc, ORIONGBE_PSC0,
	    ORIONGBE_PSC0_ANFC |			/* Enable Auto-Neg Flow Ctrl */
	    ORIONGBE_PSC0_RESERVED |		/* Must be set to 1 */
	    ORIONGBE_PSC0_FLFAIL |		/* Do NOT Force Link Fail */
	    ORIONGBE_PSC0_MRU(ORIONGBE_PSC0_MRU_1518) |
	    ORIONGBE_PSC0_SETFULLDX);		/* Set_FullDx */

	ORIONGBE_WRITE_4(sc, ORIONGBE_CRDP(0), ORIONGBE_RX_RING_ADDR(sc, 0));
	ORIONGBE_WRITE_4(sc, ORIONGBE_TCQDP, ORIONGBE_TX_RING_ADDR(sc, 0));

	if (sc->sc_version == 2) val = 0xffff7fff;
	else val = 0x3fffffff;

	for (i = 0; i < 8; i++) {
		/*
		 * Queue 0 must be programmed to 0x3fffffff. Queue 1 through 7
		 * must be programmed to 0x00000000.
		 */
		ORIONGBE_WRITE_4(sc, ORIONGBE_TQTBCOUNT(i), 0x3fffffff);
		ORIONGBE_WRITE_4(sc, ORIONGBE_TQTBCONFIG(i), val);
		val = 0x00000000;
	}

	ORIONGBE_WRITE_4(sc, ORIONGBE_PXC, ORIONGBE_PXC_RXCS);
	ORIONGBE_WRITE_4(sc, ORIONGBE_PXCX, 0);
	ORIONGBE_WRITE_4(sc, ORIONGBE_SDC,
	    ORIONGBE_SDC_RXBSZ_16_64BITWORDS |
	    ORIONGBE_SDC_BLMR |	/* Big/Litlle Endian Receive Mode: No swap */
	    ORIONGBE_SDC_BLMT | /* Big/Litlle Endian Transmit Mode: No swap */
	    ORIONGBE_SDC_TXBSZ_16_64BITWORDS);

	mii_mediachg(mii);

	/* Enable port */
	reg = ORIONGBE_READ_4(sc, ORIONGBE_PSC0);
	ORIONGBE_WRITE_4(sc, ORIONGBE_PSC0, reg | ORIONGBE_PSC0_PORTEN);

	/* If Link is UP, Start RX and TX traffic */
	if (ORIONGBE_READ_4(sc, ORIONGBE_PS0) & ORIONGBE_PS0_LINKUP) {
		/* Enable port RX/TX. */
		ORIONGBE_WRITE_4(sc, ORIONGBE_RQC, ORIONGBE_RQC_ENQ(0));
		ORIONGBE_WRITE_4(sc, ORIONGBE_TQC, ORIONGBE_TQC_ENQ);
	}

	/* Enable interrupt masks */
	ORIONGBE_WRITE_4(sc, ORIONGBE_PIM,
	    ORIONGBE_IC_RXBUF |
	    ORIONGBE_IC_EXTEND |
	    ORIONGBE_IC_RXBUFQ_MASK |
	    ORIONGBE_IC_RXERROR |
	    ORIONGBE_IC_RXERRQ_MASK);
	ORIONGBE_WRITE_4(sc, ORIONGBE_PEIM,
	    ORIONGBE_ICE_TXBUF |
	    ORIONGBE_ICE_TXERR |
	    ORIONGBE_ICE_LINKCHG);

	ifp->if_flags |= IFF_RUNNING;
	ifp->if_flags &= ~IFF_OACTIVE;

	splx(s);

	return 0;
}

/* ARGSUSED */
static void
oriongbe_stop(struct ifnet *ifp, int disable)
{
	struct oriongbe_softc *sc = ifp->if_softc;
	struct oriongbe_chain_data *cdata = &sc->sc_cdata;
	uint32_t reg;
	int i, cnt;

	DPRINTFN(2, ("oriongbe_stop\n"));

	/* Stop Rx port activity. Check port Rx activity. */
	reg = ORIONGBE_READ_4(sc, ORIONGBE_RQC);
	if (reg & ORIONGBE_RQC_ENQ_MASK)
		/* Issue stop command for active channels only */
		ORIONGBE_WRITE_4(sc, ORIONGBE_RQC,
		    ORIONGBE_RQC_DISQ_DISABLE(reg));

	/* Stop Tx port activity. Check port Tx activity. */
	if (ORIONGBE_READ_4(sc, ORIONGBE_TQC) & ORIONGBE_TQC_ENQ)
		ORIONGBE_WRITE_4(sc, ORIONGBE_TQC, ORIONGBE_TQC_DISQ);

	/* Force link down */
	reg = ORIONGBE_READ_4(sc, ORIONGBE_PSC0);
	ORIONGBE_WRITE_4(sc, ORIONGBE_PSC0, reg & ~ORIONGBE_PSC0_FLFAIL);

#define RX_DISABLE_TIMEOUT          0x1000000
#define TX_FIFO_EMPTY_TIMEOUT       0x1000000
	/* Wait for all Rx activity to terminate. */
	cnt = 0;
	do {
		if (cnt >= RX_DISABLE_TIMEOUT) {
			aprint_error_ifnet(ifp,
			    "timeout for RX stopped. rqc 0x%x\n", reg);
			break;
		}
		cnt++;

		/*
		 * Check Receive Queue Command register that all Rx queues
		 * are stopped
		 */
		reg = ORIONGBE_READ_4(sc, ORIONGBE_RQC);
	} while (reg & 0xff);

	/* Double check to verify that TX FIFO is empty */
	cnt = 0;
	while (1) {
		do {
			if (cnt >= TX_FIFO_EMPTY_TIMEOUT) {
				aprint_error_ifnet(ifp,
				    "timeout for TX FIFO empty. status 0x%x\n",
				    reg);
				break;
			}
			cnt++;

			reg = ORIONGBE_READ_4(sc, ORIONGBE_PS0);
		}
		while (!(reg & ORIONGBE_PS0_TXFIFOEMP) ||
		    reg & ORIONGBE_PS0_TXINPROG);

		if (cnt >= TX_FIFO_EMPTY_TIMEOUT)
			break;

		/* Double check */
		reg = ORIONGBE_READ_4(sc, ORIONGBE_PS0);
		if (reg & ORIONGBE_PS0_TXFIFOEMP &&
		    !(reg & ORIONGBE_PS0_TXINPROG))
			break;
		else
			aprint_error_ifnet(ifp,
			    "TX FIFO empty double check failed."
			    " %d loops, status 0x%x\n", cnt, reg);
	}

	/* Do NOT force link down */
	reg = ORIONGBE_READ_4(sc, ORIONGBE_PSC0);
	ORIONGBE_WRITE_4(sc, ORIONGBE_PSC0, reg | ORIONGBE_PSC0_FLFAIL);

	/* Wait about 2500 tclk cycles */
	delay(1000 * 1000 / mvTclk * 2500);

	/* Reset the Enable bit in the Port Serial Control Register */
	reg = ORIONGBE_READ_4(sc, ORIONGBE_PSC0);
	ORIONGBE_WRITE_4(sc, ORIONGBE_PSC0, reg & ~ORIONGBE_PSC0_PORTEN);

	/* Wait about 2500 tclk cycles */
	delay(1000 * 1000 / mvTclk * 2500);

	/* Disable interrupts */
	ORIONGBE_WRITE_4(sc, ORIONGBE_PIM, 0);
	ORIONGBE_WRITE_4(sc, ORIONGBE_PEIM, 0);

	/* Free RX and TX mbufs still in the queues. */
	for (i = 0; i < ORIONGBE_RX_RING_CNT; i++) {
		if (cdata->oriongbe_rx_chain[i].oriongbe_mbuf != NULL) {
			m_freem(cdata->oriongbe_rx_chain[i].oriongbe_mbuf);
			cdata->oriongbe_rx_chain[i].oriongbe_mbuf = NULL;
		}
	}
	for (i = 0; i < ORIONGBE_TX_RING_CNT; i++) {
		if (cdata->oriongbe_tx_chain[i].oriongbe_mbuf != NULL) {
			m_freem(cdata->oriongbe_tx_chain[i].oriongbe_mbuf);
			cdata->oriongbe_tx_chain[i].oriongbe_mbuf = NULL;
		}
	}

	ifp->if_flags &= ~(IFF_RUNNING | IFF_OACTIVE);
}

static void
fec_watchdog(struct ifnet *ifp)
{
	struct fec_softc *sc = ifp->if_softc;

	/*
	 * Reclaim first as there is a possibility of losing Tx completion
	 * interrupts.
	 */
	oriongbe_txeof(sc);
	if (sc->sc_cdata.oriongbe_tx_cnt != 0) {
		aprint_error_ifnet(ifp, "watchdog timeout\n");

		ifp->if_oerrors++;

		oriongbe_init(ifp);
	}
}


/*
 * Set media options.
 */
static int
fec_ifmedia_upd(struct ifnet *ifp)
{
	struct fec_softc *sc = ifp->if_softc;

	mii_mediachg(&sc->sc_mii);
	return 0;
}

/*
 * Report current media status.
 */
static void
fec_ifmedia_sts(struct ifnet *ifp, struct ifmediareq *ifmr)
{
	struct fec_softc *sc = ifp->if_softc;

	mii_pollstat(&sc->sc_mii);
	ifmr->ifm_active = sc->sc_mii.mii_media_active;
	ifmr->ifm_status = sc->sc_mii.mii_media_status;
}


static int
fec_miibus_readreg(device_t dev, int phy, int reg)
{
	struct oriongbe_softc *sc = device_private(dev);
	struct ifnet *ifp = &sc->sc_ethercom.ec_if;
	uint32_t smi, val;
	int i;

	for (i = 0; i < ORIONGBE_PHY_TIMEOUT; i++) {
		DELAY(1);
		if (!(ORIONGBE_READ_4(sc, ORIONGBE_SMI) & ORIONGBE_SMI_BUSY))
			break;
	}
	if (i == ORIONGBE_PHY_TIMEOUT) {
		aprint_error_ifnet(ifp, "SMI busy timeout\n");
		return -1;
	}

	smi = ORIONGBE_SMI_PHYAD(phy) | ORIONGBE_SMI_REGAD(reg) |
	    ORIONGBE_SMI_OPCODE_READ;
	ORIONGBE_WRITE_4(sc, ORIONGBE_SMI, smi);

	for (i = 0; i < ORIONGBE_PHY_TIMEOUT; i++) {
		DELAY(1);
		smi = ORIONGBE_READ_4(sc, ORIONGBE_SMI);
		if (smi & ORIONGBE_SMI_READVALID)
			break;
	}
	DPRINTFN(9, ("oriongbe_miibus_readreg: i=%d, timeout=%d\n",
	    i, ORIONGBE_PHY_TIMEOUT));

	val = smi & ORIONGBE_SMI_DATA_MASK;

	DPRINTFN(9, ("oriongbe_miibus_readreg phy=%d, reg=%#x, val=%#x\n",
	    phy, reg, val));

	return val;
}

static void
oriongbe_miibus_writereg(device_t dev, int phy, int reg, int val)
{
	struct oriongbe_softc *sc = device_private(dev);
	struct ifnet *ifp = &sc->sc_ethercom.ec_if;
	uint32_t smi;
	int i;

	DPRINTFN(9, ("oriongbe_miibus_writereg phy=%d reg=%#x val=%#x\n",
	     phy, reg, val));

	for (i = 0; i < ORIONGBE_PHY_TIMEOUT; i++) {
		DELAY(1);
		if (!(ORIONGBE_READ_4(sc, ORIONGBE_SMI) & ORIONGBE_SMI_BUSY))
			break;
	}
	if (i == ORIONGBE_PHY_TIMEOUT) {
		aprint_error_ifnet(ifp, "SMI busy timeout\n");
		return;
	}

	smi = ORIONGBE_SMI_PHYAD(phy) | ORIONGBE_SMI_REGAD(reg) |
	    ORIONGBE_SMI_OPCODE_WRITE | (val & ORIONGBE_SMI_DATA_MASK);
	ORIONGBE_WRITE_4(sc, ORIONGBE_SMI, smi);

	for (i = 0; i < ORIONGBE_PHY_TIMEOUT; i++) {
		DELAY(1);
		if (!(ORIONGBE_READ_4(sc, ORIONGBE_SMI) & ORIONGBE_SMI_BUSY))
			break;
	}
	if (i == ORIONGBE_PHY_TIMEOUT)
		aprint_error_ifnet(ifp, "phy write timed out\n");
}

static void
oriongbe_miibus_statchg(device_t dev)
{

	/* nothing to do */
}


static int
oriongbe_init_rx_ring(struct oriongbe_softc *sc)
{
	struct oriongbe_chain_data *cd = &sc->sc_cdata;
	struct oriongbe_ring_data *rd = sc->sc_rdata;
	int i;

	bzero((char *)rd->oriongbe_rx_ring,
	    sizeof(struct oriongbe_rx_desc) * ORIONGBE_RX_RING_CNT);

	for (i = 0; i < ORIONGBE_RX_RING_CNT; i++) {
		cd->oriongbe_rx_chain[i].oriongbe_desc =
		    &rd->oriongbe_rx_ring[i];
		if (i == ORIONGBE_RX_RING_CNT - 1) {
			cd->oriongbe_rx_chain[i].oriongbe_next =
			    &cd->oriongbe_rx_chain[0];
			rd->oriongbe_rx_ring[i].nextdescptr =
			    ORIONGBE_RX_RING_ADDR(sc, 0);
		} else {
			cd->oriongbe_rx_chain[i].oriongbe_next =
			    &cd->oriongbe_rx_chain[i + 1];
			rd->oriongbe_rx_ring[i].nextdescptr =
			    ORIONGBE_RX_RING_ADDR(sc, i + 1);
		}
	}

	for (i = 0; i < ORIONGBE_RX_RING_CNT; i++) {
		if (oriongbe_newbuf(sc, i, NULL,
		    sc->sc_cdata.oriongbe_rx_jumbo_map) == ENOBUFS) {
			aprint_error_ifnet(&sc->sc_ethercom.ec_if,
			    "failed alloc of %dth mbuf\n", i);
			return ENOBUFS;
		}
	}
	sc->sc_cdata.oriongbe_rx_prod = 0;
	sc->sc_cdata.oriongbe_rx_cons = 0;

	return 0;
}

static int
oriongbe_init_tx_ring(struct oriongbe_softc *sc)
{
	struct oriongbe_chain_data *cd = &sc->sc_cdata;
	struct oriongbe_ring_data *rd = sc->sc_rdata;
	int i;

	bzero((char *)sc->sc_rdata->oriongbe_tx_ring,
	    sizeof(struct oriongbe_tx_desc) * ORIONGBE_TX_RING_CNT);

	for (i = 0; i < ORIONGBE_TX_RING_CNT; i++) {
		cd->oriongbe_tx_chain[i].oriongbe_desc =
		    &rd->oriongbe_tx_ring[i];
		if (i == ORIONGBE_TX_RING_CNT - 1) {
			cd->oriongbe_tx_chain[i].oriongbe_next =
			    &cd->oriongbe_tx_chain[0];
			rd->oriongbe_tx_ring[i].nextdescptr =
			    ORIONGBE_TX_RING_ADDR(sc, 0);
		} else {
			cd->oriongbe_tx_chain[i].oriongbe_next =
			    &cd->oriongbe_tx_chain[i + 1];
			rd->oriongbe_tx_ring[i].nextdescptr =
			    ORIONGBE_TX_RING_ADDR(sc, i + 1);
		}
		rd->oriongbe_tx_ring[i].cmdsts = ORIONGBE_BUFFER_OWNED_BY_HOST;
	}

	sc->sc_cdata.oriongbe_tx_prod = 0;
	sc->sc_cdata.oriongbe_tx_cons = 0;
	sc->sc_cdata.oriongbe_tx_cnt = 0;

	ORIONGBE_CDTXSYNC(sc, 0, ORIONGBE_TX_RING_CNT,
	    BUS_DMASYNC_PREREAD | BUS_DMASYNC_PREWRITE);

	return 0;
}

static int
oriongbe_newbuf(struct oriongbe_softc *sc, int i, struct mbuf *m,
		bus_dmamap_t dmamap)
{
	struct mbuf *m_new = NULL;
	struct oriongbe_chain *c;
	struct oriongbe_rx_desc *r;
	int align;

	if (m == NULL) {
		void *buf = NULL;

		MGETHDR(m_new, M_DONTWAIT, MT_DATA);
		if (m_new == NULL) {
			aprint_error_ifnet(&sc->sc_ethercom.ec_if,
			    "no memory for rx list -- packet dropped!\n");
			return ENOBUFS;
		}

		/* Allocate the jumbo buffer */
		buf = oriongbe_jalloc(sc);
		if (buf == NULL) {
			m_freem(m_new);
			DPRINTFN(1, ("%s jumbo allocation failed -- packet "
			    "dropped!\n", sc->sc_ethercom.ec_if.if_xname));
			return ENOBUFS;
		}

		/* Attach the buffer to the mbuf */
		m_new->m_len = m_new->m_pkthdr.len = ORIONGBE_JLEN;
		MEXTADD(m_new, buf, ORIONGBE_JLEN, 0, oriongbe_jfree, sc);
	} else {
		/*
		 * We're re-using a previously allocated mbuf;
		 * be sure to re-init pointers and lengths to
		 * default values.
		 */
		m_new = m;
		m_new->m_len = m_new->m_pkthdr.len = ORIONGBE_JLEN;
		m_new->m_data = m_new->m_ext.ext_buf;
	}
	align = (u_long)m_new->m_data & ORIONGBE_BUF_MASK;
	if (align != 0)
		m_adj(m_new,  ORIONGBE_BUF_ALIGN - align);

	c = &sc->sc_cdata.oriongbe_rx_chain[i];
	r = c->oriongbe_desc;
	c->oriongbe_mbuf = m_new;
	r->bufptr = dmamap->dm_segs[0].ds_addr +
	    (((vaddr_t)m_new->m_data -
	    (vaddr_t)sc->sc_cdata.oriongbe_jumbo_buf));
	r->bufsize = ORIONGBE_JLEN & ~ORIONGBE_BUF_MASK;
	r->cmdsts = ORIONGBE_BUFFER_OWNED_BY_DMA | ORIONGBE_RX_ENABLE_INTERRUPT;

	ORIONGBE_CDRXSYNC(sc, i, BUS_DMASYNC_PREWRITE | BUS_DMASYNC_PREREAD);

	return 0;
}

/*
 * Memory management for jumbo frames.
 */

static int
oriongbe_alloc_jumbo_mem(struct oriongbe_softc *sc)
{
	char *ptr, *kva;
	bus_dma_segment_t seg;
	int i, rseg, state, error;
	struct oriongbe_jpool_entry *entry;

	state = error = 0;

	/* Grab a big chunk o' storage. */
	if (bus_dmamem_alloc(sc->sc_dmat, ORIONGBE_JMEM, PAGE_SIZE, 0,
	    &seg, 1, &rseg, BUS_DMA_NOWAIT)) {
		aprint_error("%s: can't alloc rx buffers\n",
		    device_xname(sc->sc_dev));
		return ENOBUFS;
	}

	state = 1;
	if (bus_dmamem_map(sc->sc_dmat, &seg, rseg, ORIONGBE_JMEM,
	    (void **)&kva, BUS_DMA_NOWAIT)) {
		aprint_error("%s: can't map dma buffers (%d bytes)\n",
		    device_xname(sc->sc_dev), ORIONGBE_JMEM);
		error = ENOBUFS;
		goto out;
	}

	state = 2;
	if (bus_dmamap_create(sc->sc_dmat, ORIONGBE_JMEM, 1, ORIONGBE_JMEM, 0,
	    BUS_DMA_NOWAIT, &sc->sc_cdata.oriongbe_rx_jumbo_map)) {
		aprint_error_dev(sc->sc_dev, "can't create dma map\n");
		error = ENOBUFS;
		goto out;
	}

	state = 3;
	if (bus_dmamap_load(sc->sc_dmat, sc->sc_cdata.oriongbe_rx_jumbo_map,
	    kva, ORIONGBE_JMEM, NULL, BUS_DMA_NOWAIT)) {
		aprint_error_dev(sc->sc_dev, "can't load dma map\n");
		error = ENOBUFS;
		goto out;
	}

	state = 4;
	sc->sc_cdata.oriongbe_jumbo_buf = (void *)kva;
	DPRINTFN(1,("oriongbe_jumbo_buf = 0x%p\n",
	    sc->sc_cdata.oriongbe_jumbo_buf));

	LIST_INIT(&sc->sc_jfree_listhead);
	LIST_INIT(&sc->sc_jinuse_listhead);

	/*
	 * Now divide it up into 9K pieces and save the addresses
	 * in an array.
	 */
	ptr = sc->sc_cdata.oriongbe_jumbo_buf;
	for (i = 0; i < ORIONGBE_JSLOTS; i++) {
		sc->sc_cdata.oriongbe_jslots[i] = ptr;
		ptr += ORIONGBE_JLEN;
		entry = malloc(sizeof(struct oriongbe_jpool_entry),
		    M_DEVBUF, M_NOWAIT);
		if (entry == NULL) {
			aprint_error_dev(sc->sc_dev,
			    "no memory for jumbo buffer queue!\n");
			error = ENOBUFS;
			goto out;
		}
		entry->slot = i;
		if (i)
			LIST_INSERT_HEAD(&sc->sc_jfree_listhead,
			    entry, jpool_entries);
		else
			LIST_INSERT_HEAD(&sc->sc_jinuse_listhead,
			    entry, jpool_entries);
	}
out:
	if (error != 0) {
		switch (state) {
		case 4:
			bus_dmamap_unload(sc->sc_dmat,
			    sc->sc_cdata.oriongbe_rx_jumbo_map);
		case 3:
			bus_dmamap_destroy(sc->sc_dmat,
			    sc->sc_cdata.oriongbe_rx_jumbo_map);
		case 2:
			bus_dmamem_unmap(sc->sc_dmat, kva, ORIONGBE_JMEM);
		case 1:
			bus_dmamem_free(sc->sc_dmat, &seg, rseg);
			break;
		default:
			break;
		}
	}

	return error;
}

static int
oriongbe_encap(struct oriongbe_softc *sc, struct mbuf *m_head,
	      uint32_t *txidx)
{
	struct oriongbe_tx_desc *f = NULL;
	struct oriongbe_txmap_entry *entry;
	bus_dma_segment_t *txseg;
	bus_dmamap_t txmap;
	uint32_t first, current, last, cmdsts = 0;
	int m_csumflags, i;

	DPRINTFN(3, ("oriongbe_encap\n"));

	entry = SIMPLEQ_FIRST(&sc->sc_txmap_head);
	if (entry == NULL) {
		DPRINTFN(2, ("oriongbe_encap: no txmap available\n"));
		return ENOBUFS;
	}
	txmap = entry->dmamap;

	first = current = last = *txidx;

	/*
	 * Preserve m_pkthdr.csum_flags here since m_head might be
	 * updated by m_defrag()
	 */
	m_csumflags = m_head->m_pkthdr.csum_flags;

	/*
	 * Start packing the mbufs in this chain into
	 * the fragment pointers. Stop when we run out
	 * of fragments or hit the end of the mbuf chain.
	 */
	if (bus_dmamap_load_mbuf(sc->sc_dmat, txmap, m_head, BUS_DMA_NOWAIT)) {
		DPRINTFN(1, ("oriongbe_encap: dmamap failed\n"));
		return ENOBUFS;
	}

	/* Sync the DMA map. */
	bus_dmamap_sync(sc->sc_dmat, txmap, 0, txmap->dm_mapsize,
	    BUS_DMASYNC_PREWRITE);

	if (sc->sc_cdata.oriongbe_tx_cnt + txmap->dm_nsegs >=
	    ORIONGBE_TX_RING_CNT) {
		DPRINTFN(2, ("oriongbe_encap: too few descriptors free\n"));
		bus_dmamap_unload(sc->sc_dmat, txmap);
		return ENOBUFS;
	}

	txseg = txmap->dm_segs;

	DPRINTFN(2, ("oriongbe_encap: dm_nsegs=%d\n", txmap->dm_nsegs));

	for (i = 0; i < txmap->dm_nsegs; i++) {
		f = &sc->sc_rdata->oriongbe_tx_ring[current];
		f->bufptr = txseg[i].ds_addr;
		f->bytecnt = txseg[i].ds_len;
		f->cmdsts = ORIONGBE_BUFFER_OWNED_BY_DMA;
		last = current;
		current = (current + 1) % ORIONGBE_TX_RING_CNT;
	}

	if (m_csumflags & M_CSUM_IPv4)
		cmdsts |= ORIONGBE_TX_GENERATE_IP_CHKSUM;
	if (m_csumflags & M_CSUM_TCPv4)
		cmdsts |=
		    ORIONGBE_TX_GENERATE_L4_CHKSUM | ORIONGBE_TX_L4_TYPE_TCP;
	if (m_csumflags & M_CSUM_UDPv4)
		cmdsts |=
		    ORIONGBE_TX_GENERATE_L4_CHKSUM | ORIONGBE_TX_L4_TYPE_UDP;
	if (m_csumflags & (M_CSUM_IPv4 | M_CSUM_TCPv4 | M_CSUM_UDPv4)) {
		const int iphdr_unitlen = sizeof(struct ip) / sizeof(uint32_t);

		cmdsts |= ORIONGBE_TX_IP_NO_FRAG |
		    ORIONGBE_TX_IP_HEADER_LEN(iphdr_unitlen);	/* unit is 4B */
	}
	if (txmap->dm_nsegs == 1)
		f->cmdsts = cmdsts			|
		    ORIONGBE_BUFFER_OWNED_BY_DMA	|
		    ORIONGBE_TX_GENERATE_CRC		|
		    ORIONGBE_TX_ENABLE_INTERRUPT	|
		    ORIONGBE_TX_ZERO_PADDING		|
		    ORIONGBE_TX_FIRST_DESC		|
		    ORIONGBE_TX_LAST_DESC;
	else {
		f = &sc->sc_rdata->oriongbe_tx_ring[first];
		f->cmdsts = cmdsts			|
		    ORIONGBE_BUFFER_OWNED_BY_DMA	|
		    ORIONGBE_TX_GENERATE_CRC		|
		    ORIONGBE_TX_FIRST_DESC;

		f = &sc->sc_rdata->oriongbe_tx_ring[last];
		f->cmdsts =
		    ORIONGBE_BUFFER_OWNED_BY_DMA	|
		    ORIONGBE_TX_ENABLE_INTERRUPT	|
		    ORIONGBE_TX_ZERO_PADDING		|
		    ORIONGBE_TX_LAST_DESC;
	}

	sc->sc_cdata.oriongbe_tx_chain[last].oriongbe_mbuf = m_head;
	SIMPLEQ_REMOVE_HEAD(&sc->sc_txmap_head, link);
	sc->sc_cdata.oriongbe_tx_map[last] = entry;

	/* Sync descriptors before handing to chip */
	ORIONGBE_CDTXSYNC(sc, *txidx, txmap->dm_nsegs,
	    BUS_DMASYNC_PREREAD|BUS_DMASYNC_PREWRITE);

	sc->sc_cdata.oriongbe_tx_cnt += i;
	*txidx = current;

	DPRINTFN(3, ("oriongbe_encap: completed successfully\n"));

	return 0;
}

static void
oriongbe_rxeof(struct oriongbe_softc *sc)
{
	struct oriongbe_chain_data *cdata = &sc->sc_cdata;
	struct oriongbe_rx_desc *cur_rx;
	struct ifnet *ifp = &sc->sc_ethercom.ec_if;
	struct mbuf *m;
	bus_dmamap_t dmamap;
	uint32_t rxstat;
	int idx, cur, total_len;

	idx = sc->sc_cdata.oriongbe_rx_prod;

	DPRINTFN(3, ("oriongbe_rxeof %d\n", idx));

	for (;;) {
		cur = idx;

		/* Sync the descriptor */
		ORIONGBE_CDRXSYNC(sc, idx,
		    BUS_DMASYNC_POSTREAD | BUS_DMASYNC_POSTWRITE);

		cur_rx = &sc->sc_rdata->oriongbe_rx_ring[idx];

		if ((cur_rx->cmdsts & ORIONGBE_BUFFER_OWNED_MASK) ==
		    ORIONGBE_BUFFER_OWNED_BY_DMA) {
			/* Invalidate the descriptor -- it's not ready yet */
			ORIONGBE_CDRXSYNC(sc, idx, BUS_DMASYNC_PREREAD);
			sc->sc_cdata.oriongbe_rx_prod = idx;
			break;
		}
#ifdef DIAGNOSTIC
		if ((cur_rx->cmdsts &
		    (ORIONGBE_RX_LAST_DESC | ORIONGBE_RX_FIRST_DESC)) !=
		    (ORIONGBE_RX_LAST_DESC | ORIONGBE_RX_FIRST_DESC))
			panic("oriongbe_rxeof:"
			    " buffer size is smaller than packet");
#endif

		dmamap = sc->sc_cdata.oriongbe_rx_jumbo_map;

		bus_dmamap_sync(sc->sc_dmat, dmamap, 0,
		    dmamap->dm_mapsize, BUS_DMASYNC_POSTREAD);

		m = cdata->oriongbe_rx_chain[idx].oriongbe_mbuf;
		cdata->oriongbe_rx_chain[idx].oriongbe_mbuf = NULL;
		total_len = cur_rx->bytecnt;
		rxstat = cur_rx->cmdsts;

		cdata->oriongbe_rx_map[idx] = NULL;

		idx = (idx + 1) % ORIONGBE_RX_RING_CNT;

		if (rxstat & ORIONGBE_ERROR_SUMMARY) {
#if 0
			int err = rxstat & ORIONGBE_RX_ERROR_CODE_MASK;

			if (err == ORIONGBE_RX_CRC_ERROR)
				ifp->if_ierrors++;
			if (err == ORIONGBE_RX_OVERRUN_ERROR)
				ifp->if_ierrors++;
			if (err == ORIONGBE_RX_MAX_FRAME_LEN_ERROR)
				ifp->if_ierrors++;
			if (err == ORIONGBE_RX_RESOURCE_ERROR)
				ifp->if_ierrors++;
#else
			ifp->if_ierrors++;
#endif
			oriongbe_newbuf(sc, cur, m, dmamap);
			continue;
		}

		if (total_len > ORIONGBE_RX_CSUM_MIN_BYTE) {
			/* Check IP header checksum */
			if (rxstat & ORIONGBE_RX_IP_FRAME_TYPE) {
				m->m_pkthdr.csum_flags |= M_CSUM_IPv4;
				if (!(rxstat & ORIONGBE_RX_IP_HEADER_OK))
					m->m_pkthdr.csum_flags |=
					    M_CSUM_IPv4_BAD;
			}
			/* Check TCP/UDP checksum */
			if (rxstat & ORIONGBE_RX_L4_TYPE_TCP)
				m->m_pkthdr.csum_flags |= M_CSUM_TCPv4;
			else if (rxstat & ORIONGBE_RX_L4_TYPE_UDP)
				m->m_pkthdr.csum_flags |= M_CSUM_UDPv4;
			if (!(rxstat & ORIONGBE_RX_L4_CHECKSUM))
				m->m_pkthdr.csum_flags |= M_CSUM_TCP_UDP_BAD;
		}

		/*
		 * Try to allocate a new jumbo buffer. If that
		 * fails, copy the packet to mbufs and put the
		 * jumbo buffer back in the ring so it can be
		 * re-used. If allocating mbufs fails, then we
		 * have to drop the packet.
		 */
		if (oriongbe_newbuf(sc, cur, NULL, dmamap) == ENOBUFS) {
			struct mbuf *m0;

			m0 = m_devget(mtod(m, char *), total_len, 0, ifp, NULL);
			oriongbe_newbuf(sc, cur, m, dmamap);
			if (m0 == NULL) {
				aprint_error_ifnet(ifp, "no receive buffers "
				    "available -- packet dropped!\n");
				ifp->if_ierrors++;
				continue;
			}
			m = m0;
		} else {
			m->m_pkthdr.rcvif = ifp;
			m->m_pkthdr.len = m->m_len = total_len;
		}

		/* Skip on first 2byte (HW header) */
		m_adj(m,  ORIONGBE_HWHEADER_SIZE);
		m->m_flags |= M_HASFCS;

		ifp->if_ipackets++;

#if NBPFILTER > 0
		if (ifp->if_bpf)
			bpf_mtap(ifp->if_bpf, m);
#endif
		/* pass it on. */
		(*ifp->if_input)(ifp, m);
	}
}

static void
oriongbe_txeof(struct oriongbe_softc *sc)
{
	struct oriongbe_chain_data *cdata = &sc->sc_cdata;
	struct oriongbe_tx_desc *cur_tx;
	struct ifnet *ifp = &sc->sc_ethercom.ec_if;
	struct oriongbe_txmap_entry *entry;
	int idx;

	DPRINTFN(3, ("oriongbe_txeof\n"));

	/*
	 * Go through our tx ring and free mbufs for those
	 * frames that have been sent.
	 */
	idx = cdata->oriongbe_tx_cons;
	while (idx != cdata->oriongbe_tx_prod) {
		ORIONGBE_CDTXSYNC(sc, idx, 1,
		    BUS_DMASYNC_POSTREAD | BUS_DMASYNC_POSTWRITE);

		cur_tx = &sc->sc_rdata->oriongbe_tx_ring[idx];
#ifdef ORIONGBE_DEBUG
		if (oriongbe_debug >= 3)
			oriongbe_dump_txdesc(cur_tx, idx);
#endif
		if ((cur_tx->cmdsts & ORIONGBE_BUFFER_OWNED_MASK) ==
		    ORIONGBE_BUFFER_OWNED_BY_DMA) {
			ORIONGBE_CDTXSYNC(sc, idx, 1, BUS_DMASYNC_PREREAD);
			break;
		}
		if (cur_tx->cmdsts & ORIONGBE_TX_LAST_DESC)
			ifp->if_opackets++;
		if (cur_tx->cmdsts & ORIONGBE_ERROR_SUMMARY) {
			int err = cur_tx->cmdsts & ORIONGBE_TX_ERROR_CODE_MASK;

			if (err == ORIONGBE_TX_LATE_COLLISION_ERROR)
				ifp->if_collisions++;
			if (err == ORIONGBE_TX_UNDERRUN_ERROR)
				ifp->if_oerrors++;
			if (err == ORIONGBE_TX_EXCESSIVE_COLLISION_ERRO)
				ifp->if_collisions++;
		}
		if (cdata->oriongbe_tx_chain[idx].oriongbe_mbuf != NULL) {
			entry = cdata->oriongbe_tx_map[idx];

			m_freem(cdata->oriongbe_tx_chain[idx].oriongbe_mbuf);
			cdata->oriongbe_tx_chain[idx].oriongbe_mbuf = NULL;

			bus_dmamap_sync(sc->sc_dmat, entry->dmamap, 0,
			    entry->dmamap->dm_mapsize, BUS_DMASYNC_POSTWRITE);

			bus_dmamap_unload(sc->sc_dmat, entry->dmamap);
			SIMPLEQ_INSERT_TAIL(&sc->sc_txmap_head, entry, link);
			cdata->oriongbe_tx_map[idx] = NULL;
		}
		cdata->oriongbe_tx_cnt--;
		idx = (idx + 1) % ORIONGBE_TX_RING_CNT;
	}
	if (cdata->oriongbe_tx_cnt == 0)
		ifp->if_timer = 0;

	if (cdata->oriongbe_tx_cnt < ORIONGBE_TX_RING_CNT - 2)
		ifp->if_flags &= ~IFF_OACTIVE;

	cdata->oriongbe_tx_cons = idx;
}

static void
fec_setmulti(struct fec_softc *sc)
{
	struct ifnet *ifp= &sc->sc_ethercom.ec_if;
	uint32_t pxc, dfut, upm = 0, reg, filter = 0;
	uint8_t ln = sc->sc_enaddr[5] & 0xf;		/* last nibble */

	if (ifp->if_flags & IFF_PROMISC) {
		upm = ORIONGBE_PXC_UPM;
		filter =
		    ORIONGBE_DF(0, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS) |
		    ORIONGBE_DF(1, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS) |
		    ORIONGBE_DF(2, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS) |
		    ORIONGBE_DF(3, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS);
	} else if (ifp->if_flags & IFF_ALLMULTI) {
		filter =
		    ORIONGBE_DF(0, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS) |
		    ORIONGBE_DF(1, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS) |
		    ORIONGBE_DF(2, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS) |
		    ORIONGBE_DF(3, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS);
	}

	/* Set Unicast Promiscuous mode */
	pxc = ORIONGBE_READ_4(sc, ORIONGBE_PXC);
	pxc &= ~ORIONGBE_PXC_UPM;
	pxc |= upm;
	ORIONGBE_WRITE_4(sc, ORIONGBE_PXC, pxc);

	/* Set Destination Address Filter Multicast Tables */
	for (reg = ORIONGBE_DFSMT_BEGIN; reg <= ORIONGBE_DFSMT_END; reg += 4)
		ORIONGBE_WRITE_4(sc, reg, filter);
	for (reg = ORIONGBE_DFOMT_BEGIN; reg <= ORIONGBE_DFOMT_END; reg += 4)
		ORIONGBE_WRITE_4(sc, reg, filter);

	if (ifp->if_flags & IFF_PROMISC) {
		/* necessary ? */
		for (reg = ORIONGBE_DFUT_BEGIN; reg <= ORIONGBE_DFUT_END;
		    reg += 4)
			ORIONGBE_WRITE_4(sc, reg, filter);
		return;
	}

	/* Set Destination Address Filter Unicast Table */
	reg = ORIONGBE_DFUT_BEGIN + (ln & 0x0c);
	dfut = ORIONGBE_READ_4(sc, reg);
	dfut &= ~ORIONGBE_DF(ln & 0x03, ORIONGBE_DF_QUEUE_MASK);;
	dfut |= ORIONGBE_DF(ln & 0x03, ORIONGBE_DF_QUEUE(0) | ORIONGBE_DF_PASS);
	ORIONGBE_WRITE_4(sc, reg, dfut);
}

