/*	$NetBSD: $ */

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*-
 * Copyright (c) 1998, 1999, 2004, 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Charles M. Hannum.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 1991 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)com.c	7.5 (Berkeley) 5/16/91
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include "opt_ddb.h"
#include "opt_kgdb.h"
#include "opt_lockdebug.h"
#include "opt_multiprocessor.h"
#include "opt_ntp.h"

#include "rnd.h"
#if NRND > 0 && defined(RND_COM)
#include <sys/rnd.h>
#endif

/*
 * Override cnmagic(9) macro before including <sys/systm.h>.
 * We need to know if cn_check_magic triggered debugger, so set a flag.
 * Callers of cn_check_magic must declare int cn_trapped = 0;
 * XXX: this is *ugly*!
 */
#define cn_trap()				\
	do {					\
		console_debugger();		\
		cn_trapped = 1;			\
	} while (/* CONSTCOND */ 0)

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/poll.h>
#include <sys/tty.h>
#include <sys/proc.h>
#include <sys/user.h>
#include <sys/conf.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/kernel.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/timepps.h>
#include <sys/vnode.h>
#include <sys/kauth.h>

#include <machine/intr.h>
#include <machine/bus.h>

#include <machine/pio.h>

#include <dev/ofw/openfirm.h>

#include <machine/autoconf.h>

#include <arch/powerpc/mpc5200/mpc5200reg.h>
#include <arch/powerpc/mpc5200/mpc5200var.h>
#include <arch/powerpc/mpc5200/pscreg.h>
#include <arch/powerpc/mpc5200/pscvar.h>
#include <dev/cons.h>

static void psc_enable_debugport(struct psc_softc *);

void	psc_config(struct psc_softc *);
void	psc_shutdown(struct psc_softc *);
int	pscspeed(long, long);
#if 0
static	u_char	cflag2lcr(tcflag_t);
#endif
int	pscparam(struct tty *, struct termios *);
void	pscstart(struct tty *);
int	pschwiflow(struct tty *, int);

void	psc_loadchannelregs(struct psc_softc *);
void	psc_hwiflow(struct psc_softc *);
void	psc_break(struct psc_softc *, int);
void	psc_modem(struct psc_softc *, int);
void	tiocm_to_psc(struct psc_softc *, u_long, int);
int	psc_to_tiocm(struct psc_softc *);
void	psc_iflush(struct psc_softc *);

int	psc_common_getc(dev_t, bus_space_tag_t, bus_space_handle_t);
void	psc_common_putc(dev_t, bus_space_tag_t, bus_space_handle_t, int);

int	pscinit(bus_space_tag_t, bus_addr_t, int, int, tcflag_t,
		bus_space_handle_t *);

int	psccngetc(dev_t);
void	psccnputc(dev_t, int);
void	psccnpollc(dev_t, int);

#define	integrate	static inline
void 	pscsoft(void *);
integrate void psc_rxsoft(struct psc_softc *, struct tty *);
integrate void psc_txsoft(struct psc_softc *, struct tty *);
integrate void psc_stsoft(struct psc_softc *, struct tty *);
integrate void psc_schedrx(struct psc_softc *);
void	pscdiag(void *);

extern struct cfdriver psc_cd;

dev_type_open(pscopen);
dev_type_close(pscclose);
dev_type_read(pscread);
dev_type_write(pscwrite);
dev_type_ioctl(pscioctl);
dev_type_stop(pscstop);
dev_type_tty(psctty);
dev_type_poll(pscpoll);

const struct cdevsw psc_cdevsw = {
	pscopen, pscclose, pscread, pscwrite, pscioctl,
	pscstop, psctty, pscpoll, nommap, ttykqfilter, D_TTY
};

/*
 * Make this an option variable one can patch.
 * But be warned:  this must be a power of 2!
 */
u_int psc_rbuf_size = PSC_RING_SIZE;

/* Stop input when 3/4 of the ring is full; restart when only 1/4 is full. */
u_int psc_rbuf_hiwat = (PSC_RING_SIZE * 1) / 4;
u_int psc_rbuf_lowat = (PSC_RING_SIZE * 3) / 4;

static int pscconsattached;
static int pscconsrate;
static bus_space_tag_t pscconsiot;
static bus_space_handle_t pscconsioh;
static bus_addr_t pscconsiobase;
static tcflag_t pscconscflag;
static struct cnm_state psc_cnm_state;

#ifdef KGDB
#include <sys/kgdb.h>

static bus_space_tag_t psckgdbiot;
static bus_space_handle_t psckgdbioh;
static bus_addr_t psckgdbiobase;
static int psc_kgdb_attached;

int	psc_kgdb_getc(void *);
void	psc_kgdb_putc(void *, int);
#endif /* KGDB */

#define	PSCUNIT_MASK	0x7ffff
#define	PSCDIALOUT_MASK	0x80000

#define	PSCUNIT(x)	(minor(x) & PSCUNIT_MASK)
#define	PSCDIALOUT(x)	(minor(x) & PSCDIALOUT_MASK)

#define	PSC_ISALIVE(sc)	((sc)->enabled != 0 && \
			 device_is_active((sc)->sc_dev))

#define	BR	BUS_SPACE_BARRIER_READ
#define	BW	BUS_SPACE_BARRIER_WRITE
#define PSC_BARRIER(r, f) \
	bus_space_barrier((r)->cr_iot, (r)->cr_ioh, 0, (r)->cr_nports, (f))

static int psc_match(device_t, cfdata_t, void *);
static void psc_attach(device_t, device_t, void *);

CFATTACH_DECL_NEW(psc, sizeof(struct psc_softc),
	psc_match, psc_attach, psc_detach, psc_activate);

static int
psc_match(device_t parent, cfdata_t cf, void *aux)
{
	struct confargs *ca = aux;
	char name[32];

	if (strcmp(ca->ca_name, "serial") != 0)
		return 0;

	OF_getprop(ca->ca_node, "compatible", name, sizeof(name));
	if (strcmp(name, "mpc5200-psc-uart") != 0)
		return 0;
	return 1;
}

static void
psc_attach(device_t parent, device_t self, void *aux)
{
	struct psc_softc *sc = device_private(self);
	struct confargs *ca = aux;
	struct tty *tp;
	struct reg {
		uint32_t addr;
		uint32_t size;
	} range, *rp = &range;
	uint32_t interrupts[3];
	int irq = 65;		/* XXX */

	sc->sc_dev = self;
	sc->sc_iot = ca->ca_tag;

	aprint_normal("\n");

	if (OF_getprop(ca->ca_node, "reg", rp, sizeof(range))
	    != sizeof(range))
		return;

	if (OF_getprop(ca->ca_node, "interrupts", interrupts,
		       sizeof(interrupts)) != sizeof(interrupts))
		return;

	if (bus_space_map(sc->sc_iot, rp->addr, rp->size, 0, &sc->sc_ioh) != 0)
		aprint_normal_dev(self, "map failed\n");

	callout_init(&sc->sc_diag_callout, 0);
	mutex_init(&sc->sc_lock, MUTEX_DEFAULT, IPL_HIGH);

	sc->sc_imr = 0;
	bus_space_write_2(sc->sc_iot, sc->sc_ioh, PSC_IMR, sc->sc_imr);

	if (sc->sc_iot == pscconsiot && rp->addr == pscconsiobase) {
		pscconsattached = 1;

		/* Make sure the console is always "hardwired". */
		delay(10000);			/* wait for output to finish */
		SET(sc->sc_hwflags, PSC_HW_CONSOLE);
		SET(sc->sc_swflags, TIOCFLAG_SOFTCAR);
	} else {
		SET(sc->sc_hwflags, PSC_HW_CONSOLE);
	}

	tp = ttymalloc();
	tp->t_oproc = pscstart;
	tp->t_param = pscparam;
	tp->t_hwiflow = pschwiflow;

	/* XXX - enable fifo ? */

	sc->sc_tty = tp;
	sc->sc_rbuf = malloc(psc_rbuf_size << 1, M_DEVBUF, M_NOWAIT);
	sc->sc_rbput = sc->sc_rbget = sc->sc_rbuf;
	sc->sc_rbavail = psc_rbuf_size;
	if (sc->sc_rbuf == NULL) {
		aprint_error_dev(sc->sc_dev,
		    "unable to allocate ring buffer\n");
		return;
	}
	sc->sc_ebuf = sc->sc_rbuf + (psc_rbuf_size << 1);

	tty_attach(tp);

	if (ISSET(sc->sc_hwflags, PSC_HW_CONSOLE)) {
		int maj;

		/* locate the major number */
		maj = cdevsw_lookup_major(&psc_cdevsw);

		tp->t_dev = cn_tab->cn_dev = makedev(maj,
						     device_unit(sc->sc_dev));

		aprint_normal_dev(sc->sc_dev, "console\n");
	}

	sc->sc_si = softint_establish(SOFTINT_SERIAL, pscsoft, sc);

#if NRND > 0 && defined(RND_COM)
	rnd_attach_source(&sc->rnd_source, device_xname(sc->sc_dev),
			  RND_TYPE_TTY, 0);
#endif

	/* if there are no enable/disable functions, assume the device
	   is always enabled */
	if (!sc->enable)
		sc->enabled = 1;

	psc_config(sc);

	SET(sc->sc_hwflags, PSC_HW_DEV_OK);

	if (!pmf_device_register1(self, psc_suspend, psc_resume, psc_cleanup))
		aprint_error_dev(self, "couldn't establish power handler\n");

	irq = (interrupts[0] * 32) + interrupts[1];
	sc->sc_irq = irq;
	sc->sc_ih = intr_establish(sc->sc_irq, IST_EDGE, IPL_SERIAL,
				   pscintr, sc);
}

/*
 * No locking in this routine; it is only called during attach,
 * or with the port already locked.
 */
static void
psc_enable_debugport(struct psc_softc *sc)
{

	/* Turn on line break interrupt, set carrier. */
	sc->sc_imr = INT_RXRDY;
	bus_space_write_2(sc->sc_iot, sc->sc_ioh, PSC_IMR, sc->sc_imr);
#if 0
	SET(sc->sc_mcr, MCR_DTR | MCR_RTS);
	CSR_WRITE_1(&sc->sc_regs, COM_REG_MCR, sc->sc_mcr);
#endif
}

void
psc_config(struct psc_softc *sc)
{

	/* Disable interrupts before configuring the device. */
	sc->sc_imr = 0;
	bus_space_write_1(sc->sc_iot, sc->sc_ioh, PSC_IMR, sc->sc_imr);

	if (ISSET(sc->sc_hwflags, PSC_HW_CONSOLE|PSC_HW_KGDB))
		psc_enable_debugport(sc);
}

int
psc_detach(device_t self, int flags)
{
	struct psc_softc *sc = device_private(self);
	int maj, mn;

        if (ISSET(sc->sc_hwflags, PSC_HW_CONSOLE))
		return EBUSY;

	if (sc->sc_ih != NULL)
		intr_disestablish(sc->sc_ih);

	pmf_device_deregister(self);

	/* locate the major number */
	maj = cdevsw_lookup_major(&psc_cdevsw);

	/* Nuke the vnodes for any open instances. */
	mn = device_unit(self);
	vdevgone(maj, mn, mn, VCHR);

	mn |= PSCDIALOUT_MASK;
	vdevgone(maj, mn, mn, VCHR);

	if (sc->sc_rbuf == NULL) {
		/*
		 * Ring buffer allocation failed in the psc_attach_subr,
		 * only the tty is allocated, and nothing else.
		 */
		ttyfree(sc->sc_tty);
		return 0;
	}

	/* Free the receive buffer. */
	free(sc->sc_rbuf, M_DEVBUF);

	/* Detach and free the tty. */
	tty_detach(sc->sc_tty);
	ttyfree(sc->sc_tty);

	/* Unhook the soft interrupt handler. */
	softint_disestablish(sc->sc_si);

#if NRND > 0 && defined(RND_PSC)
	/* Unhook the entropy source. */
	rnd_detach_source(&sc->rnd_source);
#endif
	callout_destroy(&sc->sc_diag_callout);

	/* Destroy the lock. */
	mutex_destroy(&sc->sc_lock);

	return (0);
}

int
psc_activate(device_t self, enum devact act)
{
	struct psc_softc *sc = device_private(self);
	int rv = 0;

	switch (act) {
	case DVACT_ACTIVATE:
		rv = EOPNOTSUPP;
		break;

	case DVACT_DEACTIVATE:
		if (sc->sc_hwflags & (PSC_HW_CONSOLE|PSC_HW_KGDB)) {
			rv = EBUSY;
			break;
		}

		if (sc->disable != NULL && sc->enabled != 0) {
			(*sc->disable)(sc);
			sc->enabled = 0;
		}
		break;
	}

	return (rv);
}

void
psc_shutdown(struct psc_softc *sc)
{
	struct tty *tp = sc->sc_tty;

	mutex_spin_enter(&sc->sc_lock);

	/* If we were asserting flow control, then deassert it. */
	SET(sc->sc_rx_flags, RX_IBUF_BLOCKED);
	psc_hwiflow(sc);

	/* Clear any break condition set with TIOCSBRK. */
	psc_break(sc, 0);

	/*
	 * Hang up if necessary.  Wait a bit, so the other side has time to
	 * notice even if we immediately open the port again.
	 * Avoid tsleeping above splhigh().
	 */
	if (ISSET(tp->t_cflag, HUPCL)) {
		psc_modem(sc, 0);
		mutex_spin_exit(&sc->sc_lock);
		/* XXX will only timeout */
		(void) kpause(ttclos, false, hz, NULL);
		mutex_spin_enter(&sc->sc_lock);
	}

	/* Turn off interrupts. */
	if (ISSET(sc->sc_hwflags, PSC_HW_CONSOLE)) {
		sc->sc_imr = INT_DB; /* interrupt on break */
	} else
		sc->sc_imr = 0;

	bus_space_write_2(sc->sc_iot, sc->sc_ioh, PSC_IMR, sc->sc_imr);

	mutex_spin_exit(&sc->sc_lock);

	if (sc->disable) {
#ifdef DIAGNOSTIC
		if (!sc->enabled)
			panic("psc_shutdown: not enabled?");
#endif
		(*sc->disable)(sc);
		sc->enabled = 0;
	}
}

int
pscopen(dev_t dev, int flag, int mode, struct lwp *l)
{
	struct psc_softc *sc;
	struct tty *tp;
	int s;
	int error;

	sc = device_lookup_private(&psc_cd, PSCUNIT(dev));
	if (sc == NULL || !ISSET(sc->sc_hwflags, PSC_HW_DEV_OK) ||
		sc->sc_rbuf == NULL)
		return (ENXIO);

	if (!device_is_active(sc->sc_dev))
		return (ENXIO);

#ifdef KGDB
	/*
	 * If this is the kgdb port, no other use is permitted.
	 */
	if (ISSET(sc->sc_hwflags, PSC_HW_KGDB))
		return (EBUSY);
#endif

	tp = sc->sc_tty;

	if (kauth_authorize_device_tty(l->l_cred, KAUTH_DEVICE_TTY_OPEN, tp))
		return (EBUSY);

	s = spltty();

	/*
	 * Do the following iff this is a first open.
	 */
	if (!ISSET(tp->t_state, TS_ISOPEN) && tp->t_wopen == 0) {
		struct termios t;

		tp->t_dev = dev;


		if (sc->enable) {
			if ((*sc->enable)(sc)) {
				splx(s);
				aprint_error_dev(sc->sc_dev,
				    "device enable failed\n");
				return (EIO);
			}
			mutex_spin_enter(&sc->sc_lock);
			sc->enabled = 1;
			psc_config(sc);
		} else {
			mutex_spin_enter(&sc->sc_lock);
		}

		/* Turn on interrupts. */
		sc->sc_imr = INT_RXRDY; /* | INTIER_ERLS | IER_EMSC; */
		bus_space_write_2(sc->sc_iot, sc->sc_ioh, PSC_IMR,
				  sc->sc_imr);

		/* Fetch the current modem control status, needed later. */
		sc->sc_sr = bus_space_read_2(sc->sc_iot, sc->sc_ioh, PSC_SR);

		/* Clear PPS capture state on first open. */
		mutex_spin_enter(&timecounter_lock);
		memset(&sc->sc_pps_state, 0, sizeof(sc->sc_pps_state));
		sc->sc_pps_state.ppscap = PPS_CAPTUREASSERT | PPS_CAPTURECLEAR;
		pps_init(&sc->sc_pps_state);
		mutex_spin_exit(&timecounter_lock);

		mutex_spin_exit(&sc->sc_lock);

		/*
		 * Initialize the termios status to the defaults.  Add in the
		 * sticky bits from TIOCSFLAGS.
		 */
		if (ISSET(sc->sc_hwflags, PSC_HW_CONSOLE)) {
			t.c_ospeed = pscconsrate;
			t.c_cflag = pscconscflag;
		} else {
			t.c_ospeed = TTYDEF_SPEED;
			t.c_cflag = TTYDEF_CFLAG;
		}
		t.c_ispeed = t.c_ospeed;
		if (ISSET(sc->sc_swflags, TIOCFLAG_CLOCAL))
			SET(t.c_cflag, CLOCAL);
		if (ISSET(sc->sc_swflags, TIOCFLAG_CRTSCTS))
			SET(t.c_cflag, CRTSCTS);
		if (ISSET(sc->sc_swflags, TIOCFLAG_MDMBUF))
			SET(t.c_cflag, MDMBUF);
		/* Make sure pscparam() will do something. */
		tp->t_ospeed = 0;
		(void) pscparam(tp, &t);
		tp->t_iflag = TTYDEF_IFLAG;
		tp->t_oflag = TTYDEF_OFLAG;
		tp->t_lflag = TTYDEF_LFLAG;
		ttychars(tp);
		ttsetwater(tp);

		mutex_spin_enter(&sc->sc_lock);

		/*
		 * Turn on DTR.  We must always do this, even if carrier is not
		 * present, because otherwise we'd have to use TIOCSDTR
		 * immediately after setting CLOCAL, which applications do not
		 * expect.  We always assert DTR while the device is open
		 * unless explicitly requested to deassert it.
		 */
		psc_modem(sc, 1);

		/* Clear the input ring, and unblock. */
		sc->sc_rbput = sc->sc_rbget = sc->sc_rbuf;
		sc->sc_rbavail = psc_rbuf_size;
		psc_iflush(sc);
		CLR(sc->sc_rx_flags, RX_ANY_BLOCK);
		psc_hwiflow(sc);

#ifdef PSC_DEBUG
		if (psc_debug)
			pscstatus(sc, "pscopen  ");
#endif

		mutex_spin_exit(&sc->sc_lock);
	}

	splx(s);

	error = ttyopen(tp, PSCDIALOUT(dev), ISSET(flag, O_NONBLOCK));
	if (error)
		goto bad;

	error = (*tp->t_linesw->l_open)(dev, tp);
	if (error)
		goto bad;

	return (0);

bad:
	if (!ISSET(tp->t_state, TS_ISOPEN) && tp->t_wopen == 0) {
		/*
		 * We failed to open the device, and nobody else had it opened.
		 * Clean up the state as appropriate.
		 */
		psc_shutdown(sc);
	}

	return (error);
}

int
pscclose(dev_t dev, int flag, int mode, struct lwp *l)
{
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(dev));
	struct tty *tp = sc->sc_tty;

	/* XXX This is for cons.c. */
	if (!ISSET(tp->t_state, TS_ISOPEN))
		return (0);

	(*tp->t_linesw->l_close)(tp, flag);
	ttyclose(tp);

	if (PSC_ISALIVE(sc) == 0)
		return (0);

	if (!ISSET(tp->t_state, TS_ISOPEN) && tp->t_wopen == 0) {
		/*
		 * Although we got a last close, the device may still be in
		 * use; e.g. if this was the dialout node, and there are still
		 * processes waiting for carrier on the non-dialout node.
		 */
		psc_shutdown(sc);
	}

	return (0);
}

int
pscread(dev_t dev, struct uio *uio, int flag)
{
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(dev));
	struct tty *tp = sc->sc_tty;

	if (PSC_ISALIVE(sc) == 0)
		return (EIO);

	return ((*tp->t_linesw->l_read)(tp, uio, flag));
}

int
pscwrite(dev_t dev, struct uio *uio, int flag)
{
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(dev));
	struct tty *tp = sc->sc_tty;

	if (PSC_ISALIVE(sc) == 0)
		return (EIO);

	return ((*tp->t_linesw->l_write)(tp, uio, flag));
}

int
pscpoll(dev_t dev, int events, struct lwp *l)
{
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(dev));
	struct tty *tp = sc->sc_tty;

	if (PSC_ISALIVE(sc) == 0)
		return (POLLHUP);

	return ((*tp->t_linesw->l_poll)(tp, events, l));
}

struct tty *
psctty(dev_t dev)
{
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(dev));
	struct tty *tp = sc->sc_tty;

	return (tp);
}

int
pscioctl(dev_t dev, u_long cmd, void *data, int flag, struct lwp *l)
{
	struct psc_softc *sc;
	struct tty *tp;
	int error;

	sc = device_lookup_private(&psc_cd, PSCUNIT(dev));
	if (sc == NULL)
		return ENXIO;
	if (PSC_ISALIVE(sc) == 0)
		return (EIO);

	tp = sc->sc_tty;

	error = (*tp->t_linesw->l_ioctl)(tp, cmd, data, flag, l);
	if (error != EPASSTHROUGH)
		return (error);

	error = ttioctl(tp, cmd, data, flag, l);
	if (error != EPASSTHROUGH)
		return (error);

	error = 0;
	switch (cmd) {
	case TIOCSFLAGS:
		error = kauth_authorize_device_tty(l->l_cred,
		    KAUTH_DEVICE_TTY_PRIVSET, tp);
		break;
	default:
		/* nothing */
		break;
	}
	if (error) {
		return error;
	}

	mutex_spin_enter(&sc->sc_lock);

	switch (cmd) {
	case TIOCSBRK:
		psc_break(sc, 1);
		break;

	case TIOCCBRK:
		psc_break(sc, 0);
		break;

	case TIOCSDTR:
		psc_modem(sc, 1);
		break;

	case TIOCCDTR:
		psc_modem(sc, 0);
		break;

	case TIOCGFLAGS:
		*(int *)data = sc->sc_swflags;
		break;

	case TIOCSFLAGS:
		sc->sc_swflags = *(int *)data;
		break;

	case TIOCMSET:
	case TIOCMBIS:
	case TIOCMBIC:
		tiocm_to_psc(sc, cmd, *(int *)data);
		break;

	case TIOCMGET:
		*(int *)data = psc_to_tiocm(sc);
		break;

	case PPS_IOC_CREATE:
	case PPS_IOC_DESTROY:
	case PPS_IOC_GETPARAMS:
	case PPS_IOC_SETPARAMS:
	case PPS_IOC_GETCAP:
	case PPS_IOC_FETCH:
#ifdef PPS_SYNC
	case PPS_IOC_KCBIND:
#endif
		mutex_spin_enter(&timecounter_lock);
		error = pps_ioctl(cmd, data, &sc->sc_pps_state);
		mutex_spin_exit(&timecounter_lock);
		break;

	case TIOCDCDTIMESTAMP:	/* XXX old, overloaded  API used by xntpd v3 */
		mutex_spin_enter(&timecounter_lock);
#ifndef PPS_TRAILING_EDGE
		TIMESPEC_TO_TIMEVAL((struct timeval *)data,
		    &sc->sc_pps_state.ppsinfo.assert_timestamp);
#else
		TIMESPEC_TO_TIMEVAL((struct timeval *)data,
		    &sc->sc_pps_state.ppsinfo.clear_timestamp);
#endif
		mutex_spin_exit(&timecounter_lock);
		break;

	default:
		error = EPASSTHROUGH;
		break;
	}

	mutex_spin_exit(&sc->sc_lock);

#ifdef PSC_DEBUG
	if (psc_debug)
		pscstatus(sc, "pscioctl ");
#endif

	return (error);
}

static inline void
psc_schedrx(struct psc_softc *sc)
{

	sc->sc_rx_ready = 1;

	/* Wake up the poller. */
	softint_schedule(sc->sc_si);
}

void
psc_break(struct psc_softc *sc, int onoff)
{

	if (onoff)
		sc->sc_cr = CMD_START_BRK;
	else
		sc->sc_cr = CMD_STOP_BRK;

	if (!sc->sc_heldchange) {
		if (sc->sc_tx_busy) {
			sc->sc_heldtbc = sc->sc_tbc;
			sc->sc_tbc = 0;
			sc->sc_heldchange = 1;
		} else
			psc_loadchannelregs(sc);
	}
}

void
psc_modem(struct psc_softc *sc, int onoff)
{
#if 0
	if (sc->sc_mcr_dtr == 0)
		return;

	if (onoff)
		SET(sc->sc_mcr, sc->sc_mcr_dtr);
	else
		CLR(sc->sc_mcr, sc->sc_mcr_dtr);

	if (!sc->sc_heldchange) {
		if (sc->sc_tx_busy) {
			sc->sc_heldtbc = sc->sc_tbc;
			sc->sc_tbc = 0;
			sc->sc_heldchange = 1;
		} else
			psc_loadchannelregs(sc);
	}
#endif
}

void
tiocm_to_psc(struct psc_softc *sc, u_long how, int ttybits)
{
#if 0
	u_char pscbits;

	pscbits = 0;
	if (ISSET(ttybits, TIOCM_DTR))
		SET(pscbits, MCR_DTR);
	if (ISSET(ttybits, TIOCM_RTS))
		SET(pscbits, MCR_RTS);

	switch (how) {
	case TIOCMBIC:
		CLR(sc->sc_mcr, pscbits);
		break;

	case TIOCMBIS:
		SET(sc->sc_mcr, pscbits);
		break;

	case TIOCMSET:
		CLR(sc->sc_mcr, MCR_DTR | MCR_RTS);
		SET(sc->sc_mcr, pscbits);
		break;
	}

	if (!sc->sc_heldchange) {
		if (sc->sc_tx_busy) {
			sc->sc_heldtbc = sc->sc_tbc;
			sc->sc_tbc = 0;
			sc->sc_heldchange = 1;
		} else
			psc_loadchannelregs(sc);
	}
#endif
}

int
psc_to_tiocm(struct psc_softc *sc)
{

	u_char pscbits;
	int ttybits = 0;
#if 0
	pscbits = sc->sc_mcr;
	if (ISSET(pscbits, MCR_DTR))
		SET(ttybits, TIOCM_DTR);
	if (ISSET(pscbits, MCR_RTS))
		SET(ttybits, TIOCM_RTS);
#endif
	pscbits = sc->sc_ipcr;
	if (ISSET(pscbits, IPCR_DCD))
		SET(ttybits, TIOCM_CD);
	if (ISSET(pscbits, IPCR_CTS))
		SET(ttybits, TIOCM_CTS);

#if 0
	if (ISSET(pscbits, MSR_DSR))
		SET(ttybits, TIOCM_DSR);
	if (ISSET(pscbits, MSR_RI | MSR_TERI))
		SET(ttybits, TIOCM_RI);
	if (ISSET(sc->sc_isr, INT_RXRDY | INT_TXRDY | IER_ERLS | IER_EMSC))
		SET(ttybits, TIOCM_LE);
#endif
	return (ttybits);
}

#if 0
static u_char
cflag2lcr(tcflag_t cflag)
{
	u_char lcr = 0;

	switch (ISSET(cflag, CSIZE)) {
	case CS5:
		SET(lcr, LCR_5BITS);
		break;
	case CS6:
		SET(lcr, LCR_6BITS);
		break;
	case CS7:
		SET(lcr, LCR_7BITS);
		break;
	case CS8:
		SET(lcr, LCR_8BITS);
		break;
	}
	if (ISSET(cflag, PARENB)) {
		SET(lcr, LCR_PENAB);
		if (!ISSET(cflag, PARODD))
			SET(lcr, LCR_PEVEN);
	}
	if (ISSET(cflag, CSTOPB))
		SET(lcr, LCR_STOPB);

	return (lcr);
}
#endif

int
pscspeed(long speed, long frequency)
{
#define	divrnd(n, q)	(((n)*2/(q)+1)/2)	/* divide and round off */

	int x, err;
	int divisor = 16;

#if 0
	if (speed == 0)
		return (0);
#endif
	if (speed <= 0)
		return (-1);
	x = divrnd(frequency / divisor, speed);
	if (x <= 0)
		return (-1);
	err = divrnd(((quad_t)frequency) * 1000 / divisor, speed * x) - 1000;
	if (err < 0)
		err = -err;
	if (err > PSC_TOLERANCE)
		return (-1);
	return (x);

#undef	divrnd
}

int
pscparam(struct tty *tp, struct termios *t)
{
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(tp->t_dev));
	int ospeed;
#if 0
	u_char cr;
#endif

	if (PSC_ISALIVE(sc) == 0)
		return (EIO);

	ospeed = pscspeed(t->c_ospeed, sc->sc_frequency);

	/* Check requested parameters. */
	if (ospeed < 0)
		return (EINVAL);
	if (t->c_ispeed && t->c_ispeed != t->c_ospeed)
		return (EINVAL);

	/*
	 * For the console, always force CLOCAL and !HUPCL, so that the port
	 * is always active.
	 */
	if (ISSET(sc->sc_swflags, TIOCFLAG_SOFTCAR) ||
	    ISSET(sc->sc_hwflags, PSC_HW_CONSOLE)) {
		SET(t->c_cflag, CLOCAL);
		CLR(t->c_cflag, HUPCL);
	}

	/*
	 * If there were no changes, don't do anything.  This avoids dropping
	 * input and improves performance when all we did was frob things like
	 * VMIN and VTIME.
	 */
	if (tp->t_ospeed == t->c_ospeed &&
	    tp->t_cflag == t->c_cflag)
		return (0);

#if 0
	cr = ISSET(sc->sc_cr, LCR_SBREAK) | cflag2cr(t->c_cflag);
#endif
	mutex_spin_enter(&sc->sc_lock);
#if 0
	sc->sc_lcr = lcr;
#endif
	/*
	 * If we're not in a mode that assumes a connection is present, then
	 * ignore carrier changes.
	 */
	if (ISSET(t->c_cflag, CLOCAL | MDMBUF))
		sc->sc_ipcr_dcd = 0;
	else
		sc->sc_ipcr_dcd = IPCR_DELTA_DCD;

	/*
	 * Set the flow control pins depending on the current flow control
	 * mode.
	 */
	if (ISSET(t->c_cflag, CRTSCTS)) {
#if 0
		sc->sc_mcr_dtr = MCR_DTR;
		sc->sc_mcr_rts = MCR_RTS;
#endif
		sc->sc_ipcr_cts = IPCR_DELTA_CTS;
		/* Set MR2 */
#if 0
		sc->sc_efr = EFR_AUTORTS | EFR_AUTOCTS;
#endif
	} else if (ISSET(t->c_cflag, MDMBUF)) {
		/*
		 * For DTR/DCD flow control, make sure we don't toggle DTR for
		 * carrier detection.
		 */
#if 0
		sc->sc_mcr_dtr = 0;
		sc->sc_mcr_rts = MCR_DTR;
#endif
		sc->sc_ipcr_cts = IPCR_DELTA_DCD;
#if 0
		sc->sc_efr = 0;
#endif
	} else {
		/*
		 * If no flow control, then always set RTS.  This will make
		 * the other side happy if it mistakenly thinks we're doing
		 * RTS/CTS flow control.
		 */
#if 0
		sc->sc_mcr_dtr = MCR_DTR | MCR_RTS;
		sc->sc_mcr_rts = 0;
#endif
		sc->sc_ipcr_cts = 0;
#if 0
		sc->sc_efr = 0;
#endif
#if 0
		if (ISSET(sc->sc_mcr, MCR_DTR))
			SET(sc->sc_mcr, MCR_RTS);
		else
			CLR(sc->sc_mcr, MCR_RTS);
#endif

	}
	sc->sc_ipcr_mask = sc->sc_ipcr_cts | sc->sc_ipcr_dcd;
#if 0
	if (ospeed == 0)
		CLR(sc->sc_mcr, sc->sc_mcr_dtr);
	else
		SET(sc->sc_mcr, sc->sc_mcr_dtr);
#endif

#if 0
	sc->sc_dlbl = ospeed;
	sc->sc_dlbh = ospeed >> 8;

	/*
	 * Set the FIFO threshold based on the receive speed.
	 *
	 *  * If it's a low speed, it's probably a mouse or some other
	 *    interactive device, so set the threshold low.
	 *  * If it's a high speed, trim the trigger level down to prevent
	 *    overflows.
	 *  * Otherwise set it a bit higher.
	 */
	if (ISSET(sc->sc_hwflags, PSC_HW_FIFO)) {
		if (t->c_ospeed <= 1200)
			sc->sc_fifo = FIFO_ENABLE | FIFO_TRIGGER_1;
		else if (t->c_ospeed <= 38400)
			sc->sc_fifo = FIFO_ENABLE | FIFO_TRIGGER_8;
		else
			sc->sc_fifo = FIFO_ENABLE | FIFO_TRIGGER_4;
	} else
		sc->sc_fifo = 0;
#endif
	/* And copy to tty. */
	tp->t_ispeed = t->c_ospeed;
	tp->t_ospeed = t->c_ospeed;
	tp->t_cflag = t->c_cflag;

	if (!sc->sc_heldchange) {
		if (sc->sc_tx_busy) {
			sc->sc_heldtbc = sc->sc_tbc;
			sc->sc_tbc = 0;
			sc->sc_heldchange = 1;
		} else
			psc_loadchannelregs(sc);
	}

	if (!ISSET(t->c_cflag, CHWFLOW)) {
		/* Disable the high water mark. */
		sc->sc_r_hiwat = 0;
		sc->sc_r_lowat = 0;
		if (ISSET(sc->sc_rx_flags, RX_TTY_OVERFLOWED)) {
			CLR(sc->sc_rx_flags, RX_TTY_OVERFLOWED);
			psc_schedrx(sc);
		}
		if (ISSET(sc->sc_rx_flags, RX_TTY_BLOCKED|RX_IBUF_BLOCKED)) {
			CLR(sc->sc_rx_flags, RX_TTY_BLOCKED|RX_IBUF_BLOCKED);
			psc_hwiflow(sc);
		}
	} else {
		sc->sc_r_hiwat = psc_rbuf_hiwat;
		sc->sc_r_lowat = psc_rbuf_lowat;
	}

	mutex_spin_exit(&sc->sc_lock);

	/*
	 * Update the tty layer's idea of the carrier bit, in case we changed
	 * CLOCAL or MDMBUF.  We don't hang up here; we only do that by
	 * explicit request.
	 */
	(void) (*tp->t_linesw->l_modem)(tp, ISSET(sc->sc_ipcr, IPCR_DCD));

#ifdef PSC_DEBUG
	if (psc_debug)
		pscstatus(sc, "pscparam ");
#endif

	if (!ISSET(t->c_cflag, CHWFLOW)) {
		if (sc->sc_tx_stopped) {
			sc->sc_tx_stopped = 0;
			pscstart(tp);
		}
	}

	return (0);
}

void
psc_iflush(struct psc_softc *sc)
{
#ifdef DIAGNOSTIC
	int reg;
#endif
	int timo;

#ifdef DIAGNOSTIC
	reg = 0xffff;
#endif
	timo = 50000;
	/* flush any pending I/O */
	while (ISSET(bus_space_read_2(sc->sc_iot, sc->sc_ioh, PSC_SR), SR_RXRDY)
	    && --timo)
#ifdef DIAGNOSTIC
		reg =
#else
		    (void)
#endif
		    bus_space_read_1(sc->sc_iot, sc->sc_ioh, PSC_RB);
#ifdef DIAGNOSTIC
	if (!timo)
		aprint_error_dev(sc->sc_dev, "psc_iflush timeout %02x\n", reg);
#endif
}

void
psc_loadchannelregs(struct psc_softc *sc)
{

	/* XXXXX necessary? */
	psc_iflush(sc);

	bus_space_write_1(sc->sc_iot, sc->sc_ioh, PSC_CR, sc->sc_cr);
	sc->sc_cr = 0;
#if 0
	CSR_WRITE_1(regsp, PSC_REG_IER, 0);

	if (ISSET(sc->sc_hwflags, PSC_HW_FLOW)) {
		KASSERT(sc->sc_type != PSC_TYPE_AU1x00);
		KASSERT(sc->sc_type != PSC_TYPE_16550_NOERS);
		/* no EFR on alchemy */
		CSR_WRITE_1(regsp, PSC_REG_EFR, sc->sc_efr);
		CSR_WRITE_1(regsp, PSC_REG_LCR, LCR_EERS);
	}
	if (sc->sc_type == PSC_TYPE_AU1x00) {
		/* alchemy has single separate 16-bit clock divisor register */
		CSR_WRITE_2(regsp, PSC_REG_DLBL, sc->sc_dlbl +
		    (sc->sc_dlbh << 8));
	} else {
		CSR_WRITE_1(regsp, PSC_REG_LCR, sc->sc_lcr | LCR_DLAB);
		CSR_WRITE_1(regsp, PSC_REG_DLBL, sc->sc_dlbl);
		CSR_WRITE_1(regsp, PSC_REG_DLBH, sc->sc_dlbh);
	}
	CSR_WRITE_1(regsp, PSC_REG_LCR, sc->sc_lcr);
	CSR_WRITE_1(regsp, PSC_REG_MCR, sc->sc_mcr_active = sc->sc_mcr);
	CSR_WRITE_1(regsp, PSC_REG_FIFO, sc->sc_fifo);
	if (sc->sc_type == PSC_TYPE_OMAP) {
		/* setup the fifos.  the FCR value is not used as long
		   as SCR[6] and SCR[7] are 0, which they are at reset
		   and we never touch the SCR register */
		uint8_t rx_fifo_trig = 40;
		uint8_t tx_fifo_trig = 60;
		uint8_t rx_start = 8;
		uint8_t rx_halt = 60;
		uint8_t tlr_value = ((rx_fifo_trig>>2) << 4) | (tx_fifo_trig>>2);
		uint8_t tcr_value = ((rx_start>>2) << 4) | (rx_halt>>2);

		/* enable access to TCR & TLR */
		CSR_WRITE_1(regsp, PSC_REG_MCR, sc->sc_mcr | MCR_TCR_TLR);

		/* write tcr and tlr values */
		CSR_WRITE_1(regsp, PSC_REG_TLR, tlr_value);
		CSR_WRITE_1(regsp, PSC_REG_TCR, tcr_value);

		/* disable access to TCR & TLR */
		CSR_WRITE_1(regsp, PSC_REG_MCR, sc->sc_mcr);

		/* enable again, but mode is based on speed */
		if (sc->sc_tty->t_termios.c_ospeed > 230400) {
			CSR_WRITE_1(regsp, PSC_REG_MDR1, MDR1_MODE_UART_13X);
		} else {
			CSR_WRITE_1(regsp, PSC_REG_MDR1, MDR1_MODE_UART_16X);
		}
	}

	CSR_WRITE_1(regsp, PSC_REG_IER, sc->sc_ier);
#endif
}

int
pschwiflow(struct tty *tp, int block)
{
#if 0
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(tp->t_dev));

	if (PSC_ISALIVE(sc) == 0)
		return (0);

	if (sc->sc_mcr_rts == 0)
		return (0);

	mutex_spin_enter(&sc->sc_lock);

	if (block) {
		if (!ISSET(sc->sc_rx_flags, RX_TTY_BLOCKED)) {
			SET(sc->sc_rx_flags, RX_TTY_BLOCKED);
			psc_hwiflow(sc);
		}
	} else {
		if (ISSET(sc->sc_rx_flags, RX_TTY_OVERFLOWED)) {
			CLR(sc->sc_rx_flags, RX_TTY_OVERFLOWED);
			psc_schedrx(sc);
		}
		if (ISSET(sc->sc_rx_flags, RX_TTY_BLOCKED)) {
			CLR(sc->sc_rx_flags, RX_TTY_BLOCKED);
			psc_hwiflow(sc);
		}
	}

	mutex_spin_exit(&sc->sc_lock);
#endif
	return (1);
}

/*
 * (un)block input via hw flowcontrol
 */
void
psc_hwiflow(struct psc_softc *sc)
{
#if 0
	if (sc->sc_mcr_rts == 0)
		return;

	if (ISSET(sc->sc_rx_flags, RX_ANY_BLOCK)) {
		CLR(sc->sc_mcr, sc->sc_mcr_rts);
		CLR(sc->sc_mcr_active, sc->sc_mcr_rts);
	} else {
		SET(sc->sc_mcr, sc->sc_mcr_rts);
		SET(sc->sc_mcr_active, sc->sc_mcr_rts);
	}
	bus_space_write_1(regsp, PSC_REG_MCR, sc->sc_mcr_active);
#endif
}


void
pscstart(struct tty *tp)
{
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(tp->t_dev));
	int s;

	if (PSC_ISALIVE(sc) == 0)
		return;

	s = spltty();
	if (ISSET(tp->t_state, TS_BUSY | TS_TIMEOUT | TS_TTSTOP))
		goto out;
	if (sc->sc_tx_stopped)
		goto out;
	if (!ttypull(tp))
		goto out;

	/* Grab the first contiguous region of buffer space. */
	{
		u_char *tba;
		int tbc;

		tba = tp->t_outq.c_cf;
		tbc = ndqb(&tp->t_outq, 0);

		mutex_spin_enter(&sc->sc_lock);

		sc->sc_tba = tba;
		sc->sc_tbc = tbc;
	}

	SET(tp->t_state, TS_BUSY);
	sc->sc_tx_busy = 1;

	/* Enable transmit completion interrupts if necessary. */
	if (!ISSET(sc->sc_imr, INT_TXRDY)) {
		SET(sc->sc_imr, INT_TXRDY);
		bus_space_write_2(sc->sc_iot, sc->sc_ioh,
				  PSC_IMR, sc->sc_imr);
	}

	/* Output the first chunk of the contiguous buffer. */
	if (!ISSET(sc->sc_hwflags, PSC_HW_NO_TXPRELOAD)) {
		u_int n;

		n = sc->sc_tbc;
		if (n > PSC_FIFOLEN)
			n = PSC_FIFOLEN;
		bus_space_write_multi_1(sc->sc_iot, sc->sc_ioh,
				      PSC_TB, sc->sc_tba, n);
		sc->sc_tbc -= n;
		sc->sc_tba += n;
	}

	mutex_spin_exit(&sc->sc_lock);
out:
	splx(s);
	return;
}

/*
 * Stop output on a line.
 */
void
pscstop(struct tty *tp, int flag)
{
	struct psc_softc *sc =
	    device_lookup_private(&psc_cd, PSCUNIT(tp->t_dev));

	mutex_spin_enter(&sc->sc_lock);
	if (ISSET(tp->t_state, TS_BUSY)) {
		/* Stop transmitting at the next chunk. */
		sc->sc_tbc = 0;
		sc->sc_heldtbc = 0;
		if (!ISSET(tp->t_state, TS_TTSTOP))
			SET(tp->t_state, TS_FLUSH);
	}
	mutex_spin_exit(&sc->sc_lock);
}

void
pscdiag(void *arg)
{
	struct psc_softc *sc = arg;
	int overflows, floods;

	mutex_spin_enter(&sc->sc_lock);
	overflows = sc->sc_overflows;
	sc->sc_overflows = 0;
	floods = sc->sc_floods;
	sc->sc_floods = 0;
	sc->sc_errors = 0;
	mutex_spin_exit(&sc->sc_lock);

	log(LOG_WARNING, "%s: %d silo overflow%s, %d ibuf flood%s\n",
	    device_xname(sc->sc_dev),
	    overflows, overflows == 1 ? "" : "s",
	    floods, floods == 1 ? "" : "s");
}

static inline void
psc_rxsoft(struct psc_softc *sc, struct tty *tp)
{
	int (*rint)(int, struct tty *) = tp->t_linesw->l_rint;
	u_char *get, *end;
	u_int cc, scc;
	uint16_t sr;
	int code;

	end = sc->sc_ebuf;
	get = sc->sc_rbget;
	scc = cc = psc_rbuf_size - sc->sc_rbavail;

	if (cc == psc_rbuf_size) {
		sc->sc_floods++;
		if (sc->sc_errors++ == 0)
			callout_reset(&sc->sc_diag_callout, 60 * hz,
			    pscdiag, sc);
	}

	/* If not yet open, drop the entire buffer content here */
	if (!ISSET(tp->t_state, TS_ISOPEN)) {
		get += cc << 1;
		if (get >= end)
			get -= psc_rbuf_size << 1;
		cc = 0;
	}
	while (cc) {
		code = get[0];
		sr = get[1] | (get[2] << 8);
		if (ISSET(sr, SR_ORERR | SR_RB | SR_FE | SR_PE)) {
			if (ISSET(sr, SR_ORERR)) {
				sc->sc_overflows++;
				if (sc->sc_errors++ == 0)
					callout_reset(&sc->sc_diag_callout,
					    60 * hz, pscdiag, sc);
			}
			if (ISSET(sr, SR_RB | SR_FE))
				SET(code, TTY_FE);
			if (ISSET(sr, SR_PE))
				SET(code, TTY_PE);
		}
		if ((*rint)(code, tp) == -1) {
			/*
			 * The line discipline's buffer is out of space.
			 */
			if (!ISSET(sc->sc_rx_flags, RX_TTY_BLOCKED)) {
				/*
				 * We're either not using flow control, or the
				 * line discipline didn't tell us to block for
				 * some reason.  Either way, we have no way to
				 * know when there's more space available, so
				 * just drop the rest of the data.
				 */
				get += cc << 1;
				if (get >= end)
					get -= psc_rbuf_size << 1;
				cc = 0;
			} else {
				/*
				 * Don't schedule any more receive processing
				 * until the line discipline tells us there's
				 * space available (through pschwiflow()).
				 * Leave the rest of the data in the input
				 * buffer.
				 */
				SET(sc->sc_rx_flags, RX_TTY_OVERFLOWED);
			}
			break;
		}
		get += 3;
		if (get >= end)
			get = sc->sc_rbuf;
		cc--;
	}

	if (cc != scc) {
		sc->sc_rbget = get;
		mutex_spin_enter(&sc->sc_lock);

		cc = sc->sc_rbavail += scc - cc;
		/* Buffers should be ok again, release possible block. */
		if (cc >= sc->sc_r_lowat) {
			if (ISSET(sc->sc_rx_flags, RX_IBUF_OVERFLOWED)) {
				CLR(sc->sc_rx_flags, RX_IBUF_OVERFLOWED);
				SET(sc->sc_imr, INT_RXRDY);
				bus_space_write_2(sc->sc_iot, sc->sc_ioh,
						  PSC_IMR, sc->sc_imr);
			}
			if (ISSET(sc->sc_rx_flags, RX_IBUF_BLOCKED)) {
				CLR(sc->sc_rx_flags, RX_IBUF_BLOCKED);
				psc_hwiflow(sc);
			}
		}
		mutex_spin_exit(&sc->sc_lock);
	}
}

integrate void
psc_txsoft(struct psc_softc *sc, struct tty *tp)
{

	CLR(tp->t_state, TS_BUSY);
	if (ISSET(tp->t_state, TS_FLUSH))
		CLR(tp->t_state, TS_FLUSH);
	else
		ndflush(&tp->t_outq, (int)(sc->sc_tba - tp->t_outq.c_cf));
	(*tp->t_linesw->l_start)(tp);
}

integrate void
psc_stsoft(struct psc_softc *sc, struct tty *tp)
{
	u_char ipcr;

	mutex_spin_enter(&sc->sc_lock);
	ipcr = sc->sc_ipcr;
	mutex_spin_exit(&sc->sc_lock);

	if (ISSET(ipcr, sc->sc_ipcr_dcd)) {
		/*
		 * Inform the tty layer that carrier detect changed.
		 */
		(void) (*tp->t_linesw->l_modem)(tp, ISSET(ipcr, IPCR_DCD));
	}

	if (ISSET(ipcr, sc->sc_ipcr_cts)) {
		/* Block or unblock output according to flow control. */
		if (ISSET(ipcr, sc->sc_ipcr_cts)) {
			sc->sc_tx_stopped = 0;
			(*tp->t_linesw->l_start)(tp);
		} else {
			sc->sc_tx_stopped = 1;
		}
	}

#ifdef PSC_DEBUG
	if (psc_debug)
		pscstatus(sc, "psc_stsoft");
#endif
}

void
pscsoft(void *arg)
{
	struct psc_softc *sc = arg;
	struct tty *tp;

	if (PSC_ISALIVE(sc) == 0)
		return;

	tp = sc->sc_tty;

	if (sc->sc_rx_ready) {
		sc->sc_rx_ready = 0;
		psc_rxsoft(sc, tp);
	}

	if (sc->sc_st_check) {
		sc->sc_st_check = 0;
		psc_stsoft(sc, tp);
	}

	if (sc->sc_tx_done) {
		sc->sc_tx_done = 0;
		psc_txsoft(sc, tp);
	}
}

int
pscintr(void *arg)
{
	struct psc_softc *sc = arg;
	u_char *put, *end;
	u_int cc;
	uint16_t isr, sr;
	uint8_t ipcr;

	if (PSC_ISALIVE(sc) == 0)
		return (0);

	mutex_spin_enter(&sc->sc_lock);
	isr = bus_space_read_2(sc->sc_iot, sc->sc_ioh, PSC_ISR);

	end = sc->sc_ebuf;
	put = sc->sc_rbput;
	cc = sc->sc_rbavail;

again:	do {
		if (ISSET(isr, INT_DB)) {
			int cn_trapped = 0;

			cn_check_magic(sc->sc_tty->t_dev,
				       CNC_BREAK, psc_cnm_state);
			if (cn_trapped)
				continue;
#if defined(KGDB) && !defined(DDB)
			if (ISSET(sc->sc_hwflags, PSC_HW_KGDB)) {
				kgdb_connect(1);
				continue;
			}
#endif
		}

		sr = bus_space_read_2(sc->sc_iot, sc->sc_ioh, PSC_SR);
		if (ISSET(sr, SR_RCV_MASK) &&
		    !ISSET(sc->sc_rx_flags, RX_IBUF_OVERFLOWED)) {
			while (cc > 0) {
				int cn_trapped = 0;
				put[0] = bus_space_read_1(sc->sc_iot, sc->sc_ioh, PSC_RB);
				put[1] = sr & 0xff;
				put[2] = sr >> 8;
				cn_check_magic(sc->sc_tty->t_dev,
					       put[0], psc_cnm_state);
				if (cn_trapped)
					goto next;
				put += 3;
				if (put >= end)
					put = sc->sc_rbuf;
				cc--;
			next:
				sr = bus_space_read_2(sc->sc_iot, sc->sc_ioh,
						      PSC_SR);
				if (!ISSET(sr, SR_RCV_MASK))
					break;
			}

			/*
			 * Current string of inpscing characters ended because
			 * no more data was available or we ran out of space.
			 * Schedule a receive event if any data was received.
			 * If we're out of space, turn off receive interrupts.
			 */
			sc->sc_rbput = put;
			sc->sc_rbavail = cc;
			if (!ISSET(sc->sc_rx_flags, RX_TTY_OVERFLOWED))
				sc->sc_rx_ready = 1;

			/*
			 * See if we are in danger of overflowing a buffer. If
			 * so, use hardware flow control to ease the pressure.
			 */
			if (!ISSET(sc->sc_rx_flags, RX_IBUF_BLOCKED) &&
			    cc < sc->sc_r_hiwat) {
				SET(sc->sc_rx_flags, RX_IBUF_BLOCKED);
				psc_hwiflow(sc);
			}

			/*
			 * If we're out of space, disable receive interrupts
			 * until the queue has drained a bit.
			 */
			if (!cc) {
				SET(sc->sc_rx_flags, RX_IBUF_OVERFLOWED);
					CLR(sc->sc_imr, INT_RXRDY);
				bus_space_write_2(sc->sc_iot, sc->sc_ioh,
						  PSC_IMR, sc->sc_imr);
			}
		} else {
			if ((isr & (INT_RXRDY|INT_TXRDY)) == INT_RXRDY) {
				(void) bus_space_read_1(sc->sc_iot, sc->sc_ioh,
							PSC_RB);
				continue;
			}
		}

		ipcr = bus_space_read_1(sc->sc_iot, sc->sc_ioh, PSC_IPCR);
		sc->sc_ipcr = ipcr;
		if ((sc->sc_pps_state.ppsparam.mode & PPS_CAPTUREBOTH) &&
		    (ipcr & IPCR_DELTA_DCD)) {
			mutex_spin_enter(&timecounter_lock);
			pps_capture(&sc->sc_pps_state);
			pps_event(&sc->sc_pps_state,
			    (ipcr & IPCR_DCD) ?
			    PPS_CAPTUREASSERT :
			    PPS_CAPTURECLEAR);
			mutex_spin_exit(&timecounter_lock);
		}

		/*
		 * Process normal status changes
		 */
		if (ISSET(ipcr, sc->sc_ipcr_mask)) {
			/*
			 * Stop output immediately if we lose the output
			 * flow control signal or carrier detect.
			 */
			if (ISSET(~ipcr, sc->sc_ipcr_mask)) {
				sc->sc_tbc = 0;
				sc->sc_heldtbc = 0;
#ifdef PSC_DEBUG
				if (psc_debug)
					pscstatus(sc, "pscintr  ");
#endif
			}

			sc->sc_st_check = 1;
		}
	} while (bus_space_read_2(sc->sc_iot, sc->sc_ioh, PSC_ISR));

	/*
	 * Read SR again, since there may be an interrupt between
	 * the last SR read and ISR read above.
	 */
	sr = bus_space_read_2(sc->sc_iot, sc->sc_ioh, PSC_SR);

	/*
	 * See if data can be transmitted as well.
	 * Schedule tx done event if no data left
	 * and tty was marked busy.
	 */
	if (ISSET(sr, SR_TXRDY)) {
		/*
		 * If we've delayed a parameter change, do it now, and restart
		 * output.
		 */
		if (sc->sc_heldchange) {
			psc_loadchannelregs(sc);
			sc->sc_heldchange = 0;
			sc->sc_tbc = sc->sc_heldtbc;
			sc->sc_heldtbc = 0;
		}

		/* Output the next chunk of the contiguous buffer, if any. */
		if (sc->sc_tbc > 0) {
			u_int n;

			n = sc->sc_tbc;
			if (n > PSC_FIFOLEN)
				n = PSC_FIFOLEN;
			bus_space_write_multi_1(sc->sc_iot, sc->sc_ioh,
					      PSC_TB, sc->sc_tba, n);
			sc->sc_tbc -= n;
			sc->sc_tba += n;
		} else {
			/* Disable transmit completion interrupts if necessary. */
			if (ISSET(sc->sc_imr, INT_TXRDY)) {
				CLR(sc->sc_imr, INT_TXRDY);
				bus_space_write_2(sc->sc_iot, sc->sc_ioh,
						  PSC_IMR, sc->sc_imr);
			}
			if (sc->sc_tx_busy) {
				sc->sc_tx_busy = 0;
				sc->sc_tx_done = 1;
			}
		}
	}

	if ((isr = bus_space_read_2(sc->sc_iot, sc->sc_ioh, PSC_ISR)))
		goto again;

	mutex_spin_exit(&sc->sc_lock);

	/* Wake up the poller. */
	softint_schedule(sc->sc_si);

#if NRND > 0 && defined(RND_PSC)
	rnd_add_uint32(&sc->rnd_source, iir | lsr);
#endif

	return (1);
}

/*
 * The following functions are polled getc and putc routines, shared
 * by the console and kgdb glue.
 *
 * The read-ahead code is so that you can detect pending in-band
 * cn_magic in polled mode while doing output rather than having to
 * wait until the kernel decides it needs input.
 */

#define MAX_READAHEAD	20
static int psc_readahead[MAX_READAHEAD];
static int psc_readaheadcount = 0;

int
psc_common_getc(dev_t dev, bus_space_tag_t iot, bus_space_handle_t ioh)
{
	int s = splserial();
	u_char stat, c;

	/* got a character from reading things earlier */
	if (psc_readaheadcount > 0) {
		int i;

		c = psc_readahead[0];
		for (i = 1; i < psc_readaheadcount; i++) {
			psc_readahead[i-1] = psc_readahead[i];
		}
		psc_readaheadcount--;
		splx(s);
		return (c);
	}

	/* block until a character bepsces available */
	while (!ISSET(stat = bus_space_read_2(iot, ioh, PSC_SR), SR_RXRDY))
		;

	c = bus_space_read_1(iot, ioh, PSC_RB);
	{
		int cn_trapped = 0; /* unused */
#ifdef DDB
		extern int db_active;
		if (!db_active)
#endif
			cn_check_magic(dev, c, psc_cnm_state);
	}
	splx(s);
	return (c);
}

void
psc_common_putc(dev_t dev, bus_space_tag_t iot, bus_space_handle_t ioh, int c)
{
	int s = splserial();
	int cin, timo;

	if (psc_readaheadcount < MAX_READAHEAD
	     && ISSET(bus_space_read_2(iot, ioh, PSC_SR), SR_RXRDY)) {
		int cn_trapped = 0;
		cin = bus_space_read_1(iot, ioh, PSC_RB);
		cn_check_magic(dev, cin, psc_cnm_state);
		psc_readahead[psc_readaheadcount++] = cin;
	}

	/* wait for any pending transmission to finish */
	timo = 150000;
	while (!ISSET(bus_space_read_2(iot, ioh, PSC_SR), SR_TXRDY) && --timo)
		continue;

	bus_space_write_1(iot, ioh, PSC_TB, c);
	bus_space_barrier(iot, ioh, 0, PSC_NPORTS, BR | BW);

	splx(s);
}

/*
 * Initialize UART for use as console or KGDB line.
 */
int
pscinit(bus_space_tag_t iot, bus_addr_t iobase, int rate, int frequency,
	tcflag_t cflag, bus_space_handle_t *ioh)
{

	if (bus_space_map(iot, iobase, PSC_NPORTS, 0, ioh))
		return (ENOMEM); /* ??? */

#if 0
	rate = pscspeed(rate, frequency);
	CSR_WRITE_1(regsp, PSC_REG_DLBL, rate);
	CSR_WRITE_1(regsp, PSC_REG_LCR, cflag2lcr(cflag));
	CSR_WRITE_1(regsp, PSC_REG_MCR, MCR_DTR | MCR_RTS);
	CSR_WRITE_1(regsp, PSC_REG_FIFO,
	    FIFO_ENABLE | FIFO_RCV_RST | FIFO_XMT_RST | FIFO_TRIGGER_1);

	if (type == PSC_TYPE_OMAP) {
		/* setup the fifos.  the FCR value is not used as long
		   as SCR[6] and SCR[7] are 0, which they are at reset
		   and we never touch the SCR register */
		uint8_t rx_fifo_trig = 40;
		uint8_t tx_fifo_trig = 60;
		uint8_t rx_start = 8;
		uint8_t rx_halt = 60;
		uint8_t tlr_value = ((rx_fifo_trig>>2) << 4) | (tx_fifo_trig>>2);
		uint8_t tcr_value = ((rx_start>>2) << 4) | (rx_halt>>2);

		/* enable access to TCR & TLR */
		CSR_WRITE_1(regsp, PSC_REG_MCR, MCR_DTR | MCR_RTS | MCR_TCR_TLR);

		/* write tcr and tlr values */
		CSR_WRITE_1(regsp, PSC_REG_TLR, tlr_value);
		CSR_WRITE_1(regsp, PSC_REG_TCR, tcr_value);

		/* disable access to TCR & TLR */
		CSR_WRITE_1(regsp, PSC_REG_MCR, MCR_DTR | MCR_RTS);

		/* enable again, but mode is based on speed */
		if (rate > 230400) {
			CSR_WRITE_1(regsp, PSC_REG_MDR1, MDR1_MODE_UART_13X);
		} else {
			CSR_WRITE_1(regsp, PSC_REG_MDR1, MDR1_MODE_UART_16X);
		}
	}
#endif
	bus_space_write_2(iot, *ioh, PSC_IMR, 0);

	return (0);
}

/*
 * Following are all routines needed for PSC to act as console
 */
struct consdev psccons = {
	NULL, NULL, psccngetc, psccnputc, psccnpollc, NULL, NULL, NULL,
	NODEV, CN_NORMAL
};


int
psccnattach(bus_space_tag_t iot, bus_addr_t iobase, int rate, int frequency,
	    tcflag_t cflag)
{
	int res;

	res = pscinit(iot, iobase, rate, frequency, cflag, &pscconsioh);
	if (res)
		return (res);

	cn_tab = &psccons;
	cn_init_magic(&psc_cnm_state);
	cn_set_magic("\047\001"); /* default magic is BREAK */

	pscconsrate = rate;
	pscconscflag = cflag;
	pscconsiobase = iobase;

	return (0);
}

int
psccngetc(dev_t dev)
{

	return (psc_common_getc(dev, pscconsiot, pscconsioh));
}

/*
 * Console kernel output character routine.
 */
void
psccnputc(dev_t dev, int c)
{

	psc_common_putc(dev, pscconsiot, pscconsioh, c);
}

void
psccnpollc(dev_t dev, int on)
{

}

#ifdef KGDB
int
psc_kgdb_attach(bus_space_tag_t iot, bus_addr_t iobase, int rate,
		int frequency, tcflag_t cflag)
{
	int res;

	if (iot == pscconsiot && iobase == pscconsiobase) {
#if !defined(DDB)
		return (EBUSY); /* cannot share with console */
#else
		psckgdbiot = iot;
		psckgdbiobase = iobase;
		psckgdbioh = pscconsioh;
#endif
	} else {
		psckgdbregs = *regsp;
		res = pscinit(psckgdbiot, psckgdbiobase, rate, frequency,
			      cflag, &psckgdbioh);
		if (res)
			return (res);

		/*
		 * XXXfvdl this shouldn't be needed, but the cn_magic goo
		 * expects this to be initialized
		 */
		cn_init_magic(&psc_cnm_state);
		cn_set_magic("\047\001");
	}

	kgdb_attach(psc_kgdb_getc, psc_kgdb_putc, NULL);
	kgdb_dev = 123; /* unneeded, only to satisfy some tests */

	return (0);
}

/* ARGSUSED */
int
psc_kgdb_getc(void *arg)
{

	return (psc_pscmon_getc(NODEV, psckgdbiot, psckgdbioh));
}

/* ARGSUSED */
void
psc_kgdb_putc(void *arg, int c)
{

	psc_pscmon_putc(NODEV, psckgdbiot, psckgdbioh, c);
}
#endif /* KGDB */

#if 0
/* helper function to identify the psc ports used by
 console or KGDB (and not yet autoconf attached) */
int
psc_is_console(bus_space_tag_t iot, bus_addr_t iobase, bus_space_handle_t *ioh)
{
	bus_space_handle_t help;

	if (!pscconsattached &&
	    iot == pscconsregs.cr_iot && iobase == pscconsregs.cr_iobase)
		help = pscconsregs.cr_ioh;
#ifdef KGDB
	else if (!psc_kgdb_attached &&
	    iot == psckgdbregs.cr_iot && iobase == psckgdbregs.cr_iobase)
		help = psckgdbregs.cr_ioh;
#endif
	else
		return (0);

	if (ioh)
		*ioh = help;
	return (1);
}
#endif

/* 
 * this routine exists to serve as a shutdown hook for systems that
 * have firmware which doesn't interact properly with a com device in
 * FIFO mode.
 */
bool
psc_cleanup(device_t self, int how)
{
#if 0
        struct psc_softc *sc = device_private(self);
 
        if (ISSET(sc->sc_hwflags, PSC_HW_FIFO))
                CSR_WRITE_1(&sc->sc_regs, PSC_REG_FIFO, 0);
#endif 
        return true;
}

bool
psc_suspend(device_t self PMF_FN_ARGS)
{
	struct psc_softc *sc = device_private(self);

	bus_space_write_2(sc->sc_iot, sc->sc_ioh, PSC_IMR, 0);

	intr_disestablish(sc->sc_ih);

	return true;
}

bool
psc_resume(device_t self PMF_FN_ARGS)
{
	struct psc_softc *sc = device_private(self);

	sc->sc_ih = intr_establish(sc->sc_irq, IST_EDGE, IPL_SERIAL,
				   pscintr, sc);

	mutex_spin_enter(&sc->sc_lock);
	psc_loadchannelregs(sc);
	mutex_spin_exit(&sc->sc_lock);

	return true;
}
