/*	$NetBSD: $	*/

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include "bpfilter.h"
#include "rnd.h"

#include <sys/param.h>
#include <sys/types.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/socket.h>

#include <uvm/uvm_extern.h>

#if NRND > 0
#include <sys/rnd.h>
#endif

#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_arp.h>
#include <net/if_media.h>
#include <net/if_ether.h>

#include <machine/bus.h>

#include <dev/mii/mii.h>
#include <dev/mii/miivar.h>

#include <powerpc/mpc5200/fecvar.h>

static void	fec_init(struct fec_softc *sc);
void	fec_putpacket(struct fec_softc *sc, u_int len);
int	fec_dmaintr(void *arg);
void	fec_reset_rxdma(struct fec_softc *sc);
void	fec_reset_txdma(struct fec_softc *sc);
void	fec_select_utp(struct fec_softc *sc);
void	fec_select_aui(struct fec_softc *sc);
int	fec_mediachange(struct fec_softc *sc);
void	fec_mediastatus(struct fec_softc *sc, struct ifmediareq *);

int fec_supmedia[] = {
	IFM_ETHER | IFM_10_T,
	IFM_ETHER | IFM_10_5,
	/*IFM_ETHER | IFM_AUTO,*/
};

#define N_SUPMEDIA (sizeof(fec_supmedia) / sizeof(int));

/* Bus-specific initialization */
void
fec_init(struct fec_softc *sc)
{
	fec_reset_rxdma(sc);
	fec_reset_txdma(sc);
}

void
fec_putpacket(struct fec_softc *sc, u_int len)
{
	dbdma_command_t *cmd = sc->sc_txdmacmd;

	DBDMA_BUILD(cmd, DBDMA_CMD_OUT_LAST, 0, len, sc->sc_txbuf_phys,
		DBDMA_INT_NEVER, DBDMA_WAIT_NEVER, DBDMA_BRANCH_NEVER);

	dbdma_start(sc->sc_txdma, sc->sc_txdmacmd);
}

/*
 * Interrupt handler for the MACE DMA completion interrupts
 */
int
fec_dmaintr(void *arg)
{
	struct fec_softc *sc = arg;
	int status, offset, statoff;
	int datalen, resid;
	int i, n;
	dbdma_command_t *cmd;

	/* We've received some packets from the MACE */

	/* Loop through, processing each of the packets */
	i = sc->sc_tail;
	for (n = 0; n < FEC_RXDMABUFS; n++, i++) {
		if (i == FEC_RXDMABUFS)
			i = 0;

		cmd = &sc->sc_rxdmacmd[i];
		/* flushcache(cmd, sizeof(dbdma_command_t)); */
		status = in16rb(&cmd->d_status);
		resid = in16rb(&cmd->d_resid);

		/*if ((status & D_ACTIVE) == 0)*/
		if ((status & 0x40) == 0)
			continue;

#if 1
		if (in16rb(&cmd->d_count) != ETHERMTU + 22)
			printf("bad d_count\n");
#endif

		datalen = in16rb(&cmd->d_count) - resid;
		datalen -= 4;	/* 4 == status bytes */

		if (datalen < 4 + sizeof(struct ether_header)) {
			printf("short packet len=%d\n", datalen);
			/* continue; */
			goto next;
		}

		offset = i * FEC_BUFSIZE;
		statoff = offset + datalen;

		DBDMA_BUILD_CMD(cmd, DBDMA_CMD_STOP, 0, 0, 0, 0);
		__asm volatile("eieio");

		/* flushcache(sc->sc_rxbuf + offset, datalen + 4); */

		sc->sc_rxframe.rx_rcvcnt = sc->sc_rxbuf[statoff + 0];
		sc->sc_rxframe.rx_rcvsts = sc->sc_rxbuf[statoff + 1];
		sc->sc_rxframe.rx_rntpc  = sc->sc_rxbuf[statoff + 2];
		sc->sc_rxframe.rx_rcvcc  = sc->sc_rxbuf[statoff + 3];
		sc->sc_rxframe.rx_frame  = sc->sc_rxbuf + offset;

		fec_rint(sc);

next:
		DBDMA_BUILD_CMD(cmd, DBDMA_CMD_IN_LAST, 0, DBDMA_INT_ALWAYS,
			DBDMA_WAIT_NEVER, DBDMA_BRANCH_NEVER);
		__asm volatile("eieio");
		cmd->d_status = 0;
		cmd->d_resid = 0;
		sc->sc_tail = i + 1;
	}

	dbdma_continue(sc->sc_rxdma);

	return 1;
}

void
fec_reset_rxdma(struct fec_softc *sc)
{
	dbdma_command_t *cmd = sc->sc_rxdmacmd;
	dbdma_regmap_t *dmareg = sc->sc_rxdma;
	int i;
	u_int8_t maccc;

	/* Disable receiver, reset the DMA channels */
	maccc = NIC_GET(sc, MACE_MACCC);
	NIC_PUT(sc, MACE_MACCC, maccc & ~ENRCV);

	dbdma_reset(dmareg);

	for (i = 0; i < FEC_RXDMABUFS; i++) {
		DBDMA_BUILD(cmd, DBDMA_CMD_IN_LAST, 0, ETHERMTU + 22,
			sc->sc_rxbuf_phys + FEC_BUFSIZE * i, DBDMA_INT_ALWAYS,
			DBDMA_WAIT_NEVER, DBDMA_BRANCH_NEVER);
		cmd++;
	}

	DBDMA_BUILD(cmd, DBDMA_CMD_NOP, 0, 0, 0,
		DBDMA_INT_NEVER, DBDMA_WAIT_NEVER, DBDMA_BRANCH_ALWAYS);
	out32(&cmd->d_cmddep, kvtop((void *)sc->sc_rxdmacmd));
	cmd++;

	dbdma_start(dmareg, sc->sc_rxdmacmd);

	sc->sc_tail = 0;

	/* Reenable receiver, reenable DMA */
	NIC_PUT(sc, MACE_MACCC, maccc);
}

void
fec_reset_txdma(struct fec_softc *sc)
{
	dbdma_command_t *cmd = sc->sc_txdmacmd;
	dbdma_regmap_t *dmareg = sc->sc_txdma;
	u_int8_t maccc;

	/* disable transmitter */
	maccc = NIC_GET(sc, MACE_MACCC);
	NIC_PUT(sc, MACE_MACCC, maccc & ~ENXMT);

	dbdma_reset(dmareg);

	DBDMA_BUILD(cmd, DBDMA_CMD_OUT_LAST, 0, 0, sc->sc_txbuf_phys,
		DBDMA_INT_NEVER, DBDMA_WAIT_NEVER, DBDMA_BRANCH_NEVER);
	cmd++;
	DBDMA_BUILD(cmd, DBDMA_CMD_STOP, 0, 0, 0,
		DBDMA_INT_NEVER, DBDMA_WAIT_NEVER, DBDMA_BRANCH_NEVER);

	out32(&dmareg->d_cmdptrhi, 0);
	out32(&dmareg->d_cmdptrlo, kvtop((void *)sc->sc_txdmacmd));

	/* restore old value */
	NIC_PUT(sc, MACE_MACCC, maccc);
}

void
fec_select_utp(struct fec_softc *sc)
{
	sc->sc_plscc = PORTSEL_GPSI | ENPLSIO;
}

void
fec_select_aui(struct fec_softc *sc)
{
	sc->sc_plscc = PORTSEL_AUI;
}

int
fec_mediachange(struct fec_softc *sc)
{
	struct ifmedia *ifm = &sc->sc_media;

	if (IFM_TYPE(ifm->ifm_media) != IFM_ETHER)
		return EINVAL;

	switch (IFM_SUBTYPE(ifm->ifm_media)) {

	case IFM_10_T:
		fec_select_utp(sc);
		break;

	case IFM_10_5:
		fec_select_aui(sc);
		break;

	default:
		return EINVAL;
	}

	return 0;
}

void
fec_mediastatus(struct fec_softc *sc, struct ifmediareq *ifmr)
{
	if (sc->sc_plscc == PORTSEL_AUI)
		ifmr->ifm_active = IFM_ETHER | IFM_10_5;
	else
		ifmr->ifm_active = IFM_ETHER | IFM_10_T;
}
