/* $NetBSD: $ */
/* $Id: mipsock.c,v 1.27 2007/06/14 12:09:42 itojun Exp $ */

/*
 * Copyright (C) 2004 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#ifdef _KERNEL_OPT
#include "opt_mip6.h"
#endif

#include "mip.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/domain.h>
#include <sys/protosw.h>
#include <sys/syslog.h>
#include <sys/proc.h>
#include <sys/kauth.h>
#include <sys/kmem.h>

#include <net/if.h>
#include <net/mipsock.h>
#include <net/if_mip.h>
#include <net/raw_cb.h>
#include <net/route.h>


#include <netinet/in.h>
#include <netinet/ip6mh.h>
#include <netinet6/in6_var.h>
#include <netinet6/mip6.h>
#include <netinet6/mip6_var.h>

DOMAIN_DEFINE(mipdomain);	/* foward declare and add to link set */

static struct	sockaddr mips_dst = { .sa_len = 2, .sa_family = PF_MOBILITY, };
static struct	sockaddr mips_src = { .sa_len = 2, .sa_family = PF_MOBILITY, };
static struct	sockproto mips_proto = { .sp_family = PF_MOBILITY, };

static int mips_output(struct mbuf *, struct socket *);
static struct mbuf *mips_msg1(int type, int len);

static int
mips_attach(struct socket *so, int proto)
{
	struct rawcb *rp;
	int s, error;

	KASSERT(sotorawcb(so) == NULL);
	rp = kmem_zalloc(sizeof(*rp), KM_SLEEP);
	rp->rcb_len = sizeof(*rp);
	so->so_pcb = rp;

	s = splsoftnet();
	error = raw_attach(so, proto);
	if (error) {
		splx(s);
		kmem_free(rp, sizeof(*rp));
		so->so_pcb = NULL;
		return error;
	}

	rp->rcb_laddr = &mips_src;
	rp->rcb_faddr = &mips_dst;
	soisconnected(so);
	so->so_options |= SO_USELOOPBACK;
	KASSERT(solocked(so));
	splx(s);
	return error;
}

static void
mips_detach(struct socket *so)
{
	int s;

	KASSERT(solocked(so));

	s = splsoftnet();
	raw_detach(so);
	splx(s);
}

static int
mips_accept(struct socket *so, struct sockaddr *nam)
{
	KASSERT(solocked(so));

	panic("mips_accept");

	return EOPNOTSUPP;
}

static int
mips_bind(struct socket *so, struct sockaddr *nam, struct lwp *l)
{
	KASSERT(solocked(so));

	return EOPNOTSUPP;
}

static int
mips_listen(struct socket *so, struct lwp *l)
{
	KASSERT(solocked(so));

	return EOPNOTSUPP;
}

static int
mips_connect(struct socket *so, struct sockaddr *nam, struct lwp *l)
{
	KASSERT(solocked(so));

	return EOPNOTSUPP;
}

static int
mips_connect2(struct socket *so, struct socket *so2)
{
	KASSERT(solocked(so));

	return EOPNOTSUPP;
}

static int
mips_disconnect(struct socket *so)
{
	struct rawcb *rp = sotorawcb(so);
	int s;

	KASSERT(solocked(so));
	KASSERT(rp != NULL);

	s = splsoftnet();
	soisdisconnected(so);
	raw_disconnect(rp);
	splx(s);

	return 0;
}

static int
mips_shutdown(struct socket *so)
{
	int s;

	KASSERT(solocked(so));

	/*
	 * Mark the connection as being incapable of further input.
	 */
	s = splsoftnet();
	socantsendmore(so);
	splx(s);
	return 0;
}

static int
mips_abort(struct socket *so)
{
	KASSERT(solocked(so));

	panic("mips_abort");

	return EOPNOTSUPP;
}

static int
mips_ioctl(struct socket *so, u_long cmd, void *nam, struct ifnet * ifp)
{
	return EOPNOTSUPP;
}

static int
mips_stat(struct socket *so, struct stat *ub)
{
	KASSERT(solocked(so));

	return 0;
}

static int
mips_peeraddr(struct socket *so, struct sockaddr *nam)
{
	struct rawcb *rp = sotorawcb(so);

	KASSERT(solocked(so));
	KASSERT(rp != NULL);
	KASSERT(nam != NULL);

	if (rp->rcb_faddr == NULL)
		return ENOTCONN;

	memcpy(nam, rp->rcb_faddr, rp->rcb_faddr->sa_len);
	return 0;
}

static int
mips_sockaddr(struct socket *so, struct sockaddr *nam)
{
	struct rawcb *rp = sotorawcb(so);

	KASSERT(solocked(so));
	KASSERT(rp != NULL);
	KASSERT(nam != NULL);

	if (rp->rcb_faddr == NULL)
		return ENOTCONN;

	memcpy(nam, rp->rcb_laddr, rp->rcb_laddr->sa_len);
	return 0;
}

static int
mips_rcvd(struct socket *so, int flags, struct lwp *l)
{
	KASSERT(solocked(so));

	return EOPNOTSUPP;
}

static int
mips_recvoob(struct socket *so, struct mbuf *m, int flags)
{
	KASSERT(solocked(so));

	return EOPNOTSUPP;
}

static int
mips_send(struct socket *so, struct mbuf *m,
    struct sockaddr *nam, struct mbuf *control, struct lwp *l)
{
	int error = 0;
	int s;

	KASSERT(solocked(so));

	s = splsoftnet();
	error = raw_send(so, m, nam, control, l, &mips_output);
	splx(s);

	return error;
}

static int
mips_sendoob(struct socket *so, struct mbuf *m, struct mbuf *control)
{
	KASSERT(solocked(so));

	m_freem(m);
	m_freem(control);

	return EOPNOTSUPP;
}

static int
mips_purgeif(struct socket *so, struct ifnet *ifp)
{

	panic("mips_purgeif");

	return EOPNOTSUPP;
}

#if 0
/*ARGSUSED*/
int
mips_usrreq(struct socket *so, int req, struct mbuf *m, struct mbuf *nam,
	struct mbuf *control, struct lwp *l)
{
	int error = 0;
	struct rawcb *rp = sotorawcb(so);
	int s;

	if (req == PRU_ATTACH) {
		rp = malloc(sizeof(*rp), M_PCB, M_WAITOK|M_ZERO);
	}
#if 0
	if (req == PRU_DETACH && rp)
		rt_adjustcount(rp->rcb_proto.sp_protocol, -1);
#endif
	s = splsoftnet();

	/*
	 * Don't call raw_usrreq() in the attach case, because
	 * we want to allow non-privileged processes to listen on
	 * and send "safe" commands to the routing socket.
	 */
	if (req == PRU_ATTACH) {
		if (l == 0)
			error = EACCES;
		else
			error = raw_attach(so, (int)(long)nam);
	} else
		error = raw_usrreq(so, req, m, nam, control, l);

	rp = sotorawcb(so);
	if (req == PRU_ATTACH && rp) {
		if (error) {
			free(rp, M_PCB);
			splx(s);
			return (error);
		}
		/* rt_adjustcount(rp->rcb_proto.sp_protocol, 1); */
		rp->rcb_laddr = &mips_src;
		rp->rcb_faddr = &mips_dst;
		soisconnected(so);
		so->so_options |= SO_USELOOPBACK;
	}
	splx(s);
	return (error);
}
#endif

/*ARGSUSED*/
static int
mips_output(struct mbuf *m, struct socket *so)
{
	int len, error = 0;
	struct mip_msghdr *miph = NULL;
	struct mipm_bc_info *mipc = NULL;
	struct mipm_nodetype_info *mipmni = NULL;
#if NMIP > 0
	struct mipm_bul_info *mipu = NULL;
	struct mip6_bul_internal *mbul = NULL;
        struct mipm_md_info *mipmd = NULL;
#endif
	struct mipm_dad *mipmdad = NULL;
	struct sockaddr_storage hoa, coa, cnaddr;
	u_int16_t bid = 0;

#define senderr(e) do { error = e; goto flush;} while (/*CONSTCOND*/ 0)
	if (m == 0 || ((m->m_len < sizeof(int32_t)) &&
	    (m = m_pullup(m, sizeof(int32_t))) == 0))
		return (ENOBUFS);
	if ((m->m_flags & M_PKTHDR) == 0)
		panic("mips_output");
	len = m->m_pkthdr.len;
	if (len < sizeof(struct mip_msghdr) ||
	    len != mtod(m, struct mip_msghdr *)->miph_msglen) {
		senderr(EINVAL);
	}
	miph = mtod(m, struct mip_msghdr *);

	/*
	 * Perform permission checking, only privileged sockets
	 * may perform operations other than RTM_GET
	 */
	/* XXX is KAUTH_NETWORK_ROUTE correct? */
	if ((kauth_authorize_network(curlwp->l_cred, KAUTH_NETWORK_ROUTE,
		    0, miph, NULL, NULL) != 0) && (error = EACCES)) {
		senderr(error);
	}

#ifndef __APPLE__
	miph->miph_pid = curproc->p_pid;
#else
	miph->miph_pid = proc_selfpid();
#endif

	switch (miph->miph_type) {
	case MIPM_BC_ADD:
/*	case MIPM_BC_UPDATE:*/
		mipc = (struct mipm_bc_info *)miph;
		bcopy(MIPC_CNADDR(mipc), &cnaddr, MIPC_CNADDR(mipc)->sa_len);
		bcopy(MIPC_HOA(mipc), &hoa, MIPC_HOA(mipc)->sa_len);
		bcopy(MIPC_COA(mipc), &coa, MIPC_COA(mipc)->sa_len);
		switch (hoa.ss_family) {
		case AF_INET6:
#ifdef MIP6_MCOA
			bid = ((struct sockaddr_in6 *)&coa)->sin6_port;
#endif /* MIP6_MCOA */
			error = mip6_bce_update((struct sockaddr_in6 *)&cnaddr,
						(struct sockaddr_in6 *)&hoa,
						(struct sockaddr_in6 *)&coa,
						mipc->mipmci_flags, bid);
			break;
		default:
			error = EPFNOSUPPORT;
			break;
		}
		break;

	case MIPM_BC_REMOVE:
		mipc = (struct mipm_bc_info *)miph;
		bcopy(MIPC_CNADDR(mipc), &cnaddr, MIPC_CNADDR(mipc)->sa_len);
		bcopy(MIPC_HOA(mipc), &hoa, MIPC_HOA(mipc)->sa_len);
		bcopy(MIPC_COA(mipc), &coa, MIPC_COA(mipc)->sa_len);
		switch (hoa.ss_family) {
		case AF_INET6:
#ifdef MIP6_MCOA
			bid = ((struct sockaddr_in6 *)&coa)->sin6_port;
#endif /* MIP6_MCOA */
			error = mip6_bce_remove_addr((struct sockaddr_in6 *)&cnaddr,
						     (struct sockaddr_in6 *)&hoa,
						     (struct sockaddr_in6 *)&coa,
						     mipc->mipmci_flags, bid);
			break;
		default:
			error = EPFNOSUPPORT;
			break;
		}
		break;

	case MIPM_BC_FLUSH:
		mip6_bce_remove_all();
		break;

	case MIPM_NODETYPE_INFO:
		mipmni = (struct mipm_nodetype_info *)miph;
		if (mipmni->mipmni_enable) {
			if (MIP6_IS_MN
			    && (mipmni->mipmni_nodetype
				& MIP6_NODETYPE_HOME_AGENT))
				error = EINVAL;
			if (MIP6_IS_HA
			    && ((mipmni->mipmni_nodetype
				    & MIP6_NODETYPE_MOBILE_NODE)
				|| (mipmni->mipmni_nodetype
				    & MIP6_NODETYPE_MOBILE_ROUTER)))
				error = EINVAL;
			mip6_nodetype |= mipmni->mipmni_nodetype;
		} else {
			if (mipmni->mipmni_nodetype
			    == MIP6_NODETYPE_NONE)
				error = EINVAL;
			mip6_nodetype &= ~mipmni->mipmni_nodetype;
		}
		break;

#if NMIP > 0
	case MIPM_BUL_ADD:
/*	case MIPM_BUL_UPDATE: */
		mipu = (struct mipm_bul_info *)miph;
		/* Non IPv6 address is not supported (only for MIP6) */
		bcopy(MIPU_PEERADDR(mipu), &cnaddr, MIPU_PEERADDR(mipu)->sa_len);
		bcopy(MIPU_HOA(mipu), &hoa, MIPU_HOA(mipu)->sa_len);
		bcopy(MIPU_COA(mipu), &coa, MIPU_COA(mipu)->sa_len);
		switch (hoa.ss_family) {
		case AF_INET6:
			if ((MIPU_PEERADDR(mipu))->sa_family != AF_INET6 ||
			    (MIPU_COA(mipu))->sa_family != AF_INET6) {
				error = EPFNOSUPPORT; /* XXX ? */
				break;
			}
#ifdef MIP6_MCOA
			bid = ((struct sockaddr_in6 *)&coa)->sin6_port;
#endif /* MIP6_MCOA */
			error = mip6_bul_add(&((struct sockaddr_in6 *)&cnaddr)->sin6_addr,
					     &((struct sockaddr_in6 *)&hoa)->sin6_addr,
					     &((struct sockaddr_in6 *)&coa)->sin6_addr,
					     mipu->mipmui_hoa_ifindex,
					     mipu->mipmui_flags,
					     mipu->mipmui_state, bid);
			break;
		default:
			error = EPFNOSUPPORT;
			break;
		}
		break;

	case MIPM_BUL_REMOVE:
		mipu = (struct mipm_bul_info *)miph;
		bcopy(MIPU_PEERADDR(mipu), &cnaddr, MIPU_PEERADDR(mipu)->sa_len);
		bcopy(MIPU_HOA(mipu), &hoa, MIPU_HOA(mipu)->sa_len);
#ifdef MIP6_MCOA
		bcopy(&((struct sockaddr_in6 *)MIPU_COA(mipu))->sin6_port,
		      &bid, sizeof(bid));
#endif /* MIP6_MCOA */
		mbul = mip6_bul_get(&((struct sockaddr_in6 *)&hoa)->sin6_addr, &((struct sockaddr_in6 *)&cnaddr)->sin6_addr, bid);
		if (mbul == NULL) 
			return (ENOENT);

		mip6_bul_remove(mbul);
		break;

	case MIPM_BUL_FLUSH:
		mip6_bul_remove_all();
		break;

	case MIPM_HOME_HINT:
	case MIPM_MD_INFO:
                mipmd = (struct mipm_md_info *)miph;
                if (mipmd->mipmmi_command == MIPM_MD_SCAN) {
                        mip6_md_scan(mipmd->mipmmi_ifindex);
                } 
                break;
#endif /* NMIP > 0 */

		/* MIPM_DAD_DO/STOP are issued from userland */
		/* MIPM_DAD_SUCCESS or MIPM_DAD_FAIL is returned as a result of the DAD */
	case MIPM_DAD:
		mipmdad = (struct mipm_dad *)miph;
		if (mipmdad->mipmdad_message == MIPM_DAD_DO) {
			mip6_do_dad(&mipmdad->mipmdad_addr6, mipmdad->mipmdad_ifindex);
		} else if (mipmdad->mipmdad_message == MIPM_DAD_STOP) {
			mip6_stop_dad(&mipmdad->mipmdad_addr6, mipmdad->mipmdad_ifindex);
		} else if (mipmdad->mipmdad_message == MIPM_DAD_LINKLOCAL) {
			mip6_do_dad_lladdr(mipmdad->mipmdad_ifindex);
		}
		break;

	default:
		return (0);
	}

 flush:
	if (miph) {
		if (error)
			miph->miph_errno = error;
/*
		else
			miph->miph_flags |= RTF_DONE;
*/
	}

	raw_input(m, &mips_proto, &mips_src, &mips_dst);
	return (error);
}

static struct mbuf *
mips_msg1(int type, int len)
{
	struct mip_msghdr *miph;
	struct mbuf *m;

	if (len > MCLBYTES)
		panic("mips_msg1");
	m = m_gethdr(M_DONTWAIT, MT_DATA);
	if (m && (size_t)len > MHLEN) {
		MCLGET(m, M_DONTWAIT);
		if ((m->m_flags & M_EXT) == 0) {
			m_free(m);
			m = NULL;
		}
	}
	if (m == NULL)
		return (m);
	m->m_pkthdr.len = m->m_len = len;
	m_reset_rcvif(m);
	miph = mtod(m, struct mip_msghdr *);
	memset(miph, 0, len);
	if (m->m_pkthdr.len != len) {
		m_freem(m);
		return (NULL);
	}
	miph->miph_msglen = len;
	miph->miph_version = MIP_VERSION;
	miph->miph_type = type;
	return (m);
}


void
mips_notify_home_hint(u_int16_t ifindex, struct sockaddr *prefix,
	u_int16_t prefixlen) 
{
	struct mipm_home_hint *hint;
	struct mbuf *m;
	int len = sizeof(struct mipm_home_hint) + prefix->sa_len;

	m = mips_msg1(MIPM_HOME_HINT, len);
	if (m == NULL)
		return;

        hint = mtod(m, struct mipm_home_hint *);

	hint->mipmhh_seq = 0;
	hint->mipmhh_ifindex = ifindex;
	hint->mipmhh_prefixlen = prefixlen;
	bcopy(prefix, (char *) hint->mipmhh_prefix, prefix->sa_len);

	raw_input(m, &mips_proto, &mips_src, &mips_dst);
}

/*
 * notify bi-directional tunneling event so that a moblie node can
 * initiate RR procedure.
 */
void
mips_notify_rr_hint(struct sockaddr *hoa, struct sockaddr *peeraddr)
{
	struct mipm_rr_hint *rr_hint;
	struct mbuf *m;
	u_short len = sizeof(struct mipm_rr_hint)
	    + hoa->sa_len + peeraddr->sa_len;

	m = mips_msg1(MIPM_RR_HINT, len);
	if (m == NULL)
		return;
	rr_hint = mtod(m, struct mipm_rr_hint *);
	rr_hint->mipmrh_seq = 0;

	bcopy(hoa, (void *)MIPMRH_HOA(rr_hint), hoa->sa_len);
	bcopy(peeraddr, (void *)MIPMRH_PEERADDR(rr_hint), peeraddr->sa_len);

	raw_input(m, &mips_proto, &mips_src, &mips_dst);
}

/*
 * notify a hint to send a binding error message.  this message is
 * usually sent when an invalid home address is received.
 */
void
mips_notify_be_hint(struct sockaddr *src, struct sockaddr *coa,
	struct sockaddr *hoa, u_int8_t status)
{
	struct mipm_be_hint *be_hint;
	struct mbuf *m;
	u_short len = sizeof(struct mipm_be_hint)
	    + src->sa_len + coa->sa_len + hoa->sa_len;

	m = mips_msg1(MIPM_BE_HINT, len);
	if (m == NULL)
		return;
	be_hint = mtod(m, struct mipm_be_hint *);
	be_hint->mipmbeh_seq = 0;

	be_hint->mipmbeh_status = status;

	bcopy(src, (void *)MIPMBEH_PEERADDR(be_hint), src->sa_len);
	bcopy(coa, (void *)MIPMBEH_COA(be_hint), coa->sa_len);
	bcopy(hoa, (void *)MIPMBEH_HOA(be_hint), hoa->sa_len);

	raw_input(m, &mips_proto, &mips_src, &mips_dst);
}

void
mips_notify_dad_result(int message, struct in6_addr *addr, int ifindex)
{
	struct mipm_dad *mipmdad;
	struct mbuf *m;
	
	m = mips_msg1(MIPM_DAD, sizeof(struct mipm_dad));
	if (m == NULL)
		return;
	mipmdad = mtod(m, struct mipm_dad *);
	mipmdad->mipmdad_message = message;
	mipmdad->mipmdad_ifindex = ifindex;
	memcpy(&mipmdad->mipmdad_addr6, addr, sizeof(struct in6_addr));

	raw_input(m, &mips_proto, &mips_src, &mips_dst);
}

/*
 * Definitions of protocols supported in the MOBILITY domain.
 */

PR_WRAP_USRREQS(mips)
#define	mips_attach	mips_attach_wrapper
#define	mips_detach	mips_detach_wrapper
#define mips_accept	mips_accept_wrapper
#define mips_bind	mips_bind_wrapper
#define mips_listen	mips_listen_wrapper
#define mips_connect	mips_connect_wrapper
#define mips_connect2	mips_connect2_wrapper
#define mips_disconnect	mips_disconnect_wrapper
#define mips_shutdown	mips_shutdown_wrapper
#define mips_abort	mips_abort_wrapper
#define	mips_ioctl	mips_ioctl_wrapper
#define	mips_stat	mips_stat_wrapper
#define mips_peeraddr	mips_peeraddr_wrapper
#define mips_sockaddr	mips_sockaddr_wrapper
#define mips_rcvd	mips_rcvd_wrapper
#define mips_recvoob	mips_recvoob_wrapper
#define mips_send	mips_send_wrapper
#define mips_sendoob	mips_sendoob_wrapper
#define mips_purgeif	mips_purgeif_wrapper

const struct pr_usrreqs mips_usrreqs = {
	.pr_attach	= mips_attach,
	.pr_detach	= mips_detach,
	.pr_accept	= mips_accept,
	.pr_bind	= mips_bind,
	.pr_listen	= mips_listen,
	.pr_connect	= mips_connect,
	.pr_connect2	= mips_connect2,
	.pr_disconnect	= mips_disconnect,
	.pr_shutdown	= mips_shutdown,
	.pr_abort	= mips_abort,
	.pr_ioctl	= mips_ioctl,
	.pr_stat	= mips_stat,
	.pr_peeraddr	= mips_peeraddr,
	.pr_sockaddr	= mips_sockaddr,
	.pr_rcvd	= mips_rcvd,
	.pr_recvoob	= mips_recvoob,
	.pr_send	= mips_send,
	.pr_sendoob	= mips_sendoob,
	.pr_purgeif	= mips_purgeif,
};
extern struct domain mipdomain;		/* or at least forward */

static struct protosw mipsw[] = {
{
	.pr_type = SOCK_RAW,
	.pr_domain = &mipdomain,
	.pr_flags = PR_ATOMIC|PR_ADDR,
	.pr_input = raw_input,
	.pr_ctlinput = raw_ctlinput,
	.pr_usrreqs = &mips_usrreqs,
	.pr_init = raw_init,
}
};

struct domain mipdomain = {
	.dom_family = PF_MOBILITY,
	.dom_name = "mip",
	.dom_protosw = mipsw,
	.dom_protoswNPROTOSW = &mipsw[__arraycount(mipsw)],
};

