/*	$KAME: if_nemo.c,v 1.9 2007/06/14 12:09:42 itojun Exp $	*/
/*	$NetBSD: $ */

/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#ifdef _KERNEL_OPT
#include "opt_inet.h"
#include "opt_mip6.h"
#endif /* _KERNEL_OPT */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/sockio.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/syslog.h>
#include <sys/intr.h>
#include <sys/protosw.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/bpf.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#ifdef	INET
#include <netinet/in_var.h>
#include <netinet/in_gif.h>
#endif	/* INET */

#ifdef INET6
#ifndef INET
#include <netinet/in.h>
#endif
#include <netinet6/in6_var.h>
#include <netinet/ip6.h>
#include <netinet6/ip6_var.h>
#include <netinet6/in6_gif.h>
#include <netinet6/ip6protosw.h>
#include <netinet6/scope6_var.h>
#endif /* INET6 */

#include <netinet/ip_encap.h>
#include <net/if_nemo.h>

#include <net/net_osdep.h>

LIST_HEAD(, nemo_softc) nemo_softc_list;

void nemoattach(int);


static int	nemo_clone_create(struct if_clone *, int);
static int	nemo_clone_destroy(struct ifnet *);
static void	nemo_start(struct ifnet *);

static struct if_clone nemo_cloner =
    IF_CLONE_INITIALIZER("nemo", nemo_clone_create, nemo_clone_destroy);

/*
 * nemo global variable definitions
 */

void
nemoattach(int dummy)
{
	LIST_INIT(&nemo_softc_list);
	if_clone_attach(&nemo_cloner);
}

static int
nemo_clone_create(struct if_clone *ifc, int unit)
{
	struct nemo_softc *sc;

	sc = malloc(sizeof(struct nemo_softc), M_DEVBUF, M_WAITOK|M_ZERO);
	if_initname(&sc->nemo_if, "nemo", unit);
	sc->encap_cookie4 = sc->encap_cookie6 = NULL;

	sc->nemo_if.if_addrlen = 0;
	sc->nemo_if.if_mtu    = GIF_MTU;
	sc->nemo_if.if_flags  = IFF_POINTOPOINT | IFF_MULTICAST;
	/* turn off ingress filter */
	sc->nemo_if.if_flags  |= IFF_LINK2;
	sc->nemo_if.if_ioctl  = nemo_ioctl;
	sc->nemo_if.if_output = nemo_output;
	sc->nemo_if.if_type   = IFT_GIF;
	sc->nemo_if.if_dlt = DLT_NULL;
	IFQ_SET_READY(&sc->nemo_if.if_snd);
	if_initialize(&sc->nemo_if);
	if_register(&sc->nemo_if);
	if_alloc_sadl(&sc->nemo_if);
	bpf_attach(&sc->nemo_if, DLT_NULL, sizeof(u_int));
#ifdef MIP6
	sc->nemo_nexthop = NULL;
#endif
	LIST_INSERT_HEAD(&nemo_softc_list, sc, nemo_list);
	return 0;
}

static int
nemo_clone_destroy(struct ifnet *ifp)
{
	struct nemo_softc * sc = ifp->if_softc;

	LIST_REMOVE(sc, nemo_list);
	bpf_detach(ifp);
	if_detach(ifp);
	free(sc, M_DEVBUF);

	return (0);
}

#ifdef GIF_ENCAPCHECK
int
nemo_encapcheck(const struct mbuf *m, int off, int proto, void *arg)
{
	struct ip ip;
	struct nemo_softc *sc;

	sc = (struct nemo_softc *)arg;
	if (sc == NULL)
		return 0;

	if ((sc->nemo_if.if_flags & IFF_UP) == 0)
		return 0;

	/* no physical address */
	if (!sc->nemo_psrc || !sc->nemo_pdst)
		return 0;

	switch (proto) {
#ifdef INET
	case IPPROTO_IPV4:
		break;
#endif
#ifdef INET6
	case IPPROTO_IPV6:
		break;
#endif
	default:
		return 0;
	}

	/* Bail on short packets */
	if (m->m_pkthdr.len < sizeof(ip))
		return 0;

	/* LINTED const cast */
	m_copydata((struct mbuf *)m, 0, sizeof(ip), &ip);

	switch (ip.ip_v) {
#ifdef INET
	case 4:
		if (sc->nemo_psrc->sa_family != AF_INET ||
		    sc->nemo_pdst->sa_family != AF_INET)
			return 0;
		return nemo_encapcheck4(m, off, proto, arg);
#endif
#ifdef INET6
	case 6:
		if (m->m_pkthdr.len < sizeof(struct ip6_hdr))
			return 0;
		if (sc->nemo_psrc->sa_family != AF_INET6 ||
		    sc->nemo_pdst->sa_family != AF_INET6)
			return 0;
		return nemo_encapcheck6(m, off, proto, arg);
#endif
	default:
		return 0;
	}
}
#endif

/* rt - added in net2 */
int
nemo_output(struct ifnet *ifp, struct mbuf *m, const struct sockaddr *dst,
	const struct rtentry *rt)
{
	struct nemo_softc *sc = (struct nemo_softc*)ifp;
	int error = 0;
	int s;
	struct m_tag *mtag;

	IFQ_CLASSIFY(&ifp->if_snd, m, dst->sa_family);

	/*
	 * nemo may cause infinite recursion calls when misconfigured.
	 * We'll prevent this by limiting packets from going through a nemo
	 * interface up to once.
	 */
	for (mtag = m_tag_find(m, PACKET_TAG_TUNNEL_INFO, NULL); mtag;
	     mtag = m_tag_find(m, PACKET_TAG_TUNNEL_INFO, mtag)) {
		if (memcmp((mtag + 1), &ifp, sizeof(struct ifnet *))
		    == 0) {
			m_freem(m);
			error = EIO;	/* is there better errno? */
			goto end;
		}
	}

	mtag = m_tag_get(PACKET_TAG_TUNNEL_INFO, sizeof(struct ifnet *),
	    M_NOWAIT);
	if (!mtag) {
		IF_DROP(&ifp->if_snd);
		m_freem(m);
		error = ENOMEM;
		goto end;

	}
	memcpy(&ifp, (mtag + 1), sizeof(struct ifnet *));
	m_tag_prepend(m, mtag);

	m->m_flags &= ~(M_BCAST|M_MCAST);
	if (!(ifp->if_flags & IFF_UP) ||
	    sc->nemo_psrc == NULL || sc->nemo_pdst == NULL) {
		m_freem(m);
		error = ENETDOWN;
		goto end;
	}

	/* XXX should we check if our outer source is legal? */

	/* use DLT_NULL encapsulation here to pass inner af type */
	M_PREPEND(m, sizeof(int), M_DONTWAIT);
	if (!m) {
		error = ENOBUFS;
		goto end;
	}
	*mtod(m, int *) = dst->sa_family;

	s = splnet();
	IFQ_ENQUEUE(&ifp->if_snd, m, error);
	if (error) {
		splx(s);
		goto end;
	}
	splx(s);

	nemo_start(ifp);
	error = 0;

  end:
	if (error)
		ifp->if_oerrors++;
	return error;
}

static void
nemo_start(struct ifnet *ifp)
{
	struct nemo_softc *sc;
	struct mbuf *m;
	int family;
	int len;
	int s;
	int error;

	sc = ifp->if_softc;

	/* output processing */
	while (1) {
		s = splnet();
		IFQ_DEQUEUE(&sc->nemo_if.if_snd, m);
		splx(s);
		if (m == NULL)
			break;

		/* grab and chop off inner af type */
		if (sizeof(int) > m->m_len) {
			m = m_pullup(m, sizeof(int));
			if (!m) {
				ifp->if_oerrors++;
				continue;
			}
		}
		family = *mtod(m, int *);
		bpf_mtap(ifp, m);
		m_adj(m, sizeof(int));

		len = m->m_pkthdr.len;

		/* dispatch to output logic based on outer AF */
		switch (sc->nemo_psrc->sa_family) {
#ifdef INET
		case AF_INET:
			error = in_gif_output(ifp, family, m);
			break;
#endif
#ifdef INET6
		case AF_INET6:
			error = in6_gif_output(ifp, family, m);
			break;
#endif
		default:
			m_freem(m);		
			error = ENETDOWN;
			break;
		}

		if (error)
			ifp->if_oerrors++;
		else {
			ifp->if_opackets++;	
			ifp->if_obytes += len;
		}
	}
}

void
nemo_input(struct mbuf *m, int af, struct ifnet *ifp)
{
	pktqueue_t *pktq;
	size_t pktlen;
	int s;

	if (ifp == NULL) {
		/* just in case */
		m_freem(m);
		return;
	}

	m_set_rcvif(m, ifp);
	pktlen = m->m_pkthdr.len;
	
	bpf_mtap_af(ifp, af, m);
	/*
	 * Put the packet to the network layer input queue according to the
	 * specified address family.
	 * Note: older versions of nemo_input directly called network layer
	 * input functions, e.g. ip6_input, here.  We changed the policy to
	 * prevent too many recursive calls of such input functions, which
	 * might cause kernel panic.  But the change may introduce another
	 * problem; if the input queue is full, packets are discarded.
	 * The kernel stack overflow really happened, and we believed
	 * queue-full rarely occurs, so we changed the policy.
	 */
	switch (af) {
#ifdef INET
	case AF_INET:
		pktq = ip_pktq;
		break;
#endif
#ifdef INET6
	case AF_INET6:
		pktq = ip6_pktq;
		break;
#endif
	default:
		m_freem(m);
		return;
	}

	s = splnet();
	if (__predict_true(pktq_enqueue(pktq, m, 0))) {
		ifp->if_ibytes += pktlen;
		ifp->if_ipackets++;
	} else {
		m_freem(m);
	}
	splx(s);
}

/* XXX how should we handle IPv6 scope on SIOC[GS]IFPHYADDR? */
int
nemo_ioctl(struct ifnet *ifp, unsigned long cmd, void *data)
{
	struct nemo_softc *sc  = (struct nemo_softc*)ifp;
	struct ifreq     *ifr = (struct ifreq*)data;
	int error = 0, size;
	struct sockaddr *dst, *src;
#ifdef	SIOCSIFMTU /* xxx */
	u_long mtu;
#endif

	switch (cmd) {
	case SIOCINITIFADDR:
		ifp->if_flags |= IFF_UP;
		break;
		
	case SIOCSIFDSTADDR:
		break;

	case SIOCADDMULTI:
	case SIOCDELMULTI:
		switch (ifr->ifr_addr.sa_family) {
#ifdef INET
		case AF_INET:	/* IP supports Multicast */
			break;
#endif /* INET */
#ifdef INET6
		case AF_INET6:	/* IP6 supports Multicast */
			break;
#endif /* INET6 */
		default:  /* Other protocols doesn't support Multicast */
			error = EAFNOSUPPORT;
			break;
		}
		break;

#ifdef	SIOCSIFMTU /* xxx */
	case SIOCGIFMTU:
		break;

	case SIOCSIFMTU:
		mtu = ifr->ifr_mtu;
		if (mtu < GIF_MTU_MIN || mtu > GIF_MTU_MAX)
			return (EINVAL);
		ifp->if_mtu = mtu;
		break;
#endif /* SIOCSIFMTU */

#ifdef INET
	case SIOCSIFPHYADDR:
#endif
#ifdef INET6
	case SIOCSIFPHYADDR_IN6:
#endif /* INET6 */
	case SIOCSLIFPHYADDR:
		switch (cmd) {
#ifdef INET
		case SIOCSIFPHYADDR:
			src = (struct sockaddr *)
				&(((struct in_aliasreq *)data)->ifra_addr);
			dst = (struct sockaddr *)
				&(((struct in_aliasreq *)data)->ifra_dstaddr);
			break;
#endif
#ifdef INET6
		case SIOCSIFPHYADDR_IN6:
			src = (struct sockaddr *)
				&(((struct in6_aliasreq *)data)->ifra_addr);
			dst = (struct sockaddr *)
				&(((struct in6_aliasreq *)data)->ifra_dstaddr);
			break;
#endif
		case SIOCSLIFPHYADDR:
			src = (struct sockaddr *)
				&(((struct if_laddrreq *)data)->addr);
			dst = (struct sockaddr *)
				&(((struct if_laddrreq *)data)->dstaddr);
			break;
		default:
			return EINVAL;
		}

		/* sa_family must be equal */
		if (src->sa_family != dst->sa_family)
			return EINVAL;

		/* validate sa_len */
		switch (src->sa_family) {
#ifdef INET
		case AF_INET:
			if (src->sa_len != sizeof(struct sockaddr_in))
				return EINVAL;
			break;
#endif
#ifdef INET6
		case AF_INET6:
			if (src->sa_len != sizeof(struct sockaddr_in6))
				return EINVAL;
			break;
#endif
		default:
			return EAFNOSUPPORT;
		}
		switch (dst->sa_family) {
#ifdef INET
		case AF_INET:
			if (dst->sa_len != sizeof(struct sockaddr_in))
				return EINVAL;
			break;
#endif
#ifdef INET6
		case AF_INET6:
			if (dst->sa_len != sizeof(struct sockaddr_in6))
				return EINVAL;
			break;
#endif
		default:
			return EAFNOSUPPORT;
		}

		/* check sa_family looks sane for the cmd */
		switch (cmd) {
		case SIOCSIFPHYADDR:
			if (src->sa_family == AF_INET)
				break;
			return EAFNOSUPPORT;
#ifdef INET6
		case SIOCSIFPHYADDR_IN6:
			if (src->sa_family == AF_INET6)
				break;
			return EAFNOSUPPORT;
#endif /* INET6 */
		case SIOCSLIFPHYADDR:
			/* checks done in the above */
			break;
		}

		error = nemo_set_tunnel(&sc->nemo_if, src, dst);
		break;

#ifdef SIOCDIFPHYADDR
	case SIOCDIFPHYADDR:
		nemo_delete_tunnel(&sc->nemo_if);
		break;
#endif
			
	case SIOCGIFPSRCADDR:
#ifdef INET6
	case SIOCGIFPSRCADDR_IN6:
#endif /* INET6 */
		if (sc->nemo_psrc == NULL) {
			error = EADDRNOTAVAIL;
			goto bad;
		}
		src = sc->nemo_psrc;
		switch (cmd) {
#ifdef INET
		case SIOCGIFPSRCADDR:
			dst = &ifr->ifr_addr;
			size = sizeof(ifr->ifr_addr);
			break;
#endif /* INET */
#ifdef INET6
		case SIOCGIFPSRCADDR_IN6:
			dst = (struct sockaddr *)
				&(((struct in6_ifreq *)data)->ifr_addr);
			size = sizeof(((struct in6_ifreq *)data)->ifr_addr);
			break;
#endif /* INET6 */
		default:
			error = EADDRNOTAVAIL;
			goto bad;
		}
		if (src->sa_len > size)
			return EINVAL;
		memcpy(dst, src, src->sa_len);
#ifdef INET6
		if (dst->sa_family == AF_INET6) {
			error = sa6_recoverscope((struct sockaddr_in6 *)dst);
			if (error != 0)
				goto bad;
		}
#endif
		break;
			
	case SIOCGIFPDSTADDR:
#ifdef INET6
	case SIOCGIFPDSTADDR_IN6:
#endif /* INET6 */
		if (sc->nemo_pdst == NULL) {
			error = EADDRNOTAVAIL;
			goto bad;
		}
		src = sc->nemo_pdst;
		switch (cmd) {
#ifdef INET
		case SIOCGIFPDSTADDR:
			dst = &ifr->ifr_addr;
			size = sizeof(ifr->ifr_addr);
			break;
#endif /* INET */
#ifdef INET6
		case SIOCGIFPDSTADDR_IN6:
			dst = (struct sockaddr *)
				&(((struct in6_ifreq *)data)->ifr_addr);
			size = sizeof(((struct in6_ifreq *)data)->ifr_addr);
			break;
#endif /* INET6 */
		default:
			error = EADDRNOTAVAIL;
			goto bad;
		}
		if (src->sa_len > size)
			return EINVAL;
		memcpy(dst, src, src->sa_len);
#ifdef INET6
		if (dst->sa_family == AF_INET6) {
			error = sa6_recoverscope((struct sockaddr_in6 *)dst);
			if (error != 0)
				goto bad;
		}
#endif
		break;

	case SIOCGLIFPHYADDR:
		if (sc->nemo_psrc == NULL || sc->nemo_pdst == NULL) {
			error = EADDRNOTAVAIL;
			goto bad;
		}

		/* copy src */
		src = sc->nemo_psrc;
		dst = (struct sockaddr *)
			&(((struct if_laddrreq *)data)->addr);
		size = sizeof(((struct if_laddrreq *)data)->addr);
		if (src->sa_len > size)
			return EINVAL;
		memcpy(dst, src, src->sa_len);

		/* copy dst */
		src = sc->nemo_pdst;
		dst = (struct sockaddr *)
			&(((struct if_laddrreq *)data)->dstaddr);
		size = sizeof(((struct if_laddrreq *)data)->dstaddr);
		if (src->sa_len > size)
			return EINVAL;
		memcpy(dst, src, src->sa_len);
		break;

	case SIOCSIFFLAGS:
		/* if_ioctl() takes care of it */
		break;
#ifdef MIP6
	case SIOCSIFPHYNEXTHOP: 
#ifdef INET6
	case SIOCSIFPHYNEXTHOP_IN6: {
#endif /* INET6 */
		struct sockaddr *nh = NULL;
		int nhlen = 0;

		switch (ifr->ifr_addr.sa_family) {
#ifdef INET
		case AF_INET:	/* IP supports Multicast */
			error = EAFNOSUPPORT;
			break;
#endif /* INET */
#ifdef INET6
		case AF_INET6:	/* IP6 supports Multicast */
			nh = (struct sockaddr *)
				&(((struct in6_ifreq *)data)->ifr_addr);
			nhlen = sizeof(((struct in6_ifreq *)data)->ifr_addr);
			break;
#endif /* INET6 */
		default:  /* Other protocols doesn't support Multicast */
			error = EAFNOSUPPORT;
			break;
		}

		if (error)
			return error;

		/* if pointer is null, allocate memory */
		if (sc->nemo_nexthop == NULL) {
			sc->nemo_nexthop = (struct sockaddr *)malloc(nhlen, M_IFADDR, M_WAITOK);
			if (sc->nemo_nexthop == NULL)
				return ENOMEM;

			bzero(sc->nemo_nexthop, nhlen);
		}
		/* set request address into nemo_nexthop */
		bcopy(nh, sc->nemo_nexthop, nhlen);
		sa6_embedscope(satosin6(sc->nemo_nexthop), ip6_use_defzone);
		break;
	}
	case SIOCGIFPHYNEXTHOP: 
#ifdef INET6
	case SIOCGIFPHYNEXTHOP_IN6: {
#endif /* INET6 */
		if (sc->nemo_nexthop == NULL) {
			error = EADDRNOTAVAIL;
			goto bad;
		}
		src = sc->nemo_nexthop;
		switch (cmd) {
#ifdef INET
		case SIOCGIFPHYNEXTHOP:
			dst = &ifr->ifr_addr;
			size = sizeof(ifr->ifr_addr);
			break;
#endif /* INET */
#ifdef INET6
		case SIOCGIFPHYNEXTHOP_IN6:
			dst = (struct sockaddr *)
				&(((struct in6_ifreq *)data)->ifr_addr);
			size = sizeof(((struct in6_ifreq *)data)->ifr_addr);
			break;
#endif /* INET6 */
		default:
			error = EADDRNOTAVAIL;
			goto bad;
		}
		if (src->sa_len > size)
			return EINVAL;
		memcpy(dst, src, src->sa_len);
#ifdef INET6
		if (dst->sa_family == AF_INET6) {
			error = sa6_recoverscope((struct sockaddr_in6 *)dst);
			if (error != 0)
				goto bad;
		}
#endif
		break;
	}
	case SIOCDIFPHYNEXTHOP: 
		/* if pointer is not null, free the memory */
		if (sc->nemo_nexthop) 
			free(sc->nemo_nexthop, M_IFADDR);
		sc->nemo_nexthop = NULL;
		break;
#endif
	default:
		error = ifioctl_common(ifp, cmd, data);
		break;
	}
 bad:
	return error;
}

int
nemo_set_tunnel(struct ifnet *ifp, struct sockaddr *src, struct sockaddr *dst)
{
	struct nemo_softc *sc = (struct nemo_softc *)ifp;
	struct nemo_softc *sc2;
	struct sockaddr *osrc, *odst, *sa;
	int s;
	int error = 0; 

	s = splsoftnet();

	for (sc2 = LIST_FIRST(&nemo_softc_list); sc2 != NULL;
	     sc2 = LIST_NEXT(sc2, nemo_list)) {
		if (sc2 == sc)
			continue;
		if (!sc2->nemo_pdst || !sc2->nemo_psrc)
			continue;
		if (sc2->nemo_pdst->sa_family != dst->sa_family ||
		    sc2->nemo_pdst->sa_len != dst->sa_len ||
		    sc2->nemo_psrc->sa_family != src->sa_family ||
		    sc2->nemo_psrc->sa_len != src->sa_len)
			continue;
#ifndef XBONEHACK
		/* can't configure same pair of address onto two nemos */
		if (bcmp(sc2->nemo_pdst, dst, dst->sa_len) == 0 &&
		    bcmp(sc2->nemo_psrc, src, src->sa_len) == 0) {
			error = EADDRNOTAVAIL;
			goto bad;
		}
#endif

		/* XXX both end must be valid? (I mean, not 0.0.0.0) */
	}

	/* XXX we can detach from both, but be polite just in case */
	if (sc->nemo_psrc) {
		switch (sc->nemo_psrc->sa_family) {
#ifdef INET
		case AF_INET:
			(void)in_gif_detach((struct gif_softc *)sc);
			break;
#endif
#ifdef INET6
		case AF_INET6:
			(void)in6_gif_detach((struct gif_softc *)sc);
			break;
#endif
		}
	}

	osrc = sc->nemo_psrc;
	sa = (struct sockaddr *)malloc(src->sa_len, M_IFADDR, M_WAITOK);
	memcpy(sa, src, src->sa_len);
	sc->nemo_psrc = sa;

	odst = sc->nemo_pdst;
	sa = (struct sockaddr *)malloc(dst->sa_len, M_IFADDR, M_WAITOK);
	memcpy(sa, dst, dst->sa_len);
	sc->nemo_pdst = sa;

	switch (sc->nemo_psrc->sa_family) {
#ifdef INET
	case AF_INET:
		error = in_gif_attach((struct gif_softc *)sc);
		break;
#endif
#ifdef INET6
	case AF_INET6:
		/* Check validity of the scope zone ID of the addresses. */
		if ((error = sa6_embedscope(
		    (struct sockaddr_in6 *)sc->nemo_psrc, ip6_use_defzone))
		    != 0)
			break;
		if ((error = sa6_embedscope(
		    (struct sockaddr_in6 *)sc->nemo_pdst, ip6_use_defzone))
		    != 0)
			break;
		error = in6_gif_attach((struct gif_softc *)sc);
		break;
#endif
	}
	if (error) {
		/* rollback */
		free(sc->nemo_psrc, M_IFADDR);
		free(sc->nemo_pdst, M_IFADDR);
		sc->nemo_psrc = osrc;
		sc->nemo_pdst = odst;
		goto bad;
	}

	if (osrc)
		free(osrc, M_IFADDR);
	if (odst)
		free(odst, M_IFADDR);

	if (sc->nemo_psrc && sc->nemo_pdst)
		ifp->if_flags |= IFF_RUNNING;
	else
		ifp->if_flags &= ~IFF_RUNNING;
	splx(s);

	return 0;

 bad:
	if (sc->nemo_psrc && sc->nemo_pdst)
		ifp->if_flags |= IFF_RUNNING;
	else
		ifp->if_flags &= ~IFF_RUNNING;
	splx(s);

	return error;
}

void
nemo_delete_tunnel(struct ifnet *ifp)
{
	struct nemo_softc *sc = (struct nemo_softc *)ifp;
	int s;

	s = splsoftnet();

	if (sc->nemo_psrc) {
		free(sc->nemo_psrc, M_IFADDR);
		sc->nemo_psrc = NULL;
	}
	if (sc->nemo_pdst) {
		free(sc->nemo_pdst, M_IFADDR);
		sc->nemo_pdst = NULL;
	}
#ifdef MIP6
	if (sc->nemo_nexthop) {
		free(sc->nemo_nexthop, M_IFADDR);
		sc->nemo_nexthop = NULL;
	}
#endif /* MIP6 */

	/* it is safe to detach from both */
#ifdef INET
	(void)in_gif_detach((struct gif_softc *)sc);
#endif
#ifdef INET6
	(void)in6_gif_detach((struct gif_softc *)sc);
#endif

	if (sc->nemo_psrc && sc->nemo_pdst)
		ifp->if_flags |= IFF_RUNNING;
	else
		ifp->if_flags &= ~IFF_RUNNING;
	splx(s);
}
