#include <sys/cdefs.h>
#include <sys/types.h>
#include <sys/stdint.h>
#include <sys/endian.h>

#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <err.h>

#include "amiga_disklabel.h"

#include "rdb.h"
#include "io.h"
#include "util.h"

#define min(a,b) (((a)<(b))?(a):(b))

/* convert id string into id word */
static unsigned long
parseid(char *s)
{
	unsigned long l;
	int i, n, v;
	char c;

	l = 0;
	for (i=0; i<4; ++i) {
		if (*s == '\0')
			break;
		if (*s == '\\') {
			c = *++s;
			if (c == '\0')
				break;
			if (c >= '0' && c <= '9') {
				v = 0;
				n = 0;
				while (c >= '0' && c <= '9') {
					v = v*10 + (c - '0');
					c = *++s;
					++n;
					if (n >= 3)
						break;
				}
				c = (char)v;
			}
		} else
			c = *s++;
		l |= (unsigned long)(unsigned char)c << 8 * (3-i);
	}

	return l;
}

/* convert id word into id string */
static void
sprintid(char *s, unsigned long l)
{
	char buf[10], c;
	int i;

	s[0] = '\0';
	for (i=0; i<4; ++i) {
		c = l >> 8 * (3-i);
		if (c < ' ' || c >= '\x7f')
			sprintf(buf,"\\%u",(unsigned char)c);
		else
			sprintf(buf,"%c",c);
		strcat(s, buf);
	}
}

/* retrieve buffer from bstr */
static void
getbstr(u_char *bstr, char *buf, size_t n)
{
	int i;

	for (i=0; i<bstr[0] && i<n; ++i)
		buf[i] = bstr[i+1];
	for (; i<n; ++i)
		buf[i] = '\0';
}

/* store buffer in bstr */
static void
putbstr(char *buf, size_t n, u_char *bstr)
{
	int i;

	for (i=0; buf[i] && i<n; ++i)
		bstr[i+1] = buf[i];
	bstr[0] = i;
	for (; i<n; ++i)
		bstr[i+1] = '\0';
}

/* swap block words depending on type */
static void
swap_block1(uint32_t *buf, int s, int e)
{
	int i;
	for (i=s; i<e; ++i)
		buf[i] = be32toh(buf[i]);
}

static void
swap_block(uint32_t *buf, char *what)
{
#define O(t,f) (offsetof(struct t, f)/sizeof(uint32_t))
	if (strcmp(what,"RDSK") == 0) {
		swap_block1(buf,0,O(rdblock, diskvendor));
	} else if (strcmp(what,"PART") == 0) {
		swap_block1(buf,0,O(partblock, partname));
		swap_block1(buf,O(partblock, resv2),128);
	} else {
		swap_block1(buf,0,128);
	}
#undef O
}

/* compute RDB checksums */
static uint32_t
xsum(uint32_t *buf, char *what)
{
	uint32_t sum;
	int i, n;

	n = buf[1];

	swap_block(buf, what);

	sum = 0;
	for (i = 0; i < n; ++i)
		sum += bswap32(buf[i]);

	swap_block(buf, what);

	return sum;
}

/* IO and swap */
static int
rdbread_block(int fd, void *buf, uint32_t key, char *what)
{
	int rc;
	rc = read_block(fd, buf, key, what);
	if (rc == 0)
		swap_block(buf, what);
	return rc;
}
static int
rdbwrite_block(int fd, void *buf, uint32_t key, char *what)
{
	swap_block(buf, what);
	return write_block(fd, buf, key, what);
}

/*
 * Free RDB memory
 */
void
free_rdb(struct rdb_mem *rdbm)
{
	struct part_mem *pbm, *npbm;
	struct lseg_mem *lsm, *nlsm;
	struct fsh_mem *fshm, *nfshm;


	for (fshm=rdbm->fshs; fshm != NULL; fshm = nfshm) {
		nfshm = fshm->next;
		free(fshm);
	}
	for (lsm=rdbm->drvini; lsm != NULL; lsm = nlsm) {
		nlsm = lsm->next;
		free(lsm);
	}
	for (pbm=rdbm->parts; pbm != NULL; pbm = npbm) {
		npbm = pbm->next;
		free(pbm);
	}
	free(rdbm);
}

/*
 * Adjust RDB to geometry
 */
void
size_rdb(struct rdb_mem *rdbm, struct diskgeometry *g)
{
	struct rdblock *rdb;

	rdbm->modified = 1;
	rdb = (struct rdblock *)rdbm->data;

	if (rdb->ncylinders == 0 || rdb->nheads == 0 || rdb->nsectors == 0) {
		rdb->ncylinders = g->blocks / (g->sectors * g->heads);
		rdb->nheads = g->heads;
		rdb->nsectors = g->sectors;
	} else {
		rdb->ncylinders = g->blocks / (rdb->nsectors * rdb->nheads);
	}

	if (rdb->ncylinders * rdb->nheads * rdb->nsectors != g->blocks)
		printf("Invalid geometry %lu * %lu * %lu != %lu\n",
			(unsigned long)rdb->ncylinders,
			(unsigned long)rdb->nheads,
			(unsigned long)rdb->nsectors,
			(unsigned long)g->blocks);

	rdb->secpercyl = rdb->nheads * rdb->nsectors;
	rdb->lowcyl    = 1;
	rdb->highcyl   = rdb->ncylinders - 1;
	rdb->rdblowb   = 0;
	rdb->rdbhighb  = min(rdb->nsectors, RDB_MAXBLOCKS);
}


/*
 * Create a default RDB
 */
struct rdb_mem *
init_rdb(int start, struct diskgeometry *g)
{
	struct rdblock *rdb;
	struct rdb_mem *rdbm;

	rdbm = calloc(1, sizeof(struct rdb_mem));
	if (rdbm == NULL)
		errx(1, "Unable to allocate RDB memory");

	rdbm->dskblk = start;
	rdbm->modified = 1;

	rdb = (struct rdblock *)rdbm->data;
	rdb->id = RDBLOCK_ID;
	rdb->nsumlong = g->blocksize / 4;
	rdb->hostid = 7; /* SCSI device id for host adapter */
	rdb->nbytes = g->blocksize;
	rdb->flags = RDBF_CTRLID | RDBF_SYNC;
	rdb->badbhead = RDBNULL;
	rdb->partbhead = RDBNULL;
	rdb->fsbhead = RDBNULL;
	rdb->driveinit = RDBNULL;
	rdb->resv1[0] = RDBNULL;
	rdb->resv1[1] = RDBNULL;
	rdb->resv1[2] = RDBNULL;
	rdb->resv1[3] = RDBNULL;
	rdb->resv1[4] = RDBNULL;
	rdb->resv1[5] = RDBNULL;
	rdb->interleave = 1;
	strcpy(rdb->contvendor, "NetBSD");
	strcpy(rdb->contproduct, "SCSI driver");
	strcpy(rdb->contrevision, "0");

	size_rdb(rdbm, g);

	strncpy(rdb->diskvendor, "Hardfile", 8);
	strncpy(rdb->diskproduct, "Hardfile", 16);
	strncpy(rdb->diskrevision, "", 4);

	rdb->flags |= RDBF_DISKID;

	return rdbm;
}

struct part_mem *
add_part(struct rdb_mem *rdbm, int block, struct part_mem *mark, uint32_t numcyl)
{
	struct rdblock *rdb = (struct rdblock *)rdbm->data;
	struct partblock *pb;
	struct part_mem *pbm, *pbmprev;
	int partnum;
	uint32_t lowcyl, highcyl;
	char partbuf[31];

	partnum = 1;
	pbmprev = NULL;
	for (pbm=rdbm->parts; pbm; pbm=pbm->next) {
		if (pbm == mark)
			break;
		++partnum;
		pbmprev = pbm;
	}
	mark = pbm;

	lowcyl = rdb->lowcyl;
	highcyl = rdb->highcyl;
	if (pbmprev) {
		lowcyl = ((struct partblock *)pbmprev->data)->e.highcyl+1;
		if (mark)
			highcyl = ((struct partblock *)mark->data)->e.lowcyl-1;
	} else {
		if (rdbm->parts)
			highcyl = ((struct partblock *)rdbm->parts->data)->e.lowcyl-1;
	}

	if (highcyl < lowcyl) {
		printf("No space for %lu cylinders\n",
			(unsigned long)numcyl);
		return NULL;
	}

	if (numcyl == 0) {
		numcyl = highcyl - lowcyl + 1;
		printf("Autosize partition to %lu cylinders\n",
			(unsigned long)numcyl);
	} else
		printf("Creating partition with %lu cylinders\n",
			(unsigned long)numcyl);

	if (highcyl - lowcyl + 1 < numcyl) {
		printf("Not enough space for %lu cylinders\n",
			(unsigned long)numcyl);
		return NULL;
	}

	pbm = calloc(1, sizeof(struct part_mem));
	if (pbm == NULL)
		errx(1, "Unable to allocate PART memory");

	pbm->dskblk = block;
	pbm->modified = 1;
	pbm->next = NULL;

	pb = (struct partblock *)pbm->data;
	pb->id = PARTBLOCK_ID;
	pb->nsumlong = rdb->nsumlong;
	pb->hostid = rdb->hostid;
	pb->e.tabsize = 16;
	pb->e.sizeblock = rdb->nbytes/4;
	pb->e.numheads = rdb->nheads;
	pb->e.secperblk = 1;
	pb->e.secpertrk = rdb->nsectors;
	pb->e.resvblocks = 0;
	pb->e.lowcyl = lowcyl;
	pb->e.highcyl = lowcyl + numcyl - 1;
	pb->e.mask = 0xfffffffe;	/***************************/
	pb->e.dostype = parseid("NBU\\7");
	sprintf(partbuf, "part%d", partnum);
	putbstr(partbuf, 31, pb->partname);
	pb->next = RDBNULL;

	if (pbmprev) {
		if (pbmprev->next)
			pb->next = pbmprev->next->dskblk;
		((struct partblock *)pbmprev->data)->next = pbm->dskblk;
		pbm->next = pbmprev->next;
		pbmprev->next = pbm;
		pbmprev->modified = 1;
	} else {
		pb->next = rdb->partbhead;
		rdb->partbhead = pbm->dskblk;
		pbm->next = rdbm->parts;
		rdbm->parts = pbm;
		rdbm->modified = 1;
	}

	return pbm;
}

int
delete_part(struct rdb_mem *rdbm, struct part_mem *mark)
{
	struct part_mem *pbm, *pbmprev;
	struct rdblock *rdb;
	struct partblock *pb;

	pbmprev = NULL;
	for (pbm=rdbm->parts; pbm; pbm=pbm->next) {
		if (pbm == mark)
			break;
		pbmprev = pbm;
	}
	if (pbm == NULL)
		return 0;

	rdb = (struct rdblock *)rdbm->data;
	pb = (struct partblock *)pbm->data;

	if (pbmprev) {
		((struct partblock *)pbmprev->data)->next = pb->next;
		pbmprev->next = pbm->next;
		pbmprev->modified = 1;
	} else {
		rdb->partbhead = pb->next;
		rdbm->parts = pbm->next;
		rdbm->modified = 1;
	}

	free(pbm);
	return 1;
}

/*
 * scan for RDB and load into memory
 */
static void
load_badb(int fd, struct rdb_mem *rdbm, uint32_t blk)
{
	struct badblock *bb;
	struct badb_mem *bbm, **bbmp;

	bbmp = &rdbm->badblocks;
	while (blk != RDBNULL) {
		bbm = calloc(1, sizeof (struct badb_mem));
		if (bbm == NULL)
			errx(1, "BADB malloc");

		bbm->dskblk = blk;
		bbm->modified = 0;
		bbm->next = NULL;

		bb = (struct badblock *)bbm->data;
		if (rdbread_block(fd, bb, blk, "BADB"))
			errx(1, "BADB read");
		if (bb->id != BADBLOCK_ID)
			errx(1, "BADBLOCK invalid ID");
		if (xsum((uint32_t *)bb, "BADB"))
			fprintf(stderr, "BADB Checksum Error\n");

		blk = bb->next;

		*bbmp = bbm;
		bbmp = &bbm->next;
	}
}

static void
load_part(int fd, struct rdb_mem *rdbm, uint32_t blk)
{
	struct partblock *pb;
	struct part_mem *pbm, **pbmp;

	pbmp = &rdbm->parts;
	while (blk != RDBNULL) {
		pbm = calloc(1, sizeof (struct part_mem));
		if (pbm == NULL)
			errx(1, "PART malloc");

		pbm->dskblk = blk;
		pbm->modified = 0;
		pbm->next = NULL;

		pb = (struct partblock *)pbm->data;
		if (rdbread_block(fd, pb, blk, "PART"))
			errx(1, "PART read");
		if (pb->id != PARTBLOCK_ID)
			errx(1, "PARTBLOCK invalid ID");
		if (xsum((uint32_t *)pb, "PART"))
			fprintf(stderr, "PART Checksum Error\n");

		blk = pb->next;

		*pbmp = pbm;
		pbmp = &pbm->next;
	}
}

static void
load_fs(int fd, struct rdb_mem *rdbm, uint32_t blk)
{
	struct fsh_mem *fsm, **fsmp;
	struct fsblock *fs;
	struct lseg_mem *lsm, **lsmp;
	struct lsegblock *ls;

	fsmp = &rdbm->fshs;
	while (blk != RDBNULL) {
		fsm = calloc(1, sizeof (struct fsh_mem));
		if (fsm == NULL)
			errx(1, "FSH malloc");

		fsm->lsegs = NULL;
		fsm->dskblk = blk;
		fsm->modified = 0;
		fsm->next = NULL;

		fs = (struct fsblock *)fsm->data;
		if (rdbread_block(fd, fs, blk, "FSHD"))
			errx(1, "FSH read");
		if (fs->id != FSBLOCK_ID)
			errx(1, "FSBLOCK invalid ID");
		if (xsum((uint32_t *)fs, "FSHD"))
			fprintf(stderr, "FSH Checksum Error\n");

		blk = fs->lsegblocks;

		lsmp = &fsm->lsegs;
		while (blk != RDBNULL) {
			lsm = calloc(1, sizeof (struct lseg_mem));
			if (lsm == NULL)
				errx(1, "FS LSEG malloc");

			lsm->dskblk = blk;
			lsm->modified = 0;
			lsm->next = NULL;

			ls = (struct lsegblock *)lsm->data;
			if (rdbread_block(fd, ls, blk, "LSEG"))
				errx(1, "FS LSEG read");
			if (ls->id != LSEGBLOCK_ID)
				errx(1, "FS LSEGBLOCK invalid ID");
			if (xsum((uint32_t *)ls, "LSEG"))
				fprintf(stderr, "FS LSEG Checksum Error\n");
			blk = ls->next;

			*lsmp = lsm;
			lsmp = &lsm->next;
		}

		blk = fs->next;

		*fsmp = fsm;
		fsmp = &fsm->next;
	}
}

static void
load_drvini(int fd, struct rdb_mem *rdbm, uint32_t blk)
{
	struct lseg_mem *lsm, **lsmp;
	struct lsegblock *ls;

 	lsmp = &rdbm->drvini;
	while (blk != RDBNULL) {
		lsm = calloc(1, sizeof (struct lseg_mem));
		if (lsm == NULL)
			errx(1, "DRVINI malloc");

		lsm->dskblk = blk;
		lsm->modified = 0;
		lsm->next = NULL;

		ls = (struct lsegblock *)lsm->data;
		if (rdbread_block(fd, ls, blk, "LSEG"))
			errx(1, "DRVINI read");
		if (ls->id != LSEGBLOCK_ID)
			errx(1, "DRVINI LSEGBLOCK invalid ID");
		if (xsum((uint32_t *)ls, "LSEG"))
			fprintf(stderr, "DRVINI Checksum Error\n");
		blk = ls->next;

		*lsmp = lsm;
		lsmp = &lsm->next;
	}
}

struct rdb_mem *
load_rdb(int fd)
{
	struct rdb_mem *rdbm;
	struct rdblock *rdb;

	int blk;

	rdbm = calloc(1, sizeof(struct rdb_mem));
	if (rdbm == NULL)
		errx(1, "RDB malloc");

	rdbm->parts = NULL;
	rdbm->fshs = NULL;
	rdbm->drvini = NULL;
	rdb = (struct rdblock *)rdbm->data;

	for (blk = 0; blk < RDB_MAXBLOCKS; ++blk) {
		if (rdbread_block(fd, rdb, blk, "RDSK"))
			errx(1, "RDB read");
		if (rdb->id == RDBLOCK_ID)
			break;
	}
	if (blk >= RDB_MAXBLOCKS) {
		fprintf(stderr, "RDB not found\n");
		free(rdbm);
		return NULL;
	}

	if (xsum((uint32_t *)rdb, "RDSK"))
		fprintf(stderr, "RDB Checksum error\n");

	rdbm->dskblk = blk;
	rdbm->modified = 0;
	if (rdb->badbhead != RDBNULL)
		load_badb(fd, rdbm, rdb->badbhead);
	if (rdb->partbhead != RDBNULL)
		load_part(fd, rdbm, rdb->partbhead);
	if (rdb->fsbhead != RDBNULL)
		load_fs(fd, rdbm, rdb->fsbhead);
	if (rdb->driveinit != RDBNULL)
		load_drvini(fd, rdbm, rdb->driveinit);

	return rdbm;
}

/*
 * scan for RDB and trash it by invalidating checksum
 */
int
erase_rdb(int fd, struct rdb_mem *rdbm)
{
	struct rdb_mem *trdbm;
	struct rdblock *rdb;
	uint32_t blk;

	if (rdbm->dskblk == 0)
		return 0;

	trdbm = calloc(1, sizeof(struct rdb_mem));
	if (trdbm == NULL)
		errx(1, "Temporary RDB malloc");
	rdb = (struct rdblock *)trdbm->data;

	for (blk = 0; blk < rdbm->dskblk; ++blk) {
		if (rdbread_block(fd, rdb, blk, "RDSK"))
			break;
		if (rdb->id == RDBLOCK_ID) {
			printf("Erasing RDSK block @ %lu\n",
				(unsigned long)blk);
			rdb->id = 0x2F44534B;
			rdb->chksum = 0;
			rdb->chksum = -xsum((uint32_t *)rdb, "RDSK");
			if (rdbwrite_block (fd, rdb, blk, "RDSK"))
				return -1;
		}
	}

	free(trdbm);
	return 0;
}


/*
 * write RDB to disk
 */
int
write_rdb(int fd, struct rdb_mem *rdbm)
{
	struct badb_mem *bbm;
	struct part_mem *pbm;
	struct fsh_mem *fsm;
	struct lseg_mem *lsm;

	if (rdbm->modified) {
		struct rdblock *rdb = (struct rdblock *)rdbm->data;
		printf("Writing RDSK block @ %lu\n",
			(unsigned long)rdbm->dskblk);
		rdb->chksum = 0;
		rdb->chksum = -xsum((uint32_t *)rdb, "RDSK");
		if (rdbwrite_block(fd, rdb, rdbm->dskblk, "RDSK"))
			return -1;
	}

	bbm = rdbm->badblocks;
	while (bbm) {
		if (bbm->modified) {
			struct badblock *bb = (struct badblock *)bbm->data;
			printf("Writing BADB block @ %lu\n",
				(unsigned long)bbm->dskblk);
			bb->chksum = 0;
			bb->chksum = -xsum((uint32_t *)bb, "BADB");
			if (rdbwrite_block(fd, bb, bbm->dskblk, "BADB"))
				return -1;
		}
		bbm = bbm->next;
	}

	pbm = rdbm->parts;
	while (pbm) {
		if (pbm->modified) {
			struct partblock *pb = (struct partblock *)pbm->data;
			printf("Writing PART block @ %lu\n",
				(unsigned long)pbm->dskblk);
			pb->chksum = 0;
			pb->chksum = -xsum((uint32_t *)pb, "PART");
			if (rdbwrite_block(fd, pb, pbm->dskblk, "PART"))
				return -1;
		}
		pbm = pbm->next;
	}

	fsm = rdbm->fshs;
	while (fsm) {
		if (fsm->modified) {
			struct fsblock *fs = (struct fsblock *)fsm->data;
			printf("Writing FSHD block @ %lu\n",
				(unsigned long)fsm->dskblk);
			fs->chksum = 0;
			fs->chksum = -xsum((uint32_t *)fs, "FSHD");
			if (rdbwrite_block(fd, fs, fsm->dskblk, "FSHD"))
				return -1;
		}
		fsm = fsm->next;
	}

	lsm = rdbm->drvini;
	while (lsm) {
		if (lsm->modified) {
			struct lsegblock *ls = (struct lsegblock *)lsm->data;
			printf("Writing LSEG block @ %lu\n",
				(unsigned long)lsm->dskblk);
			ls->chksum = 0;
			ls->chksum = -xsum((uint32_t *)ls, "LSEG");
			if (rdbwrite_block(fd, ls, lsm->dskblk, "LSEG"))
				return -1;
		}
		lsm = lsm->next;
	}

	return 0;
}

/*
 * show RDB data
 */
static void
display_part1(uint32_t dskblk, struct partblock *pb)
{
	char partbuf[32];
	char dostype[20];
	char *sep = "";

	sprintid(dostype, pb->e.dostype);

	getbstr(pb->partname, partbuf, 31);
	partbuf[31] = '\0';

	printf("PART %-2lu %6lu-%-6lu %2lu %6lu %4lu %3lu %08lx  %4ld %2lu %-5s %s",
		(unsigned long)dskblk,
		(unsigned long)pb->e.lowcyl,
		(unsigned long)pb->e.highcyl,
		(unsigned long)pb->e.numheads,
		(unsigned long)pb->e.secpertrk,
		(unsigned long)pb->e.sizeblock,
		(unsigned long)pb->e.resvblocks,
		(unsigned long)pb->e.mask,
		(long)(int32_t)pb->e.bootpri,
		(unsigned long)(pb->e.tabsize >=19 ? pb->e.bootblocks : 0),
		dostype,
		partbuf);

	if (pb->flags) {
		printf(" (");
		if (pb->flags & PBF_BOOTABLE) {
			printf("Bootable");
			sep=",";
		}
		if (pb->flags & PBF_NOMOUNT)
			printf("%sNoMount", sep);
		if (pb->flags & ~(PBF_BOOTABLE|PBF_NOMOUNT))
			printf("%s0x%04lx", sep, (unsigned long)pb->flags);
		printf(")");
	}

	printf("\n");
}
static void
display_part2(int dskblk, struct partblock *pb)
{
	printf(" ts=%lu so=%lu spb=%lu prefac=%lu inter=%lu nbufs=%lu memt=%08lx max=%lu\n",
		(unsigned long)pb->e.tabsize,
		(unsigned long)pb->e.secorg,
		(unsigned long)pb->e.secperblk,
		(unsigned long)pb->e.prefac,
		(unsigned long)pb->e.interleave,
		(unsigned long)pb->e.numbufs,
		(unsigned long)pb->e.membuftype,
		(unsigned long)pb->e.maxtrans);
	printf(" nsum=%lu hostid=%lu next=%lu",
		(unsigned long)pb->nsumlong,
		(unsigned long)pb->hostid,
		(unsigned long)pb->next);
	if (pb->e.tabsize >= 20)
		printf(" fsize=%lu",(unsigned long)pb->e.fsize);
	if (pb->e.tabsize >= 21)
		printf(" frag=%lu",(unsigned long)pb->e.frag);
	if (pb->e.tabsize >= 22)
		printf(" cpg=%lu",(unsigned long)pb->e.cpg);
	printf("\n");
}

static void
display_part(struct part_mem *pbm, struct partblock *pb1)
{
	struct partblock *pb;

	printf("     %-2s %6s-%-6s %2s %6s %4s %3s %8s  %4s %2s %5s %s\n",
		"#",
		"Low","High",
		"Hd", "SpT", "Siz",
		"Res", "Mask", "Pri", "Bb",
		"Type",
		"Name");

	while (pbm) {
		pb = (struct partblock *)pbm->data;
		if (pb->id != PARTBLOCK_ID)
			break;
		if (pb1 == NULL || pb1 == pb) {
			display_part1(pbm->dskblk, pb);
			if (pb1 != NULL)
				display_part2(pbm->dskblk, pb);
		}
		pbm = pbm->next;
	}
}

static void
display_fsh(struct fsh_mem *fsm)
{
	char dostype[20];
	struct fsblock *fs;
	struct lsegblock *ls;
	struct lseg_mem *lsm, *span;
	int nblks, nbytes;
	const char *sep;

	while (fsm) {
		fs = (struct fsblock *)fsm->data;
		if (fs->id != FSBLOCK_ID) {
			fprintf(stderr, "FileSystem Header bad\n");
			break;
		}
		printf("FSH  %-2lu", (unsigned long)fsm->dskblk);
		sprintid(dostype, fs->dostype);
		printf(" dostype %-4s", dostype);
		nblks = nbytes = 0;
		lsm = fsm->lsegs;
		printf(" lseg [");
		sep = "";
		span = 0;
		while (lsm) {
			ls = (struct lsegblock *)lsm->data;
			if (ls->id == LSEGBLOCK_ID) {
				nblks++;
				nbytes += (ls->nsumlong - 5) * 4;
				span = lsm;
				while (lsm->next && lsm->next->dskblk == lsm->dskblk+1) {
					ls = (struct lsegblock *)lsm->data;
					nblks++;
					nbytes += (ls->nsumlong - 5) * 4;
					lsm = lsm->next;
				}
				if (lsm != span)
					printf("%s%lu-%lu",sep,
						(unsigned long)span->dskblk,
						(unsigned long)lsm->dskblk);
				else
					printf("%s%lu",sep,
						(unsigned long)lsm->dskblk);
			} else
				printf("%s!%lu",sep,
					(unsigned long)lsm->dskblk);
			sep = ",";
			lsm = lsm->next;
		}
		printf ("] blocks %d bytes %d[0x%x]\n", nblks, nbytes, nbytes);
		fsm = fsm->next;
	}
}

static void
display_drvini(struct lseg_mem *drvm)
{
	struct lsegblock *ls;
	int nblks, nbytes;

	if (drvm == NULL)
		return;

	printf("DriveInit @ %lu:  ", (unsigned long)drvm->dskblk);
	nblks = nbytes = 0;
	while (drvm) {
		ls = (struct lsegblock *)drvm->data;
		if (ls->id != LSEGBLOCK_ID)
			break;
		++nblks;
		nbytes += (ls->nsumlong - 5) * 4;
		drvm = drvm->next;
	}
	printf("%d blocks %d[%x] bytes\n", nblks, nbytes, nbytes);
}

static void
display_rdb(struct rdb_mem *rdbm)
{
	struct rdblock *rdb;
	char str[20];

	printf("RDSK %-2lu\n Flags", (unsigned long)rdbm->dskblk);
	rdb = (struct rdblock *)rdbm->data;
	str[0] = ' ';
	if (rdb->flags & RDBF_LAST) {
		printf(" LAST");
		str[0] = ',';
	}
	if (rdb->flags & RDBF_LASTLUN) {
		printf("%cLASTLUN", str[0]);
		str[0] = ',';
	}
	if (rdb->flags & RDBF_LASTUNIT) {
		printf("%cLASTID", str[0]);
		str[0] = ',';
	}
	if (rdb->flags & RDBF_NORESELECT) {
		printf("%cNORESELECT", str[0]);
		str[0] = ',';
	}
	if (rdb->flags & RDBF_DISKID) {
		printf("%cDISKID", str[0]);
		str[0] = ',';
	}
	if (rdb->flags & RDBF_CTRLID) {
		printf("%cCTRLID", str[0]);
		str[0] = ',';
	}
	if (rdb->flags & RDBF_SYNC) {
		printf("%cSYNC", str[0]);
		str[0] = ',';
	}
	printf("\n");
	if (rdb->flags & RDBF_DISKID) {
		strncpy(str, rdb->diskvendor, 8);
		str[8] = 0;
		printf(" Disk Vendor '%s'", str);
		strncpy(str, rdb->diskproduct, 16);
		str[16] = 0;
		printf(" Product '%s'", str);
		strncpy(str, rdb->diskrevision, 4);
		str[4] = 0;
		printf(" Revision '%s'\n", str);
	}
	if (rdb->flags & RDBF_CTRLID) {
		strncpy(str, rdb->contvendor, 8);
		str[8] = 0;
		printf(" Controller Vendor '%s'", str);
		strncpy(str, rdb->contproduct, 16);
		str[16] = 0;
		printf(" Product '%s'", str);
		strncpy(str, rdb->contrevision, 4);
		str[4] = 0;
		printf(" Revision '%s'\n", str);
	}
	printf(" Cylinders %lu Heads %lu Sectors %lu",
		(unsigned long)rdb->ncylinders,
		(unsigned long)rdb->nheads,
		(unsigned long)rdb->nsectors);
	printf(" LowCyl %lu HighCyl %lu Sect/Cyl %lu\n",
		(unsigned long)rdb->lowcyl,
		(unsigned long)rdb->highcyl,
		(unsigned long)rdb->secpercyl);
	printf(" nsum=%lu hostid=%lu nbytes=%lu inter=%lu park=%lu wpre=%lu reduce=%lu step=%lu psecs=%lu\n",
		(unsigned long)rdb->nsumlong,
		(unsigned long)rdb->hostid,
		(unsigned long)rdb->nbytes,
		(unsigned long)rdb->interleave,
		(unsigned long)rdb->park,
		(unsigned long)rdb->wprecomp,
		(unsigned long)rdb->reducedwrite,
		(unsigned long)rdb->steprate,
		(unsigned long)rdb->parkseconds);
	printf(" rdblowb=%lu rdbhighb=%lu\n",
		(unsigned long)rdb->rdblowb,
		(unsigned long)rdb->rdbhighb);
}

void display_all(struct rdb_mem *rdbm)
{
	display_rdb(rdbm);
	display_part(rdbm->parts, NULL);
	display_fsh(rdbm->fshs);
	display_drvini(rdbm->drvini);
}

uint32_t find_blk(struct rdb_mem *rdbm)
{
	struct rdblock *rdb = (struct rdblock *)rdbm->data;
	struct badb_mem *bbm;
	struct part_mem *pbm;
	struct fsh_mem *fsm;
	struct lseg_mem *lsm;
	uint32_t last;

	last = rdbm->dskblk;

	for (bbm=rdbm->badblocks; bbm; bbm=bbm->next) {
		if (bbm->dskblk > last)
			last = bbm->dskblk;
	}
	for (pbm=rdbm->parts; pbm; pbm=pbm->next) {
		if (pbm->dskblk > last)
			last = pbm->dskblk;
	}
	for (fsm=rdbm->fshs; fsm; fsm=fsm->next) {
		if (fsm->dskblk > last)
			last = fsm->dskblk;
		for (lsm=fsm->lsegs; lsm; lsm=lsm->next) {
			if (lsm->dskblk > last)
				last = lsm->dskblk;
		}
	}
	for (lsm=rdbm->drvini; lsm; lsm=lsm->next) {
		if (lsm->dskblk > last)
			last = lsm->dskblk;
	}

	last = last+1;

	if (last >= rdb->lowcyl * rdb->secpercyl)
		return 0;

	return last;
}

struct part_mem *find_part(struct rdb_mem *rdbm, uint32_t blk)
{
	struct part_mem *pbm;

	for (pbm=rdbm->parts; pbm; pbm=pbm->next) {
		if (pbm->dskblk == blk)
			break;
	}
	return pbm;
}

static struct mitem menu_rdsk[] = {
	{ 'i', "hostid ID" },
	{ 'b', "nbytes NUM" },
	{ 'C', "ncylinders NUM" },
	{ 'S', "nsectors NUM" },
	{ 'H', "nheads NUM" },
	{ 'f', "flags [last,lastlun,lastunit,noreselect,diskid,ctrlid,sync]" },
	{ 'l', "lowcyl NUM" },
	{ 'h', "highcyl NUM" },
	{ 'v', "disk vendor STRING" },
	{ 'p', "disk product STRING" },
	{ 'r', "disk revision STRING" },
	{ 'V', "controller vendor STRING" },
	{ 'P', "controller product STRING" },
	{ 'R', "controller revision STRING" },
	{ 'x', "check" },
	{ 'X', "set geometry and check" },
	{ '.', "print" },
	{ 'a', "abort" },
	{ 'q', "quit" }
};

int
edit_rdb(struct rdb_mem *rdbm, struct diskgeometry *g)
{
	struct rdblock *rdb;
	struct rdblock RDB;
	int c, mod;

	rdb = (struct rdblock *)rdbm->data;
	RDB = *rdb;
	mod = 0;

	display_rdb(rdbm);

	while ((c = MENU(menu_rdsk)) != EOF) {
		switch (c) {
		case 'q':
			if (memcmp(rdb, &RDB, sizeof(RDB)))
				mod = 1;
			break;
		case 'a':
			*rdb = RDB;
			mod = 0;
			break;
		case '.':
			display_rdb(rdbm);
			break;
		case 'X':
			size_rdb(rdbm, g);
			/* FALLTHROUGH */
		case 'x':
			rdb->rdblowb = 0;
			rdb->rdbhighb = RDB_MAXBLOCKS;
			rdb->interleave = 1;
			rdb->nsumlong = rdb->nbytes/4;
			rdb->secpercyl = rdb->nsectors * rdb->nheads;

			break;
		case 'i': GETUINT(&rdb->hostid);break;
		case 'b': GETUINT(&rdb->nbytes);break;
		case 'C': GETUINT(&rdb->ncylinders);break;
		case 'S': GETUINT(&rdb->nsectors);break;
		case 'H': GETUINT(&rdb->nheads);break;
		case 'f': GETFLAGS(&rdb->flags,"last,lastlun,lastunit,noreselect,diskid,ctrlid,sync");break;
		case 'n': GETUINT(&rdb->nbytes); break;
		case 'l': GETUINT(&rdb->lowcyl); break;
		case 'h': GETUINT(&rdb->highcyl); break;
		case 'v': GETBUF(rdb->diskvendor); break;
		case 'p': GETBUF(rdb->diskproduct); break;
		case 'r': GETBUF(rdb->diskrevision); break;
		case 'V': GETBUF(rdb->contvendor); break;
		case 'P': GETBUF(rdb->contproduct); break;
		case 'R': GETBUF(rdb->contrevision); break;
		default:
			printf("invalid input: '%c'\n",c);
			break;
		}
		if (c == 'a' || c == 'q')
			break;
	}

	if (mod)
		rdbm->modified = 1;
	return rdbm->modified;
}

static struct mitem menu_part[] = {
	{ 'f', "flags [bootable,nomount]" },
	{ 'n', "partname STRING" },
	{ 'r', "resvblocks NUM" },
	{ 'v', "interleave NUM" },
	{ 'l', "lowcyl NUM" },
	{ 'h', "highcyl NUM" },
	{ 'b', "numbufs NUM" },
	{ 'B', "membuftype [chip,fast,local,24bitdam]" },
	{ 'M', "maxtrans NUM" },
	{ 'm', "mask NUM" },
	{ 'p', "bootpri NUM" },
	{ 't', "dostype STRING" },
	{ 'o', "bootblocks NUM" },
	{ 'S', "fsize NUM" },
	{ 'F', "frag NUM" },
	{ 'C', "cpg NUM" },
	{ 'x', "check" },
	{ '.', "print" },
	{ 'a', "abort" },
	{ 'q', "quit" }
};

static int
makeenviron(struct ados_environ *e)
{
	uint32_t tabsize = e->tabsize;
	int mod = 0;

#define s(a,n,i) ((a)->tabsize >= (i) ? (a)->n : 0)
	if (s(e,fsize,20) || s(e,frag,21) || s(e,cpg,22)) {
		e->fsize = s(e,fsize,20);
		e->frag = s(e,frag,21);
		e->cpg = s(e,cpg,22);
		e->tabsize = 23; /* XXX */
		mod = 1;
	} else if (s(e,bootblocks,19))
		e->tabsize = 19;
	else if (s(e,control,18))
		e->tabsize = 18;
	else if (s(e,baud,17))
		e->tabsize = 17;
#undef s
	if (tabsize != e->tabsize)
		mod = 1;
	return mod;
}

int
edit_part(struct part_mem *pbm, struct rdb_mem *rdbm)
{
	struct rdblock *rdb;
	struct partblock *pb, PB;
	int c, mod;
	long l;
	char partbuf[31];
	char dostype[20];

	rdb = (struct rdblock *)rdbm->data;
	pb  = (struct partblock *)pbm->data;
	PB  = *pb;
	mod = 0;

#define set(x,l,h,v) if((x)<(l)||(x)>(h)){(x)=(v);mod=1;}
	set(pb->e.tabsize,0,23,23); /* XXX */
	set(pb->e.tabsize,16,23,16);
	set(pb->e.secorg,0,0,0);
	set(pb->e.secperblk,1,1,1);
	set(pb->e.prefac,0,0,0);
#undef set

	if (!batch) display_part(pbm, pb);

	while ((c = MENU(menu_part)) != EOF) {
		switch (c) {
		case 'q':
			if (memcmp(pb, &PB, sizeof(PB)))
				mod = 1;
			break;
		case 'a':
			*pb = PB;
			break;
		case '.':
			display_part(pbm, pb);
			break;
		case 'x':
			pb->nsumlong = 128;
			pb->hostid = rdb->hostid;
			pb->e.sizeblock = rdb->nbytes / 4;
			pb->e.secpertrk = rdb->nsectors;
			pb->e.numheads = rdb->nheads;
			break;
		case 'f': GETFLAGS(&pb->flags,"bootable,nomount");break;
		case 'n':
			GETBUF(partbuf);
			putbstr(partbuf, 31, pb->partname);
			break;
		case 's': GETUINT(&pb->e.sizeblock); break;
		case 'r': GETUINT(&pb->e.resvblocks); break;
		case 'v': GETUINT(&pb->e.interleave); break;
		case 'l': GETUINT(&pb->e.lowcyl); break;
		case 'h': GETUINT(&pb->e.highcyl); break;
		case 'b': GETUINT(&pb->e.numbufs); break;
		case 'B': GETFLAGS(&pb->e.membuftype,",chip,fast,,,,,,local,24bitdma"); break;
		case 'M': GETUINT(&pb->e.maxtrans); break;
		case 'm': GETUINT(&pb->e.mask); break;
		case 'p': if (getint(&l)) pb->e.bootpri = l; break;
		case 't':
			GETBUF(dostype);
			dostype[19] = '\0';
			pb->e.dostype = parseid(dostype);
			break;
		case 'o':
			GETUINT(&pb->e.bootblocks);
			if (pb->e.tabsize < 19)
				pb->e.tabsize = 19;
			break;
		case 'S':
			GETUINT(&pb->e.fsize);
			if (pb->e.tabsize < 20)
				pb->e.tabsize = 20;
			break;
		case 'F':
			GETUINT(&pb->e.frag);
			if (pb->e.tabsize < 21)
				pb->e.tabsize = 21;
			break;
		case 'C':
			GETUINT(&pb->e.cpg);
			if (pb->e.tabsize < 22)
				pb->e.tabsize = 22;
			break;
		default:
			printf("invalid input: '%c'\n",c);
			break;
		}
		if (c == 'a' || c == 'q')
			break;
	}

	if (makeenviron(&pb->e))
		mod = 1;

	if (mod)
		pbm->modified = 1;
	return pbm->modified;
}

void
print_disktab(struct rdb_mem *rdbm)
{
	struct part_mem *pbm;
	struct rdblock *rdb;
	struct partblock *pb, *parts[8], PARTC, PARTD;
	int nparts, i;
	char c;
	unsigned long spc;
	uint32_t dtroot = parseid("NBR\\7");
	uint32_t dtswap = parseid("NBS\\1");
	uint32_t dtuser = parseid("NBU\\7");
	char *fstype;
	unsigned long bsize, fsize;

	rdb = (struct rdblock *)rdbm->data;

	for (i=0; i<8; ++i)
		parts[i] = NULL;

	nparts = 4;
	for (pbm=rdbm->parts; pbm; pbm=pbm->next) {
		pb = (struct partblock *)pbm->data;
		if (pb->e.dostype == dtroot) {
			if (parts[0] == NULL)
				parts[0] = pb;
			continue;
		}
		if (pb->e.dostype == dtswap) {
			if (parts[1] == NULL)
				parts[1] = pb;
			continue;
		}
		if (nparts < 8)
			parts[nparts++] = pb;
	}

	/* XXX i386 convention */
	PARTC.e.secpertrk = rdb->nsectors;
	PARTC.e.numheads  = rdb->nheads;
	PARTC.e.sizeblock = rdb->nbytes / 4;
	PARTC.e.lowcyl    = 0;
	PARTC.e.highcyl   = rdb->ncylinders-1;
	PARTC.e.dostype   = 0;
	parts[2] = &PARTC;
	PARTD.e.secpertrk = rdb->nsectors;
	PARTD.e.numheads  = rdb->nheads;
	PARTD.e.sizeblock = rdb->nbytes / 4;
	PARTD.e.lowcyl    = rdb->lowcyl;
	PARTD.e.highcyl   = rdb->highcyl;
	PARTD.e.dostype   = 0;
	parts[3] = &PARTD;

	printf("%s%s%s:\\\n\t:dt=%s:se#%lu:ns#%lu:nt#%lu:sc#%lu:nc#%lu:\\\n",
		"hardfile",
		rdb->flags & RDBF_DISKID ? "|" : "",
		rdb->flags & RDBF_DISKID ? rdb->diskproduct : "",
		"unknown",
		(unsigned long)rdb->nbytes,
		(unsigned long)rdb->nsectors,
		(unsigned long)rdb->nheads,
		(unsigned long)rdb->secpercyl,
		(unsigned long)rdb->ncylinders);

	for (i=0; i<nparts; ++i) {
		pb = parts[i];
		if (pb == NULL)
			continue;
		spc = pb->e.secpertrk * pb->e.numheads;
		if (pb->e.dostype == dtswap) {
			fstype = "swap";
			bsize = fsize = 0;
		} else if (pb->e.dostype == dtroot
		           || pb->e.dostype == dtuser) {
			fstype = "4.2BSD";
			fsize = pb->e.sizeblock * 4;
			if (fsize < 1024) fsize = 1024;
			bsize = fsize <= 8192 ? fsize * 8
				: fsize <= 16384 ? fsize * 4
				: fsize <= 32768 ? fsize * 2
				: fsize;
			if (pb->e.tabsize >= 20)
				bsize = pb->e.fsize;
			if (pb->e.tabsize >= 21)
				fsize = pb->e.frag;
		} else if (pb->e.dostype == 0) {
			fstype = NULL;
			fsize = bsize = 0;
		} else {
			fstype = "ADOS";
			fsize = bsize = 0;
		}
		c = 'a' + i;
		printf("\t:p%c#%lu:o%c#%lu",
			c, (pb->e.highcyl - pb->e.lowcyl + 1) * spc,
			c, pb->e.lowcyl * spc);
		if (fstype != NULL) printf(":t%c=%s", c, fstype);
		if (bsize != 0) printf(":b%c#%lu", c, bsize);
		if (fsize != 0) printf(":f%c#%lu", c, fsize);
		printf(":%s\n", i+1 >= nparts ? "" : "\\");
	}
}
