/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Alistair Crooks (agc@NetBSD.org)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <dirent.h>
#include <err.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "detach.h"

/* find the correct Unix-domain socket name - returned in ap->sockname */
static int
sockexists(attachment_t *ap, char *sockdir, pid_t pid)
{
	struct dirent	*dp;
	struct stat	 st;
	DIR		*dirp;
	int		 found;

	if (pid != 0) {
		/* if the pid is non-zero, then we know what to look for */
		(void) snprintf(ap->sockname, sizeof(ap->sockname), "%s/window-%d", sockdir, pid);
		if (stat(ap->sockname, &st) < 0) {
			ap->sockname[0] = 0x0;
			return 0;
		}
		if (!S_ISSOCK(st.st_mode)) {
			ap->sockname[0] = 0x0;
			return 0;
		}
		return 1;
	}
	/* otherwise we have to search the directory for it */
	if ((dirp = opendir(sockdir)) == NULL) {
		warn("can't open directory for \"%s\"", sockdir);
		return 0;
	}
	for (found = 0 ; (dp = readdir(dirp)) != NULL ; ) {
		if (strcmp(dp->d_name, ".") == 0 || strcmp(dp->d_name, "..") == 0) {
			continue;
		}
		if (strncmp(dp->d_name, "window-", 7) != 0) {
			continue;
		}
		(void) snprintf(ap->sockname, sizeof(ap->sockname), "%s/%s", sockdir, dp->d_name);
		if (stat(ap->sockname, &st) < 0) {
			continue;
		}
		if (!S_ISSOCK(st.st_mode)) {
			continue;
		}
		found = 1;
		break;
	}
	(void) closedir(dirp);
	return found;
}

/* make a new server-side socket */
static int
newserversocket(attachment_t *ap)
{
	struct sockaddr_un	server;
	int			sd;

	if ((sd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
		warn("can't get socket on server side");
		return -1;
	}
	(void) memset(&server, 0x0, sizeof(server));
	server.sun_family = AF_UNIX;
	(void) strlcpy(server.sun_path, ap->sockname, sizeof(server.sun_path));
	if (connect(sd, (struct sockaddr *)&server, strlen(ap->sockname) + 2) == 0) {
		warn("socket \"%s\" already exists", server.sun_path);
		(void) close(sd);
		return -1;
	}
	if (bind(sd, (struct sockaddr *)&server, strlen(ap->sockname) + 2) < 0) {
		warn("can't bind to socket \"%s\"", ap->sockname);
		(void) close(sd);
		return -1;
	}
	(void) chmod(ap->sockname, 0700);
	if (listen(sd, 10) < 0) {
		warn("can't listen to socket \"%s\"", ap->sockname);
		(void) close(sd);
		return -1;
	}
	return sd;
}

/* make a new client-side socket */
static int
newclientsocket(attachment_t *ap)
{
	struct sockaddr_un	server;
	int			sd;

	if ((sd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
		warn("can't get socket on client side");
		return -1;
	}
	(void) memset(&server, 0x0, sizeof(server));
	server.sun_family = AF_UNIX;
	(void) strlcpy(server.sun_path, ap->sockname, sizeof(server.sun_path));
	if (connect(sd, (struct sockaddr *)&server, strlen(ap->sockname) + 2) < 0) {
		warn("can't connect to \"%s\"", server.sun_path);
		(void) close(sd);
		return -1;
	}
	return sd;
}

/* fill in the base socket directory name */
static void
fillbasedir(char *buf, size_t size, char *base)
{
	struct passwd	*pwp;
	char		*name;

	name = NULL;
	if ((pwp = getpwuid(geteuid())) != NULL) {
		name = pwp->pw_name;
	}
	(void) snprintf(buf, size, "%s/windows/%s", base,
			(name) ? name : "unknown");
}

/* create the socket */
int
createsocket(attachment_t *ap)
{
	fillbasedir(ap->basedir, sizeof(ap->basedir), WW_SOCKETDIR);
	if (sockexists(ap, ap->basedir, getpid())) {
		warn("can't create a new socket \"%s\"", ap->sockname);
		return 0;
	}
	return newserversocket(ap);
}

/* attach to the server socket */
int
attachsocket(attachment_t *ap, pid_t pid)
{
	int	sd;

	fillbasedir(ap->basedir, sizeof(ap->basedir), WW_SOCKETDIR);
	if (!sockexists(ap, ap->basedir, pid)) {
		warn("can't create a new socket \"%s\"", ap->sockname);
		return 0;
	}
	if ((sd = newclientsocket(ap)) < 0) {
		warn("can't attach to socket \"%s\"", ap->sockname);
		return 0;
	}
	return sd;
}
