/*	$NetBSD: win.c,v 1.13 2003/08/07 11:17:34 agc Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#ifndef lint
__RCSID("$NetBSD: win.c,v 1.13 2003/08/07 11:17:34 agc Exp $");
#endif /* not lint */

#include <stdlib.h>
#include <string.h>

#include "defs.h"
#include "char.h"

static void
labelwin(winvars_t *winvars, ww_t *w)
{
	int mode;

	mode = w == winvars->selwin ? WWM_REV : 0;

	if (!ISSET(w->ww_uflags, WWU_HASFRAME))
		return;
	if (w->ww_id >= 0) {
		char buf[2];

		buf[0] = w->ww_id + '1';
		buf[1] = 0;
		wwlabel(winvars, w, winvars->framewin, 1, buf, mode);
	}
	if (w->ww_label) {
		int col;

		if (ISSET(w->ww_uflags, WWU_CENTER)) {
			col = (w->ww_w.nc - strlen(w->ww_label)) / 2;
			col = MAX(3, col);
		} else
			col = 3;
		wwlabel(winvars, w, winvars->framewin, col, w->ww_label, mode);
	}
}

static void
wwclose(winvars_t *winvars, ww_t *w)
{
	winvars->wwindex[w->ww_index] = 0;
	if (w->ww_pty >= 0)
		(void) close(w->ww_pty);
	if (w->ww_socket >= 0)
		(void) close(w->ww_socket);
	wwfree(w->ww_win, w->ww_w.t);
	wwfree((char **)(void *)w->ww_buf, w->ww_b.t);
	if (w->ww_fmap != 0)
		wwfree(w->ww_fmap, w->ww_w.t);
	free(w->ww_nvis + w->ww_w.t);
	if (w->ww_ob != 0)
		free(w->ww_ob);
	free(w);
}

/*
 * Higher level routines for dealing with windows.
 *
 * There are two types of windows: user window, and information window.
 * User windows are the ones with a pty and shell.  Information windows
 * are for displaying error messages, and other information.
 *
 * The windows are doubly linked in overlapping order and divided into
 * two groups: foreground and normal.  Information
 * windows are always foreground.  User windows can be either.
 * Addwin() adds a window to the list at the top of one of the two groups.
 * Deletewin() deletes a window.  Front() moves a window to the front
 * of its group.  Wwopen(), wwadd(), and wwdelete() should never be called
 * directly.
 */

/*
 * Open a user window.
 */
ww_t *
openwin(winvars_t *winvars, int id, int row, int col, int nrow, int ncol,
	int nline, char *label, int type, int uflags, char *shf, char **sh)
{
	ww_t *w;

	if (id < 0 && (id = findid(winvars)) < 0)
		return 0;
	if (row + nrow <= 0 || row > winvars->wwnrow - 1 ||
	    col + ncol <= 0 || col > winvars->wwncol - 1) {
		error("Illegal window position.");
		return 0;
	}
	w = wwopen(winvars, type, 0, nrow, ncol, row, col, nline);
	if (w == NULL) {
		error("Can't open window: %s.", wwerror(winvars));
		return 0;
	}
	w->ww_id = id;
	winvars->window[id] = w;
	CLR(w->ww_uflags, WWU_ALLFLAGS);
	SET(w->ww_uflags, uflags);
	w->ww_alt = w->ww_w;
	if (label != NULL && setlabel(w, label) < 0)
		error("No memory for label.");
	wwcursor(winvars, w, 1);
	/*
	 * We have to do this little maneuver to make sure
	 * addwin() puts w at the top, so we don't waste an
	 * insert and delete operation.
	 */
	setselwin(winvars, NULL);
	addwin(w, 0);
	setselwin(winvars, w);
	if (wwspawn(winvars, w, shf, sh) < 0) {
		error("Can't execute %s: %s.", shf, wwerror(winvars));
		closewin(winvars, w);
		return NULL;
	}
	return w;
}

int
findid(winvars_t *winvars)
{
	int i;

	for (i = 0; i < MAX_NUM_WINDOWS && winvars->window[i] != 0; i++) {
	}
	if (i >= MAX_NUM_WINDOWS) {
		error("Too many windows.");
		return -1;
	}
	return i;
}

ww_t *
findselwin(winvars_t *winvars)
{
	ww_t *w, *s = 0;
	int i;

	for (i = 0; i < MAX_NUM_WINDOWS; i++)
		if ((w = winvars->window[i]) != NULL && w != winvars->selwin &&
		    (s == NULL ||
		     (!ISFG(winvars, w) && (w->ww_order < s->ww_order || ISFG(winvars, s)))))
			s = w;
	return s;
}

/*
 * Close a user window.  Close all if w == 0.
 */
void
closewin(winvars_t *winvars, ww_t *w)
{
	char didit = 0;
	int i;

	if (w == NULL) {
		for (i = 0; i < MAX_NUM_WINDOWS; i++) {
			if ((w = winvars->window[i]) != NULL) {
				closewin1(w);
				didit++;
			}
		}
	} else {
		closewin1(w);
		didit++;
	}
	if (didit) {
		if (winvars->selwin == NULL) {
			if (winvars->lastselwin != NULL) {
				setselwin(winvars, winvars->lastselwin);
				winvars->lastselwin = NULL;
			} else if ((w = findselwin(winvars)) != NULL) {
				setselwin(winvars, w);
			}
		}
		if (winvars->lastselwin == NULL && winvars->selwin) {
			if ((w = findselwin(winvars)) != NULL) {
				winvars->lastselwin = w;
			}
		}
		reframe();
	}
}

/*
 * Open an information (display) window.
 */
ww_t *
openiwin(winvars_t *winvars, int nrow, const char *label)
{
	ww_t *w;

	if ((w = wwopen(winvars, WWT_INTERNAL, 0, nrow, winvars->wwncol, 2, 0, 0)) == NULL)
		return NULL;
	SET(w->ww_wflags, WWW_MAPNL | WWW_NOINTR | WWW_NOUPDATE | WWW_UNCTRL);
	SET(w->ww_uflags, WWU_HASFRAME | WWU_CENTER);
	w->ww_id = -1;
	(void) setlabel(w, label);
	addwin(w, 1);
	reframe();
	return w;
}

/*
 * Close an information window.
 */
void
closeiwin(ww_t *w)
{
	closewin1(w);
	reframe();
}

void
closewin1(ww_t *w)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	if (w == winvars->selwin)
		winvars->selwin = NULL;
	if (w == winvars->lastselwin)
		winvars->lastselwin = NULL;
	if (w->ww_id >= 0 && w->ww_id < MAX_NUM_WINDOWS)
		winvars->window[w->ww_id] = NULL;
	if (w->ww_label)
		free(w->ww_label);
	deletewin(w);
	wwclose(winvars, w);
}

/*
 * Move the window to the top of its group.
 * Don't do it if already fully visible.
 * Wwvisible() doesn't work for tinted windows.
 * But anything to make it faster.
 * Always reframe() if doreframe is true.
 */
void
front(ww_t *w, char doreframe)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	if (w->ww_back != (ISFG(winvars, w) ? winvars->framewin : winvars->fgwin) && !wwvisible(winvars, w)) {
		deletewin(w);
		addwin(w, ISFG(winvars, w));
		doreframe = 1;
	}
	if (doreframe)
		reframe();
}

/*
 * Add a window at the top of normal windows or foreground windows.
 * For normal windows, we put it behind the current window.
 */
void
addwin(ww_t *w, char fg)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	if (fg) {
		wwadd(winvars, w, winvars->framewin);
		if (winvars->fgwin == winvars->framewin)
			winvars->fgwin = w;
	} else
		wwadd(winvars, w, winvars->selwin != 0 && winvars->selwin != w && !ISFG(winvars, winvars->selwin)
				? winvars->selwin : winvars->fgwin);
}

/*
 * Delete a window.
 */
void
deletewin(ww_t *w)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	if (winvars->fgwin == w)
		winvars->fgwin = w->ww_back;
	wwdelete(winvars, w);
}

void
reframe(void)
{
	ww_t *w;
	winvars_t	*winvars;

	winvars = get_winvars();
	wwunframe(winvars, winvars->framewin);
	for (w = winvars->wwhead.ww_back; w != &winvars->wwhead; w = w->ww_back)
		if (ISSET(w->ww_uflags, WWU_HASFRAME)) {
			wwframe(winvars, w, winvars->framewin);
			labelwin(winvars, w);
		}
}

void
stopwin(ww_t *w)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	if (w->ww_pty >= 0 && w->ww_type == WWT_PTY && wwstoptty(winvars, w->ww_pty) < 0)
		error("Can't stop output: %s.", wwerror(winvars));
	else
		SET(w->ww_pflags, WWP_STOPPED);
}

void
startwin(ww_t *w)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	if (w->ww_pty >= 0 && w->ww_type == WWT_PTY &&
	    wwstarttty(winvars, w->ww_pty) < 0)
		error("Can't start output: %s.", wwerror(winvars));
	else
		CLR(w->ww_pflags, WWP_STOPPED);
}

void
sizewin(ww_t *w, int nrow, int ncol)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	ww_t *back = w->ww_back;

	w->ww_alt.nr = w->ww_w.nr;
	w->ww_alt.nc = w->ww_w.nc;
	wwdelete(winvars, w);
	if (wwsize(winvars, w, nrow, ncol) < 0)
		error("Can't resize window: %s.", wwerror(winvars));
	wwadd(winvars, w, back);
	reframe();
}

void
waitnl(ww_t *w)
{
	(void) waitnl1(w, "[Type any key to continue]");
}

int
more(ww_t *w, char always)
{
	int c;
	int uc = ISSET(w->ww_wflags, WWW_UNCTRL);
	winvars_t	*winvars;

	winvars = get_winvars();
	if (!always && w->ww_cur.r < w->ww_w.b - 2)
		return 0;
	c = waitnl1(w, "[Type escape to abort, any other key to continue]");
	CLR(w->ww_wflags, WWW_UNCTRL);
	wwputs(winvars, "\033E", w);
	SET(w->ww_wflags, uc);
	return c == CONTROL('[') ? 2 : 1;
}

int
waitnl1(ww_t *w, const char *prompt)
{
	int uc = ISSET(w->ww_wflags, WWW_UNCTRL);
	winvars_t	*winvars;

	winvars = get_winvars();
	CLR(w->ww_wflags, WWW_UNCTRL);
	front(w, 0);
	wwprintf(w, "\033Y%c%c\033sA%s\033rA ",
		w->ww_w.nr - 1 + ' ', ' ', prompt);	/* print on last line */
	WWCURTOWIN(winvars, w);
	while (WWPEEKC(winvars) < 0)
		wwiomux(winvars);
	SET(w->ww_wflags, uc);
	return WWGETC(winvars);
}
