/*	$NetBSD: var.c,v 1.9 2003/08/07 11:17:33 agc Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#ifndef lint
__RCSID("$NetBSD: var.c,v 1.9 2003/08/07 11:17:33 agc Exp $");
#endif /* not lint */

#include <stdlib.h>
#include <string.h>
#include "value.h"
#include "var.h"

static variable_t	*var_head;
static variable_t	*alias_head;


variable_t **
get_var_list(const char *type)
{
	if (strcmp(type, "var") == 0) {
		return &var_head;
	}
	if (strcmp(type, "alias") == 0) {
		return &alias_head;
	}
	return NULL;
}

variable_t *
var_set1(variable_t **head, const char *name, value_t *v)
{
	variable_t **p;
	variable_t *r;
	value_t val;

	/* do this first, easier to recover */
	val = *v;
	if (val.v_type == V_STR && val.v_str != 0 &&
	    (val.v_str = strdup(val.v_str)) == 0)
		return 0;
	if (*(p = var_lookup1(head, name)) == 0) {
		r = (variable_t *) malloc(sizeof (variable_t));
		if (r == 0) {
			VAL_FREE(val);
			return 0;
		}
		if ((r->r_name = strdup(name)) == 0) {
			VAL_FREE(val);
			free(r);
			return 0;
		}
		r->r_left = r->r_right = 0;
		*p = r;
	} else {
		r = *p;
		VAL_FREE(r->r_val);
	}
	r->r_val = val;
	return r;
}

variable_t *
var_setstr1(variable_t **head, const char *name, char *str)
{
	value_t v;

	v.v_type = V_STR;
	v.v_str = str;
	return var_set1(head, name, &v);
}

variable_t *
var_setnum1(variable_t **head, const char *name, int num)
{
	value_t v;

	v.v_type = V_NUM;
	v.v_num = num;
	return var_set1(head, name, &v);
}

int
var_unset1(variable_t **head, char *name)
{
	variable_t **p;
	variable_t *r;

	if (*(p = var_lookup1(head, name)) == 0)
		return -1;
	r = *p;
	*p = r->r_left;
	while (*p != 0)
		p = &(*p)->r_right;
	*p = r->r_right;
	VAL_FREE(r->r_val);
	free(r->r_name);
	free(r);
	return 0;
}

variable_t **
var_lookup1(variable_t **p, const char *name)
{
	int cmp;

	while (*p != 0) {
		if ((cmp = strcmp(name, (*p)->r_name)) < 0)
			p = &(*p)->r_left;
		else if (cmp > 0)
			p = &(*p)->r_right;
		else
			break;
	}
	return p;
}

int
var_walk1(variable_t *r, int (*func) (void *, variable_t *), void *a)
{
	if (r == 0)
		return 0;
	if (var_walk1(r->r_left, func, a) < 0 || (*func)(a, r) < 0
	    || var_walk1(r->r_right, func, a) < 0)
		return -1;
	return 0;
}
