/* $NetBSD$ */

/*
 * Copyright  2005 Alistair Crooks.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"

#ifdef HAVE_SYS_CDEFS_H
#include <sys/cdefs.h>
#endif

#ifndef lint
__COPYRIGHT("@(#) Copyright  2005 \
	        Alistair Crooks.  All rights reserved.");
__RCSID("$NetBSD$");
#endif

#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#ifdef HAVE_ERR_H
#include <err.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <unistd.h>

#include "defs.h"
#include "aa.h"

/* put all the files from the file `filelist' into an array of files */
static int
expandfilelist(char *filelist, strv_t *newargs)
{
	FILE	*fp;
	char	 buf[MAXPATHLEN];

	if ((fp = fopen(filelist, "r")) == NULL) {
		warn("can't open \"%s\"", filelist);
		return 0;
	}
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		ALLOC(char *, newargs->v, newargs->size, newargs->c, 30, 30, "expandfilelist", exit(EXIT_FAILURE));
		newargs->v[newargs->c++] = strnsave(buf, strlen(buf) - 1);
	}
	(void) fclose(fp);
	return 1;
}

/* work out what max size they want */
static int64_t
getsize(const char *s)
{
	int64_t	 ret;
	char	*to;

	if ((ret = strtoull(s, &to, 10)) <= 0) {
		return ARCHANGEL_DEFAULT_MAXFSIZE;
	}
	switch(*to) {
	case 'm':
	case 'M':
		ret = MB(ret);
		break;
	case 'k':
	case 'K':
		ret = KB(ret);
	}
	if (ret < ARCHANGEL_LOW_MAXFSIZE || ret > ARCHANGEL_HIGH_MAXFSIZE) {
		ret = ARCHANGEL_DEFAULT_MAXFSIZE;
	}
	return ret;
}


int
main(int argc, char **argv)
{
	int64_t	 maxfsize;
	strv_t	 newargs;
	char	 cwd[MAXPATHLEN];
	char	*filelist;
	char	*logfile;
	char	*name;
	char	*dir;
	char	*cp;
	aa_t	 aa;
	int	 verbose;
	int	 action;
	int	 ex;
	int	 i;

	archangel_init(&aa);
	aa.ztype = ARCHANGEL_BZLIB;
	aa.protection = ARCHANGEL_CLEAR;
	setlocale(LC_ALL, "");
	dir = filelist = NULL;
	action = 0;
	name = NULL;
	logfile = NULL;
	ex = EXIT_SUCCESS;
	maxfsize = ARCHANGEL_DEFAULT_MAXFSIZE;
	if (*argv[1] != '-') {
		i = strlen(argv[1]) + 1 + 1;
		NEWARRAY(char, cp, i, "main", exit(EXIT_FAILURE));
		cp[0] = '-';
		(void) strlcpy(cp + 1, argv[1], i - 1);
		argv[1] = cp;
	}
	while ((i = getopt(argc, argv, "AC:L:S:T:X:cf:o:s:tvx")) != -1) {
		switch(i) {
		case 'A':
			action = ARCHANGEL_APPEND;
			break;
		case 'C':
			dir = optarg;
			break;
		case 'L':
			logfile = optarg;
			break;
		case 'S':
			maxfsize = getsize(optarg);
			break;
		case 'T':
			filelist = optarg;
			break;
		case 'X':
			archangel_add_pattern(&aa, "exclude", optarg);
			break;
		case 'c':
			action = ARCHANGEL_ADD;
			break;
		case 'f':
			name = optarg;
			break;
		case 'o':
			if (strcmp(optarg, "signed") == 0) {
				aa.protection |= ARCHANGEL_SIGNED;
			} else if (strcmp(optarg, "encrypted") == 0) {
				aa.protection |= ARCHANGEL_ENCRYPTED;
			} else {
				warnx("unrecognised option \"%s\"", optarg);
			}
			break;
		case 's':
			archangel_add_pattern(&aa, "transform", optarg);
			break;
		case 't':
			action = ARCHANGEL_LIST;
			break;
		case 'v':
			verbose += 1;
			break;
		case 'x':
			action = ARCHANGEL_EXTRACT;
			break;
		default:
			errx(EXIT_FAILURE, "Unknown flag \"%c\"", i);
		}
	}
	if (!archangel_open(&aa, name, action, maxfsize, verbose)) {
		errx(EXIT_FAILURE, "can't open \"%s\" archive", name);
	}
	if (dir != NULL && chdir(dir) < 0) {
		warn("can't chdir to \"%s\"", dir);
	}
	if (filelist != NULL) {
		(void) getcwd(cwd, sizeof(cwd));
		(void) memset(&newargs, 0x0, sizeof(newargs));
		expandfilelist(filelist, &newargs);
		argc = newargs.c;
		argv = newargs.v;
		optind = 0;
		if (chdir(cwd) < 0) {
			warn("can't chdir to \"%s\"", cwd);
		}
	}
	switch(action) {
	case ARCHANGEL_ADD:
	case ARCHANGEL_APPEND:
		if (logfile && (aa.logfile = fopen(logfile, "a")) == NULL) {
			err(EXIT_FAILURE, "can't open logfile \"%s\"", logfile);
		}
		if (!archangel_add(&aa, argc - optind, argv + optind)) {
			ex = EXIT_FAILURE;
		}
		break;
	case ARCHANGEL_LIST:
		archangel_list(&aa, argc - optind, argv + optind);
		break;
	case ARCHANGEL_EXTRACT:
		if (!archangel_extract(&aa, argc - optind, argv + optind)) {
			ex = EXIT_FAILURE;
		}
		break;
	default:
		errx(EXIT_FAILURE, "Unknown action to perform\n");
	}
	archangel_close(&aa);
	exit(ex);
}
