/* $NetBSD$ */

/*
 * Copyright  2005 Alistair Crooks.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef AA_H_
#define AA_H_

#define KB(x)	((x) * 1024)
#define MB(x)	((x) * 1024 * 1024)

/* actions */
enum {
	ARCHANGEL_ADD = 1,
	ARCHANGEL_APPEND,
	ARCHANGEL_DELETE,
	ARCHANGEL_EXTRACT,
	ARCHANGEL_LIST
};

/* compression styles */
enum {
	ARCHANGEL_LITERAL = 'L',
	ARCHANGEL_ZLIB = 'z',
	ARCHANGEL_BZLIB = 'b'
};

/* protection types */
enum {
	ARCHANGEL_CLEAR = 0,
	ARCHANGEL_SIGNED = 0x01,
	ARCHANGEL_ENCRYPTED = 0x02
};

/* size of arrays for user names */
enum {
	ARCHANGEL_USER_SIZE = 32,
	ARCHANGEL_GROUP_SIZE = 32
};

/* magic number for entries */
enum {
	ARCHANGEL_MAGIC1 = 0x280459ac,
	ARCHANGEL_MAGIC2 = 0xac010495,
};

enum {
	ARCHANGEL_FILE = 'f',
	ARCHANGEL_DIR = 'd',
	ARCHANGEL_BLOCK_SPECIAL = 'b',
	ARCHANGEL_CHAR_SPECIAL = 'c',
	ARCHANGEL_SYMLINK = 'l',
	ARCHANGEL_SPARSE_FILE = ' '
};

enum {
	ARCHANGEL_VERSION = 0x20050610
};

enum {
	ARCHANGEL_ID_LEN = 40,
	ARCHANGEL_PHRASE_LEN = 128
};

enum {
	ARCHANGEL_DEFAULT_MAXFSIZE = MB(1),
	ARCHANGEL_LOW_MAXFSIZE = KB(1),
	ARCHANGEL_HIGH_MAXFSIZE = MB(256),
	ARCHANGEL_MAXPATHLEN = 1024
};

DEFINE_ARRAY(strv_t, char *);

/* this struct defines an archangel subentry */
typedef struct aasubent_t {
	int64_t		 size;		/* size of uncompressed entry in bytes */
	int64_t		 ino;		/* its inode number */
	int32_t		 mode;		/* mode of entry */
	int32_t		 dev;		/* device of entry */
	int32_t		 nlink;		/* number of links */
	char		 user[ARCHANGEL_USER_SIZE];	/* user's name */
	char		 group[ARCHANGEL_GROUP_SIZE];	/* his primary group */
	int64_t		 mtime;		/* last modification time */
	int64_t		 atime;		/* last access time */
	int64_t		 ctime;		/* last inode changed time */
	int64_t		 birth;		/* birth time */
	int64_t		 blocks;	/* number of blocks used */
	int64_t		 zsize;		/* compressed size */
	uint64_t	 padding;	/* number of bytes of extra padding */
	int32_t		 flags;		/* any flags */
	int32_t		 gen;		/* generation number */
	int8_t		 compression;	/* compression type used */
	int8_t		 type;		/* type of entry */
	uint8_t		 lastpart;	/* last part */
	int8_t		 pad;		/* padding for 32-bit alignment */
	uint64_t	 offset;	/* offset from start of entry */
	uint32_t	 part;		/* part # */
	uint32_t	 namelen;	/* length of name */
	char		*name;		/* in-memory pointer to name */
	char		*value;		/* in-memory pointer to contents */
	uint32_t	 xattrlen;	/* length of extended attributes */
	char		*xattr;		/* in-memory pointer to extended attributes */
} aasubent_t;

/* this struct defines an archangel entry */
typedef struct aaent_t {
	int32_t		 magic1;	/* magic number */
	int64_t		 next;		/* offset to next entry */
	int8_t		 protection;	/* type of protection used */
	int8_t		 pad[3];	/* padding for 32-bit alignment */
	aasubent_t	 sub;		/* the subentry for this entry */
	int32_t		 magic2;	/* magic number #2 */
} aaent_t;

/* this struct defines an archangel archive (in memory) */
typedef struct aa_t {
	int32_t	 	 action;	/* action to be performed */
	int8_t	 	 protection;	/* type of protection used */
	int8_t	 	 verbose;	/* verbose output */
	int8_t	 	 ztype;		/* type of compression */
	int32_t		 maxpadding;	/* maximum size of padding (o means no padding) */
	int64_t	 	 off;		/* current file offset */
	strv_t	 	 excludes;	/* patterns to exclude */
	strv_t	 	 xformfrom;	/* patterns to transform from */
	strv_t	 	 xformto;	/* patterns to transform to */
	int32_t	 	 fd;		/* in-memory file descriptor */
	int64_t		 dev;		/* archive's dev */
	int64_t		 inode;		/* archive's inode */
	int64_t		 maxfsize;	/* maximum file size */
	char		*name;		/* archive name */
	void		*cxform;	/* compiled xform regexps */
	char		 id[ARCHANGEL_ID_LEN];	/* associated id */
	char		 phrase[ARCHANGEL_PHRASE_LEN];	/* and something else related to it */
	FILE		*logfile;	/* where to log information */
} aa_t;

/* our own endian functions */
uint16_t archangel_htobe16(uint16_t);
uint16_t archangel_be16toh(uint16_t);
uint32_t archangel_htobe32(uint32_t);
uint32_t archangel_be32toh(uint32_t);
uint64_t archangel_htobe64(uint64_t);
uint64_t archangel_be64toh(uint64_t);

char *strnsave(const char *, ssize_t);

void archangel_init(aa_t *);

int archangel_open(aa_t *, char *, int, uint64_t, int);
void archangel_close(aa_t *);

int archangel_add_pattern(aa_t *, const char *, char *);

int archangel_add_entry(aa_t *, char *, size_t);
int archangel_add(aa_t *, int, char **);

int archangel_list_entry(aa_t *, aaent_t *, int, char *);
int archangel_list(aa_t *, int, char **);

int archangel_extract_entry(aa_t *, aaent_t *, int, char *);
int archangel_extract(aa_t *, int, char **);

#endif /* !AA_H_ */
