/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ctype.h>
#include <string.h>

#include "paneld.h"

static int	text_open(struct menu *m);
static int	text_close(struct menu *m);
static int	text_next(struct menu *m);
static int	text_prev(struct menu *m);
static int	text_inc(struct menu *m);
static int	text_dec(struct menu *m);
static int	text_select(struct menu *m);
static char*	text_allocbuf(int *sz);

static int
text_open(struct menu *m)
{
	int len;
	if (exec_read(m) != E_OK)
		return E_IO;
	len = strlen(m->io.buf);
	if (len < m->io.len) {
		memset(m->io.buf + len, ' ', m->io.len - len);
		m->io.buf[m->io.len] = 0;
	}

	return E_OK;
}

static int
text_close(struct menu *m)
{
	char *ptr = m->io.buf + m->io.len - 1;
	while (*ptr == ' ')
		ptr--;
	*++ptr = 0;
	return exec_write(m);
}

static int
text_next(struct menu *m)
{
	return window->edit()->right();
}

static int
text_prev(struct menu *m)
{
	return window->edit()->left();
}

static int
text_inc(struct menu *m)
{
	int c;
	if (window->edit()->getchr(&c) == E_OK) {
		if (c > 125)
			return window->edit()->putchr(' ');
		else
			return window->edit()->putchr(++c);
	}
	return E_IO;
}

static int
text_dec(struct menu *m)
{
	int c;
	if (window->edit()->getchr(&c) == E_OK) {
		if (c == ' ')
			return window->edit()->putchr(126);
		else
			return window->edit()->putchr(--c);
	}
	return E_IO;
}

static int
text_select(struct menu *m)
{
	if (text_open(m) == E_OK) {
		window->set_title(m->nm);
		window->edit()->setbuf(m->io.buf, ALIGN_ANY);

		if (enter_menu(m) == E_SELECT)
			return text_close(m);
	}

	return E_IO;
}

static char*
text_allocbuf(int *sz)
{
	return (char*)cf_malloc(*sz + 1);
}

struct menu_ops*
get_text_ops(void)
{
	static struct menu_ops text_ops = {
		text_next,
		text_prev,
		text_inc,
		text_dec,
		text_select,
		text_allocbuf
	};
	return &text_ops;
}
