/*	$NetBSD$	*/

/*
 * Copyright (c) 1997 Michael L. Hitch
 * Copyright (c) 1995 Daniel Widenfalk
 * Copyright (c) 1994 Christian E. Hopps
 * Copyright (c) 1982, 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Daniel Widenfalk
 *	and Michael L. Hitch.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*-
 * Copyright (c) 1998 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This software contains code contributed to The NetBSD Foundation
 * by Mark Brinicombe.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the NetBSD
 *	Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SCSI driver for the Connect32 SCSI II interface
 *
 *	from: amiga/dev/flsc.c,v 1.21 1998/07/04 22:18:16
 */

#define COSC_POLL

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/device.h>
#include <sys/buf.h>
#include <sys/proc.h>
#include <sys/user.h>
#include <sys/queue.h>

#include <dev/scsipi/scsi_all.h>
#include <dev/scsipi/scsipi_all.h>
#include <dev/scsipi/scsiconf.h>
#include <dev/scsipi/scsi_message.h>

#include <machine/bootconfig.h>
#include <machine/cpu.h>
#include <machine/param.h>
#include <machine/io.h>

#include <dev/ic/ncr53c9xreg.h>
/*#include <dev/ic/ncr53c9xvar.h>*/
#include <arm32/dev/ncr53c9xvar.h>

#include <arm32/podulebus/podulebus.h>
#include <arm32/podulebus/podules.h>
#include <arm32/podulebus/coscvar.h>

void	coscattach	__P((struct device *, struct device *, void *));
int	coscmatch	__P((struct device *, struct cfdata *, void *));
int	cosc_scsi_cmd	__P((struct scsipi_xfer *xs));

/* Linkup to the rest of the kernel */
struct cfattach cosc_ca = {
	sizeof(struct cosc_softc), coscmatch, coscattach
};

struct scsipi_adapter cosc_switch = {
#ifdef COSC_DEBUG
	cosc_scsi_cmd,
#else
	ncr53c9x_scsi_cmd,
#endif
	minphys,		/* no max at this level; handled by DMA code */
	NULL,
	NULL,
};

struct scsipi_device cosc_dev = {
	NULL,			/* Use default error handler */
	NULL,			/* have a queue, served by this */
	NULL,			/* have no async handler */
	NULL,			/* Use default 'done' routine */
};

/*
 * Functions and the switch for the MI code.
 */
u_char	cosc_read_reg __P((struct ncr53c9x_softc *, int));
void	cosc_write_reg __P((struct ncr53c9x_softc *, int, u_char));
int	cosc_dma_isintr __P((struct ncr53c9x_softc *));
void	cosc_dma_reset __P((struct ncr53c9x_softc *));
int	cosc_dma_intr __P((struct ncr53c9x_softc *));
int	cosc_dma_setup __P((struct ncr53c9x_softc *, caddr_t *,
	    size_t *, int, size_t *));
void	cosc_dma_go __P((struct ncr53c9x_softc *));
void	cosc_dma_stop __P((struct ncr53c9x_softc *));
int	cosc_dma_isactive __P((struct ncr53c9x_softc *));
void	cosc_clear_latched_intr __P((struct ncr53c9x_softc *));
int	cosc_intr __P((void *arg));

struct ncr53c9x_glue cosc_glue = {
	cosc_read_reg,
	cosc_write_reg,
	cosc_dma_isintr,
	cosc_dma_reset,
	cosc_dma_intr,
	cosc_dma_setup,
	cosc_dma_go,
	cosc_dma_stop,
	cosc_dma_isactive,
	NULL,
};

extern int ncr53c9x_debug;

/*
 * if we are a Connect32 card
 */
int
coscmatch(parent, cf, aux)
	struct device	*parent;
	struct cfdata	*cf;
	void		*aux;
{
	struct podule_attach_args *pa = (struct podule_attach_args *)aux;

	/* Look for the card */
	if (matchpodule(pa, MANUFACTURER_MCS, PODULE_MCS_SCSI, -1) != 0)
		return(1);

	/* Old versions of the ROM on this card could have the wrong ID */
	if (matchpodule(pa, MANUFACTURER_ACORN, PODULE_ACORN_SCSI, -1) == 0)
		return(0);
	if (strncmp(pa->pa_podule->description, "MCS", 3) != 0)
		return(0);

	return(1);
}

/*
 * Attach this instance, and then all the sub-devices
 */
void
coscattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct podule_attach_args *pa = (struct podule_attach_args *)aux;
	struct cosc_softc *cosc = (void *)self;
	struct ncr53c9x_softc *sc = &cosc->sc_ncr53c9x;
	volatile u_char *iobase;

	if (pa->pa_podule_number == -1)
		panic("Podule has disappeared !");

	if (pa->pa_podule->manufacturer == MANUFACTURER_ACORN
	    && pa->pa_podule->product == PODULE_ACORN_SCSI)
		printf(" Faulty expansion card identity\n");

	/*
	 * Set up the glue for MI code early; we use some of it here.
	 */
	sc->sc_glue = &cosc_glue;

	/*
	 * Set up the register pointers
	 */
	iobase = (volatile u_char *)pa->pa_podule->fast_base;
	cosc->sc_reg = iobase + COSC_ESCOFFSET_BASE;
	cosc->sc_term = iobase + COSC_TERMINATION_CONTROL;

	sc->sc_freq = 40;		/* Clocked at 40Mhz */

	/* Select page zero (so we can see the config info) */
	iobase[COSC_PAGE_REGISTER] = 0;

	sc->sc_id = 7;
	/* Provide an override for the host id */
	(void)get_bootconf_option(boot_args, "cosc.hostid",
	    BOOTOPT_TYPE_INT, &sc->sc_id);

	/* Set termination power */
	if (iobase[COSC_CONFIG_TERMINATION] & COSC_CONFIG_TERMINATION_ON) {
		printf(" termpwr on");
		*cosc->sc_term = COSC_TERMINATION_ON;
	} else {
		printf(" termpwr off");
		*cosc->sc_term = COSC_TERMINATION_OFF;
	}

	/*
	 * It is necessary to try to load the 2nd config register here,
	 * to find out what rev the flsc chip is, else the flsc_reset
	 * will not set up the defaults correctly.
	 */
	sc->sc_cfg1 = sc->sc_id | NCRCFG1_PARENB;
	sc->sc_cfg2 = NCRCFG2_SCSI2 | NCRCFG2_FE;
	sc->sc_cfg3 = 0x08 /*FCLK*/ | NCRESPCFG3_FSCSI | NCRESPCFG3_CDB;
	sc->sc_rev = NCR_VARIANT_NCR53C94;

	/*
	 * This is the value used to start sync negotiations
	 * Note that the NCR register "SYNCTP" is programmed
	 * in "clocks per byte", and has a minimum value of 4.
	 * The SCSI period used in negotiation is one-fourth
	 * of the time (in nanoseconds) needed to transfer one byte.
	 * Since the chip's clock is given in MHz, we have the following
	 * formula: 4 * period = (1000 / freq) * 4
	 */
/*	sc->sc_minsync = 1000 / sc->sc_freq;*/
	sc->sc_minsync = 0;

	/* Really no limit, but since we want to fit into the TCR... */
	sc->sc_maxxfer = 64 * 1024;

/*	ncr53c9x_debug |= 0xffff;*/

#ifndef COSC_POLL
	/*
	 * Configure interrupts.
	 */
#ifdef COSC_DEBUG
	cosc->sc_ih.ih_func = (void *)cosc_intr;
#else
	cosc->sc_ih.ih_func = (void *)ncr53c9x_intr;
#endif
	cosc->sc_ih.ih_arg  = sc;
	cosc->sc_ih.ih_level = IPL_BIO;
	cosc->sc_ih.ih_name = "cosc";
	cosc->sc_ih.ih_maskaddr = pa->pa_podule->irq_addr;
	cosc->sc_ih.ih_maskbits = pa->pa_podule->irq_mask;

	if (irq_claim(IRQ_PODULE, &cosc->sc_ih))
		panic("%s: Cannot install IRQ handler\n", self->dv_xname);

	/* Enable interrupts */
/*	*cosc->sc_inten = COSC_INTERRUPTS_ON;*/
	sc->sc_features = 0;

#else
	/* Disable interrupts and flag no-interrupt mode */
/*	*cosc->sc_inten = COSC_INTERRUPTS_OFF;*/
	sc->sc_features = NCR_F_NOINTR;
	printf(" polling");
#endif
	/* Turn LED off */
/*	*cosc->sc_led = cosc_LED_OFF;*/

	/*
	 * Now try to attach all the sub-devices
	 */
	ncr53c9x_attach(sc, &cosc_switch, &cosc_dev);
}

/*
 * Glue functions.
 */

u_char
cosc_read_reg(sc, reg)
	struct ncr53c9x_softc *sc;
	int reg;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;

	return cosc->sc_reg[reg * COSC_FAS_REG_SPACING];
}

void
cosc_write_reg(sc, reg, val)
	struct ncr53c9x_softc *sc;
	int reg;
	u_char val;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;
	struct ncr53c9x_tinfo *ti;
	u_char v = val;

	if (cosc->sc_piomode && reg == NCR_CMD &&
	    v == (NCRCMD_TRANS|NCRCMD_DMA)) {
		v = NCRCMD_TRANS;
	}
	/*
	 * Can't do synchronous transfers in SCSI_POLL mode:
	 * If starting SCSI_POLL command, clear defer sync negotiation
	 * by clearing the T_NEGOTIATE flag.  If starting SCSI_POLL and
	 * the device is currently running synchronous, force another
	 * T_NEGOTIATE with 0 offset.
	 */
	if (reg == NCR_SELID) {
		ti = &sc->sc_tinfo[
		    sc->sc_nexus->xs->sc_link->scsipi_scsi.target];
		if (sc->sc_nexus->xs->flags & SCSI_POLL) {
			if (ti->flags & T_SYNCMODE) {
				ti->flags ^= T_SYNCMODE | T_NEGOTIATE;
			} else if (ti->flags & T_NEGOTIATE) {
				ti->flags ^= T_NEGOTIATE | T_SYNCHOFF;
				/* save T_NEGOTIATE in private flags? */
			}
		} else {
			/*
			 * If we haven't attempted sync negotiation yet,
			 * do it now.
			 */
			if ((ti->flags & (T_SYNCMODE | T_SYNCHOFF)) ==
			    T_SYNCHOFF &&
			    sc->sc_minsync != 0)	/* XXX */
				ti->flags ^= T_NEGOTIATE | T_SYNCHOFF;
		}
	}
	if (reg == NCR_CMD && v == NCRCMD_SETATN  &&
	    sc->sc_flags & NCR_SYNCHNEGO &&
	     sc->sc_nexus->xs->flags & SCSI_POLL) {
		ti = &sc->sc_tinfo[
		    sc->sc_nexus->xs->sc_link->scsipi_scsi.target];
		ti->offset = 0;
	}
	cosc->sc_reg[reg * COSC_FAS_REG_SPACING] = v;
}

int
cosc_dma_isintr(sc)
	struct ncr53c9x_softc *sc;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;

/*	if (sc->sc_state == NCR_CONNECTED || sc->sc_state == NCR_SELECTING)
		*cosc->sc_led = cosc_LED_ON;
	else
		*cosc->sc_led = cosc_LED_OFF;*/

	if (cosc->sc_reg[NCR_STAT * COSC_FAS_REG_SPACING] & NCRSTAT_INT)
		return 1;

	return 0;
}

#if 0
void
cosc_clear_latched_intr(sc)
	struct ncr53c9x_softc *sc;
{
/*	struct cosc_softc *cosc = (struct cosc_softc *)sc;*/

	/* Interrupt cleared via ncr53c9x */
}
#endif

void
cosc_dma_reset(sc)
	struct ncr53c9x_softc *sc;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;
	struct ncr53c9x_tinfo *ti;

	if (sc->sc_nexus)
		ti = &sc->sc_tinfo[sc->sc_nexus->xs->sc_link->scsipi_scsi.target];
	else
		ti = &sc->sc_tinfo[1];	/* XXX */
	if (cosc->sc_active) {
		printf("dmaaddr %p dmasize %d stat %x flags %x off %d per %d ff %x",
	            *cosc->sc_dmaaddr, cosc->sc_dmasize,
		    cosc->sc_reg[NCR_STAT * COSC_FAS_REG_SPACING], ti->flags, ti->offset,
		    ti->period, cosc->sc_reg[NCR_FFLAG * COSC_FAS_REG_SPACING]);
		printf(" intr %x\n", cosc->sc_reg[NCR_INTR * COSC_FAS_REG_SPACING]);
	}
	cosc->sc_active = 0;
	cosc->sc_piomode = 0;
}

int
cosc_dma_intr(sc)
	struct ncr53c9x_softc *sc;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;
	u_char	*p;
	volatile u_char *cmdreg, *intrreg, *statreg, *fiforeg;
	u_int	coscphase, coscstat, coscintr;
	int	cnt;

	NCR_DMA(("cosc_dma_intr: pio %d cnt %d int %x stat %x fifo %d ",
	    cosc->sc_piomode, cosc->sc_dmasize, sc->sc_espintr, sc->sc_espstat,
	    cosc->sc_reg[NCR_FFLAG * COSC_FAS_REG_SPACING] & NCRFIFO_FF));
	if (cosc->sc_active == 0) {
		printf("cosc_intr--inactive DMA\n");
		return -1;
	}

	/* if DMA transfer, update sc_dmaaddr and sc_pdmalen, else PIO xfer */
	if (cosc->sc_piomode == 0) {
		panic("cosc_dma_intr");
	}

	if ((sc->sc_espintr & NCRINTR_BS) == 0) {
		cosc->sc_active = 0;
		cosc->sc_piomode = 0;
		NCR_DMA(("no NCRINTR_BS\n"));
		return 0;
	}

	cnt = cosc->sc_dmasize;
#if 0
	if (cnt == 0) {
		printf("data interrupt, but no count left.");
	}
#endif

	p = *cosc->sc_dmaaddr;
	coscphase = sc->sc_phase;
	coscstat = (u_int) sc->sc_espstat;
	coscintr = (u_int) sc->sc_espintr;
	cmdreg = cosc->sc_reg + NCR_CMD * COSC_FAS_REG_SPACING;
	fiforeg = cosc->sc_reg + NCR_FIFO * COSC_FAS_REG_SPACING;
	statreg = cosc->sc_reg + NCR_STAT * COSC_FAS_REG_SPACING;
	intrreg = cosc->sc_reg + NCR_INTR * COSC_FAS_REG_SPACING;
	NCR_DMA(("PIO %d datain %d phase %d stat %x intr %x\n",
	    cnt, cosc->sc_datain, coscphase, coscstat, coscintr));
	do {
		if (cosc->sc_datain) {
			*p++ = *fiforeg;
			cnt--;
			if (coscphase == DATA_IN_PHASE) {
				*cmdreg = NCRCMD_TRANS;
			} else {
				cosc->sc_active = 0;
			}
	 	} else {
			NCR_DMA(("cosc_dma_intr: PIO out- phase %d cnt %d active %d\n",
			    coscphase, cnt, cosc->sc_active));
			if (   (coscphase == DATA_OUT_PHASE)
			    || (coscphase == MESSAGE_OUT_PHASE)) {
				int n;
				n = 16 - (cosc->sc_reg[NCR_FFLAG * COSC_FAS_REG_SPACING] & NCRFIFO_FF);
				if (n > cnt)
					n = cnt;
				cnt -= n;
				while (n-- > 0)
					*fiforeg = *p++;
				*cmdreg = NCRCMD_TRANS;
			} else {
				cosc->sc_active = 0;
			}
		}

		if (cosc->sc_active && cnt) {
			while (!(*statreg & 0x80));
			coscstat = *statreg;
			coscintr = *intrreg;
			coscphase = (coscintr & NCRINTR_DIS)
				    ? /* Disconnected */ BUSFREE_PHASE
				    : coscstat & PHASE_MASK;
		}
	} while (cnt && cosc->sc_active && (coscintr & NCRINTR_BS));
#if 1
	if (cosc->sc_dmasize < 8 && cnt)
		printf("cosc_dma_intr: short transfer: dmasize %d cnt %d\n",
		    cosc->sc_dmasize, cnt);
#endif
	NCR_DMA(("cosc_dma_intr: PIO transfer [%d], %d->%d phase %d stat %x intr %x\n",
	    *cosc->sc_pdmalen, cosc->sc_dmasize, cnt, coscphase, coscstat, coscintr));
	sc->sc_phase = coscphase;
	sc->sc_espstat = (u_char) coscstat;
	sc->sc_espintr = (u_char) coscintr;
	*cosc->sc_dmaaddr = p;
	*cosc->sc_pdmalen -= cosc->sc_dmasize - cnt;
	cosc->sc_dmasize = cnt;

	if (*cosc->sc_pdmalen == 0) {
		sc->sc_espstat |= NCRSTAT_TC;
		cosc->sc_piomode = 0;
	}
	return 0;
}

int
cosc_dma_setup(sc, addr, len, datain, dmasize)
	struct ncr53c9x_softc *sc;
	caddr_t *addr;
	size_t *len;
	int datain;
	size_t *dmasize;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;

	cosc->sc_dmaaddr = addr;
	cosc->sc_pdmalen = len;
	cosc->sc_datain = datain;
	cosc->sc_dmasize = *dmasize;

	cosc->sc_piomode = 1;
	/* IF pio or POLL */
	if (cosc->sc_piomode) {
		*dmasize = cosc->sc_dmasize;
		NCR_DMA(("cosc_dma_setup: PIO %p/%d [%d]\n", *addr,
		    cosc->sc_dmasize, *len));
		if (datain == 0) {
			int n;
			n = cosc->sc_dmasize;
			if (n > 16)
				n = 16;
			while (n-- > 0) {
				cosc->sc_reg[NCR_FIFO * COSC_FAS_REG_SPACING] = **cosc->sc_dmaaddr;
				(*cosc->sc_pdmalen)--;
				(*cosc->sc_dmaaddr)++;
				--cosc->sc_dmasize;
			}
		}
		return 0;
	}
	/* ELSE DMA */
	panic("cosc_dma_setup");
}

void
cosc_dma_go(sc)
	struct ncr53c9x_softc *sc;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;

	NCR_DMA(("cosc_dma_go: datain %d size %d\n", cosc->sc_datain,
	    cosc->sc_dmasize));
	if (sc->sc_nexus->xs->flags & SCSI_POLL) {
		cosc->sc_active = 1;
		return;
	} else if (cosc->sc_piomode == 0) {
		/* NO DMA */
		panic("cosc_dma_go");
	} else
		cosc->sc_active = 1;
}

void
cosc_dma_stop(sc)
	struct ncr53c9x_softc *sc;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;

	cosc->sc_piomode = 0;
}

int
cosc_dma_isactive(sc)
	struct ncr53c9x_softc *sc;
{
	struct cosc_softc *cosc = (struct cosc_softc *)sc;

	return cosc->sc_active;
}

/* Only used for debugging */
#ifdef COSC_DEBUG
int
cosc_scsi_cmd(xs)
	struct scsipi_xfer *xs;
{
	struct scsipi_link *sc_link = xs->sc_link;

	if (sc_link->scsipi_scsi.lun == 0)
	printf("id=%d lun=%d cmdlen=%d datalen=%d opcode=%02x flags=%08x status=%02x blk=%02x %02x timeo=%d\n",
	    sc_link->scsipi_scsi.target, sc_link->scsipi_scsi.lun, xs->cmdlen, xs->datalen, xs->cmd->opcode,
	    xs->flags, xs->status, xs->cmd->bytes[0], xs->cmd->bytes[1],
	    xs->timeout);

	return(ncr53c9x_scsi_cmd(xs));
}

/* Only used to cleanup trace output as interrupt is shared */
int
cosc_intr(arg)
	void *arg;
{
	struct ncr53c9x_softc *sc = arg;

	if (cosc_dma_isintr(sc))
		return(ncr53c9x_intr(sc));
	return(0);
}
#endif
