#include <lib/libsa/stand.h>
#include <lib/libkern/libkern.h>

#include <sys/cdefs.h>
#include <sys/types.h>

#define MIPS3 1
#include <uvm/uvm_extern.h>
#include <mips/cpuregs.h>
#include <mips/pte.h>
extern void mips3_TLBRead __P((int, struct tlb *));
extern int mips3_GetWIRED __P((void));

#if 0
#include <arc/arc/arcbios.h>
#else
#include "arcbios.h"
#endif

#include <stdio.h>
#include "dumpconf.h"

#define ARRAY_LENGTH(array)	(sizeof(array)/sizeof(array[0]))

void dump_argv(FILE *file, int argc, char **argv);
void dump_environ(FILE *file, char **envp);
void dump_tlb_lo(FILE *file, unsigned int);
void dump_tlb(FILE *file);
void dump_all(FILE *file, int argc, char **argv, char **envp);
int parse_command(char *command, char **cmdv, int maxcmdc);
void __main(void);
int main(int argc, char **argv, char **envp);

void
dump_argv(FILE *file, int argc, char **argv)
{
	int i;

	fprintf(file, "argc = %d\n", argc);
	for (i = 0; i < argc; i++)
		fprintf(file, "argv[%d] = `%s'\n", i, argv[i]);
}

void
dump_environ(FILE *file, char **envp)
{
	while (*envp)
		fprintf(file, "[%s]\n", *envp++);
}

void
dump_tlb_lo(FILE *file, unsigned int tlb_lo)
{
	if (tlb_lo & 0xfc000000) {
		fprintf(file, "0x%x ", tlb_lo >> 26);
	} else {
		fprintf(file, "0x");
	}
	fprintf(file, "%x(%d%c%c%c)", (tlb_lo & 0x03ffffc0) << 6,
		(tlb_lo & MIPS3_TLB_ATTR_MASK) >> MIPS3_TLB_ATTR_SHIFT,
		(tlb_lo & MIPS3_TLB_DIRTY_BIT) ? 'D' : '.',
		(tlb_lo & MIPS3_TLB_VALID_BIT) ? 'V' : '.',
		(tlb_lo & MIPS3_TLB_GLOBAL_BIT) ? 'G' : '.');
}

void
dump_tlb(FILE *file)
{
	int i;

	fprintf(file, "wired TLB = %d\n", mips3_GetWIRED());
	for (i = 0; i < MIPS3_TLB_NUM_TLB_ENTRIES; i++) {
		struct tlb tlb;

		mips3_TLBRead(i, &tlb);

		fprintf(file, "%s%d: ", i < 10 ? " " : "", i);
		switch (tlb.tlb_mask & MIPS3_PG_SIZE_16M) {
		case MIPS3_PG_SIZE_4K:
			fprintf(file, "4K");
			break;
		case MIPS3_PG_SIZE_16K:
			fprintf(file, "16K");
			break;
		case MIPS3_PG_SIZE_64K:
			fprintf(file, "64K");
			break;
		case MIPS3_PG_SIZE_256K:
			fprintf(file, "256K");
			break;
		case MIPS3_PG_SIZE_1M:
			fprintf(file, "1M");
			break;
		case MIPS3_PG_SIZE_4M:
			fprintf(file, "4M");
			break;
		case MIPS3_PG_SIZE_16M:
			fprintf(file, "16M");
			break;
		default:
			fprintf(file, "unknown(0x%x)",
				tlb.tlb_mask & MIPS3_PG_SIZE_16M);
			break;
		}

		fprintf(file, ", asid: 0x%x, v: 0x%x, ",
			tlb.tlb_hi & MIPS3_PG_ASID,
			tlb.tlb_hi & MIPS3_PG_HVPN);
		if (tlb.tlb_hi & ~(MIPS3_PG_HVPN|MIPS3_PG_ASID))
			fprintf(file, ", rfuhi: 0x%x, ",
				tlb.tlb_hi & ~(MIPS3_PG_HVPN|MIPS3_PG_ASID));
		fprintf(file, "p0: ");
		dump_tlb_lo(file, tlb.tlb_lo0);
		fprintf(file, ", p1: ");
		dump_tlb_lo(file, tlb.tlb_lo1);
		fprintf(file, "\n");
	}
}

void
dump_all(FILE *file, int argc, char **argv, char **envp)
{
	fprintf(file, "Command Arguments:\n");
	dump_argv(file, argc, argv);
	fprintf(file, "\nEnvironment Variables:\n");
	dump_environ(file, envp);
	fprintf(file, "\nSystem Parameter Block:\n");
	bios_dump_param_blk(file);
	fprintf(file, "\nSystem Identifier:\n");
	bios_dump_system_id(file);
	fprintf(file, "\nSystem Configuration Tree:\n");
	bios_dump_configuration(file);
	fprintf(file, "\nMemory Layout:\n");
	bios_dump_memory(file);
	fprintf(file, "\nDisplay Status:\n");
	bios_dump_display_status(file, 1);
#if 0 /* PICA-61 sometimes misbehaves, if write-count isn't multple of 512 */
	fflush(file); /* dump_tlb() depends on R4000/R4400 */
#endif
	fprintf(file, "\nTLB:\n");
	dump_tlb(file);
}

int parse_command(char *command, char **cmdv, int maxcmdc)
{
	int cmdc = 0;

	while (*command) {
		while (*command == ' ' || *command == '\t')
			command++;
		if (*command == '\0')
			break;
		if (cmdc >= maxcmdc)
			break;
		cmdv[cmdc++] = command;
		while (*command && *command != ' ' && *command != '\t')
			command++;
		if (*command)
			*command++ = '\0';
	}
	return (cmdc);
}

void
__main()
{}


int
main(argc, argv, envp)
	int argc;
	char **argv;
	char **envp;
{
	int i;
	char buffer[100];
	int cmdc;
	char *cmdv[10];
	static char dumpfilename[80] =
		"scsi()disk(0)rdisk()partition(1)\\arcdiag.out";
	static char dumpfileext[] = ".out";

	if (!bios_magic())
		return (1);

	if (strlen(argv[0]) + sizeof(dumpfileext) <= sizeof(dumpfilename)) {
		strcpy(dumpfilename, argv[0]);
		strcat(dumpfilename, dumpfileext);
	}
	printf("default dumpfile = `%s'\n", dumpfilename);

	for (;;) {
		printf("arcdiag> ");
		if (fgets(buffer, sizeof(buffer), stdin) == NULL)
			break;
		i = strlen(buffer);
		if (i > 0 && buffer[i - 1] == '\n')
			buffer[i - 1] = '\0';
		cmdc = parse_command(buffer, cmdv, ARRAY_LENGTH(cmdv));
		if (cmdc == 0)
			continue;
		if (strcmp(cmdv[0], "exit") == 0) {
			break;
		} else if (strcmp(cmdv[0], "file") == 0) {
			if (cmdc > 1) {
				if (strlen(cmdv[1]) >= sizeof(dumpfilename))
					printf("%s: too long filename",
					       cmdv[1]);
				else
					strcpy(dumpfilename, cmdv[1]);
			}
			printf("dumpfile = `%s'\n", dumpfilename);
		} else if (strcmp(cmdv[0], "dump") == 0) {
			FILE *fp = fopen(dumpfilename, "w");
			if (fp == NULL) {
				printf("%s: error %d: %s\n", dumpfilename,
				       errno, arc_strerror(errno));
			} else {
				dump_all(fp, argc, argv, envp);
				fclose(fp);
			}
		} else if (strcmp(cmdv[0], "argv") == 0) {
			dump_argv(stdout, argc, argv);
		} else if (strcmp(cmdv[0], "printenv") == 0) {
			dump_environ(stdout, envp);
		} else if (strcmp(cmdv[0], "sysparam") == 0) {
			bios_dump_param_blk(stdout);
		} else if (strcmp(cmdv[0], "sysid") == 0) {
			bios_dump_system_id(stdout);
		} else if (strcmp(cmdv[0], "config") == 0) {
			bios_dump_configuration(stdout);
		} else if (strcmp(cmdv[0], "memory") == 0) {
			bios_dump_memory(stdout);
		} else if (strcmp(cmdv[0], "disp") == 0) {
			bios_dump_display_status(stdout, 1);
		} else if (strcmp(cmdv[0], "tlb") == 0) {
			dump_tlb(stdout);
		} else {
			printf("Usage:\n");
			printf("\texit            : exit program\n");
			printf("\tfile [filename] : set/print dumpfile\n");
			printf("\tdump            : save all info to file\n");
			printf("\targv            : print argv\n");
			printf("\tprintenv        : print environ\n");
			printf("\tsysparam        : print sys param block\n");
			printf("\tsysid           : print system id\n");
			printf("\tconfig          : print config tree\n");
			printf("\tmemory          : print memory layout\n");
			printf("\tdisp            : print display status\n");
			printf("\ttlb             : print TLB\n");
		}
	}
	return (0);
}
