/* $NetBSD: h_sha2hmac.c,v 1.1 2019/12/03 04:20:45 hikaru Exp $ */

/*-
 * Copyright (c) 2014 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>

#include <sys/ioctl.h>
#include <sys/time.h>

#include <crypto/cryptodev.h>

/* Test data from RFC4868 */
const struct {
	int num;
	int alg;
	size_t key_len;
	size_t len;
	size_t mac_len;
	unsigned char key[80];
	unsigned char data[80];
	unsigned char mac[80];
} tests[] = {
	{ 1, CRYPTO_SHA2_256_HMAC, 32, 8, 16,
	  { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b },
	  "Hi There",
	  { 0x19, 0x8a, 0x60, 0x7e, 0xb4, 0x4b, 0xfb, 0xc6,
	    0x99, 0x03, 0xa0, 0xf1, 0xcf, 0x2b, 0xbd, 0xc5 }
	},
	{ 2, CRYPTO_SHA2_256_HMAC, 32, 28, 16,
	  "JefeJefeJefeJefe"
	  "JefeJefeJefeJefe",
	  "what do ya want "
	  "for nothing?",
	  { 0x16, 0x7f, 0x92, 0x85, 0x88, 0xc5, 0xcc, 0x2e,
	    0xef, 0x8e, 0x30, 0x93, 0xca, 0xa0, 0xe8, 0x7c }
	},
	{ 3, CRYPTO_SHA2_256_HMAC, 32, 50, 16,
	  { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
	  { 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd },
          { 0xcd, 0xcb, 0x12, 0x20, 0xd1, 0xec, 0xcc, 0xea,
	    0x91, 0xe5, 0x3a, 0xba, 0x30, 0x92, 0xf9, 0x62 }
	},
	{ 4, CRYPTO_SHA2_256_HMAC, 32, 50, 16,
	  { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
	    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
	    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20 },
	  { 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd },
	  { 0x37, 0x2e, 0xfc, 0xf9, 0xb4, 0x0b, 0x35, 0xc2,
	    0x11, 0x5b, 0x13, 0x46, 0x90, 0x3d, 0x2e, 0xf4 }
	},
	{ 5, CRYPTO_SHA2_384_HMAC, 48, 8, 24,
	  { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b },
	  "Hi There",
	  { 0xb6, 0xa8, 0xd5, 0x63, 0x6f, 0x5c, 0x6a, 0x72,
	    0x24, 0xf9, 0x97, 0x7d, 0xcf, 0x7e, 0xe6, 0xc7,
	    0xfb, 0x6d, 0x0c, 0x48, 0xcb, 0xde, 0xe9, 0x73 }
	},
	{ 6, CRYPTO_SHA2_384_HMAC, 48, 28, 24,
	  "JefeJefeJefeJefe"
	  "JefeJefeJefeJefe"
	  "JefeJefeJefeJefe",
	  "what do ya want "
	  "for nothing?",
	  { 0x2c, 0x73, 0x53, 0x97, 0x4f, 0x18, 0x42, 0xfd,
	    0x66, 0xd5, 0x3c, 0x45, 0x2c, 0xa4, 0x21, 0x22,
	    0xb2, 0x8c, 0x0b, 0x59, 0x4c, 0xfb, 0x18, 0x4d }
	},
	{ 7, CRYPTO_SHA2_384_HMAC, 48, 50, 24,
	  { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
	  { 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd },
	  { 0x80, 0x9f, 0x43, 0x9b, 0xe0, 0x02, 0x74, 0x32,
	    0x1d, 0x4a, 0x53, 0x86, 0x52, 0x16, 0x4b, 0x53,
	    0x55, 0x4a, 0x50, 0x81, 0x84, 0xa0, 0xc3, 0x16 }
	},
	{ 8, CRYPTO_SHA2_384_HMAC, 48, 50, 24,
	  { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
	    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
	    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
	    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11,
	    0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19 },
	  { 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd },
	  { 0x5b, 0x54, 0x00, 0x85, 0xc6, 0xe6, 0x35, 0x80,
	    0x96, 0x53, 0x2b, 0x24, 0x93, 0x60, 0x9e, 0xd1,
	    0xcb, 0x29, 0x8f, 0x77, 0x4f, 0x87, 0xbb, 0x5c }
	},
	{ 9, CRYPTO_SHA2_512_HMAC, 64, 8, 32,
	  { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b },
	  "Hi There",
	  { 0x63, 0x7e, 0xdc, 0x6e, 0x01, 0xdc, 0xe7, 0xe6,
	    0x74, 0x2a, 0x99, 0x45, 0x1a, 0xae, 0x82, 0xdf,
	    0x23, 0xda, 0x3e, 0x92, 0x43, 0x9e, 0x59, 0x0e,
	    0x43, 0xe7, 0x61, 0xb3, 0x3e, 0x91, 0x0f, 0xb8 }
	},
	{ 10, CRYPTO_SHA2_512_HMAC, 64, 28, 32,
	  "JefeJefeJefeJefe"
	  "JefeJefeJefeJefe"
	  "JefeJefeJefeJefe"
	  "JefeJefeJefeJefe",
	  "what do ya want "
	  "for nothing?",
	  { 0xcb, 0x37, 0x09, 0x17, 0xae, 0x8a, 0x7c, 0xe2,
	    0x8c, 0xfd, 0x1d, 0x8f, 0x47, 0x05, 0xd6, 0x14,
	    0x1c, 0x17, 0x3b, 0x2a, 0x93, 0x62, 0xc1, 0x5d,
	    0xf2, 0x35, 0xdf, 0xb2, 0x51, 0xb1, 0x54, 0x54 }
	},
	{ 11, CRYPTO_SHA2_512_HMAC, 64, 50, 32,
	  { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
	    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
	  { 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
	    0xdd, 0xdd },
	  { 0x2e, 0xe7, 0xac, 0xd7, 0x83, 0x62, 0x4c, 0xa9,
	    0x39, 0x87, 0x10, 0xf3, 0xee, 0x05, 0xae, 0x41,
	    0xb9, 0xf9, 0xb0, 0x51, 0x0c, 0x87, 0xe4, 0x9e,
	    0x58, 0x6c, 0xc9, 0xbf, 0x96, 0x17, 0x33, 0xd8
	  }
	},
	{ 12, CRYPTO_SHA2_512_HMAC, 64, 50, 32,
	  { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
	    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
	    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
	    0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
	    0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
	    0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
	    0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40 },
	  { 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
	    0xcd, 0xcd },
	  { 0x5e, 0x66, 0x88, 0xe5, 0xa3, 0xda, 0xec, 0x82,
	    0x6c, 0xa3, 0x2e, 0xae, 0xa2, 0x24, 0xef, 0xf5,
	    0xe7, 0x00, 0x62, 0x89, 0x47, 0x47, 0x0e, 0x13,
	    0xad, 0x01, 0x30, 0x25, 0x61, 0xba, 0xb1, 0x08 }
	},
};

int
main(void)
{
	size_t i;
	int fd, res;
	struct session_op cs;
	struct crypt_op co;
	unsigned char buf[80];

	fd = open("/dev/crypto", O_RDWR, 0);
	if (fd < 0)
		err(1, "open");
	for (i = 0; i < __arraycount(tests); i++) {
		memset(&cs, 0, sizeof(cs));
		cs.mac = tests[i].alg;
		cs.mackeylen = tests[i].key_len;
		cs.mackey = __UNCONST(&tests[i].key);
		res = ioctl(fd, CIOCGSESSION, &cs);
		if (res < 0)
			err(1, "CIOCGSESSION test %d", tests[i].num);

		memset(&co, 0, sizeof(co));
		memset(buf, 0, sizeof(buf));
		co.ses = cs.ses;
		co.op = COP_ENCRYPT;
		co.len = tests[i].len;
		co.src = __UNCONST(&tests[i].data);
		co.mac = buf;
		res = ioctl(fd, CIOCCRYPT, &co);
		if (res < 0)
			err(1, "CIOCCRYPT test %d", tests[i].num);

		/* compare with trailing zeros */
		if (memcmp(co.mac, &tests[i].mac, sizeof(tests[i].mac)))
			errx(1, "verification failed test %d", tests[i].num);

		res = ioctl(fd, CIOCFSESSION, &cs.ses);
		if (res < 0)
			err(1, "CIOCFSESSION test %d", tests[i].num);
	}
	return 0;
}
