/* query-view.c
 *
 * Copyright (C) 2002 Vivien Malerba
 * Copyright (C) 2002 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include "query.h"
#include "marshal.h"

struct _QueryViewPrivate {
	Query     *query;
	GObject   *obj;       /* Query or DbTable */
	gboolean   obj_weak_ref;

	gint       occ;       /* occurrence in case of aliases, starts at 0 */
	gchar     *alias;     /* new name given as alias, if any */
	gchar     *alias_exp; /* new name given as alias, if any */

	/* Activation */
	gchar     *ref;       /* may be used to find a reference to the Query */
	gboolean   activated;

	/* signals to catch creation and destruction of DbField or QueryField objects */
	gulong     creation_sig;
	gulong     removal_sig;
	gulong     modif_sig;
};

static void query_view_class_init (QueryViewClass * class);
static void query_view_init       (QueryView * qv);
static void query_view_dispose    (GObject   * object);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass   *parent_class = NULL;

enum
{
	CONTENT_ADDED,
	CONTENT_REMOVED,
	STATUS_CHANGED,
	LAST_SIGNAL
};

static gint query_view_signals[LAST_SIGNAL] = { 0, 0, 0 };

guint
query_view_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryViewClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_view_class_init,
			NULL,
			NULL,
			sizeof (QueryView),
			0,
			(GInstanceInitFunc) query_view_init
		};		

		type = g_type_register_static (G_TYPE_OBJECT, "QueryView", &info, 0);
	}
	return type;
}

static void
query_view_class_init (QueryViewClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	query_view_signals[CONTENT_ADDED] =
		g_signal_new ("content_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryViewClass, content_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE, 1,
			      G_TYPE_POINTER);
	
	query_view_signals[CONTENT_REMOVED] =
		g_signal_new ("content_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryViewClass, content_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE, 1,
			      G_TYPE_POINTER);

	query_view_signals[STATUS_CHANGED] =
		g_signal_new ("status_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryViewClass, status_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	class->content_added = NULL;
	class->content_removed = NULL;
	class->status_changed = NULL;

	object_class->dispose = query_view_dispose;
}


static void
query_view_init (QueryView * qv)
{
	qv->priv = g_new0 (QueryViewPrivate, 1);
	qv->priv->obj = NULL;
	qv->priv->obj_weak_ref = FALSE;
	qv->priv->occ = 0;
	qv->priv->alias = NULL;
	qv->priv->alias_exp = NULL;
	qv->priv->ref = NULL;
	qv->priv->activated = FALSE;
	qv->priv->query = NULL;
}



GObject   *
query_view_new (Query *q) 
{
	GObject   *obj;
	QueryView *qv;

	g_return_val_if_fail (q, NULL);
	g_return_val_if_fail (IS_QUERY (q), NULL);


	obj = g_object_new (QUERY_VIEW_TYPE, NULL);
	qv = QUERY_VIEW (obj);

	qv->priv->query = q;

	return obj;
}


GObject   *
query_view_new_copy (QueryView *qv)
{
	QueryView *newqv;

	g_return_val_if_fail (qv, NULL);
	g_return_val_if_fail (IS_QUERY_VIEW (qv), NULL);

	newqv = QUERY_VIEW (query_view_new (qv->priv->query));
	query_view_set_view_obj (newqv, qv->priv->obj);
	newqv->priv->occ = qv->priv->occ;
	if (qv->priv->alias)
		newqv->priv->alias = g_strdup (qv->priv->alias);
	if (qv->priv->ref)
		newqv->priv->ref = g_strdup (qv->priv->ref);
	return G_OBJECT (newqv);
}


static void view_obj_destroy_cb (QueryView *qv, GObject   *obj); /* GWeak Notify */
static void
query_view_dispose (GObject   * object)
{
	QueryView *qv;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_QUERY_VIEW (object));

	qv = QUERY_VIEW (object);

	if (qv->priv) {
#ifdef debug
		g_print ("Disposing of QueryView %p\n", qv);
#endif
		query_view_deactivate (qv);
		
		if (qv->priv->alias) {
			g_free (qv->priv->alias);
			qv->priv->alias = NULL;
		}
		if (qv->priv->alias_exp) {
			g_free (qv->priv->alias_exp);
			qv->priv->alias_exp = NULL;
		}
		if (qv->priv->ref) {
			g_free (qv->priv->ref);
			qv->priv->ref = NULL;
		}
		qv->priv->occ = -1;
		g_free (qv->priv);
		qv->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}


void 
query_view_deactivate (QueryView *qv)
{
	g_return_if_fail (qv && IS_QUERY_VIEW (qv));

	if (!qv->priv->activated)
		return;

	query_view_set_view_obj (qv, NULL);
	qv->priv->activated = FALSE;

#ifdef debug_signal
	g_print (">> 'STATUS_CHANGED' from query_view_deactivate\n");
#endif
	g_signal_emit (G_OBJECT (qv), query_view_signals[STATUS_CHANGED], 0);
#ifdef debug_signal
	g_print ("<< 'STATUS_CHANGED' from query_view_deactivate\n");
#endif
}

void 
query_view_activate (QueryView *qv)
{
	gpointer ptr;
	
	g_return_if_fail (qv && IS_QUERY_VIEW (qv));

	if (qv->priv->activated)
		return;

	if (qv->priv->obj) {
		qv->priv->activated = TRUE;
#ifdef debug_signal
		g_print (">> 'STATUS_CHANGED' from query_view_activate\n");
#endif
		g_signal_emit (G_OBJECT (qv), query_view_signals[STATUS_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'STATUS_CHANGED' from query_view_activate\n");
#endif
	}
	else {
		GObject *obj;
		ptr = query_find_from_xml_name (qv->priv->query->conf, NULL, qv->priv->ref);
		obj = NULL;
		if (ptr)
			obj = G_OBJECT (ptr);
		else {
			ptr = database_find_table_from_xml_name (qv->priv->query->conf->db, 
								 qv->priv->ref);
			if (ptr)
				obj = G_OBJECT (ptr);
		}
		
		if (obj) {
			g_free (qv->priv->ref);
			qv->priv->ref = NULL;
			query_view_set_view_obj (qv, obj);
			
			qv->priv->activated = TRUE;
#ifdef debug_signal
			g_print (">> 'STATUS_CHANGED' from query_view_activate\n");
#endif
			g_signal_emit (G_OBJECT (qv), query_view_signals[STATUS_CHANGED], 0);
#ifdef debug_signal
			g_print ("<< 'STATUS_CHANGED' from query_view_activate\n");
#endif
		}
	}
}

gboolean 
query_view_get_activated (QueryView *qv)
{
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), FALSE);
	return qv->priv->activated;
}



void query_view_set_query (QueryView *qv, Query *q)
{
	g_return_if_fail (qv && IS_QUERY_VIEW (qv));
	g_return_if_fail (q && IS_QUERY (q));
	
	qv->priv->query = q;
}

static void qv_obj_created_cb (GObject *obj, GObject *content, QueryView *qv);
static void qv_obj_dropped_cb (GObject *obj, GObject *content, QueryView *qv);
static void qv_obj_modified_cb (GObject *obj, GObject *content, QueryView *qv);
void
query_view_set_view_obj (QueryView *qv, GObject   *obj)
{
	g_return_if_fail (qv && IS_QUERY_VIEW (qv));

	if (qv->priv->obj && qv->priv->obj_weak_ref) {
		g_object_weak_unref (qv->priv->obj, (GWeakNotify) view_obj_destroy_cb, qv);
		g_signal_handler_disconnect (qv->priv->obj, qv->priv->creation_sig);
		g_signal_handler_disconnect (qv->priv->obj, qv->priv->removal_sig);
		if (IS_QUERY (qv->priv->obj))
			g_signal_handler_disconnect (qv->priv->obj, qv->priv->modif_sig);
		qv->priv->obj = NULL;
		qv->priv->obj_weak_ref = FALSE;
	}
	
	if (obj) {
		g_return_if_fail (IS_DB_TABLE (obj) || IS_QUERY (obj));

		qv->priv->obj = obj;
		g_object_weak_ref (qv->priv->obj, (GWeakNotify) view_obj_destroy_cb, qv);

		/* signal names common to Query and DbTable classes */
		qv->priv->creation_sig = g_signal_connect (G_OBJECT (obj), "field_created",
							   G_CALLBACK (qv_obj_created_cb), qv);
		qv->priv->removal_sig = g_signal_connect (G_OBJECT (obj), "field_dropped",
							  G_CALLBACK (qv_obj_dropped_cb), qv);

		/* specific to Query */
		if (IS_QUERY (obj))
			qv->priv->modif_sig = g_signal_connect (G_OBJECT (obj), "field_print_modified",
							  G_CALLBACK (qv_obj_modified_cb), qv);
		qv->priv->obj_weak_ref = TRUE;
	}
}

static void
qv_obj_created_cb (GObject *obj, GObject *content, QueryView *qv)
{
	gboolean do_signal = TRUE;
	
	if (IS_QUERY_FIELD (content) && !query_field_get_is_printed (QUERY_FIELD (content)))
		do_signal = FALSE;
	
	if (do_signal) {
#ifdef debug_signal
		g_print (">> 'CONTENT_ADDED' from qv_obj_created_cb\n");
#endif
		g_signal_emit (G_OBJECT (qv), query_view_signals[CONTENT_ADDED], 0, content);
#ifdef debug_signal
		g_print ("<< 'CONTENT_ADDED' from qv_obj_created_cb\n");
#endif
	}
}

static void
qv_obj_dropped_cb (GObject *obj, GObject *content, QueryView *qv)
{

#ifdef debug_signal
	g_print (">> 'CONTENT_REMOVED' from qv_obj_created_cb\n");
#endif
	g_signal_emit (G_OBJECT (qv), query_view_signals[CONTENT_REMOVED], 0, content);
#ifdef debug_signal
	g_print ("<< 'CONTENT_REMOVED' from qv_obj_created_cb\n");
#endif
}

static void
qv_obj_modified_cb (GObject *obj, GObject *content, QueryView *qv)
{
	if (IS_QUERY_FIELD (content)) {
		if (query_field_get_is_printed (QUERY_FIELD (content)))
			qv_obj_created_cb (obj, content, qv);
		else
			qv_obj_dropped_cb (obj, content, qv);
	}
}

static void 
view_obj_destroy_cb (QueryView *qv, GObject   *obj)
{
	qv->priv->obj_weak_ref = FALSE;
	query_view_deactivate (qv);
}



/*
 * XML Related stuff 
 */

gchar *
query_view_get_xml_id (QueryView *qv)
{
	gchar *str, *id;

	id = query_get_xml_id (qv->priv->query);
	str = g_strdup_printf ("%s:QV%d", id, g_slist_index (qv->priv->query->views, qv));
	g_free (id);

	return str;
}

xmlNodePtr 
query_view_save_to_xml (QueryView *qv)
{
	xmlNodePtr node;
	gchar *str;

	node = xmlNewNode (NULL, "QueryView");

	str = query_view_get_xml_id (qv);
	xmlSetProp (node, "id", str);
	g_free (str);

	if (IS_DB_TABLE (qv->priv->obj))
		str = db_table_get_xml_id (DB_TABLE (qv->priv->obj));
	else
		str = query_get_xml_id (QUERY (qv->priv->obj));
	xmlSetProp (node, "object", str);
	g_free (str);
	
	str = g_strdup_printf ("%d", qv->priv->occ);
	xmlSetProp (node, "occ", str);
	g_free (str);
	
	if (qv->priv->alias) 
		xmlSetProp (node, "alias", qv->priv->alias);

	return node;
}

void query_view_load_from_xml (QueryView *qv, xmlNodePtr node)
{
	gchar *str;

	str = xmlGetProp (node, "object");
	if (str) {
		GObject   *obj = NULL;
		gpointer ptr;

		/* QueryView is a table ?, a Query ? */
		ptr = database_find_table_from_xml_name (qv->priv->query->conf->db, str);
		if (ptr)
			obj = G_OBJECT (ptr);
		else {
			ptr = query_find_from_xml_name (qv->priv->query->conf, NULL, str);
			if (ptr)
				obj = G_OBJECT (ptr);
		}
				
		if (obj) {
			query_view_set_view_obj (qv, obj);
			g_free (str);
		}
		else {
			/* do it later */
			query_view_set_view_obj (qv, NULL);
			qv->priv->ref = str; /* later we try to find this */
		}
	}
			
	str = xmlGetProp (node, "occ");
	if (str) {
		qv->priv->occ = atoi (str);
		g_free (str);
	}
	else
		qv->priv->occ = 0;
			
	str = xmlGetProp (node, "alias");
	qv->priv->alias = str;
}

QueryView *
query_view_find_from_xml_name (ConfManager * conf, Query * start_query, gchar *xmlname)
{
	gchar *copy, *ptr;
	Query *q;
	QueryView *qv = NULL;

	copy = g_strdup (xmlname);
	ptr = strtok (copy , ":");
	q = query_find_from_xml_name (conf, start_query, ptr);
	if (q) {
		gint i;
		ptr = strtok (NULL , ":");
		i = atoi (ptr+2);
		qv = g_slist_nth_data (q->views, i);
	}

	g_free (copy);
	return qv;
}

/* field can be DbField or QueryField */
gboolean
query_view_contains_field (QueryView *qv, GObject   *field)
{
	gboolean found = FALSE;

	g_return_val_if_fail (IS_QUERY_VIEW (qv), FALSE);
	if (IS_DB_TABLE (qv->priv->obj)) {
		if (IS_DB_FIELD (field)) 
			found = (g_slist_find (DB_TABLE (qv->priv->obj)->fields, DB_FIELD (field)) != NULL);
	}
	else {
		if (IS_QUERY (qv->priv->obj)) {
			if (IS_QUERY_FIELD (field))
				found = (g_slist_find (QUERY (qv->priv->obj)->fields, QUERY_FIELD (field)) != NULL);
		}
	}

	return found;
}

#define QUERY_VIEW_NAME(qv) IS_QUERY (qv->priv->obj) ? \
		QUERY (qv->priv->obj)->name : \
		DB_TABLE (qv->priv->obj)->name

gchar *
query_view_get_textual (QueryView *qv)
{
	gchar *str, *retval;

	if (IS_QUERY (qv->priv->obj))
		str = QUERY (qv->priv->obj)->name;
	else
		str = DB_TABLE (qv->priv->obj)->name;
	if (qv->priv->occ)
		retval = g_strdup_printf ("%s (%d)", str, qv->priv->occ);
	else
		retval = g_strdup (str);

	return retval;
}			


/* query_view_is_alias
 *
 * returns TRUE if the QueryView holds a duplicate instance of an
 * existing Query or Table in the current query. FALSE if it is a
 * real Query or Table
 *
 */

gboolean
query_view_is_alias (QueryView *qv)
{
	return qv->priv->occ;
}

/* query_view_get_name
 *
 * if the QueryView is an alias, returns the alias. Otherwise,
 * returns the real name of its Table or Query. No need to free the returned string.
 * 
 *
 */

const gchar *
query_view_get_name (QueryView *qv)
{
	const gchar *str;

	if (qv->priv->occ)
		str = qv->priv->alias;
	else
		str = QUERY_VIEW_NAME(qv);

	return str;
}


/* query_view_get_real_name
 *
 * returns the name of its Table or Query, even if the QueryView is an alias, ie,
 * the alias name is ignored. No need to free the returned string.
 *
 */

const gchar *
query_view_get_real_name (QueryView *qv)
{
	return QUERY_VIEW_NAME(qv);
}


/* query_view_get_aliased_name
 *
 * returns the SQL expression to define the alias, e.g., table AS table_1 or simply
 * the table name if there is no alias. No need to free the returned string.
 *
 * Implementation note: for efficiency reasons and user code simplification, the first time
 * this function is called, the alias expression is created (if there is an alias) and 
 * stored in the query view object
 *
 */

void 
query_view_set_alias (QueryView *qv, const gchar *alias)
{
	g_return_if_fail (qv && IS_QUERY_VIEW (qv));

	if (qv->priv->alias)
		g_free (qv->priv->alias);
	qv->priv->alias = NULL;

	if (alias) 
		qv->priv->alias = g_strdup (alias);

	/* REM: there is no "changed" signal */
}

const gchar *
query_view_get_alias (QueryView *qv)
{
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);
	return qv->priv->alias;
}

const gchar *
query_view_get_alias_expr (QueryView *qv)
{
	gchar *str;

	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);

	if (qv->priv->occ) {
		if (qv->priv->alias_exp)
			g_free (qv->priv->alias_exp);

		qv->priv->alias_exp = g_strdup_printf ("%s AS %s", QUERY_VIEW_NAME(qv), qv->priv->alias);
		str = qv->priv->alias_exp;
	}
	else
		str = QUERY_VIEW_NAME(qv);

	return str;
}

gchar *
query_view_get_field_name (QueryView *qv, GObject   *qvf)
{
	gchar *str = NULL;

	if (IS_QUERY (qv->priv->obj)) {
		if (qvf && IS_QUERY_FIELD (qvf) && 
		    (query_field_get_query (QUERY_FIELD (qvf)) == QUERY (qv->priv->obj))) {
			if (query_field_get_name (QUERY_FIELD (qvf)))
				str = g_strdup (query_field_get_name (QUERY_FIELD (qvf)));
			else {
				if (query_field_get_is_printed (QUERY_FIELD (qvf)))
					str = g_strdup (query_field_get_alias (QUERY_FIELD (qvf)));
				else
					str = query_field_get_xml_id (QUERY_FIELD (qvf));
			}
		}
	}
	
	if (!str && IS_DB_TABLE (qv->priv->obj)) {
		if (qvf && IS_DB_FIELD (qvf) && g_slist_find (DB_TABLE (qv->priv->obj)->fields, qvf)) {
			str = g_strdup (DB_FIELD (qvf)->name);
		}
	}

	if (!str)
		g_warning ("query_view_get_field_name returns a NULL value");

	return str;
}

GObject *
query_view_get_view_obj (QueryView *qv)
{
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);
	return qv->priv->obj;
}

Query *
query_view_get_query (QueryView *qv)
{
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);
	return qv->priv->query;
}

void 
query_view_set_occ (QueryView *qv, gint occ)
{
	g_return_if_fail (qv && IS_QUERY_VIEW (qv));
	qv->priv->occ = occ;
	/* REM: there is no "changed" signal */
}

gint 
query_view_get_occ (QueryView *qv)
{
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), -1);
	return qv->priv->occ;
}

gboolean
query_view_get_is_activated (QueryView *qv)
{
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), FALSE);
	g_return_val_if_fail (qv->priv, FALSE);

	return qv->priv->activated;
}
