/* query-exec.c
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "query-env.h"
#include "query-exec.h"
#include "data-entry.h"
#include "marshal.h"
#include "data-grid.h"
#include "data-form.h"


static void query_exec_class_init (QueryExecClass * class);
static void query_exec_init (QueryExec * qx);
static void query_exec_dispose (GObject   * object);


typedef struct _MissingDataNode MissingDataNode;
typedef struct _ExecUi          ExecUi;

struct _QueryExecPriv {
	Query           *q;	/* q is a copy of env->q, so here is a ref to the q to be used */
	ServerResultset *sres;
	DbTable         *modif_table;
	guint            actions; /* possible user actions */
	gboolean         run_done;

	GSList          *missing_data;
	gchar           *executed_query; /* SQL (or XML) query which is executed or going to be executed */
	GSList          *exec_uis; /* list of user interfaces opened for the data */	

	gboolean         disposed; /* TRUE if we have been through the _dispose() method */
};


struct _MissingDataNode
{
	QueryField        *qf;    /* represented QueryField */
	gchar             *descr; 
	GtkWidget         *data_entry; /* DataEntry if we use a DataEntry */
};

#define MISSING_DATA_NODE(nde) ((MissingDataNode *) nde)

struct _ExecUi
{
	GtkWidget *ui;  /* Form or grid */
	GtkWidget *dlg; /* dialog in which the form or grid is */
	QueryExec *qx;
};
#define EXEC_UI(st) ((ExecUi *) st)

static void add_managed_ui (QueryExec *qx, ExecUi *eui);


/*
 *
 * Main functions for the object
 *
 */


enum
{
	NOT_VALID,
	LAST_SIGNAL
};

static gint query_exec_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass   *parent_class = NULL;

guint
query_exec_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryExecClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_exec_class_init,
			NULL,
			NULL,
			sizeof (QueryExec),
			0,
			(GInstanceInitFunc) query_exec_init
		};		
		
		type = g_type_register_static (G_TYPE_OBJECT, "QueryExec", &info, 0);
	}

	return type;
}

static void
query_exec_class_init (QueryExecClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	query_exec_signals[NOT_VALID] =
		g_signal_new ("not_valid",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryExecClass, not_valid),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	class->not_valid = NULL;

	object_class->dispose = query_exec_dispose;
}

static void
query_exec_init (QueryExec * qx)
{
	qx->priv = g_new0 (QueryExecPriv, 1);
	qx->priv->q = NULL;
	qx->priv->modif_table = NULL;
	qx->priv->actions = 0;
	qx->priv->sres = NULL;
	qx->priv->run_done = FALSE;
	qx->priv->missing_data = NULL;
	qx->priv->executed_query = NULL;
	qx->priv->exec_uis = NULL;
	qx->priv->disposed = FALSE;
}

static void prepare_query (QueryExec * qx);
static void query_changed (Query * q, QueryExec * qx);
GObject *
query_exec_new (QueryEnv * env)
{
	GObject   *obj;
	QueryExec *qx;

	g_return_val_if_fail (env != NULL, NULL);
	g_return_val_if_fail (IS_QUERY_ENV (env), NULL);

	obj = g_object_new (QUERY_EXEC_TYPE, NULL);
	qx = QUERY_EXEC (obj);

	/* working on a copy of the query, so the user can work on the query at the same time as
	   the query is executed. The QueryExec object has a reference on the Query object. */
	qx->priv->q = QUERY (query_new_copy (env->q));
	qx->priv->modif_table = env->modif_table;
	qx->priv->actions = env->actions;

	/* register to the QueryEnv */
	query_env_register_exec_obj (env, G_OBJECT (qx));

	prepare_query (qx);

	/* connection to signals from the Query */
	g_signal_connect (G_OBJECT (qx->priv->q), "changed",
			  G_CALLBACK (query_changed), qx);

	return obj;
}

/* This function operates some modifications on the structure of the query:
   - just after any QF ALLFIELDS it inserts QF of type FIELD for all the fields of the 
     corresponding QueryView and hides the ALLFIELDS QF.
   - it inserts some QF of type FIELD for all the primary keys of all the DbTable present
     in all the QueryViews, if they are not yet present
*/ 
static void 
prepare_query (QueryExec * qx)
{
	GSList *list;
	QueryField *qf;

	/* ALLFIELDS QF -> multiple FIELD QFs */
	list = qx->priv->q->fields;
	while (list) {
		qf = QUERY_FIELD (list->data);
		if (qf->field_type == QUERY_FIELD_ALLFIELDS) {
			GSList *nextone = g_slist_next (list);
			QueryView *view = query_field_allfields_get_queryview (qf);

			if (view && IS_DB_TABLE (view->obj)) {
				DbTable *table;
				QueryField *nqf;
				GSList *fields;

				table = DB_TABLE (view->obj);
				fields = table->fields;
				while (fields) {
					nqf = query_field_field_dbfield_new (qx->priv->q,
									     DB_FIELD (fields->data)->name,
									     view, DB_FIELD (fields->data));
					query_field_set_is_printed (nqf, TRUE);
					query_add_field_before (qx->priv->q, nqf, qf);
					fields = g_slist_next (fields);
				}
				query_field_set_is_hidden (qf, TRUE);
			}
		
			list = nextone;
		}
		else
			list = g_slist_next (list);
	}

	/* insertion of Primary key fields if there is none */
	list = qx->priv->q->views;
	while (list) {
		QueryView *view = QUERY_VIEW (list->data);

		if (IS_DB_TABLE (view->obj)) {
			DbTable *table = DB_TABLE (view->obj);
			gboolean all_pk_found = TRUE;
			gboolean has_a_field = FALSE;
			GSList *fields;
			
			fields = table->fields;
			while (fields) {
				fields = g_slist_next (fields);
			}
			/* FIXME */
		}
		list = g_slist_next (list);
	}

#ifdef debug
	query_dump_contents (qx->priv->q);
#endif
}

static void
query_changed (Query * q, QueryExec * qx)
{
	/* FIXME: emit the "not_valid" signal and freeze the dialogs related to this Exec */
}

/* in case a QueryField is destroyed */
static void query_field_notify (QueryExec *qx, QueryField *qf);
static void dispose_missing_data (QueryExec *qx);
static void
query_exec_dispose (GObject   * object)
{
	QueryExec *qx;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_QUERY_EXEC (object));

	qx = QUERY_EXEC (object);

	if (qx->priv) {
		qx->priv->disposed = TRUE;
		dispose_missing_data (qx);
		
		while (qx->priv->exec_uis) {
			ExecUi *eui;
			
			eui = EXEC_UI (qx->priv->exec_uis->data);
			gtk_widget_destroy (GTK_WIDGET (eui->dlg));	/* => destroys the form or grid as well */
		}
		
		if (qx->priv->sres) {
			g_object_unref (G_OBJECT (qx->priv->sres));
			qx->priv->sres = NULL;
		}
		
		if (qx->priv->q) {
			g_object_unref (G_OBJECT (qx->priv->q));
			qx->priv->q = NULL;
		}
		
		if (qx->priv->executed_query) {
			g_free (qx->priv->executed_query);
			qx->priv->executed_query = NULL;
		}
		g_free (qx->priv);
		qx->priv = NULL;
	}
	

	/* for the parent class */
        parent_class->dispose (object);
}

void 
query_exec_free (QueryExec * exec)
{
	g_return_if_fail (exec && IS_QUERY_EXEC (exec));
	g_object_unref (G_OBJECT (exec));
}

static void 
query_field_notify (QueryExec *qx, QueryField *qf)
{
	GSList *list;
	MissingDataNode *mdn = NULL;

	list = qx->priv->missing_data;
	while (list && !mdn) {
		if (MISSING_DATA_NODE (list->data)->qf == qf) 
			mdn = MISSING_DATA_NODE (list->data);
		list = g_slist_next (list);
	}
	
	if (mdn) {
		mdn->qf = NULL;
		if (mdn->descr) {
			g_free (mdn->descr);
			mdn->descr = NULL;
		}
		if (mdn->data_entry) {
			gtk_widget_destroy (mdn->data_entry);
			mdn->data_entry = NULL;
		}
		qx->priv->missing_data = g_slist_remove (qx->priv->missing_data, mdn);
		g_free (mdn);
	}
}

static void
dispose_missing_data (QueryExec *qx)
{
	GSList *list;
	g_return_if_fail (qx && IS_QUERY_EXEC (qx));

	list = qx->priv->missing_data;
	while (list) {
		MissingDataNode *mdn;

		mdn = MISSING_DATA_NODE (qx->priv->missing_data->data);
		if (mdn->qf) {
			g_object_weak_unref (G_OBJECT (mdn->qf), (GWeakNotify) query_field_notify, qx);
			mdn->qf = NULL;
		}
		if (mdn->descr) {
			g_free (mdn->descr);
			mdn->descr = NULL;
		}
		if (mdn->data_entry) {
			gtk_widget_destroy (mdn->data_entry);
			mdn->data_entry = NULL;
		}
		g_free (mdn);
		list = g_slist_next (list);
	}

	g_slist_free (qx->priv->missing_data);
	qx->priv->missing_data = NULL;
}

/*
 * management of the opened widgets refering to the QX
 */

static void managed_ui_dlg_destroyed (QueryExec *qx, GtkWidget *wid);
static void managed_ui_ui_destroyed (QueryExec *qx, GtkWidget *wid);
static void 
add_managed_ui (QueryExec *qx, ExecUi *eui)
{
	qx->priv->exec_uis = g_slist_append (qx->priv->exec_uis, eui);
	g_object_weak_ref (G_OBJECT (eui->dlg), (GWeakNotify) managed_ui_dlg_destroyed, qx);
	g_object_weak_ref (G_OBJECT (eui->ui), (GWeakNotify) managed_ui_ui_destroyed, qx);
}

static void 
managed_ui_dlg_destroyed (QueryExec *qx, GtkWidget *wid)
{
	GSList *list;
	ExecUi *eui = NULL;

	list = qx->priv->exec_uis;
	while (list && !eui) {
		if (EXEC_UI (list->data)->dlg == wid)
			eui = EXEC_UI (list->data);
		list = g_slist_next (list);
	}

	if (eui) {
		if (eui->ui) {
			g_object_weak_unref (G_OBJECT (eui->ui), (GWeakNotify) managed_ui_ui_destroyed, qx);
			eui->ui = NULL;
		}
		qx->priv->exec_uis = g_slist_remove (qx->priv->exec_uis, eui);
		g_free (eui);
	}
	else
		g_warning ("EUI could not be found!");

	if (!qx->priv->exec_uis && !qx->priv->disposed) /* time to destroy itself */
		query_exec_free (qx);
}

static void 
managed_ui_ui_destroyed (QueryExec *qx, GtkWidget *wid)
{
	GSList *list;
	ExecUi *eui = NULL;

	list = qx->priv->exec_uis;
	while (list && !eui) {
		if (EXEC_UI (list->data)->ui == wid)
			eui = EXEC_UI (list->data);
		list = g_slist_next (list);
	}

	if (eui) {
		if (eui->dlg) {
			g_object_weak_unref (G_OBJECT (eui->dlg), (GWeakNotify) managed_ui_dlg_destroyed, qx);
			gtk_widget_destroy (GTK_WIDGET (eui->dlg));
			eui->dlg = NULL;
		}
		qx->priv->exec_uis = g_slist_remove (qx->priv->exec_uis, eui);
		g_free (eui);
	}
	else
		g_warning ("EUI could not be found!");

	if (!qx->priv->exec_uis && !qx->priv->disposed) /* time to destroy itself */
		query_exec_free (qx);
}






/*
 * Recordset initialisation
 */


static gboolean real_run_the_query (QueryExec * qx, ServerAccessQueryType qtype);

/* The object asks for values before if applicable and runs the query,
   return TRUE if the query has been executed correctly */
static gboolean
query_exec_run (QueryExec * qx)
{
	GSList *hold, *list, *missing_list = NULL;
	gboolean retval = FALSE;

	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), FALSE);
	g_return_val_if_fail (qx->priv->q && IS_QUERY (qx->priv->q), FALSE);

	/* Rem: We assume that all the QueryField in the Query are used for the query
	   (either in the FROM, in the WHERE part or in the joins part),
	   so for any QUERY_FIELD_VALUE, we will ask for a value if it is needed */

	/* if we have a SQL text only query, then run it, we don't need any values */
	if (qx->priv->q->type == QUERY_TYPE_SQL) {
		return real_run_the_query (qx, SERVER_ACCESS_QUERY_SQL);
	}

	/* FIXME: if the query is not a SELECT query, then take care of it here */

	/* list the missing value */
	list = query_get_missing_value_fields (qx->priv->q);
	hold = list;
	while (list) {
		DataHandler *dh;
		QueryField *qf = QUERY_FIELD (list->data);
		MissingDataNode *mdn = g_new0 (MissingDataNode, 1);

		mdn->qf = qf;
		g_object_weak_ref (G_OBJECT (qf), (GWeakNotify) query_field_notify, qx);
		
		mdn->descr = g_strdup_printf ("%s:", qf->name);
		
		/* FIXME: in the next function, use the right 2nd argument in relation of the implementation
		   of the QUERY_FIELD_VALUE QueryField  */
		dh = server_access_get_object_handler (qx->priv->q->conf->srv, G_OBJECT (qf));
		/* FIXME: use the default value defined by the QUERY_FIELD_VALUE QueryField 
		   instead of ""*/
		mdn->data_entry = GTK_WIDGET (data_handler_get_widget_from_value (dh,
						       (query_field_value_get_default_value (qf))));
		missing_list = g_slist_append (missing_list, mdn);
		list = g_slist_next (list);
	}
	if (hold)
		g_slist_free (hold);


	if (missing_list == NULL) {
		retval = real_run_the_query (qx, SERVER_ACCESS_QUERY_SQL);

		qx->priv->run_done = retval;
		return retval;
	}
	else {
		/* build the dialog to fill in values */
		GtkWidget *dlg, *table, *label;
		guint i = 0;
		gchar *str;
		gint button;
		
		/* dialog itself */
		dlg = gtk_dialog_new_with_buttons (_("Enter necessary values"), NULL, 0,
						   GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
						   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);
		
		gtk_dialog_set_default_response (GTK_DIALOG (dlg), GTK_RESPONSE_REJECT);
		
		/* dialog contents */
		str = g_strdup_printf (_("Fill the following parameters to execute the\n"
					 "'%s' query"), qx->priv->q->name);
		label = gtk_label_new (str);
		g_free (str);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), label, FALSE, TRUE, GNOME_PAD);
		
		table = gtk_table_new (g_slist_length (missing_list), 2, FALSE);
		gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD / 2.);
		gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD / 2.);
		gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD / 2.);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), table, TRUE, TRUE, GNOME_PAD);
		
		
		list = missing_list;
		while (list) {
			MissingDataNode *mdn = MISSING_DATA_NODE (list->data);
			label = gtk_label_new (mdn->descr);
			gtk_table_attach (GTK_TABLE (table), label, 0, 1, i, i + 1,
					  0, 0, 0, 0);
			gtk_table_attach_defaults (GTK_TABLE (table),
						   mdn->data_entry, 1, 2, i,
						   i + 1);
			i++;
			list = g_slist_next (list);
		}
		gtk_widget_show_all (GTK_DIALOG (dlg)->vbox);
		button = gtk_dialog_run (GTK_DIALOG (dlg));
		qx->priv->missing_data = missing_list;

		switch (button) {
		case GTK_RESPONSE_ACCEPT:
			gtk_widget_hide (dlg); /* to avoid the destruction of the widgets in the MDN */
			retval = real_run_the_query (qx, SERVER_ACCESS_QUERY_SQL);
			break;
		default:
			retval = FALSE;
			break;
		}
		gtk_widget_destroy (dlg);

		qx->priv->run_done = retval;
		return retval;
	}
}
	
/* 
 * Now really run the query
 */
static gboolean
real_run_the_query (QueryExec * qx, ServerAccessQueryType qtype)
{
	GSList *hold, *mlist, *query_param_values = NULL;
	QueryParamValue *qmv;

	mlist = qx->priv->missing_data;
	while (mlist) {
		MissingDataNode *mdn;
		GdaValue *value;

		mdn = MISSING_DATA_NODE (mlist->data);
		value = data_entry_get_gdavalue (DATA_ENTRY (mdn->data_entry));

		/* the widgets are destroyed when the dialog is closed,
		   so we set them to NULL here to avoid trying to destroy them
		   again. */
		mdn->data_entry = NULL;

		qmv = g_new0 (QueryParamValue, 1);
		qmv->field = mdn->qf;
		qmv->value = value;
		query_param_values = g_slist_append (query_param_values, qmv);
		mlist = g_slist_next (mlist);
	}

	qx->priv->executed_query = query_get_select_query (qx->priv->q, query_param_values);

	/* free the list of missing_data (the widgets) */
	dispose_missing_data (qx);

	/* free the list of query missing values */
	while (query_param_values) {
		qmv = QUERY_PARAM_VALUE (query_param_values->data);
		gda_value_free (qmv->value);
		g_free (qmv);
		hold = query_param_values;
		query_param_values = g_slist_remove_link (query_param_values, query_param_values);
		g_slist_free_1 (hold);
	}

	/* run the query */
	qx->priv->sres = server_access_do_query (qx->priv->q->conf->srv, qx->priv->executed_query, qtype);

	return qx->priv->sres ? TRUE : FALSE;
}













/*
 * Get a ready to work dialog with
 * the right form in it to edit data
 */

static GtkWidget *prepare_ui_dialog (QueryExec *qx, gchar *title);
static GtkWidget *get_real_edit_form (QueryExec * qx, gulong numrow, gboolean empty);
static void form_contents_modified_cb (DataForm *form, ExecUi *eui);
GtkWidget *
query_exec_get_edit_form (QueryExec * qx, gulong numrow)
{
	GtkWidget *dlg = NULL, *newui;
	ExecUi *eui;
	gchar *str;
	gboolean ok = TRUE;

	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), NULL);
	g_return_val_if_fail (qx->priv->q && IS_QUERY (qx->priv->q), NULL);

	eui = g_new0 (ExecUi, 1);
	eui->qx = qx;

	/* init RS if not done */
	if (!qx->priv->run_done)
		ok = query_exec_run (qx);

	if (ok) {
		/* dialog preparation */
		str = g_strdup_printf (_("Recordset Edition (query '%s')"),
				       qx->priv->q->name);
		dlg = prepare_ui_dialog (qx, str);
		g_free (str);
		eui->dlg = dlg;
		
		newui = get_real_edit_form (qx, numrow, FALSE);
		eui->ui = newui;
		g_signal_connect (G_OBJECT (newui), "contents_modified",
				  G_CALLBACK (form_contents_modified_cb), eui);
		
		gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dlg)->vbox), newui);
		gtk_widget_show (newui);
		
		/* addtion to the list of managed dialogs */
		add_managed_ui (qx, eui);
	}

	return dlg;
}


/* This callback updates the status of the action buttons present with the form itself */
static void
form_contents_modified_cb (DataForm *form, ExecUi *eui)
{
	g_print ("TODO form_contents_modified_cb ()\n");
}


/*
 * Get a ready to work dialog with
 * the right form in it to insert new data
 */

GtkWidget *
query_exec_get_insert_form (QueryExec * qx)
{
	GtkWidget *dlg = NULL, *newui;
	ExecUi *eui;
	gchar *str;
	gboolean ok = TRUE;

	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), NULL);
	g_return_val_if_fail (qx->priv->q && IS_QUERY (qx->priv->q), NULL);

	eui = g_new0 (ExecUi, 1);
	eui->qx = qx;

	/* init RS if not done */
	if (!qx->priv->run_done)
		ok = query_exec_run (qx);

	if (ok) {
		/* dialog preparation */
		str = g_strdup_printf (_("Data insertion in '%s' (query '%s')"),
				       qx->priv->modif_table->name, qx->priv->q->name);
		dlg = prepare_ui_dialog (qx, str);
		g_free (str);
		eui->dlg = dlg;
		
		newui = get_real_edit_form (qx, 0, TRUE);
		eui->ui = newui;
		
		gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dlg)->vbox), newui);
		gtk_widget_show (newui);
		
		/* addtion to the list of managed dialogs */
		add_managed_ui (qx, eui);
	}

	return dlg;
}

gint 
query_exec_get_user_actions (QueryExec * exec)
{
	g_return_val_if_fail (exec && IS_QUERY_EXEC (exec), -1);
	g_assert (exec->priv);

	return exec->priv->actions;
}

static void sub_action_refresh (GtkWidget *wid, QueryExec * qx);
static void sub_action_insert  (GtkWidget *wid, QueryExec * qx);
static void sub_action_edit    (GtkWidget *wid, gulong row, QueryExec * qx);
static void sub_action_delete  (GtkWidget *wid, gulong row, QueryExec * qx);
static void sub_action_choose  (GtkWidget *wid, gulong row, QueryExec * qx);
static void sub_action_viewall (GtkWidget *wid, QueryExec * qx);

static GtkWidget *get_real_edit_form (QueryExec * qx, gulong numrow, gboolean empty)
{
	GtkWidget *form;
	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), NULL);
	g_return_val_if_fail (qx->priv->q && IS_QUERY (qx->priv->q), NULL);
	g_return_val_if_fail (qx->priv->sres && IS_SERVER_RESULTSET (qx->priv->sres), NULL);

	if (empty)
		form = data_form_new_insert (qx);
	else
		form = data_form_new_edit (qx, numrow);

	/* signals from the form */
	g_signal_connect (G_OBJECT (form), "action_refresh",
			  G_CALLBACK (sub_action_refresh), qx);
	g_signal_connect (G_OBJECT (form), "action_insert",
			  G_CALLBACK (sub_action_insert), qx);
	g_signal_connect (G_OBJECT (form), "action_viewall",
			  G_CALLBACK (sub_action_viewall), qx);
	g_signal_connect (G_OBJECT (form), "action_choose",
			  G_CALLBACK (sub_action_choose), qx);	

	return form;
}




/*
 * Get a ready to work dialog with
 * the right form in it to insert new data
 */

static GtkWidget * get_real_consult_grid (QueryExec * qx, gulong numrow);
GtkWidget *
query_exec_get_consult_grid   (QueryExec * exec, gulong numrow)
{
	GtkWidget *dlg = NULL, *newui;
	ExecUi *eui;
	gchar *str;
	gboolean ok = TRUE;

	g_return_val_if_fail (exec && IS_QUERY_EXEC (exec), NULL);
	g_return_val_if_fail (exec->priv->q && IS_QUERY (exec->priv->q), NULL);

	eui = g_new0 (ExecUi, 1);
	eui->qx = exec;

	/* init RS if not done */
	if (!exec->priv->run_done)
		ok = query_exec_run (exec);

	if (ok) {
		/* dialog preparation */
		str = g_strdup_printf (_("Recordset tabular view (query '%s')"),
				       exec->priv->q->name);
		dlg = prepare_ui_dialog (exec, str);
		g_free (str);
		eui->dlg = dlg;
		
		newui = get_real_consult_grid (exec, numrow);
		eui->ui = newui;
		
		gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dlg)->vbox), newui);
		gtk_widget_show (newui);
		
		/* addtion to the list of managed dialogs */
		add_managed_ui (exec, eui);
	}

	return dlg;	
}


static GtkWidget *
get_real_consult_grid (QueryExec * qx, gulong numrow)
{
	GtkWidget *grid;
	g_return_val_if_fail (qx && IS_QUERY_EXEC (qx), NULL);
	g_return_val_if_fail (qx->priv->q && IS_QUERY (qx->priv->q), NULL);

	grid = data_grid_new (qx, numrow);

	/* signals from the grid */
	g_signal_connect (G_OBJECT (grid), "action_refresh",
			  G_CALLBACK (sub_action_refresh), qx);
	g_signal_connect (G_OBJECT (grid), "action_insert",
			  G_CALLBACK (sub_action_insert), qx);
	g_signal_connect (G_OBJECT (grid), "action_edit",
			  G_CALLBACK (sub_action_edit), qx);
	g_signal_connect (G_OBJECT (grid), "action_delete",
			  G_CALLBACK (sub_action_delete), qx);
	g_signal_connect (G_OBJECT (grid), "action_choose",
			  G_CALLBACK (sub_action_choose), qx);	

	return grid;
}

static void
sub_action_refresh (GtkWidget * wid, QueryExec * qx)
{
	if (!qx->priv->executed_query)
		return;

	if (qx->priv->sres) {
		g_object_unref (G_OBJECT (qx->priv->sres));
		qx->priv->sres = NULL;
	}

	/* getting a new QueryRes */
	qx->priv->sres = server_access_do_query (qx->priv->q->conf->srv, 
						 qx->priv->executed_query, SERVER_ACCESS_QUERY_SQL);
	
	if (qx->priv->sres) {
		/* Updating all the user interfaces for this qx */
		GSList *list;
		ExecUi *eui;

		list = qx->priv->exec_uis;
		while (list) {
			eui = EXEC_UI (list->data);
			if (IS_DATA_GRID (eui->ui))
				data_grid_refresh (DATA_GRID (eui->ui));
			else if (IS_DATA_FORM (eui->ui))
				data_form_refresh (DATA_FORM (eui->ui));
			list = g_slist_next (list);
		}
	}
}

/* creates a new dialog with the right callbacks regarding its management */
static GtkWidget *prepare_ui_dialog (QueryExec *qx, gchar *title);

static void
sub_action_insert (GtkWidget * wid, QueryExec * qx)
{
	GtkWidget *dlg;

	dlg = query_exec_get_insert_form (qx);
	if (dlg)
		gtk_widget_show (dlg);
}

static void
sub_action_edit (GtkWidget * wid, gulong row, QueryExec * qx)
{
	GtkWidget *dlg;

	dlg = query_exec_get_edit_form (qx, row);
	gtk_widget_show (dlg);
}

static void
sub_action_delete (GtkWidget * wid, gulong row, QueryExec * qx)
{
	gchar *str;
	str = g_strdup_printf (_("Removal of row %ld:\nto be implemented in %s, line %d"),
			       row, __FILE__, __LINE__);
	gnome_app_message (GNOME_APP (qx->priv->q->conf->app), str);
	g_free (str);
}

static void
sub_action_choose (GtkWidget * wid, gulong row, QueryExec * qx)
{
	gchar *str;
	str = g_strdup_printf (_("Choosing row %ld:\nto be implemented in %s, line %d"),
			       row, __FILE__, __LINE__);
	gnome_app_message (GNOME_APP (qx->priv->q->conf->app), str);
	g_free (str);
}

static void
sub_action_viewall (GtkWidget * wid, QueryExec * qx)
{
	GtkWidget *dlg, *newui;
	ExecUi *eui = NULL;
	gchar *str;
	GSList *list;
	gboolean found = FALSE;

	list = qx->priv->exec_uis;
	while (list && !found) {
		eui = EXEC_UI (list->data);
		if (IS_DATA_GRID (eui->ui))
			found = TRUE;
		else
			list = g_slist_next (list);
	}

	if (found)
		gdk_window_raise (GTK_WIDGET (eui->dlg)->window);
	else {
		newui = query_exec_get_consult_grid (qx, 0);
		if (newui) {
			str = g_strdup_printf (_("Execution of query '%s'"), qx->priv->q->name);
			dlg = prepare_ui_dialog (qx, str);
			g_free (str);
			
			eui = g_new0 (ExecUi, 1);
			eui->qx = qx;
			eui->dlg = dlg;
			eui->ui = newui;
			
			gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dlg)->vbox), newui);
			gtk_widget_show (newui);
			
			gtk_widget_show (dlg);
			add_managed_ui (qx, eui);
		}
	}
}




static void ui_dialog_response_cb   (GtkDialog *dlg, gint button_number, QueryExec *qx);
static GtkWidget *
prepare_ui_dialog (QueryExec *qx, gchar *title)
{
	GtkWidget *dlg;

	dlg = gtk_dialog_new_with_buttons (title, NULL, 0,
					   GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	g_signal_connect (G_OBJECT (dlg), "response",
			  G_CALLBACK (ui_dialog_response_cb), qx);
	/* NOTE: the "destroy" signal is not used here */

	gtk_window_set_resizable (GTK_WINDOW (dlg), TRUE);
	return dlg;
}

static void 
ui_dialog_response_cb (GtkDialog *dlg, gint button_number, QueryExec *qx)
{
	/* whatever the response, we simply destroy the dialog and the associated ExecUi struct */

	gtk_widget_destroy (GTK_WIDGET (dlg));
}

/*
 * Getting information 
 */
Query *
query_exec_get_query_ref (QueryExec * exec)
{
	g_return_val_if_fail (exec && IS_QUERY_EXEC (exec), NULL);
	return exec->priv->q;
}

DbTable *
query_exec_get_modif_table (QueryExec * exec)
{
	g_return_val_if_fail (exec && IS_QUERY_EXEC (exec), NULL);
	return exec->priv->modif_table;
}

ServerResultset  *
query_exec_get_data_set (QueryExec * exec)
{
	g_return_val_if_fail (exec && IS_QUERY_EXEC (exec), NULL);
	return exec->priv->sres;
}
