/* mg-conf.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "mg-conf.h"
#include "marshal.h"
#include <libxml/parser.h>
#include <libxml/tree.h>
#include "mg-base.h"
#include "mg-server.h"
#include "mg-database.h"
#include "mg-xml-storage.h"
#include "mg-query.h"
#include "mg-referer.h"
#include "mg-entity.h"
#include "mg-db-table.h"
#include "graph/mg-graph.h"
#include "mg-custom-layout.h"

/* 
 * Main static functions 
 */
static void mg_conf_class_init (MgConfClass * class);
static void mg_conf_init (MgConf * srv);
static void mg_conf_dispose (GObject   * object);
static void mg_conf_finalize (GObject   * object);

static void mg_conf_set_property (GObject              *object,
				  guint                 param_id,
				  const GValue         *value,
				  GParamSpec           *pspec);
static void mg_conf_get_property (GObject              *object,
				  guint                 param_id,
				  GValue               *value,
				  GParamSpec           *pspec);

static gboolean mg_conf_load_queries (MgConf *conf, xmlNodePtr queries, GError **error);
static gboolean mg_conf_load_graphs (MgConf *conf, xmlNodePtr graphs, GError **error);
static gboolean mg_conf_load_custom_layouts (MgConf *conf, xmlNodePtr layouts, GError **error);

static void query_id_changed_cb (MgQuery *query, MgConf *conf);
static void query_nullified_cb (MgQuery *query, MgConf *conf);
static void query_weak_ref_notify (MgConf *conf, MgQuery *query);
static void updated_query_cb (MgQuery *query, MgConf *conf);

static void graph_id_changed_cb (MgGraph *graph, MgConf *conf);
static void graph_nullified_cb (MgGraph *graph, MgConf *conf);
static void graph_weak_ref_notify (MgConf *conf, MgGraph *graph);
static void updated_graph_cb (MgGraph *graph, MgConf *conf);

static void layout_id_changed_cb (MgCustomLayout *layout, MgConf *conf);
static void layout_nullified_cb (MgCustomLayout *layout, MgConf *conf);
static void layout_weak_ref_notify (MgConf *conf, MgCustomLayout *layout);
static void updated_layout_cb (MgCustomLayout *layout, MgConf *conf);

static void conf_changed (MgConf *conf, gpointer data);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	QUERY_ADDED,
	QUERY_REMOVED,
	QUERY_UPDATED,
	GRAPH_ADDED,
	GRAPH_REMOVED,
	GRAPH_UPDATED,
	LAYOUT_ADDED,
	LAYOUT_REMOVED,
	LAYOUT_UPDATED,
	CHANGED,
	LAST_SIGNAL
};

static gint mg_conf_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0, 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP_SERIAL_QUERY,
	PROP_SERIAL_GRAPH,
	PROP_SERIAL_LAYOUT
};


struct _MgConfPrivate
{
	guint              serial_query; /* counter */
	guint              serial_graph; /* counter */
	guint              serial_layout; /* counter */

	GSList            *assumed_queries;   /* list of MgQuery objects */
	GSList            *all_queries;

	GSList            *assumed_graphs;   /* list of MgGraph objects */
	GSList            *all_graphs;

	GSList            *assumed_layouts;   /* list of MgCustomLayout objects */
	GSList            *all_layouts;

	MgDatabase        *database;
	MgServer          *srv;

	gchar             *xml_filename;
};

/* module error */
GQuark mg_conf_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("mg_conf_error");
	return quark;
}


guint
mg_conf_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgConfClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_conf_class_init,
			NULL,
			NULL,
			sizeof (MgConf),
			0,
			(GInstanceInitFunc) mg_conf_init
		};
		
		type = g_type_register_static (G_TYPE_OBJECT, "MgConf", &info, 0);
	}
	return type;
}

static void
mg_conf_class_init (MgConfClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	mg_conf_signals[QUERY_ADDED] =
		g_signal_new ("query_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, query_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[QUERY_REMOVED] =
		g_signal_new ("query_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, query_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[QUERY_UPDATED] =
		g_signal_new ("query_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, query_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[GRAPH_ADDED] =
		g_signal_new ("graph_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, graph_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[GRAPH_REMOVED] =
		g_signal_new ("graph_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, graph_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[GRAPH_UPDATED] =
		g_signal_new ("graph_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, graph_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[LAYOUT_ADDED] =
		g_signal_new ("layout_added",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, layout_added),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[LAYOUT_REMOVED] =
		g_signal_new ("layout_removed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, layout_removed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[LAYOUT_UPDATED] =
		g_signal_new ("layout_updated",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, layout_updated),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE,
			      1, G_TYPE_POINTER);
	mg_conf_signals[CHANGED] =
		g_signal_new ("changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgConfClass, changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->query_added = (void (*) (MgConf *, MgQuery *)) conf_changed;
	class->query_removed = (void (*) (MgConf *, MgQuery *)) conf_changed;
	class->query_updated = (void (*) (MgConf *, MgQuery *)) conf_changed;
	class->graph_added = (void (*) (MgConf *, MgGraph *)) conf_changed;
	class->graph_removed = (void (*) (MgConf *, MgGraph *)) conf_changed;
	class->graph_updated = (void (*) (MgConf *, MgGraph *)) conf_changed;
	class->layout_added = (void (*) (MgConf *, MgCustomLayout *)) conf_changed;
	class->layout_removed = (void (*) (MgConf *, MgCustomLayout *)) conf_changed;
	class->layout_updated = (void (*) (MgConf *, MgCustomLayout *)) conf_changed;
	class->changed = NULL;

	/* Properties */
	object_class->set_property = mg_conf_set_property;
	object_class->get_property = mg_conf_get_property;
	g_object_class_install_property (object_class, PROP_SERIAL_QUERY,
					 g_param_spec_uint ("query_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));
	g_object_class_install_property (object_class, PROP_SERIAL_GRAPH,
					 g_param_spec_uint ("graph_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));
	g_object_class_install_property (object_class, PROP_SERIAL_LAYOUT,
					 g_param_spec_uint ("layout_serial", NULL, NULL, 
							    1, G_MAXUINT, 1, G_PARAM_READABLE));

	object_class->dispose = mg_conf_dispose;
	object_class->finalize = mg_conf_finalize;
}

/* data is unused */
static void
conf_changed (MgConf *conf, gpointer data)
{
#ifdef debug_signal
	g_print (">> 'CHANGED' from %s()\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (conf), mg_conf_signals[CHANGED], 0);
#ifdef debug_signal
	g_print ("<< 'CHANGED' from %s()\n", __FUNCTION__);
#endif	
}

static void
mg_conf_init (MgConf * conf)
{
	conf->priv = g_new0 (MgConfPrivate, 1);
	conf->priv->serial_query = 1;
	conf->priv->serial_graph = 1;
	conf->priv->serial_layout = 1;
	conf->priv->assumed_queries = NULL;
	conf->priv->all_queries = NULL;
	conf->priv->assumed_graphs = NULL;
	conf->priv->all_graphs = NULL;
	conf->priv->assumed_layouts = NULL;
	conf->priv->all_layouts = NULL;
	conf->priv->database = NULL;
	conf->priv->srv = NULL;
	conf->priv->xml_filename = NULL;
}

/**
 * mg_conf_new
 *
 * Create a new #MgConf object.
 *
 * Returns: the newly created object.
 */
GObject   *
mg_conf_new ()
{
	GObject   *obj;
	MgConf *conf;

	obj = g_object_new (MG_CONF_TYPE, NULL);
	conf = MG_CONF (obj);

	/* Server and database objects creation */
	conf->priv->srv = MG_SERVER (mg_server_new (conf));
	conf->priv->database = MG_DATABASE (mg_database_new (conf));

	return obj;
}


static void
mg_conf_dispose (GObject   * object)
{
	MgConf *conf;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CONF (object));

	conf = MG_CONF (object);
	if (conf->priv) {
		GSList *list;

		/* layouts */
		list = conf->priv->all_layouts;
		while (list) {
			g_object_weak_unref (G_OBJECT (list->data), (GWeakNotify) layout_weak_ref_notify, conf);
			
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), 
							      G_CALLBACK (layout_id_changed_cb), conf);
			list = g_slist_next (list);
		}

		if (conf->priv->all_layouts) {
			g_slist_free (conf->priv->all_layouts);
			conf->priv->all_layouts = NULL;
		}

		while (conf->priv->assumed_layouts)
			mg_base_nullify (MG_BASE (conf->priv->assumed_layouts->data));


		/* graphs */
		list = conf->priv->all_graphs;
		while (list) {
			g_object_weak_unref (G_OBJECT (list->data), (GWeakNotify) graph_weak_ref_notify, conf);
			
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), 
							      G_CALLBACK (graph_id_changed_cb), conf);
			list = g_slist_next (list);
		}

		if (conf->priv->all_graphs) {
			g_slist_free (conf->priv->all_graphs);
			conf->priv->all_graphs = NULL;
		}

		while (conf->priv->assumed_graphs)
			mg_base_nullify (MG_BASE (conf->priv->assumed_graphs->data));

		/* queries */
		list = conf->priv->all_queries;
		while (list) {
			g_object_weak_unref (G_OBJECT (list->data), (GWeakNotify) query_weak_ref_notify, conf);
			
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data), 
							      G_CALLBACK (query_id_changed_cb), conf);
			list = g_slist_next (list);
		}

		if (conf->priv->all_queries) {
			g_slist_free (conf->priv->all_queries);
			conf->priv->all_queries = NULL;
		}

		while (conf->priv->assumed_queries)
			mg_base_nullify (MG_BASE (conf->priv->assumed_queries->data));

		/* database */
		if (conf->priv->database) {
			g_object_unref (G_OBJECT (conf->priv->database));
			conf->priv->database = NULL;
		}
		
		/* server */
		if (conf->priv->srv) {
			g_object_unref (G_OBJECT (conf->priv->srv));
			conf->priv->srv = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
mg_conf_finalize (GObject   * object)
{
	MgConf *conf;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CONF (object));

	conf = MG_CONF (object);
	if (conf->priv) {
		if (conf->priv->xml_filename) {
			g_free (conf->priv->xml_filename);
			conf->priv->xml_filename = NULL;
		}

		g_free (conf->priv);
		conf->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static void 
mg_conf_set_property (GObject              *object,
		      guint                 param_id,
		      const GValue         *value,
		      GParamSpec           *pspec)
{
	MgConf *mg_conf;

	mg_conf = MG_CONF (object);
	if (mg_conf->priv) {
		switch (param_id) {
		case PROP_SERIAL_QUERY:
		case PROP_SERIAL_GRAPH:
		case PROP_SERIAL_LAYOUT:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
		}
	}
}

static void
mg_conf_get_property (GObject              *object,
		       guint                 param_id,
		       GValue               *value,
		       GParamSpec           *pspec)
{
	MgConf *mg_conf;
	mg_conf = MG_CONF (object);
	
	if (mg_conf->priv) {
		switch (param_id) {
		case PROP_SERIAL_QUERY:
			g_value_set_uint (value, mg_conf->priv->serial_query++);
			break;
		case PROP_SERIAL_GRAPH:
			g_value_set_uint (value, mg_conf->priv->serial_graph++);
			break;
		case PROP_SERIAL_LAYOUT:
			g_value_set_uint (value, mg_conf->priv->serial_layout++);
			break;
		}	
	}
}


static void xml_validity_error_func (void *ctx, const char *msg, ...);

/**
 * mg_conf_load_xml_file
 * @conf: a #MgConf object
 * @xmlfile: the name of the file to which the XML will be written to
 * @error: location to store error, or %NULL
 *
 * Loads an XML file which respects the Mergeant DTD, and creates all the necessary
 * objects that are defined within the XML file. During the creation of the other
 * objects, all the normal signals are emitted.
 *
 * If the MgConf object already has some contents, then it is first of all
 * nullified (to return its state as when it was first created).
 *
 * If an error occurs during loading then the MgConf object is left as empty
 * as when it is first created.
 *
 * Returns: TRUE if loading was successfull and FALSE otherwise.
 */
gboolean
mg_conf_load_xml_file (MgConf *conf, const gchar *xmlfile, GError **error)
{
	xmlDocPtr doc;
	xmlNodePtr node, subnode;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), FALSE);
	g_return_val_if_fail (conf->priv, FALSE);
	g_return_val_if_fail (xmlfile && *xmlfile, FALSE);

	if (! g_file_test (xmlfile, G_FILE_TEST_EXISTS)) {
		g_set_error (error,
			     MG_CONF_ERROR,
			     MG_CONF_LOAD_FILE_NOT_EXIST_ERROR,
			     "File '%s' does not exist", xmlfile);
		return FALSE;
	}

	doc = xmlParseFile (xmlfile);

	if (doc) {
		/* doc validation */
		xmlValidCtxtPtr validc;

		validc = g_new0 (xmlValidCtxt, 1);
		validc->userData = conf;
		validc->error = xml_validity_error_func;
		validc->warning = NULL; 
		xmlDoValidityCheckingDefaultValue = 1;
		if (! xmlValidateDocument (validc, doc)) {
			gchar *str;

			xmlFreeDoc (doc);
			g_free (validc);
			str = g_object_get_data (G_OBJECT (conf), "xmlerror");
			if (str) {
				g_set_error (error,
					     MG_CONF_ERROR,
					     MG_CONF_FILE_LOAD_ERROR,
					     "File '%s' does not conform to DTD:\n%s", xmlfile, str);
				g_free (str);
				g_object_set_data (G_OBJECT (conf), "xmlerror", NULL);
			}
			else 
				g_set_error (error,
					     MG_CONF_ERROR,
					     MG_CONF_FILE_LOAD_ERROR,
					     "File '%s' does not conform to DTD", xmlfile);

			return FALSE;
		}
		g_free (validc);
	}
	else {
		g_set_error (error,
			     MG_CONF_ERROR,
			     MG_CONF_FILE_LOAD_ERROR,
			     "Can't load file '%s'", xmlfile);
		return FALSE;
	}

	/* doc is now OK */
	node = xmlDocGetRootElement (doc);
	if (strcmp (node->name, "MG_CONF")) {
		g_set_error (error,
			     MG_CONF_ERROR,
				     MG_CONF_FILE_LOAD_ERROR,
			     "XML file '%s' does not have any <MG_CONF> node", xmlfile);
		return FALSE;
	}
	subnode = node->children;
	
	if (!subnode) {
		g_set_error (error,
			     MG_CONF_ERROR,
			     MG_CONF_FILE_LOAD_ERROR,
			     "XML file '%s': <MG_CONF> does not have any children",
			     xmlfile);
		return FALSE;
	}

	/* MgServer object */
	if (xmlNodeIsText (subnode)) 
		subnode = subnode->next;

	if (strcmp (subnode->name, "MG_SERVER")) {
		g_set_error (error,
			     MG_CONF_ERROR,
			     MG_CONF_FILE_LOAD_ERROR,
			     "XML file '%s': <MG_SERVER> not first child of <MG_CONF>",
			     xmlfile);
		return FALSE;
	}
	if (!mg_xml_storage_load_from_xml (MG_XML_STORAGE (conf->priv->srv), subnode, error))
		return FALSE;
	
	/* MgDatabase object */
	subnode = subnode->next;
	if (xmlNodeIsText (subnode)) 
		subnode = subnode->next;
	if (!subnode || strcmp (subnode->name, "MG_DATABASE")) {
		g_set_error (error,
			     MG_CONF_ERROR,
			     MG_CONF_FILE_LOAD_ERROR,
			     "XML file '%s': <MG_DATABASE> not second child of <MG_CONF>",
			     xmlfile);
		return FALSE;
	}
	if (!mg_xml_storage_load_from_xml (MG_XML_STORAGE (conf->priv->database), subnode, error))
		return FALSE;

	/* MgQuery objects */
	subnode = subnode->next;
	if (xmlNodeIsText (subnode)) 
		subnode = subnode->next;
	if (!subnode || strcmp (subnode->name, "MG_QUERIES")) {
		g_set_error (error,
			     MG_CONF_ERROR,
			     MG_CONF_FILE_LOAD_ERROR,
			     "XML file '%s': <MG_QUERIES> not 3rd child of <MG_CONF>",
			     xmlfile);
		return FALSE;
	}
	if (!mg_conf_load_queries (conf, subnode, error))
		return FALSE;

	/* MgGraph objects */
	subnode = subnode->next;
	if (xmlNodeIsText (subnode)) 
		subnode = subnode->next;
	if (subnode && strcmp (subnode->name, "MG_GRAPHS")) {
		g_set_error (error,
			     MG_CONF_ERROR,
			     MG_CONF_FILE_LOAD_ERROR,
			     "XML file '%s': <MG_GRAPHS> not 4th child of <MG_CONF>",
			     xmlfile);
		return FALSE;
	}
	if (subnode && !mg_conf_load_graphs (conf, subnode, error))
		return FALSE;
	

	/* MgCustomLayout objects */
	subnode = subnode->next;
	if (xmlNodeIsText (subnode)) 
		subnode = subnode->next;
	if (subnode && strcmp (subnode->name, "MG_LAYOUTS")) {
		g_set_error (error,
			     MG_CONF_ERROR,
			     MG_CONF_FILE_LOAD_ERROR,
			     "XML file '%s': <MG_LAYOUTS> not 5th child of <MG_CONF>",
			     xmlfile);
		return FALSE;
	}
	if (subnode && !mg_conf_load_custom_layouts (conf, subnode, error))
		return FALSE;

	xmlFreeDoc (doc);

	return TRUE;
}

static gboolean
mg_conf_load_queries (MgConf *conf, xmlNodePtr queries, GError **error)
{
	xmlNodePtr qnode = queries->children;
	gboolean allok = TRUE;
	
	while (qnode && allok) {
		if (!strcmp (qnode->name, "MG_QUERY")) {
			MgQuery *query;

			query = MG_QUERY (mg_query_new (conf));
			allok = mg_xml_storage_load_from_xml (MG_XML_STORAGE (query), qnode, error);
			mg_conf_assume_query (conf, query);
			g_object_unref (G_OBJECT (query));
		}
		qnode = qnode->next;
	}

	if (allok) {
		GSList *list = conf->priv->assumed_queries;
		while (list) {
			mg_referer_activate (MG_REFERER (list->data));
			list= g_slist_next (list);
		}
	}

	return allok;
}

static gboolean
mg_conf_load_graphs (MgConf *conf, xmlNodePtr graphs, GError **error)
{
	xmlNodePtr qnode = graphs->children;
	gboolean allok = TRUE;
	
	while (qnode && allok) {
		if (!strcmp (qnode->name, "MG_GRAPH")) {
			MgGraph *graph;

			graph = MG_GRAPH (mg_graph_new (conf, MG_GRAPH_DB_RELATIONS)); /* type is checked next line */
			allok = mg_xml_storage_load_from_xml (MG_XML_STORAGE (graph), qnode, error);
			mg_conf_assume_graph (conf, graph);
			g_object_unref (G_OBJECT (graph));
		}
		qnode = qnode->next;
	}

	if (0 && allok) { /* the MgGraph does not implement the MgReferer interface for now */
		GSList *list = conf->priv->assumed_graphs;
		while (list) {
			mg_referer_activate (MG_REFERER (list->data));
			list= g_slist_next (list);
		}
	}

	return allok;
}

static gboolean
mg_conf_load_custom_layouts (MgConf *conf, xmlNodePtr layouts, GError **error)
{
	xmlNodePtr qnode = layouts->children;
	gboolean allok = TRUE;

	while (qnode && allok) {
		MgCustomLayout *cl;

		if (! xmlNodeIsText (qnode)) {
			cl = (MgCustomLayout *) mg_custom_layout_new (conf);
			allok = mg_xml_storage_load_from_xml (MG_XML_STORAGE (cl), qnode, error);
			mg_conf_assume_layout (conf, cl);
			g_object_unref (G_OBJECT (cl));
		}
		    
		qnode = qnode->next;
	}

	if (0 && allok) { /* the MgGraph does not implement the MgReferer interface for now */
		GSList *list = conf->priv->assumed_layouts;
		while (list) {
			mg_referer_activate (MG_REFERER (list->data));
			list= g_slist_next (list);
		}
	}

	return allok;
}

/*
 * function called when an error occurred during the document validation
 */
static void
xml_validity_error_func (void *ctx, const char *msg, ...)
{
        xmlValidCtxtPtr pctxt;
        va_list args;
        gchar *str, *str2, *newerr;
	MgConf *conf;

	pctxt = (xmlValidCtxtPtr) ctx;
	/* FIXME: it looks like libxml does set ctx to userData... */
	/*conf = MG_CONF (pctxt->userData);*/
	conf = MG_CONF (pctxt);
	str2 = g_object_get_data (G_OBJECT (conf), "xmlerror");

        va_start (args, msg);
        str = g_strdup_vprintf (msg, args);
        va_end (args);
	
	if (str2) {
		newerr = g_strdup_printf ("%s\n%s", str2, str);
		g_free (str2);
	}
	else
		newerr = g_strdup (str);
        g_free (str);
	g_object_set_data (G_OBJECT (conf), "xmlerror", newerr);
}


/**
 * mg_conf_save_xml_file
 * @conf: a #MgConf object
 * @xmlfile: the name of the file to which the XML will be written to
 * @error: location to store error, or %NULL
 *
 * Saves the contents of a MgConf object to a file which is given as argument.
 *
 * Returns: TRUE if saving was successfull and FALSE otherwise.
 */
gboolean
mg_conf_save_xml_file (MgConf *conf, const gchar *xmlfile, GError **error)
{
	gboolean retval = TRUE;
	xmlDocPtr doc;
#define XML_LIBMG_DTD_FILE DTDINSTALLDIR"/libmergeant.dtd"

	g_return_val_if_fail (conf && IS_MG_CONF (conf), FALSE);
	g_return_val_if_fail (conf->priv, FALSE);
		
	doc = xmlNewDoc ("1.0");
	if (doc) {
		xmlNodePtr topnode, node;

		/* DTD insertion */
                xmlCreateIntSubset(doc, "MG_CONF", NULL, XML_LIBMG_DTD_FILE);

		/* Top node */
		topnode = xmlNewDocNode (doc, NULL, "MG_CONF", NULL);
		xmlDocSetRootElement (doc, topnode);

		/* MgServer */
		node = mg_xml_storage_save_to_xml (MG_XML_STORAGE (conf->priv->srv), error);
		if (node)
			xmlAddChild (topnode, node);
		else 
			/* error handling */
			retval = FALSE;
		
		/* MgDatabase */
		if (retval) {
			node = mg_xml_storage_save_to_xml (MG_XML_STORAGE (conf->priv->database), error);
			if (node)
				xmlAddChild (topnode, node);
			else 
				/* error handling */
				retval = FALSE;
		}
		
		/* MgQuery objects */
		if (retval) {
			GSList *list;
			node = xmlNewChild (topnode, NULL, "MG_QUERIES", NULL);
			list = conf->priv->assumed_queries;
			while (list) {
				if (!mg_query_get_parent_query (MG_QUERY (list->data))) {
					/* We only add nodes for queries which do not have any parent query */
					xmlNodePtr qnode;
					
					qnode = mg_xml_storage_save_to_xml (MG_XML_STORAGE (list->data), error);
					if (qnode)
						xmlAddChild (node, qnode);
					else 
						/* error handling */
						retval = FALSE;
				}
				list = g_slist_next(list);
			}
		}

		/* MgGraph objects */
		if (retval) {
			GSList *list;
			node = xmlNewChild (topnode, NULL, "MG_GRAPHS", NULL);
			list = conf->priv->assumed_graphs;
			while (list) {
				xmlNodePtr qnode;
				
				qnode = mg_xml_storage_save_to_xml (MG_XML_STORAGE (list->data), error);
				if (qnode)
					xmlAddChild (node, qnode);
				else 
					/* error handling */
					retval = FALSE;

				list = g_slist_next(list);
			}
		}

		/* MgCustomLayout objects */
		if (retval) {
			GSList *list;
			node = xmlNewChild (topnode, NULL, "MG_LAYOUTS", NULL);
			list = conf->priv->assumed_layouts;
			while (list) {
				xmlNodePtr qnode;
				
				qnode = mg_xml_storage_save_to_xml (MG_XML_STORAGE (list->data), error);
				if (qnode)
					xmlAddChild (node, qnode);
				else 
					/* error handling */
					retval = FALSE;

				list = g_slist_next(list);
			}
		}

		/* actual writing to file */
		if (retval) {
			gint i;
			i = xmlSaveFormatFile (xmlfile, doc, TRUE);
			if (i == -1) {
				/* error handling */
				g_set_error (error, MG_CONF_ERROR, MG_CONF_FILE_SAVE_ERROR,
					     _("Error writing XML file %s"), xmlfile);
				retval = FALSE;
			}
		}

		/* getting rid of the doc */
		xmlFreeDoc (doc);
	}
	else {
		/* error handling */
		g_set_error (error, MG_CONF_ERROR, MG_CONF_FILE_SAVE_ERROR,
			     _("Can't allocate memory for XML structure."));
		retval = FALSE;
	}

	return retval;
}


/**
 * mg_conf_declare_query
 * @conf: a #MgConf object
 * @query: a #MgQuery object
 *
 * Declares the existence of a new query to @conf. All the #MgQuery objects MUST
 * be declared to the corresponding #MgConf object for the library to work correctly.
 * Once @query has been declared, @conf does not hold any reference to @query. If @conf
 * must hold such a reference, then use mg_conf_assume_query().
 *
 * This functions is called automatically from each mg_query_new* function, and it should not be necessary
 * to call it except for classes extending the #MgQuery class.
 */
void
mg_conf_declare_query (MgConf *conf, MgQuery *query)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);
	g_return_if_fail (query && IS_MG_QUERY (query));
	
	/* we don't take any reference on query */
	if (g_slist_find (conf->priv->all_queries, query))
		return;	
	else {
		conf->priv->all_queries = g_slist_append (conf->priv->all_queries, query);
		g_object_weak_ref (G_OBJECT (query), (GWeakNotify) query_weak_ref_notify, conf);

		/* make sure the conf->priv->serial_query value is always 1 above this query's id */
		query_id_changed_cb (query, conf);
		g_signal_connect (G_OBJECT (query), "id_changed",
				  G_CALLBACK (query_id_changed_cb), conf);
	}
}

static void
query_id_changed_cb (MgQuery *query, MgConf *conf)
{
	if (conf->priv->serial_query <= mg_base_get_id (MG_BASE (query)))
		conf->priv->serial_query = mg_base_get_id (MG_BASE (query)) + 1;
}


static void
query_weak_ref_notify (MgConf *conf, MgQuery *query)
{
	conf->priv->all_queries = g_slist_remove (conf->priv->all_queries, query);
	g_signal_handlers_disconnect_by_func (G_OBJECT (query),
					      G_CALLBACK (query_id_changed_cb), conf);
}

static void
updated_query_cb (MgQuery *query, MgConf *conf)
{
#ifdef debug_signal
	g_print (">> 'QUERY_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (conf), "query_updated", query);
#ifdef debug_signal
	g_print ("<< 'QUERY_UPDATED' from %s\n", __FUNCTION__);
#endif	
}

/**
 * mg_conf_assume_query
 * @conf: a #MgConf object
 * @query: a #MgQuery object
 *
 * Force @conf to manage @query: it will get a reference to it.
 */
void
mg_conf_assume_query (MgConf *conf, MgQuery *query)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);
	g_return_if_fail (query && IS_MG_QUERY (query));
	
	if (g_slist_find (conf->priv->assumed_queries, query)) {
		g_warning ("MgQuery %p already assumed!", query);
		return;
	}

	mg_conf_declare_query (conf, query);

	conf->priv->assumed_queries = g_slist_append (conf->priv->assumed_queries, query);
	g_object_ref (G_OBJECT (query));
	g_signal_connect (G_OBJECT (query), "nullified",
			  G_CALLBACK (query_nullified_cb), conf);
	g_signal_connect (G_OBJECT (query), "changed",
			  G_CALLBACK (updated_query_cb), conf);

#ifdef debug_signal
	g_print (">> 'QUERY_ADDED' from mg_conf_assume_query\n");
#endif
	g_signal_emit (G_OBJECT (conf), mg_conf_signals[QUERY_ADDED], 0, query);
#ifdef debug_signal
	g_print ("<< 'QUERY_ADDED' from mg_conf_assume_query\n");
#endif
}


/* called when a MgQuery is "nullified" */
static void 
query_nullified_cb (MgQuery *query, MgConf *conf)
{
	mg_conf_unassume_query (conf, query);
}


/**
 * mg_conf_unassume_query
 * @conf: a #MgConf object
 * @query: a #MgQuery object
 *
 * Forces @conf to lose a reference it has on @query
 */
void
mg_conf_unassume_query (MgConf *conf, MgQuery *query)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);

	if (g_slist_find (conf->priv->assumed_queries, query)) {
		conf->priv->assumed_queries = g_slist_remove (conf->priv->assumed_queries, query);
		g_signal_handlers_disconnect_by_func (G_OBJECT (query),
						      G_CALLBACK (query_nullified_cb), conf);
		g_signal_handlers_disconnect_by_func (G_OBJECT (query),
						      G_CALLBACK (updated_query_cb), conf);
#ifdef debug_signal
		g_print (">> 'QUERY_REMOVED' from mg_conf_unassume_query\n");
#endif
		g_signal_emit (G_OBJECT (conf), mg_conf_signals[QUERY_REMOVED], 0, query);
#ifdef debug_signal
		g_print ("<< 'QUERY_REMOVED' from mg_conf_unassume_query\n");
#endif
		g_object_unref (G_OBJECT (query));
	}
}


/**
 * mg_conf_get_queries
 * @conf: a #MgConf object
 *
 * Get a list of all the non interdependant queries managed by @conf
 * (only queries with no parent query are listed)
 *
 * Returns: a new list of #MgQuery objects
 */
GSList *
mg_conf_get_queries (MgConf *conf)
{
	GSList *list, *retval = NULL;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (conf->priv, NULL);

	list = conf->priv->assumed_queries;
	while (list) {
		if (! mg_query_get_parent_query (MG_QUERY (list->data)))
			retval = g_slist_append (retval, list->data);
		list = g_slist_next (list);
	}

	return retval;
}

/**
 * mg_conf_get_query_by_xml_id
 * @conf: a #MgConf object
 * @xml_id: the XML Id of the query being searched
 *
 * Find a #MgQuery object from its XML Id
 *
 * Returns: the #MgQuery object, or NULL if not found
 */
MgQuery *
mg_conf_get_query_by_xml_id (MgConf *conf, const gchar *xml_id)
{
	MgQuery *query = NULL;

	GSList *list;
        gchar *str;

        g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
        g_return_val_if_fail (conf->priv, NULL);

        list = conf->priv->all_queries;
        while (list && !query) {
                str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (list->data));
                if (!strcmp (str, xml_id))
                        query = MG_QUERY (list->data);
                g_free (str);
                list = g_slist_next (list);
        }

        return query;
}

/**
 * mg_conf_declare_graph
 * @conf: a #MgConf object
 * @graph: a #MgGraph object
 *
 * Declares the existence of a new graph to @conf. All the #MgGraph objects MUST
 * be declared to the corresponding #MgConf object for the library to work correctly.
 * Once @graph has been declared, @conf does not hold any reference to @graph. If @conf
 * must hold such a reference, then use mg_conf_assume_graph().
 *
 * This functions is called automatically from each mg_graph_new* function, and it should not be necessary
 * to call it except for classes extending the #MgGraph class.
 */
void
mg_conf_declare_graph (MgConf *conf, MgGraph *graph)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);
	g_return_if_fail (graph && IS_MG_GRAPH (graph));
	
	/* we don't take any reference on graph */
	if (g_slist_find (conf->priv->all_graphs, graph))
		return;	
	else {
		conf->priv->all_graphs = g_slist_append (conf->priv->all_graphs, graph);
		g_object_weak_ref (G_OBJECT (graph), (GWeakNotify) graph_weak_ref_notify, conf);

		/* make sure the conf->priv->serial_graph value is always 1 above this graph's id */
		graph_id_changed_cb (graph, conf);
		g_signal_connect (G_OBJECT (graph), "id_changed",
				  G_CALLBACK (graph_id_changed_cb), conf);
	}
}

static void
graph_id_changed_cb (MgGraph *graph, MgConf *conf)
{
	if (conf->priv->serial_graph <= mg_base_get_id (MG_BASE (graph)))
		conf->priv->serial_graph = mg_base_get_id (MG_BASE (graph)) + 1;
}


static void
graph_weak_ref_notify (MgConf *conf, MgGraph *graph)
{
	conf->priv->all_graphs = g_slist_remove (conf->priv->all_graphs, graph);
	g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
					      G_CALLBACK (graph_id_changed_cb), conf);
}

static void
updated_graph_cb (MgGraph *graph, MgConf *conf)
{
#ifdef debug_signal
	g_print (">> 'GRAPH_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (conf), "graph_updated", graph);
#ifdef debug_signal
	g_print ("<< 'GRAPH_UPDATED' from %s\n", __FUNCTION__);
#endif
}



/**
 * mg_conf_assume_graph
 * @conf: a #MgConf object
 * @graph: a #MgGraph object
 *
 * Force @conf to manage @graph: it will get a reference to it.
 */
void
mg_conf_assume_graph (MgConf *conf, MgGraph *graph)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);
	g_return_if_fail (graph && IS_MG_GRAPH (graph));
	
	if (g_slist_find (conf->priv->assumed_graphs, graph)) {
		g_warning ("MgGraph %p already assumed!", graph);
		return;
	}

	mg_conf_declare_graph (conf, graph);

	conf->priv->assumed_graphs = g_slist_append (conf->priv->assumed_graphs, graph);
	g_object_ref (G_OBJECT (graph));
	g_signal_connect (G_OBJECT (graph), "nullified",
			  G_CALLBACK (graph_nullified_cb), conf);
	g_signal_connect (G_OBJECT (graph), "changed",
			  G_CALLBACK (updated_graph_cb), conf);

#ifdef debug_signal
	g_print (">> 'GRAPH_ADDED' from mg_conf_assume_graph\n");
#endif
	g_signal_emit (G_OBJECT (conf), mg_conf_signals[GRAPH_ADDED], 0, graph);
#ifdef debug_signal
	g_print ("<< 'GRAPH_ADDED' from mg_conf_assume_graph\n");
#endif
}


/* called when a MgGraph is "nullified" */
static void 
graph_nullified_cb (MgGraph *graph, MgConf *conf)
{
	mg_conf_unassume_graph (conf, graph);
}


/**
 * mg_conf_unassume_graph
 * @conf: a #MgConf object
 * @graph: a #MgGraph object
 *
 * Forces @conf to lose a reference it has on @graph
 */
void
mg_conf_unassume_graph (MgConf *conf, MgGraph *graph)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);

	if (g_slist_find (conf->priv->assumed_graphs, graph)) {
		conf->priv->assumed_graphs = g_slist_remove (conf->priv->assumed_graphs, graph);
		g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
						      G_CALLBACK (graph_nullified_cb), conf);
		g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
						      G_CALLBACK (updated_graph_cb), conf);
#ifdef debug_signal
		g_print (">> 'GRAPH_REMOVED' from mg_conf_unassume_graph\n");
#endif
		g_signal_emit (G_OBJECT (conf), mg_conf_signals[GRAPH_REMOVED], 0, graph);
#ifdef debug_signal
		g_print ("<< 'GRAPH_REMOVED' from mg_conf_unassume_graph\n");
#endif
		g_object_unref (G_OBJECT (graph));
	}
}


/**
 * mg_conf_get_graphs
 * @conf: a #MgConf object
 *
 * Get a list of all the graphs managed by @conf
 *
 * Returns: a new list of #MgGraph objects
 */
GSList *
mg_conf_get_graphs (MgConf *conf)
{
	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (conf->priv, NULL);

	if (conf->priv->assumed_graphs)
		return g_slist_copy (conf->priv->assumed_graphs);
	else
		return NULL;
}

/**
 * mg_conf_get_graph_by_xml_id
 * @conf: a #MgConf object
 * @xml_id: the XML Id of the graph being searched
 *
 * Find a #MgGraph object from its XML Id
 *
 * Returns: the #MgGraph object, or NULL if not found
 */
MgGraph *
mg_conf_get_graph_by_xml_id (MgConf *conf, const gchar *xml_id)
{
	MgGraph *graph = NULL;

	GSList *list;
        gchar *str;

        g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
        g_return_val_if_fail (conf->priv, NULL);

        list = conf->priv->all_graphs;
        while (list && !graph) {
                str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (list->data));
                if (!strcmp (str, xml_id))
                        graph = MG_GRAPH (list->data);
                g_free (str);
                list = g_slist_next (list);
        }

        return graph;
}

/**
 * mg_conf_get_graph_for_object
 * @conf: a #MgConf object
 * @obj: a #Gobject object
 *
 * Find a #MgGraph object guiven the object it is related to.
 *
 * Returns: the #MgGraph object, or NULL if not found
 */
MgGraph *
mg_conf_get_graph_for_object (MgConf *conf, GObject *obj)
{
	MgGraph *graph = NULL;

	GSList *list;
	GObject *ref_obj;

        g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
        g_return_val_if_fail (conf->priv, NULL);

        list = conf->priv->all_graphs;
        while (list && !graph) {
		g_object_get (G_OBJECT (list->data), "ref_object", &ref_obj, NULL);
		if (ref_obj == obj)
                        graph = MG_GRAPH (list->data);
                list = g_slist_next (list);
        }

        return graph;
}

/**
 * mg_conf_declare_layout
 * @conf: a #MgConf object
 * @layout: a #MgLayout object
 *
 * Declares the existence of a new layout to @conf. All the #MgCustomLayout objects MUST
 * be declared to the corresponding #MgConf object for the library to work correctly.
 * Once @layout has been declared, @conf does not hold any reference to @layout. If @conf
 * must hold such a reference, then use mg_conf_assume_layout().
 *
 * This functions is called automatically from each mg_layout_new* function, and it should not be necessary
 * to call it except for classes extending the #MgCustomLayout class.
 */
void
mg_conf_declare_layout (MgConf *conf, MgCustomLayout *layout)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);
	g_return_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout));
	
	/* we don't take any reference on layout */
	if (g_slist_find (conf->priv->all_layouts, layout))
		return;	
	else {
		conf->priv->all_layouts = g_slist_append (conf->priv->all_layouts, layout);
		g_object_weak_ref (G_OBJECT (layout), (GWeakNotify) layout_weak_ref_notify, conf);

		/* make sure the conf->priv->serial_layout value is always 1 above this layout's id */
		layout_id_changed_cb (layout, conf);
		g_signal_connect (G_OBJECT (layout), "id_changed",
				  G_CALLBACK (layout_id_changed_cb), conf);
	}
}

/**
 * mg_conf_assume_layout
 * @conf: a #MgConf object
 * @layout: a #MgCustomLayout object
 *
 * Force @conf to manage @layout: it will get a reference to it.
 */
void
mg_conf_assume_layout (MgConf *conf, MgCustomLayout *layout)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);
	g_return_if_fail (layout && IS_MG_CUSTOM_LAYOUT (layout));
	
	if (g_slist_find (conf->priv->assumed_layouts, layout)) {
		g_warning ("MgCustomLayout %p already assumed!", layout);
		return;
	}

	mg_conf_declare_layout (conf, layout);

	conf->priv->assumed_layouts = g_slist_append (conf->priv->assumed_layouts, layout);
	g_object_ref (G_OBJECT (layout));
	g_signal_connect (G_OBJECT (layout), "nullified",
			  G_CALLBACK (layout_nullified_cb), conf);
	g_signal_connect (G_OBJECT (layout), "changed",
			  G_CALLBACK (updated_layout_cb), conf);

#ifdef debug_signal
	g_print (">> 'LAYOUT_ADDED' from mg_conf_assume_layout\n");
#endif
	g_signal_emit (G_OBJECT (conf), mg_conf_signals[LAYOUT_ADDED], 0, layout);
#ifdef debug_signal
	g_print ("<< 'LAYOUT_ADDED' from mg_conf_assume_layout\n");
#endif
}

/**
 * mg_conf_unassume_layout
 * @conf: a #MgConf object
 * @layout: a #MgCustomLayout object
 *
 * Forces @conf to lose a reference it has on @layout
 */
void
mg_conf_unassume_layout (MgConf *conf, MgCustomLayout *layout)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);

	if (g_slist_find (conf->priv->assumed_layouts, layout)) {
		conf->priv->assumed_layouts = g_slist_remove (conf->priv->assumed_layouts, layout);
		g_signal_handlers_disconnect_by_func (G_OBJECT (layout),
						      G_CALLBACK (layout_nullified_cb), conf);
		g_signal_handlers_disconnect_by_func (G_OBJECT (layout),
						      G_CALLBACK (updated_layout_cb), conf);
#ifdef debug_signal
		g_print (">> 'LAYOUT_REMOVED' from mg_conf_unassume_layout\n");
#endif
		g_signal_emit (G_OBJECT (conf), mg_conf_signals[LAYOUT_REMOVED], 0, layout);
#ifdef debug_signal
		g_print ("<< 'LAYOUT_REMOVED' from mg_conf_unassume_layout\n");
#endif
		g_object_unref (G_OBJECT (layout));
	}
}

/**
 * mg_conf_get_layouts
 * @conf: a #MgConf object
 *
 * Get a list of all the layouts managed by @conf
 *
 * Returns: a new list of #MgCustomLayout objects
 */
GSList *
mg_conf_get_layouts (MgConf *conf)
{
	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (conf->priv, NULL);

	if (conf->priv->assumed_layouts)
		return g_slist_copy (conf->priv->assumed_layouts);
	else
		return NULL;
}

/**
 * mg_conf_get_layout_by_xml_id
 * @conf: a #MgConf object
 * @xml_id: the XML Id of the layout being searched
 *
 * Find a #MgCustomLayout object from its XML Id
 *
 * Returns: the #MgCustomLayout object, or NULL if not found
 */
MgCustomLayout *
mg_conf_get_layout_by_xml_id (MgConf *conf, const gchar *xml_id)
{
	MgCustomLayout *layout = NULL;

	GSList *list;
        gchar *str;

        g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
        g_return_val_if_fail (conf->priv, NULL);

        list = conf->priv->all_layouts;
        while (list && !layout) {
                str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (list->data));
                if (!strcmp (str, xml_id))
                        layout = MG_CUSTOM_LAYOUT (list->data);
                g_free (str);
                list = g_slist_next (list);
        }

        return layout;
}


static void
layout_id_changed_cb (MgCustomLayout *layout, MgConf *conf)
{
	if (conf->priv->serial_layout <= mg_base_get_id (MG_BASE (layout)))
		conf->priv->serial_layout = mg_base_get_id (MG_BASE (layout)) + 1;
}

static void
layout_nullified_cb (MgCustomLayout *layout, MgConf *conf)
{
	mg_conf_unassume_layout (conf, layout);
}

static void
layout_weak_ref_notify (MgConf *conf, MgCustomLayout *layout)
{
	conf->priv->all_layouts = g_slist_remove (conf->priv->all_layouts, layout);
	g_signal_handlers_disconnect_by_func (G_OBJECT (layout),
					      G_CALLBACK (layout_id_changed_cb), conf);
}

static void
updated_layout_cb (MgCustomLayout *layout, MgConf *conf)
{
#ifdef debug_signal
	g_print (">> 'LAYOUT_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (conf), "layout_updated", layout);
#ifdef debug_signal
	g_print ("<< 'LAYOUT_UPDATED' from %s\n", __FUNCTION__);
#endif	
}


/**
 * mg_conf_get_server
 * @conf: a #MgConf object
 *
 * Fetch a pointer to the MgServer used by the MgConf object.
 *
 * Returns: a pointer to the MgServer
 */
MgServer *
mg_conf_get_server (MgConf *conf)
{
	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (conf->priv, NULL);

	return conf->priv->srv;
}

/**
 * mg_conf_get_database
 * @conf: a #MgConf object
 *
 * Fetch a pointer to the MgDatabase used by the MgConf object.
 *
 * Returns: a pointer to the MgDatabase
 */
MgDatabase *
mg_conf_get_database (MgConf *conf)
{
	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (conf->priv, NULL);

	return conf->priv->database;
}

#ifdef debug
/**
 * mg_conf_dump
 * @conf: a #MgConf object
 *
 * Dumps the whole dictionnary managed by the MgConf object
 */
void
mg_conf_dump (MgConf *conf)
{
	GSList *list;

	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);

	g_print ("\n----------------- DUMPING START -----------------\n");
	g_print (D_COL_H1 "MgConf %p\n" D_COL_NOR, conf);
	if (conf->priv->srv) 
		mg_server_dump (conf->priv->srv, 0);

	if (conf->priv->database)
		mg_base_dump (MG_BASE (conf->priv->database), 0);

	list = conf->priv->assumed_queries;
	if (list)
		g_print ("Queries:\n");
	else
		g_print ("No Query defined\n");
	while (list) {
		if (!mg_query_get_parent_query (MG_QUERY (list->data)))
			mg_base_dump (MG_BASE (list->data), 0);
		list = g_slist_next (list);
	}

	list = conf->priv->assumed_graphs;
	if (list)
		g_print ("Graphs:\n");
	else
		g_print ("No Graph defined\n");
	while (list) {
		mg_base_dump (MG_BASE (list->data), 0);
		list = g_slist_next (list);
	}

	list = conf->priv->assumed_layouts;
	if (list)
		g_print ("Layouts:\n");
	else
		g_print ("No Layout defined\n");
	while (list) {
		mg_base_dump (MG_BASE (list->data), 0);
		list = g_slist_next (list);
	}

	g_print ("----------------- DUMPING END -----------------\n\n");
}
#endif


/**
 * mg_conf_compute_xml_filename
 * @conf: a #MgConf object
 * @datasource: a data source
 * @app_id: an extra identification, or %NULL
 * @error: location to store error, or %NULL
 *
 * Get the prefered filename which represents the data dictionnary associated to the @datasource data source.
 * Using the returned value in conjunction with mg_conf_load_xml_file() and mg_conf_save_xml_file has
 * the advantage of letting the library handle file naming onventions.
 *
 * The @app_id argument allows to give an extra identification to the request, when some special features
 * must be saved but not interfere with the default dictionnary.
 *
 * Returns: a new string
 */
gchar *
mg_conf_compute_xml_filename (MgConf *conf, const gchar *datasource, const gchar *app_id, GError **error)
{
	gchar *str;
	gboolean with_error = FALSE;

/* REM: for now @conf is not directly used, but another version could make use of it, so we still keep it as
 *      an argument */

/* same as for libgda */
#define LIBGDA_USER_CONFIG_DIR G_DIR_SEPARATOR_S ".libgda"

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (conf->priv, NULL);
	g_return_val_if_fail (datasource && *datasource, NULL);

	if (!app_id)
		str = g_strdup_printf ("%s%sDICT_%s_default.xml", g_get_home_dir (), LIBGDA_USER_CONFIG_DIR G_DIR_SEPARATOR_S,
				       datasource);
	else
		str = g_strdup_printf ("%s%sDICT_%s_%s.xml", g_get_home_dir (), LIBGDA_USER_CONFIG_DIR G_DIR_SEPARATOR_S,
				       datasource, app_id);

	/* create an empty file with that name */
	if (!g_file_test (str, G_FILE_TEST_EXISTS)) {
		gchar *dirpath;
		FILE *fp;
		
		dirpath = g_strdup_printf ("%s%s", g_get_home_dir (), LIBGDA_USER_CONFIG_DIR);
		if (!g_file_test (dirpath, G_FILE_TEST_IS_DIR)){
			if (mkdir (dirpath, 0700)) {
				g_set_error (error,
					     MG_CONF_PROPOSED_FILE,
					     MG_CONF_FILE_LOAD_ERROR,
					     _("Error creating directory %s"), dirpath);
				with_error = TRUE;
			}
		}
		g_free (dirpath);

		/* fp = fopen (str, "wt"); */
/* 		if (fp == NULL) { */
/* 			g_set_error (error, */
/* 				     MG_CONF_PROPOSED_FILE, */
/* 				     MG_CONF_FILE_LOAD_ERROR, */
/* 				     _("Unable to create the dictionnary file %s"), str); */
/* 			with_error = TRUE; */
/* 		} */
/* 		else */
/* 			fclose (fp); */
	}

	if (with_error) {
		g_free (str);
		str = NULL;
	}

	return str;
}


/**
 * mg_conf_set_xml_filename
 * @conf: a #MgConf object
 * @xmlfile: a file name
 *
 * Sets the filename @conf will use when mg_conf_save_xml() and mg_conf_load_xml() are called.
 */
void
mg_conf_set_xml_filename (MgConf *conf, const gchar *xmlfile)
{
	g_return_if_fail (conf && IS_MG_CONF (conf));
	g_return_if_fail (conf->priv);

	if (conf->priv->xml_filename) {
		g_free (conf->priv->xml_filename);
		conf->priv->xml_filename = NULL;
	}
	
	if (xmlfile)
		conf->priv->xml_filename = g_strdup (xmlfile);
}

/**
 * mg_conf_get_xml_filename
 * @conf: a #MgConf object
 *
 * Get the filename @conf will use when mg_conf_save_xml() and mg_conf_load_xml() are called.
 *
 * Returns: the filename, or %NULL if none have been set.
 */
const gchar *
mg_conf_get_xml_filename (MgConf *conf)
{
	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (conf->priv, NULL);

	return conf->priv->xml_filename;
}

/**
 * mg_conf_load_xml
 * @conf: a #MgConf object
 * @error: location to store error, or %NULL
 * 
 * Loads an XML file which respects the Mergeant DTD, and creates all the necessary
 * objects that are defined within the XML file. During the creation of the other
 * objects, all the normal signals are emitted.
 *
 * If the MgConf object already has some contents, then it is first of all
 * nullified (to return its state as when it was first created).
 *
 * If an error occurs during loading then the MgConf object is left as empty
 * as when it is first created.
 *
 * The file loaded is the one specified using mg_conf_set_xml_filename()
 *
 * Returns: TRUE if loading was successfull and FALSE otherwise.
 */
gboolean
mg_conf_load_xml (MgConf *conf, GError **error)
{
	g_return_val_if_fail (conf && IS_MG_CONF (conf), FALSE);
	g_return_val_if_fail (conf->priv, FALSE);

	return mg_conf_load_xml_file (conf, conf->priv->xml_filename, error);
}

/**
 * mg_conf_save_xml
 * @conf: a #MgConf object
 * @error: location to store error, or %NULL
 *
 * Saves the contents of a MgConf object to a file which is specified using the
 * mg_conf_set_xml_filename() method.
 *
 * Returns: TRUE if saving was successfull and FALSE otherwise.
 */
gboolean
mg_conf_save_xml (MgConf *conf, GError **error)
{
	g_return_val_if_fail (conf && IS_MG_CONF (conf), FALSE);
	g_return_val_if_fail (conf->priv, FALSE);

	return mg_conf_save_xml_file (conf, conf->priv->xml_filename, error);
}


/**
 * mg_conf_get_entities_fk_constraints
 * @conf: a #MgConf object
 * @entity1: an object implementing the #MgEntity interface
 * @entity2: an object implementing the #MgEntity interface
 * @entity1_has_fk: TRUE if the returned constraints are the one for which @entity1 contains the foreign key
 *
 * Get a list of all the constraints which represent a foreign constrains, between
 * @entity1 and @entity2. If @entity1 and @entity2 are #MgDbTable objects, then the
 * constraints are the ones from the database.
 *
 * Constraints are represented as #MgDbConstraint objects.
 *
 * Returns: a new list of the constraints
 */
GSList *
mg_conf_get_entities_fk_constraints (MgConf *conf, MgEntity *entity1, MgEntity *entity2,
				     gboolean entity1_has_fk)
{
	GSList *retval = NULL;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (conf->priv, NULL);
	g_return_val_if_fail (entity1 && IS_MG_ENTITY (entity1), NULL);
	g_return_val_if_fail (entity2 && IS_MG_ENTITY (entity2), NULL);
	if (entity1 == entity2)
		return NULL;

	if (IS_MG_DB_TABLE (entity1)) {
		if (IS_MG_DB_TABLE (entity2)) 
			retval = mg_database_get_tables_fk_constraints (conf->priv->database, 
									MG_DB_TABLE (entity1), MG_DB_TABLE (entity2),
									entity1_has_fk);
		else 
			TO_IMPLEMENT;
	}
	else 
		TO_IMPLEMENT;

	return retval;
}
