/* mg-canvas-fkconstraint.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include "mg-canvas.h"
#include "mg-canvas-fkconstraint.h"
#include "mg-canvas-entity.h"
#include "mg-canvas-text.h"
#include <libmergeant/mg-field.h>
#include <libmergeant/mg-db-constraint.h>

static void mg_canvas_fkconstraint_class_init (MgCanvasFkconstraintClass * class);
static void mg_canvas_fkconstraint_init       (MgCanvasFkconstraint * cc);
static void mg_canvas_fkconstraint_dispose    (GObject   * object);
static void mg_canvas_fkconstraint_finalize   (GObject   * object);

static void mg_canvas_fkconstraint_set_property    (GObject              *object,
						    guint                 param_id,
						    const GValue         *value,
						    GParamSpec           *pspec);
static void mg_canvas_fkconstraint_get_property    (GObject              *object,
						    guint                 param_id,
						    GValue               *value,
						    GParamSpec           *pspec);

static void clean_items (MgCanvasFkconstraint *cc);
static void create_items (MgCanvasFkconstraint *cc);

enum
{
	PROP_0,
	PROP_FK_CONSTRAINT
};

struct _MgCanvasFkconstraintPrivate
{
	GSList           *constraints;
	MgCanvasEntity   *fk_entity_item;
	MgCanvasEntity   *ref_pk_entity_item;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
mg_canvas_fkconstraint_get_type (void)
{
	static GType type = 0;

        if (!type) {
		static const GTypeInfo info = {
			sizeof (MgCanvasFkconstraintClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_canvas_fkconstraint_class_init,
			NULL,
			NULL,
			sizeof (MgCanvasFkconstraint),
			0,
			(GInstanceInitFunc) mg_canvas_fkconstraint_init
		};		

		type = g_type_register_static (MG_CANVAS_ITEM_TYPE, "MgCanvasFkconstraint", &info, 0);
	}

	return type;
}	

static void
mg_canvas_fkconstraint_class_init (MgCanvasFkconstraintClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = mg_canvas_fkconstraint_dispose;
	object_class->finalize = mg_canvas_fkconstraint_finalize;

	/* Properties */
	object_class->set_property = mg_canvas_fkconstraint_set_property;
	object_class->get_property = mg_canvas_fkconstraint_get_property;
	g_object_class_install_property (object_class, PROP_FK_CONSTRAINT,
					 g_param_spec_pointer ("fk_constraint", "Latest FK constraint to add", 
							       NULL, G_PARAM_WRITABLE));
       
}

static void
mg_canvas_fkconstraint_init (MgCanvasFkconstraint *cc)
{
	cc->priv = g_new0 (MgCanvasFkconstraintPrivate, 1);
	cc->priv->constraints = NULL;
	cc->priv->fk_entity_item = NULL;
	cc->priv->ref_pk_entity_item = NULL;
}


static void entity_destroy_cb (MgCanvasEntity *entity, MgCanvasFkconstraint *cc);
static void constraint_nullified_cb (MgDbConstraint *fkcons, MgCanvasFkconstraint *cc);

static void
mg_canvas_fkconstraint_dispose (GObject *object)
{
	MgCanvasFkconstraint *cc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CANVAS_FKCONSTRAINT (object));

	cc = MG_CANVAS_FKCONSTRAINT (object);

	clean_items (cc);
	if (cc->priv->constraints) {
		GSList *list = cc->priv->constraints;
		while (list) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
							      G_CALLBACK (constraint_nullified_cb), cc);
			list = g_slist_next (list);
		}
		g_slist_free (cc->priv->constraints);
		cc->priv->constraints = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}


static void
mg_canvas_fkconstraint_finalize (GObject  *object)
{
	MgCanvasFkconstraint *cc;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CANVAS_FKCONSTRAINT (object));

	cc = MG_CANVAS_FKCONSTRAINT (object);
	if (cc->priv) {
		g_free (cc->priv);
		cc->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

static void 
mg_canvas_fkconstraint_set_property    (GObject              *object,
					guint                 param_id,
					const GValue         *value,
					GParamSpec           *pspec)
{
	MgCanvasFkconstraint *cc;

	cc = MG_CANVAS_FKCONSTRAINT (object);

	switch (param_id) {
	case PROP_FK_CONSTRAINT:
		mg_canvas_fkconstraint_add_constraint (cc, g_value_get_pointer (value));
		break;
	}
}

static void 
mg_canvas_fkconstraint_get_property    (GObject              *object,
					guint                 param_id,
					GValue               *value,
					GParamSpec           *pspec)
{
	MgCanvasFkconstraint *cc;

	cc = MG_CANVAS_FKCONSTRAINT (object);

	switch (param_id) {
	default:
		g_warning ("No such property!");
		break;
	}
}

static void
constraint_nullified_cb (MgDbConstraint *fkcons, MgCanvasFkconstraint *cc)
{
	g_assert (g_slist_find (cc->priv->constraints, fkcons));
	cc->priv->constraints = g_slist_remove (cc->priv->constraints, fkcons);
	g_signal_handlers_disconnect_by_func (G_OBJECT (fkcons),
					      G_CALLBACK (constraint_nullified_cb), cc);
	
	/* destroy itself if there are no more constraint left in the end */
	if (!cc->priv->constraints)
		gtk_object_destroy (GTK_OBJECT (cc));
	else {
		clean_items (cc);
		create_items (cc);
	}
}

static void
entity_destroy_cb (MgCanvasEntity *entity, MgCanvasFkconstraint *cc)
{
	gtk_object_destroy (GTK_OBJECT (cc));
}


static gboolean single_item_event_cb (GnomeCanvasItem *ci, GdkEvent *event, MgCanvasFkconstraint *cc);
static void entity_item_moved_cb (GnomeCanvasItem *entity, MgCanvasFkconstraint *cc);

/* 
 * destroy any existing GnomeCanvasItem objects 
 */
static void 
clean_items (MgCanvasFkconstraint *cc)
{
	if (cc->priv->fk_entity_item) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->fk_entity_item),
						      G_CALLBACK (entity_item_moved_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->fk_entity_item),
						      G_CALLBACK (entity_destroy_cb), cc);
		cc->priv->fk_entity_item = NULL;
	}

	if (cc->priv->ref_pk_entity_item) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->ref_pk_entity_item),
						      G_CALLBACK (entity_item_moved_cb), cc);
		g_signal_handlers_disconnect_by_func (G_OBJECT (cc->priv->ref_pk_entity_item),
						      G_CALLBACK (entity_destroy_cb), cc);
		cc->priv->ref_pk_entity_item = NULL;
	}

	/* remove all the GnomeCanvasItem objects */
	while (GNOME_CANVAS_GROUP (cc)->item_list)
		gtk_object_destroy (GTK_OBJECT (GNOME_CANVAS_GROUP (cc)->item_list->data));
	
}

/* structure to return shapes, being either points or a path definition */
typedef struct {
	GnomeCanvasPoints  *points;
	GnomeCanvasPathDef *path_def;
} AnchorShape;
#define ANCHOR_SHAPE(x) ((AnchorShape*)(x))

static GSList *compute_anchor_shapes (MgCanvasEntity *fk_ent, MgCanvasEntity *ref_pk_ent, guint nb_anchors);
static void    free_anchor_shapes (GSList *anchor_shapes);
/*
 * create new GnomeCanvasItem objects
 *
 * It is assumed that the list of FK constraints to render is coherent (that all the constraints are
 * FK constraints, for the same table, referencing the same FK table).
 */
static void 
create_items (MgCanvasFkconstraint *cc)
{
	GnomeCanvasItem *item;
	GSList *fklist = cc->priv->constraints, *list, *anchor_shapes, *list2;
	MgDbConstraint *fkcons;
	MgDbTable *table;
	MgCanvasItem *entity_item;
	MgCanvas *canvas = mg_canvas_item_get_canvas (MG_CANVAS_ITEM (cc));
	GdkLineStyle style;
	gboolean user_constraint;

	if (!fklist)
		return;

	/* Analyse FK constraint */
	fkcons = MG_DB_CONSTRAINT (fklist->data);
	item = GNOME_CANVAS_ITEM (cc);
	table = mg_db_constraint_get_table (fkcons);
	entity_item = mg_canvas_get_item_for_object (canvas, MG_BASE (table));
	g_return_if_fail (entity_item);
	cc->priv->fk_entity_item = MG_CANVAS_ENTITY (entity_item);
	g_signal_connect (G_OBJECT (entity_item), "moving",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "moved",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "shifted",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "destroy",
			  G_CALLBACK (entity_destroy_cb), cc);

	table = mg_db_constraint_fkey_get_ref_table (fkcons);
	entity_item = mg_canvas_get_item_for_object (canvas, MG_BASE (table));
	g_return_if_fail (entity_item);
	cc->priv->ref_pk_entity_item = MG_CANVAS_ENTITY (entity_item);
	g_signal_connect (G_OBJECT (entity_item), "moving",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "moved",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "shifted",
			  G_CALLBACK (entity_item_moved_cb), cc);
	g_signal_connect (G_OBJECT (entity_item), "destroy",
			  G_CALLBACK (entity_destroy_cb), cc);

	/* actual line(s) */
	anchor_shapes = compute_anchor_shapes (cc->priv->fk_entity_item, cc->priv->ref_pk_entity_item, 
					       g_slist_length (fklist));
	list2 = anchor_shapes;
	list = fklist;
	while (list && list2) {
		AnchorShape *shape = ANCHOR_SHAPE (list2->data);

		fkcons = MG_DB_CONSTRAINT (list->data);
		g_object_get (G_OBJECT (fkcons), "user_constraint", &user_constraint, NULL);
		if (user_constraint)
			style = GDK_LINE_ON_OFF_DASH;
		else
			style = GDK_LINE_SOLID;
		
		if (shape->points) {
			item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cc),
						      GNOME_TYPE_CANVAS_LINE,
						      "points", shape->points,
						      "fill_color", "black",
						      "width_units", 2.,
						      "cap_style", GDK_CAP_ROUND,
						      "line_style", style,
						      "first_arrowhead", TRUE,
						      "arrow-shape-a", 8.,
						      "arrow-shape-b", 12.,
						      "arrow-shape-c", 5.,
						      "smooth", TRUE,
						      NULL);
			g_object_set_data (G_OBJECT (item), "fkcons", fkcons);
			g_signal_connect (G_OBJECT (item), "event", 
					  G_CALLBACK (single_item_event_cb), cc);
		}
		if (shape->path_def) {
			item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (cc),
						      gnome_canvas_bpath_get_type(),
						      "bpath", shape->path_def,
						      "fill_color", "black",
						      "width_units", 2.,
						      "cap_style", GDK_CAP_ROUND,
						      NULL);
			g_object_set_data (G_OBJECT (item), "fkcons", fkcons);
			g_signal_connect (G_OBJECT (item), "event", 
					  G_CALLBACK (single_item_event_cb), cc);
		}

		list = g_slist_next (list);
		list2 = g_slist_next (list2);
	}
	free_anchor_shapes (anchor_shapes);
}

static void popup_delete_cb (GtkMenuItem *mitem, MgCanvasFkconstraint *cc);


/*
 * item is for a single FK constraint
 */
static gboolean
single_item_event_cb (GnomeCanvasItem *ci, GdkEvent *event, MgCanvasFkconstraint *cc)
{
	MgDbConstraint *fkcons = g_object_get_data (G_OBJECT (ci), "fkcons");
	gboolean highlight = FALSE, is_user_constraint = FALSE;
	GtkWidget *menu, *entry;
	gboolean done = FALSE;
	GSList *list, *pairs;

	switch (event->type) {
	case GDK_ENTER_NOTIFY:
		highlight = TRUE;
	case GDK_LEAVE_NOTIFY:
		pairs = mg_db_constraint_fkey_get_fields (fkcons);
		list = pairs;
		while (list) {
			MgDbConstraintFkeyPair *pair = MG_DB_CONSTRAINT_FK_PAIR (list->data);
			MgCanvasField *field;
			
			field = mg_canvas_entity_get_field_item (cc->priv->fk_entity_item, 
								 MG_FIELD (pair->fkey));
			mg_canvas_text_set_highlight (MG_CANVAS_TEXT (field), highlight);
			field = mg_canvas_entity_get_field_item (cc->priv->ref_pk_entity_item, 
								 MG_FIELD (pair->ref_pkey));
			mg_canvas_text_set_highlight (MG_CANVAS_TEXT (field), highlight);
			
			g_free (pair);
			list = g_slist_next (list);
		}
		g_slist_free (pairs);
		break;
	case GDK_BUTTON_PRESS:
                menu = gtk_menu_new ();
                entry = gtk_menu_item_new_with_label (_("Remove"));
		g_object_get (G_OBJECT (fkcons), "user_constraint", &is_user_constraint, NULL);
		gtk_widget_set_sensitive (entry, is_user_constraint);
		g_object_set_data (G_OBJECT (entry), "fkcons", fkcons);
                g_signal_connect (G_OBJECT (entry), "activate", G_CALLBACK (popup_delete_cb), cc);
                gtk_menu_append (GTK_MENU (menu), entry);
                gtk_widget_show (entry);
                gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
                                NULL, NULL, ((GdkEventButton *)event)->button,
                                ((GdkEventButton *)event)->time);
		done = TRUE;
                break;
	default:
		break;
	}

	return done;
}


static void
popup_delete_cb (GtkMenuItem *mitem, MgCanvasFkconstraint *cc)
{
	MgDbConstraint *fkcons = g_object_get_data (G_OBJECT (mitem), "fkcons");
	gboolean is_user_constraint;

	g_object_get (G_OBJECT (fkcons), "user_constraint", &is_user_constraint, NULL);
	if (is_user_constraint)
		mg_base_nullify (MG_BASE (fkcons));
}

static void
entity_item_moved_cb (GnomeCanvasItem *entity, MgCanvasFkconstraint *cc)
{
	clean_items (cc);
	create_items (cc);
}


static gboolean
compute_intersect_rect_line (gdouble rectx1, gdouble recty1, gdouble rectx2, gdouble recty2,
			     gdouble P1x, gdouble P1y, gdouble P2x, gdouble P2y,
			     gdouble *R1x, gdouble *R1y, gdouble *R2x, gdouble *R2y);

/**
 * mg_canvas_fkconstraint_add_constraint
 * @cc:
 * @fkcons:
 * 
 * Add @fkcons to the list of displayed constraints
 */
void
mg_canvas_fkconstraint_add_constraint (MgCanvasFkconstraint *cc, MgDbConstraint *fkcons)
{
	g_return_if_fail (cc && IS_MG_CANVAS_FKCONSTRAINT (cc));
	g_return_if_fail (cc->priv);
	g_return_if_fail (fkcons && IS_MG_DB_CONSTRAINT (fkcons));
	g_return_if_fail (mg_db_constraint_get_constraint_type (MG_DB_CONSTRAINT (fkcons)) == 
			  CONSTRAINT_FOREIGN_KEY);

	if (g_slist_find (cc->priv->constraints, fkcons))
		return;

	if (cc->priv->constraints) {
		/* there are already some FK constraints there, so test that the new one is
		 * compatible with the existing ones*/
		
	}

	cc->priv->constraints = g_slist_append (cc->priv->constraints, fkcons);
	g_signal_connect (G_OBJECT (fkcons), "nullified",
			  G_CALLBACK (constraint_nullified_cb), cc);

	clean_items (cc);
	create_items (cc);
}

/*
 * Computes the points' coordinates of the line going from
 * @ref_pk_ent to @fk_ent (which are themselves rectangles)
 *
 * Returns a list of AnchorShape structures, use free_anchor_shapes() to free it.
 */
static GSList *
compute_anchor_shapes (MgCanvasEntity *fk_ent, MgCanvasEntity *ref_pk_ent, guint nb_anchors)
{
	GSList *retval = NULL;
	guint i;
	gdouble fx1, fy1, fx2, fy2; /* FK entity item (bounds) */
	gdouble rx1, ry1, rx2, ry2; /* REF PK entity item  (bounds) */

	gdouble rcx, rcy; /* center of ref_pk entity item */
	gdouble cx, cy;

	gdouble rux, ruy; /* current ref_pk point for the arrow line */
	gdouble dx, dy; /* increments to compute the new ref_pk point for the arrow line */

	g_return_val_if_fail (nb_anchors > 0, NULL);

	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (fk_ent), &fx1, &fy1, &fx2, &fy2);
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (ref_pk_ent), &rx1, &ry1, &rx2, &ry2);

	/* compute the cx, cy, dx and dy values */
	rcx = (rx1 + rx2) / 2.;
	rcy = (ry1 + ry2) / 2.;
	cx = (fx1 + fx2) / 2.;
	cy = (fy1 + fy2) / 2.;
	rux = rcx;
	ruy = rcy;
	dx = 0;
	dy = 0;

	for (i = 0; i < nb_anchors; i++) {
		AnchorShape *shape = g_new0 (AnchorShape, 1);

		/* TODO:
		   - detect tables overlapping
		*/		
		if ((rcx == cx) && (rcy == cy)) {
			/* tables have the same center (includes when they are equal) */
			gdouble Dy, Dx;
			GnomeCanvasPoints *ap, *points;
			
			points = gnome_canvas_points_new (4);
			ap = gnome_canvas_points_new (4);

			Dy = (ry2 - ry1) / 2. / (gdouble ) (nb_anchors + 1) * (gdouble) (i + 1);
			Dx = (rx2 - rx1) * (0.8 + 0.1 * i);
			g_assert (compute_intersect_rect_line (rx1, ry1, rx2, ry2,
							       cx, cy, cx + Dx, cy - Dy,
							       &(ap->coords[0]), &(ap->coords[1]),
							       &(ap->coords[2]), &(ap->coords[3])));
			
			if (ap->coords[0] > ap->coords[2]) {
				points->coords[0] = ap->coords[0];
				points->coords[1] = ap->coords[1];
			}
			else {
				points->coords[0] = ap->coords[2];
				points->coords[1] = ap->coords[3];
			}

			points->coords[2] = cx + Dx;
			points->coords[3] = cy - Dy;

			Dy = (fy2 - fy1) / 2. / (gdouble ) (nb_anchors + 1) * (gdouble) (i + 1);
			Dx = (fx2 - fx1) * (0.8 + 0.1 * i);
			points->coords[4] = cx + Dx;
			points->coords[5] = cy + Dy;

			g_assert (compute_intersect_rect_line (fx1, fy1, fx2, fy2,
							       cx, cy, cx + Dx, cy + Dy,
							       &(ap->coords[0]), &(ap->coords[1]),
							       &(ap->coords[2]), &(ap->coords[3])));
			
			if (ap->coords[0] > ap->coords[2]) {
				points->coords[6] = ap->coords[0];
				points->coords[7] = ap->coords[1];
			}
			else {
				points->coords[6] = ap->coords[2];
				points->coords[7] = ap->coords[3];
			}
			
			shape->points = points;
			gnome_canvas_points_free (ap);

			/* GnomeCanvasPathDef *path_def; */
			
			/* path_def = gnome_canvas_path_def_new(); */
			
			/* gnome_canvas_path_def_moveto(path_def, 500.0, 175.0); */
			/* gnome_canvas_path_def_curveto(path_def, 550.0, 175.0, 550.0, 275.0, 500.0, 275.0); */
			/* shape->path_def = path_def; */
			
		}
		else {
			GnomeCanvasPoints *ap, *points;
			
			points = gnome_canvas_points_new (2);
			ap = gnome_canvas_points_new (4);

			if (nb_anchors > 1) {
				if ((dx == 0) && (dy == 0)) {
					/* compute perpendicular to D={(rcx, rcy), (cx, cy)} */
					gdouble vx = (rcx - cx), vy = (rcy - cy);
					gdouble tmp;
					
					tmp = vx;
					vx = vy;
					vy = - tmp;
					
					/* compute intersect of ref_pkey rectangle and D=[vx, vy] passing at (rcx, rcy) */
					g_assert (compute_intersect_rect_line (rx1, ry1, rx2, ry2,
									       rcx, rcy, rcx + vx, rcy + vy,
									       &(ap->coords[0]), &(ap->coords[1]),
									       &(ap->coords[2]), &(ap->coords[3])));
					dx = (ap->coords[2] - ap->coords[0]) / (gdouble) (nb_anchors  + 1);
					dy = (ap->coords[3] - ap->coords[1]) / (gdouble) (nb_anchors  + 1);
					rux = ap->coords[0];
					ruy = ap->coords[1];
				}

				rux += dx;
				ruy += dy;
			}
			
			/* compute the 4 intersection points */
			g_assert (compute_intersect_rect_line (rx1, ry1, rx2, ry2,
							       rux, ruy, cx, cy,
							       &(ap->coords[0]), &(ap->coords[1]),
							       &(ap->coords[2]), &(ap->coords[3])));
			g_assert (compute_intersect_rect_line (fx1, fy1, fx2, fy2,
							       rux, ruy, cx, cy,
							       &(ap->coords[4]), &(ap->coords[5]),
							       &(ap->coords[6]), &(ap->coords[7])));
			
			/* choosing between point coords(0,1) and coords(2,3) */
			if (((ap->coords[0] - ap->coords[4]) * (ap->coords[0] - ap->coords[4]) + 
			     (ap->coords[1] - ap->coords[5]) * (ap->coords[1] - ap->coords[5])) <
			    ((ap->coords[2] - ap->coords[4]) * (ap->coords[2] - ap->coords[4]) + 
			     (ap->coords[3] - ap->coords[5]) * (ap->coords[3] - ap->coords[5]))) {
				points->coords[0] = ap->coords[0];
				points->coords[1] = ap->coords[1];
			}
			else {
				points->coords[0] = ap->coords[2];
				points->coords[1] = ap->coords[3];
			}
			
			/* choosing between point coords(4,5) and coords(6,7) */
			if (((points->coords[0] - ap->coords[4]) * (points->coords[0] - ap->coords[4]) +
			     (points->coords[1] - ap->coords[5]) * (points->coords[1] - ap->coords[5])) <
			    ((points->coords[0] - ap->coords[6]) * (points->coords[0] - ap->coords[6]) +
			     (points->coords[1] - ap->coords[7]) * (points->coords[1] - ap->coords[7]))) {
				points->coords[2] = ap->coords[4];
				points->coords[3] = ap->coords[5];
			}
			else {
				points->coords[2] = ap->coords[6];
				points->coords[3] = ap->coords[7];
			}
			
			shape->points = points;
			gnome_canvas_points_free (ap);
		}

		retval = g_slist_append (retval, shape);
	}

	return retval;
}

/*
 * Free the list of points
 */
static void
free_anchor_shapes (GSList *anchor_shapes)
{
	GSList *list = anchor_shapes;

	if (!list)
		return;

	while (list) {
		AnchorShape *shape = ANCHOR_SHAPE (list->data);
		if (shape->points)
			gnome_canvas_points_free (shape->points);
		if (shape->path_def)
			gnome_canvas_path_def_unref (shape->path_def);
		g_free (shape);
		list = g_slist_next (list);
	}
	g_slist_free (anchor_shapes);
}

/*
 * Computes the points of intersection between a rectangle (defined by the first 2 points)
 * and a line (defined by the next 2 points).
 *
 * The result is returned in place of the line's point definition
 *
 *             --------- -----   D1
 *             |       |
 *             |       |
 *             |       |
 *             |       |
 *             |       |
 *             |       |
 *             |       |
 *             --------- ----   D2
 *             
 *             |       |
 *             |       |
 *             D3      D4
 *
 * Returns: TRUE if the line crosses the rectangle, and FALSE if it misses it.
 */
static gboolean
compute_intersect_rect_line (gdouble rectx1, gdouble recty1, gdouble rectx2, gdouble recty2,
			     gdouble P1x, gdouble P1y, gdouble P2x, gdouble P2y,
			     gdouble *R1x, gdouble *R1y, gdouble *R2x, gdouble *R2y)
{
	gboolean retval = FALSE;
	gboolean rotated = FALSE;
	gdouble a=0.; /* line slope   y = a x + b */
	gdouble b;    /* line offset  y = a x + b */
	gdouble offset = 2.;
		
	gdouble ptsx[4]; /* points' X coordinate: 0 for intersect with D1, 1 for D2,... */
	gdouble ptsy[4]; /* points' Y coordinate */

	g_return_val_if_fail ((rectx1 != rectx2) || (recty1 != recty2), FALSE);
	g_return_val_if_fail ((rectx1 < rectx2) && (recty1 < recty2), FALSE);
	g_return_val_if_fail ((P1x != P2x) || (P1y != P2y), FALSE);

	/* rotate the coordinates to invert X and Y to avoid rounding problems ? */
	if (P1x != P2x)
		a = (P1y - P2y) / (P1x - P2x);
	if ((P1x == P2x) || (fabs (a) > 1)) {
		gdouble tmp;
		rotated = TRUE;
		tmp = rectx1; rectx1 = recty1; recty1 = tmp;
		tmp = rectx2; rectx2 = recty2; recty2 = tmp;
		tmp = P1x; P1x = P1y; P1y = tmp;
		tmp = P2x; P2x = P2y; P2y = tmp;
		a = (P1y - P2y) / (P1x - P2x);
	}

	/* here we have (P1x != P2x), non vertical line */
	b = P1y - a * P1x;

	if (a == 0) {
		/* horizontal line */

		if ((b <= recty2) && (b >= recty1)) {
			retval = TRUE;
			*R1x = rectx1 - offset; *R1y = b;
			*R2x = rectx2 + offset; *R2y = b;
		}
	}
	else {
		gdouble retx[2] = {0., 0.};
		gdouble rety[2] = {0., 0.};
		gint i = 0;

		/* non horizontal and non vertical line */
		/* D1 */
		ptsy[0] = recty1 - offset;
		ptsx[0] = (recty1 - b) / a;

		/* D2 */
		ptsy[1] = recty2 + offset;
		ptsx[1] = (recty2 - b) / a;

		/* D3 */
		ptsx[2] = rectx1 - offset;
		ptsy[2] = a * rectx1 + b;

		/* D4 */
		ptsx[3] = rectx2 + offset;
		ptsy[3] = a * rectx2 + b;
		
		if ((ptsx[0] >= rectx1) && (ptsx[0] <= rectx2)) {
			retval = TRUE;
			retx[i] = ptsx[0]; rety[i] = ptsy[0];
			i ++;
		}
		if ((ptsx[1] >= rectx1) && (ptsx[1] <= rectx2)) {
			retval = TRUE;
			retx[i] = ptsx[1]; rety[i] = ptsy[1];
			i ++;
		}
		if ((i<2) && (ptsy[2] >= recty1) && (ptsy[2] <= recty2)) {
			retval = TRUE;
			retx[i] = ptsx[2]; rety[i] = ptsy[2];
			i ++;
		}
		if ((i<2) && (ptsy[3] >= recty1) && (ptsy[3] <= recty2)) {
			retval = TRUE;
			retx[i] = ptsx[3]; rety[i] = ptsy[3];
			i++;
		}

		if (retval) {
			g_assert (i == 2); /* wee need 2 points! */
			*R1x = retx[0]; *R1y = rety[0];
			*R2x = retx[1]; *R2y = rety[1];
		}
	}

	if (retval && rotated) {
		/* rotate it back */
		gdouble tmp;

		tmp = *R1x; *R1x = *R1y; *R1y = tmp;
		tmp = *R2x; *R2x = *R2y; *R2y = tmp;
	}

	return retval;
}
