/* ws-tables.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "ws-tables.h"
#include "workspace-page.h"
#include <string.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <libgnomedb/libgnomedb.h>

/* 
 * Main static functions 
 */
static void ws_tables_class_init (WsTablesClass * class);
static void ws_tables_init (WsTables *ws);
static void ws_tables_dispose (GObject *object);
static void ws_tables_finalize (GObject *object);

/* WorkspacePage interface */
static void         ws_tables_page_init       (WorkspacePageIface *iface);
static gchar       *ws_tables_get_name        (WorkspacePage *iface);
static gchar       *ws_tables_get_description (WorkspacePage *iface);
static GtkWidget   *ws_tables_get_selector    (WorkspacePage *iface);
static GtkWidget   *ws_tables_get_work_area   (WorkspacePage *iface);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* private structure */
struct _WsTablesPrivate
{
	MgConf         *conf;
	GtkWidget      *selector;
	GtkWidget      *work_area;

	GtkWidget      *notebook;
	GtkWidget      *description;
	GtkWidget      *fields;
	GtkTextBuffer  *integ_descr;
	GtkWidget      *table_add;
	GtkWidget      *table_del;
	GtkWidget      *table_contents;

	GObject        *sel_obj;
};

guint
ws_tables_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (WsTablesClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) ws_tables_class_init,
			NULL,
			NULL,
			sizeof (WsTables),
			0,
			(GInstanceInitFunc) ws_tables_init
		};

		static const GInterfaceInfo workspace_page_info = {
			(GInterfaceInitFunc) ws_tables_page_init,
			NULL,
			NULL
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WsTables", &info, 0);
		g_type_add_interface_static (type, WORKSPACE_PAGE_TYPE, &workspace_page_info);
	}
	return type;
}

static void 
ws_tables_page_init (WorkspacePageIface *iface)
{
	iface->get_name = ws_tables_get_name;
	iface->get_description = ws_tables_get_description;
	iface->get_selector = ws_tables_get_selector;
	iface->get_work_area = ws_tables_get_work_area;
}

static void
ws_tables_class_init (WsTablesClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = ws_tables_dispose;
	object_class->finalize = ws_tables_finalize;
}

static void
ws_tables_init (WsTables *ws)
{
	ws->priv = g_new0 (WsTablesPrivate, 1);
}

static void ws_tables_initialize (WsTables *ws);

/**
 * ws_tables_new
 * @conf: a #MgConf object
 *
 * Creates a new WsTables object
 *
 * Returns: the new object
 */
GObject*
ws_tables_new (MgConf *conf)
{
	GObject  *obj;
	WsTables *ws;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);

	obj = g_object_new (WS_TABLES_TYPE, NULL);
	ws = WS_TABLES (obj);
	ws->priv->conf = conf;
	g_object_ref (G_OBJECT (conf));

	ws_tables_initialize (ws);

	return obj;
}


static void
ws_tables_dispose (GObject *object)
{
	WsTables *ws;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_WS_TABLES (object));

	ws = WS_TABLES (object);
	if (ws->priv) {
		if (ws->priv->selector) {
			gtk_widget_destroy (ws->priv->selector);
			ws->priv->selector = NULL;
		}

		if (ws->priv->work_area) {
			gtk_widget_destroy (ws->priv->work_area);
			ws->priv->work_area = NULL;
		}

		if (ws->priv->conf) {
			g_object_unref (G_OBJECT (ws->priv->conf));
			ws->priv->conf = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
ws_tables_finalize (GObject   * object)
{
	WsTables *ws;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_WS_TABLES (object));

	ws = WS_TABLES (object);
	if (ws->priv) {
		g_free (ws->priv);
		ws->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}



static void selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, WsTables *ws);
static void table_add_clicked_cb (GtkButton *button, WsTables *ws);
static void table_del_clicked_cb (GtkButton *button, WsTables *ws);
static void table_contents_clicked_cb (GtkButton *button, WsTables *ws);
static void table_updated_cb (MgDatabase *db, MgDbTable *table, WsTables *ws);
static void
ws_tables_initialize (WsTables *ws)
{
	GtkWidget *label, *vbox, *wid, *nb, *vp, *vbox2, *bbox, *button, *hbox, *sw;

	/* Selector part */
	wid = mg_selector_new (ws->priv->conf, NULL,
			       MG_SELECTOR_TABLES, MG_SELECTOR_COLUMN_TYPE);
	ws->priv->selector = wid;
	g_signal_connect (G_OBJECT (ws->priv->selector), "selection_changed", 
			  G_CALLBACK (selector_selection_changed_cb), ws);
	gtk_widget_set_size_request (wid, 200, -1);

	/* WorkArea part */
	vbox = gtk_vbox_new (FALSE, 0);
	ws->priv->work_area = vbox;
	
	nb = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	ws->priv->notebook = nb;
	
	label = gtk_label_new (_("Please select a table from the list on the left,\n"
				 "or create a new one using the 'Add' button below."));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), label, NULL);
	gtk_widget_show (label);
	
	vp = gtk_vpaned_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vp, NULL);
	gtk_widget_show (vp);

	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_paned_add1 (GTK_PANED (vp), vbox2);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Description:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	label = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	ws->priv->description = label;
	
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Fields:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	wid = mg_selector_new (ws->priv->conf, NULL,
			       MG_SELECTOR_FIELDS, 
			       MG_SELECTOR_COLUMN_TYPE | MG_SELECTOR_COLUMN_FIELD_LENGTH | MG_SELECTOR_COLUMN_FIELD_NNUL |
			       MG_SELECTOR_COLUMN_FIELD_DEFAULT | MG_SELECTOR_COLUMN_COMMENTS);
	gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	ws->priv->fields = wid;

	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_paned_add2 (GTK_PANED (vp), vbox2);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Integrity rules:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), sw, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_widget_show (sw);

	wid = gtk_text_view_new ();
	gtk_container_add (GTK_CONTAINER (sw), wid);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (wid), 5);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW (wid), 5);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (wid), FALSE);
	ws->priv->integ_descr = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wid));
	gtk_text_buffer_set_text (ws->priv->integ_descr, "", -1);
	gtk_widget_show (wid);

	gtk_text_buffer_create_tag (ws->priv->integ_descr, "header",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "red", NULL);

	gtk_text_buffer_create_tag (ws->priv->integ_descr, "section",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "blue", NULL);

	bbox = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 0);
	gtk_widget_show (bbox);

	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	ws->priv->table_add = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (table_add_clicked_cb), ws);

	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	ws->priv->table_del = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (table_del_clicked_cb), ws);

	button = gtk_button_new_from_stock (GTK_STOCK_INDEX);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_button_set_label (GTK_BUTTON (button), _("Contents"));
	ws->priv->table_contents = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (table_contents_clicked_cb), ws);

	g_signal_connect (G_OBJECT (mg_conf_get_database (ws->priv->conf)), "table_updated",
			  G_CALLBACK (table_updated_cb), ws);
}

static void table_info_display_update (MgDbTable *table, WsTables *ws);

static void
selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, WsTables *ws)
{
	if (sel_object && !IS_MG_DB_TABLE (sel_object))
		return;

	ws->priv->sel_obj = sel_object;
	gtk_notebook_set_current_page (GTK_NOTEBOOK (ws->priv->notebook), sel_object ? 1 : 0);
	gtk_widget_set_sensitive (ws->priv->table_del, sel_object ? TRUE : FALSE);
	gtk_widget_set_sensitive (ws->priv->table_contents, sel_object ? TRUE : FALSE);
	table_info_display_update ((MgDbTable *) sel_object, ws);
}

static void
table_updated_cb (MgDatabase *db, MgDbTable *table, WsTables *ws)
{
	if ((GObject *)table == ws->priv->sel_obj)
		table_info_display_update (table, ws);
}

static void
table_add_clicked_cb (GtkButton *button, WsTables *ws)
{
	NOT_YET_IMPLEMENTED (WORKSPACE_PAGE (ws), _("table creation"));
}

static void
table_del_clicked_cb (GtkButton *button, WsTables *ws)
{
	NOT_YET_IMPLEMENTED (WORKSPACE_PAGE (ws), _("table removal"));
}


/* extra action for the grid widgets */
static void extra_action_close_cb (GtkAction *action, GtkWidget *window);
static GtkActionEntry extra_actions[] = {
	{ "Close", GTK_STOCK_CLOSE, "_Close", NULL, "Close this window", G_CALLBACK (extra_action_close_cb)}
};
static void
table_contents_clicked_cb (GtkButton *button, WsTables *ws)
{
	MgQuery *query;
	gchar *sql;
	MgConf *conf = ws->priv->conf;
	GError *error = NULL;
	
	sql = g_strdup_printf ("SELECT * FROM %s\n", mg_base_get_name (MG_BASE (ws->priv->sel_obj)));
	query = (MgQuery *) mg_query_new_from_sql (conf, sql, &error);
	g_free (sql);

	if (mg_query_get_query_type (query) != MG_QUERY_TYPE_NON_PARSED_SQL) {
		GtkWidget *window, *grid, *vbox, *toolbar;
		gchar *str;
		GSList *targets;

		GtkActionGroup *grid_agroup, *extra_agroup;
		GtkUIManager *ui;

		static const gchar *grid_actions =
			"<ui>"
			"  <toolbar  name='ToolBar'>"
			"    <toolitem action='WorkWidgetNew'/>"
			"    <toolitem action='WorkWidgetDelete'/>"
			"    <toolitem action='WorkWidgetUndelete'/>"
			"    <toolitem action='WorkWidgetCommit'/>"
			"    <toolitem action='WorkWidgetReset'/>"
			"    <separator/>"
			"    <toolitem action='WorkWidgetFirstChunck'/>"
			"    <toolitem action='WorkWidgetPrevChunck'/>"
			"    <toolitem action='WorkWidgetNextChunck'/>"
			"    <toolitem action='WorkWidgetLastChunck'/>"
			"    <separator/>"
			"    <toolitem action='Close'/>"					
			"  </toolbar>"
			"</ui>";

		targets = mg_query_get_targets (query);
		grid = mg_work_grid_new (query, MG_TARGET (targets->data));
		
		g_slist_free (targets);

		str = g_strdup_printf (_("Contents of '%s'"), mg_base_get_name (MG_BASE (ws->priv->sel_obj)));
		window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title (GTK_WINDOW (window), str);
		g_free (str);

		vbox = gtk_vbox_new (FALSE, 0);
		gtk_container_add (GTK_CONTAINER (window), vbox);
		gtk_widget_show (vbox);

		ui = gtk_ui_manager_new ();
		grid_agroup = mg_work_widget_get_actions_group (MG_WORK_WIDGET (grid));
		extra_agroup = gtk_action_group_new ("ExtraActions");
		gtk_action_group_add_actions (extra_agroup, extra_actions, G_N_ELEMENTS (extra_actions), window);
		gtk_ui_manager_insert_action_group (ui, grid_agroup, 0);
		gtk_ui_manager_insert_action_group (ui, extra_agroup, 0);
		gtk_ui_manager_add_ui_from_string (ui, grid_actions, -1, NULL);
		toolbar = gtk_ui_manager_get_widget (ui, "/ToolBar");
		gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);
		gtk_widget_show (toolbar);

		g_object_set (G_OBJECT (grid), "title_visible", FALSE, NULL);
		mg_work_widget_alldata_show_actions (MG_WORK_WIDGET (grid), FALSE);
		mg_work_widget_run (MG_WORK_WIDGET (grid), MG_ACTION_ASK_CONFIRM_UPDATE | MG_ACTION_ASK_CONFIRM_DELETE |
				    MG_ACTION_ASK_CONFIRM_INSERT | MG_ACTION_REPORT_ERROR | MG_ACTION_NAVIGATION_ARROWS);
		gtk_box_pack_start (GTK_BOX (vbox), grid, TRUE, TRUE, 0);
		gtk_widget_show (grid);

		gtk_window_set_default_size (GTK_WINDOW (window), 800, 300);
		gtk_widget_show (window);

		g_object_unref (G_OBJECT (query));
	}
	else {
		if (error) {
			g_print ("ERROR: %s\n", error->message);
			g_error_free (error);
		}
	}
}

static void
extra_action_close_cb (GtkAction *action, GtkWidget *window)
{
	gtk_widget_destroy (window);
}

static void
table_info_display_update (MgDbTable *table, WsTables *ws)
{
	const gchar *str = NULL;
	gchar *title = NULL;
	GtkTextIter start, end;
	
	if (table)
		str = mg_base_get_description (MG_BASE (table));
	if (str && *str) 
		gtk_label_set_text (GTK_LABEL (ws->priv->description), str);
	else
		gtk_label_set_text (GTK_LABEL (ws->priv->description), "---");
	
	mg_selector_set_mode_columns (MG_SELECTOR (ws->priv->fields), (GObject*) table,
				      MG_SELECTOR_FIELDS, 
				      MG_SELECTOR_COLUMN_TYPE | MG_SELECTOR_COLUMN_FIELD_LENGTH | 
				      MG_SELECTOR_COLUMN_FIELD_NNUL |
				      MG_SELECTOR_COLUMN_FIELD_DEFAULT | MG_SELECTOR_COLUMN_COMMENTS);

	/* global title update */
	title = ws_tables_get_description (WORKSPACE_PAGE (ws));
	g_signal_emit_by_name (G_OBJECT (ws), "description_changed", title);
	g_free (title);

	/* integrity rules update */
	gtk_text_buffer_get_start_iter (ws->priv->integ_descr, &start);
	gtk_text_buffer_get_end_iter (ws->priv->integ_descr, &end);
	gtk_text_buffer_delete (ws->priv->integ_descr, &start, &end);
	
	if (table) {
		GtkTextIter current;
		GSList *constraints, *list;

		gtk_text_buffer_get_start_iter (ws->priv->integ_descr, &current);		
		/* constraints list */
		constraints = mg_db_table_get_constraints (table);
		
		/* PKey */
		list = constraints;
		while (list) {
			gboolean first = TRUE;
			if (mg_db_constraint_get_constraint_type (MG_DB_CONSTRAINT (list->data)) == 
			    CONSTRAINT_PRIMARY_KEY) {
				MgDbConstraint *cstr = MG_DB_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = mg_db_constraint_pkey_get_fields (cstr);
				list2 = fields;
				while (list2) {
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ws->priv->integ_descr,
											  &current, 
											  _("Primary key"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
					}

					if (first) 
						first = FALSE;
					else
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, ", ", -1);

					gtk_text_buffer_insert (ws->priv->integ_descr, &current, 
								mg_base_get_name (MG_BASE (list2->data)), -1);
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n\n", -1);
			}
			list = g_slist_next (list);
		}

		/* FKey */
		list = constraints;
		while (list) {
			if (mg_db_constraint_get_constraint_type (MG_DB_CONSTRAINT (list->data)) == 
			    CONSTRAINT_FOREIGN_KEY) {
				MgDbConstraint *cstr = MG_DB_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = mg_db_constraint_fkey_get_fields (cstr);
				list2 = fields;
				while (list2) {
					MgEntity *ent;
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ws->priv->integ_descr,
											  &current, 
											  _("Foreign key"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
					}
						
					str = mg_base_get_name (MG_BASE (MG_DB_CONSTRAINT_FK_PAIR (list2->data)->fkey));
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, " --> ", -1);
					ent = mg_field_get_entity (MG_FIELD (MG_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey));
					str = mg_base_get_name (MG_BASE (ent));
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, ".", -1);
					str = mg_base_get_name (MG_BASE (MG_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey));
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
						
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
			}
			list = g_slist_next (list);
		}

		/* Unique */
		list = constraints;
		while (list) {
			if (mg_db_constraint_get_constraint_type (MG_DB_CONSTRAINT (list->data)) == 
			    CONSTRAINT_UNIQUE) {
				MgDbConstraint *cstr = MG_DB_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = mg_db_constraint_unique_get_fields (cstr);
				list2 = fields;
				while (list2) {
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ws->priv->integ_descr,
											  &current, 
											  _("UNIQUE constraint"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
					}
					else
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, ", ", -1);

					gtk_text_buffer_insert (ws->priv->integ_descr, &current, 
								mg_base_get_name (MG_BASE (list2->data)), -1);
						
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n\n", -1);
			}
			list = g_slist_next (list);
		}

		/* check constraint: FIXME */

		g_slist_free (constraints);			
	}
}




/* 
 * WorkspacePage interface implementation
 */
static gchar *
ws_tables_get_name (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	return g_strdup_printf (_("Tables & views"));
}

static gchar *
ws_tables_get_description (WorkspacePage *iface)
{
	gchar *str = NULL;
	WsTables *ws;

	g_return_val_if_fail (iface && IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	ws = WS_TABLES (iface);
	if (ws->priv->sel_obj)
		str = g_strdup_printf ("Table: <b>%s</b>", mg_base_get_name (MG_BASE (ws->priv->sel_obj)));
	else
		str = g_strdup_printf ("<b>%s</b>", _("No table selected"));

	return str;
}

static GtkWidget *
ws_tables_get_selector (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	return WS_TABLES (iface)->priv->selector;
}
static GtkWidget *
ws_tables_get_work_area (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	return WS_TABLES (iface)->priv->work_area;
}
