/* query-editor.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "query-editor.h"
#include <libgnomedb/gnome-db-editor.h>

static void query_editor_class_init (QueryEditorClass * class);
static void query_editor_init (QueryEditor * wid);
static void query_editor_dispose (GObject   * object);


struct _QueryEditorPriv
{
	MgQuery     *query;
	
	GtkWidget   *q_name;
	GtkWidget   *q_descr;
	GtkWidget   *notebook;
	GtkWidget   *toggle;

	/* SQL edition */
	GtkWidget   *sql_editor;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
query_editor_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryEditorClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_editor_class_init,
			NULL,
			NULL,
			sizeof (QueryEditor),
			0,
			(GInstanceInitFunc) query_editor_init
		};		
		
		type = g_type_register_static (GTK_TYPE_VBOX, "QueryEditor", &info, 0);
	}

	return type;
}

static void
query_editor_class_init (QueryEditorClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = query_editor_dispose;
}

static void
query_editor_init (QueryEditor * wid)
{
	wid->priv = g_new0 (QueryEditorPriv, 1);
	wid->priv->query = NULL;
}

static void query_editor_initialize (QueryEditor *mgsel);


static void object_weak_notify (QueryEditor *mgsel, GObject *obj);
/**
 * query_editor_new
 * @conf: a #MgConf object
 *
 * Creates a new #QueryEditor widget.
 *
 * Returns: the new widget
 */
GtkWidget *
query_editor_new (MgQuery *query)
{
	GObject    *obj;
	QueryEditor *qedit;

	g_return_val_if_fail (query && IS_MG_QUERY (query), NULL);
		
	obj = g_object_new (QUERY_EDITOR_TYPE, NULL);
	qedit = QUERY_EDITOR (obj);

	qedit->priv->query = query;

	g_object_weak_ref (G_OBJECT (qedit->priv->query),
			   (GWeakNotify) object_weak_notify, qedit);
	
	query_editor_initialize (qedit);

	return GTK_WIDGET (obj);
}

static void query_changed_cb (MgQuery *query, QueryEditor *qedit);
static void
object_weak_notify (QueryEditor *qedit, GObject *obj)
{
	if (obj == (GObject*) qedit->priv->query) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (qedit->priv->query),
						      G_CALLBACK (query_changed_cb), qedit);

		/* Tell that we don't need to weak unref the MgQuery */
		qedit->priv->query = NULL;
	}
}

static void
query_editor_dispose (GObject *object)
{
	QueryEditor *qedit;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_QUERY_EDITOR (object));
	qedit = QUERY_EDITOR (object);

	if (qedit->priv) {
		/* Weak unref the MgQuery if necessary */
		if (qedit->priv->query) {
			g_object_weak_unref (G_OBJECT (qedit->priv->query),
					     (GWeakNotify) object_weak_notify, qedit);
			g_signal_handlers_disconnect_by_func (G_OBJECT (qedit->priv->query),
						      G_CALLBACK (query_changed_cb), qedit);
		}

		/* the private area itself */
		g_free (qedit->priv);
		qedit->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void entry_name_changed_cb (GtkEntry *entry, QueryEditor *qedit);
static void entry_description_changed_cb (GtkEntry *entry, QueryEditor *qedit);
static void sql_query_text_apply_cb (GtkButton *button, QueryEditor *qedit);
static void edition_mode_toggled (GtkToggleButton *button, QueryEditor *qedit);

static void
query_editor_initialize (QueryEditor *qedit)
{
	GtkWidget *wid, *label, *table, *hbox, *nb, *vbox;
	gtk_box_set_spacing (GTK_BOX (qedit), 6);
	
	/* query attributes */
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Query attributes:</b>"));
	gtk_box_pack_start (GTK_BOX (qedit), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (qedit), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	table = gtk_table_new (2, 3, FALSE);
	gtk_widget_show (table);
	gtk_box_pack_start (GTK_BOX (hbox), table, TRUE, TRUE, 0);
	gtk_table_set_row_spacings (GTK_TABLE (table), 3);
	gtk_table_set_col_spacings (GTK_TABLE (table), 5);
	
	label = gtk_label_new (_("Name:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

	wid = gtk_entry_new ();
	gtk_widget_show (wid);
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
	qedit->priv->q_name = wid;
	g_signal_connect (G_OBJECT (wid), "changed", G_CALLBACK (entry_name_changed_cb), qedit);

	label = gtk_label_new (_("Description:"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

	wid = gtk_entry_new ();
	gtk_widget_show (wid);
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 1, 2);
	qedit->priv->q_descr = wid;
	g_signal_connect (G_OBJECT (wid), "changed", G_CALLBACK (entry_description_changed_cb), qedit);

	wid = gtk_toggle_button_new_with_mnemonic (_("SQL"));
	gtk_widget_show (wid);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wid), TRUE);
	gtk_table_attach (GTK_TABLE (table), wid, 2, 3, 0, 2, GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
	qedit->priv->toggle = wid;
	g_signal_connect (G_OBJECT (wid), "toggled", G_CALLBACK (edition_mode_toggled), qedit);
	
	/* notebook */
	nb = gtk_notebook_new ();
	gtk_widget_show (nb);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (qedit), nb, TRUE, TRUE, 0);
	qedit->priv->notebook = nb;

	/* SQL edition page */
	vbox = gtk_vbox_new (FALSE, 6);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vbox, NULL);
	gtk_widget_show (vbox);

	label = gtk_label_new (_("<b>Enter the SQL statement representing the query:</b>"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	wid = gnome_db_editor_new ();
	qedit->priv->sql_editor = wid;
	gtk_widget_show (wid);
        gnome_db_editor_set_editable (GNOME_DB_EDITOR (wid), TRUE);
        gnome_db_editor_set_highlight (GNOME_DB_EDITOR (wid), FALSE); /* CHANGE ME */
        gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 0);
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);
	gtk_widget_show (hbox);
	wid = gtk_button_new_from_stock (GTK_STOCK_APPLY);
	gtk_box_pack_end (GTK_BOX (hbox), wid, FALSE, TRUE, 0);
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "clicked", G_CALLBACK (sql_query_text_apply_cb), qedit);
	
	g_signal_connect (G_OBJECT (qedit->priv->query), "changed",
			  G_CALLBACK (query_changed_cb), qedit);

	/* Point and click edition page */
	vbox = gtk_vbox_new (FALSE, 6);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vbox, NULL);
	gtk_widget_show (vbox);

	label = gtk_label_new ("Point and Click GUI is Not yet implemented");
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);

	/* initial filling of the widgets */
	query_changed_cb (qedit->priv->query, qedit);
}

static void
entry_name_changed_cb (GtkEntry *entry, QueryEditor *qedit)
{
	/* blocking recursive signal */
	g_signal_handlers_block_by_func (G_OBJECT (qedit->priv->query),
					 G_CALLBACK (query_changed_cb), qedit);
	mg_base_set_name (MG_BASE (qedit->priv->query), gtk_entry_get_text (entry));
	/* unblocking recursive signal */
	g_signal_handlers_unblock_by_func (G_OBJECT (qedit->priv->query),
					   G_CALLBACK (query_changed_cb), qedit);
}

static void
entry_description_changed_cb (GtkEntry *entry, QueryEditor *qedit)
{
	/* blocking recursive signal */
	g_signal_handlers_block_by_func (G_OBJECT (qedit->priv->query),
					 G_CALLBACK (query_changed_cb), qedit);
	mg_base_set_description (MG_BASE (qedit->priv->query), gtk_entry_get_text (entry));
	/* unblocking recursive signal */
	g_signal_handlers_unblock_by_func (G_OBJECT (qedit->priv->query),
					   G_CALLBACK (query_changed_cb), qedit);
}

static void
query_changed_cb (MgQuery *query, QueryEditor *qedit)
{
	const gchar *str;
	gchar *sql;

	/* blocking recursive signal */
	g_signal_handlers_block_by_func (G_OBJECT (qedit->priv->q_name),
					 G_CALLBACK (entry_name_changed_cb), qedit);
	g_signal_handlers_block_by_func (G_OBJECT (qedit->priv->q_descr),
					 G_CALLBACK (entry_description_changed_cb), qedit);
	/* query name */
	str = mg_base_get_name (MG_BASE (qedit->priv->query));
	if (str)
		gtk_entry_set_text (GTK_ENTRY (qedit->priv->q_name), str);
	else
		gtk_entry_set_text (GTK_ENTRY (qedit->priv->q_name), "");
	
	/* query description */
	str = mg_base_get_description (MG_BASE (qedit->priv->query));
	if (str)
		gtk_entry_set_text (GTK_ENTRY (qedit->priv->q_descr), str);
	else
		gtk_entry_set_text (GTK_ENTRY (qedit->priv->q_descr), "");

	/* SQL version */
	sql = mg_renderer_render_as_sql (MG_RENDERER (qedit->priv->query), NULL, 
					 MG_RENDERER_EXTRA_PRETTY_SQL | MG_RENDERER_EXTRA_VAL_ATTRS, NULL);
	if (sql) {
		gnome_db_editor_set_text (GNOME_DB_EDITOR (qedit->priv->sql_editor), sql, -1);
		g_free (sql);
	}
	else
		gnome_db_editor_set_text (GNOME_DB_EDITOR (qedit->priv->sql_editor), "", -1);

	/* unblocking recursive signal */
	g_signal_handlers_unblock_by_func (G_OBJECT (qedit->priv->q_name),
					 G_CALLBACK (entry_name_changed_cb), qedit);
	g_signal_handlers_unblock_by_func (G_OBJECT (qedit->priv->q_descr),
					 G_CALLBACK (entry_description_changed_cb), qedit);
}

static void
sql_query_text_apply_cb (GtkButton *button, QueryEditor *qedit)
{
	gchar *sql;
	GError *error = NULL;
	gchar *str = NULL;
	MgQuery *query = qedit->priv->query;

	sql = gnome_db_editor_get_all_text (GNOME_DB_EDITOR (qedit->priv->sql_editor));
	mg_query_set_sql_text (query, sql, &error);

	if (mg_query_get_query_type (query) ==  MG_QUERY_TYPE_NON_PARSED_SQL) {
		if (error) {
			str = g_strdup (error->message);
			g_error_free (error);
		}
		else
			str = g_strdup (_("Error (no details available)."));
	}

	if (str) {
		GtkWidget *dlg;
		gchar *msg;

		msg = g_strdup_printf (_("<big><b>SQL query analyse error:</b></big>:\n"
					 "<small>This error prevents extra treatements of this query "
					 "which will be considered as a SQL statement specific to the "
					 "connected database.</small>\n\n"
					 "%s"), str);
		dlg = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, 
					      GTK_MESSAGE_WARNING, GTK_BUTTONS_CLOSE, msg);
		gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dlg)->label), TRUE);
		g_free (msg);
		g_free (str);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
	}
}

static void
edition_mode_toggled (GtkToggleButton *button, QueryEditor *qedit)
{
	gtk_notebook_set_current_page (GTK_NOTEBOOK (qedit->priv->notebook),
				       gtk_toggle_button_get_active (button) ? 0 : 1);
}
