/* mg-entry-combo.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "mg-entry-combo.h"
#include <libmergeant/mg-server.h>
#include <libmergeant/mg-data-handler.h>
#include <libmergeant/mg-query.h>
#include <libmergeant/mg-entity.h>
#include <libmergeant/mg-field.h>
#include <libmergeant/mg-qfield.h>
#include <libmergeant/mg-parameter.h>
#include <libmergeant/mg-renderer.h>
#include <libmergeant/mg-result-set.h>
#include <libgnomedb/libgnomedb.h>
#include <libmergeant/utility.h>

#define USE_OLD_GTK_COMBO_WIDGET

static void mg_entry_combo_class_init (MgEntryComboClass *class);
static void mg_entry_combo_init (MgEntryCombo *wid);
static void mg_entry_combo_dispose (GObject   *object);

static void mg_entry_combo_set_property (GObject              *object,
					 guint                 param_id,
					 const GValue         *value,
					 GParamSpec           *pspec);
static void mg_entry_combo_get_property (GObject              *object,
					 guint                 param_id,
					 GValue               *value,
					 GParamSpec           *pspec);

static void          display_combo_model       (MgEntryCombo *combo);
static void          choose_auto_default_value (MgEntryCombo *combo);
#ifdef USE_OLD_GTK_COMBO_WIDGET
static void          set_selection_combo_model (MgEntryCombo *combo, gint pos);
#endif

static void          mg_entry_combo_emit_signal (MgEntryCombo *combo);
static void          real_combo_block_signals (MgEntryCombo *wid);
static void          real_combo_unblock_signals (MgEntryCombo *wid);

/* MgDataEntry interface (value must be a GDA_VALUE_TYPE_LIST) */
static void            mg_entry_combo_data_entry_init   (MgDataEntryIface *iface);
static void            mg_entry_combo_set_value         (MgDataEntry *de, const GdaValue * value);
static GdaValue       *mg_entry_combo_get_value         (MgDataEntry *de);
static void            mg_entry_combo_set_value_orig    (MgDataEntry *de, const GdaValue * value);
static const GdaValue *mg_entry_combo_get_value_orig    (MgDataEntry *de);
static void            mg_entry_combo_set_value_default (MgDataEntry *de, const GdaValue * value);
static void            mg_entry_combo_set_attributes    (MgDataEntry *de, guint attrs, guint mask);
static guint           mg_entry_combo_get_attributes    (MgDataEntry *de);
static gboolean        mg_entry_combo_expand_in_layout  (MgDataEntry *de);

static void            mg_conf_weak_notify (MgEntryCombo *combo, MgConf *conf);
static void            dependency_param_changed_cb (MgParameter *param, MgEntryCombo *combo);

/* signals */
enum
{
	LAST_SIGNAL
};

static gint mg_entry_combo_signals[LAST_SIGNAL] = { };

/* properties */
enum
{
        PROP_0,
        PROP_SET_DEFAULT_IF_INVALID
};

struct  _MgEntryComboPriv {
	ComboCore          *ccore;
        GtkWidget          *combo_entry;

	gboolean            data_valid;
	gboolean            null_forced;
	gboolean            default_forced;
	gboolean            null_possible;
	gboolean            default_possible;

	gboolean            show_actions;
	gboolean            set_default_if_invalid; /* use first entry when provided value is not found ? */
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
mg_entry_combo_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgEntryComboClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_entry_combo_class_init,
			NULL,
			NULL,
			sizeof (MgEntryCombo),
			0,
			(GInstanceInitFunc) mg_entry_combo_init
		};		

		static const GInterfaceInfo data_entry_info = {
			(GInterfaceInitFunc) mg_entry_combo_data_entry_init,
			NULL,
			NULL
		};

		type = g_type_register_static (MG_ENTRY_SHELL_TYPE, "MgEntryCombo", &info, 0);
		g_type_add_interface_static (type, MG_DATA_ENTRY_TYPE, &data_entry_info);
	}
	return type;
}

static void
mg_entry_combo_data_entry_init (MgDataEntryIface *iface)
{
        iface->set_value_type = NULL;
        iface->get_value_type = NULL;
        iface->set_value = mg_entry_combo_set_value;
        iface->get_value = mg_entry_combo_get_value;
        iface->set_value_orig = mg_entry_combo_set_value_orig;
        iface->get_value_orig = mg_entry_combo_get_value_orig;
        iface->set_value_default = mg_entry_combo_set_value_default;
        iface->set_attributes = mg_entry_combo_set_attributes;
        iface->get_attributes = mg_entry_combo_get_attributes;
        iface->get_handler = NULL;
        iface->expand_in_layout = mg_entry_combo_expand_in_layout;
}


static void
mg_entry_combo_class_init (MgEntryComboClass *class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = mg_entry_combo_dispose;

	/* Properties */
        object_class->set_property = mg_entry_combo_set_property;
        object_class->get_property = mg_entry_combo_get_property;
        g_object_class_install_property (object_class, PROP_SET_DEFAULT_IF_INVALID,
					 g_param_spec_boolean ("set_default_if_invalid", NULL, NULL, FALSE,
                                                               (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

#ifdef USE_OLD_GTK_COMBO_WIDGET
static void combo_contents_changed_cb (GtkWidget *entry, MgEntryCombo *combo);
#endif

static void
real_combo_block_signals (MgEntryCombo *wid)
{
#ifdef USE_OLD_GTK_COMBO_WIDGET
	g_signal_handlers_block_by_func (G_OBJECT (GTK_COMBO (wid->priv->combo_entry)->entry),
					 G_CALLBACK (combo_contents_changed_cb), wid);
#else
	TO_IMPLEMENT;
#endif
}

static void
real_combo_unblock_signals (MgEntryCombo *wid)
{
#ifdef USE_OLD_GTK_COMBO_WIDGET
	g_signal_handlers_unblock_by_func (G_OBJECT (GTK_COMBO (wid->priv->combo_entry)->entry),
					   G_CALLBACK (combo_contents_changed_cb), wid);
#else
	TO_IMPLEMENT;
#endif
}


static void
mg_entry_combo_emit_signal (MgEntryCombo *combo)
{
#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (combo), "contents_modified");
#ifdef debug_signal
	g_print ("<< 'CONTENTS_MODIFIED' from %s\n", __FUNCTION__);
#endif
}

static void
mg_entry_combo_init (MgEntryCombo *combo)
{
	/* Private structure */
	combo->priv = g_new0 (MgEntryComboPriv, 1);
	combo->priv->ccore = NULL;
	combo->priv->set_default_if_invalid = FALSE;
	combo->priv->combo_entry = NULL;
	combo->priv->data_valid = FALSE;
	combo->priv->null_forced = FALSE;
	combo->priv->default_forced = FALSE;
	combo->priv->null_possible = TRUE;
	combo->priv->default_possible = FALSE;
	combo->priv->show_actions = TRUE;
}


/**
 * mg_entry_combo_new
 * @conf: a #MgConf object
 * @context: a #MgContext object
 * @node: a #MgContextNode structure, part of @context
 *
 * Creates a new #MgEntryCombo widget. The widget is a combo box which displays a
 * selectable list of items (the items come from the result of the execution of the 'node->query'
 * SELECT query). Thus the widget allows the simultaneuos selection of one or more values
 * (one for each 'node->params') while proposing potentially "more readable" choices.
 * 
 * @node is not used afterwards.
 *
 * Returns: the new widget
 */
GtkWidget *
mg_entry_combo_new (MgConf *conf, MgContext *context, MgContextNode *node)
{
	GObject *obj;
	MgEntryCombo *combo;
	GSList *list;
	GList *values;
	MgQueryType qtype;
	GtkWidget *entry;
	gboolean null_possible;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	g_return_val_if_fail (context && IS_MG_CONTEXT (context), NULL);
	g_return_val_if_fail (node, NULL);
	g_return_val_if_fail (g_slist_find (context->nodes, node), NULL);
	g_return_val_if_fail (node->query, NULL);
	qtype = mg_query_get_query_type (node->query);
	g_return_val_if_fail ((qtype == MG_QUERY_TYPE_SELECT) ||
			      (qtype == MG_QUERY_TYPE_UNION) ||
			      (qtype == MG_QUERY_TYPE_INTERSECT), NULL);

	obj = g_object_new (MG_ENTRY_COMBO_TYPE, NULL);
	combo = MG_ENTRY_COMBO (obj);

	/* create the ComboCore structure */
	combo->priv->ccore = utility_combo_initialize_core (conf, context, node, 
							   G_CALLBACK (dependency_param_changed_cb), combo);
	g_object_weak_ref (G_OBJECT (combo->priv->ccore->conf),
			   (GWeakNotify) mg_conf_weak_notify, combo);

	/* create the combo itself */
#ifdef USE_OLD_GTK_COMBO_WIDGET
	entry = gtk_combo_new ();
	mg_entry_shell_pack_entry (MG_ENTRY_SHELL (combo), entry);
	gtk_widget_show (entry);
	combo->priv->combo_entry = entry;
	g_signal_connect (G_OBJECT (GTK_COMBO (entry)->entry), "changed",
			  G_CALLBACK (combo_contents_changed_cb), combo);
	gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (entry)->entry), FALSE);
#else
	TO_IMPLEMENT;
	entry = NULL;
#endif

	/* create the data model */
	utility_combo_compute_model (combo->priv->ccore);

	/* find an intelligent original value: the ones provided by the parameters themselves */
	null_possible = TRUE;
	values = NULL;
	list = combo->priv->ccore->nodes;
	while (list) {
		const GdaValue *init = mg_parameter_get_value (COMBO_NODE (list->data)->param);
		values = g_list_append (values, init);

		if (mg_parameter_get_not_null (COMBO_NODE (list->data)->param))
			null_possible = FALSE;
		
		list = g_slist_next (list);
	}
	combo->priv->null_possible = null_possible;
	display_combo_model (combo);
	mg_entry_combo_set_values (combo, values);
	g_list_free (values);

	return GTK_WIDGET (obj);
}


/*
 * Displays the combo->priv->ccore->data_model model in the real combo widget, without emitting
 * a single signal.
 */
static void
display_combo_model (MgEntryCombo *combo)
{
#ifdef USE_OLD_GTK_COMBO_WIDGET
	GList *strings = NULL, *list;
	GtkWidget *combo_entry = combo->priv->combo_entry;

	strings = utility_combo_compute_choice_strings (combo->priv->ccore);
	if (combo->priv->null_possible) 
		strings = g_list_prepend (strings, g_strdup (""));
	
	real_combo_block_signals (combo);
	gtk_combo_set_popdown_strings (GTK_COMBO (combo_entry), strings);	      
	gtk_combo_set_value_in_list (GTK_COMBO (combo_entry), TRUE, FALSE);
	real_combo_unblock_signals (combo);
	list = strings;
	while (list) {
		g_free (list->data);
		list = g_list_next (list);
	}
	g_list_free (strings);
#else
	/* TODO: use a widget which operates with something similar to GtkTreeIter where
	 * it is possible to add / removed the 1st empty line without making all over again the
	 * complete list of possible selectable items */
	TO_IMPLEMENT;
#endif
}


static void
choose_auto_default_value (MgEntryCombo *combo)
{
#ifdef USE_OLD_GTK_COMBO_WIDGET
	gint pos = 0;
	if (combo->priv->null_possible)
		pos ++;
	set_selection_combo_model (combo, pos);
	combo_contents_changed_cb (NULL, combo);
#else
	TO_IMPLEMENT;
#endif
}

static void
mg_conf_weak_notify (MgEntryCombo *combo, MgConf *conf)
{
	/* render non sensitive */
	gtk_widget_set_sensitive (GTK_WIDGET (combo), FALSE);
}

static void
dependency_param_changed_cb (MgParameter *param, MgEntryCombo *combo)
{
	GList *values = NULL;

	if (combo->priv->data_valid) 
		values = mg_entry_combo_get_values (combo);

	utility_combo_compute_model (combo->priv->ccore);
	display_combo_model (combo);

	if (values) {
		GList *list;

		mg_entry_combo_set_values (combo, values);
		list = values;
		while (list) {
			gda_value_free ((GdaValue *)list->data);
			list = g_list_next (list);
		}
		g_list_free (values);
	}

	if (!combo->priv->data_valid) 
		choose_auto_default_value (combo);
}

static void
mg_entry_combo_dispose (GObject *object)
{
	MgEntryCombo *combo;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_ENTRY_COMBO (object));

	combo = MG_ENTRY_COMBO (object);

	if (combo->priv) {		
		if (combo->priv->ccore) {			
			/* Weak unref the MgConf is necessary */
			if (combo->priv->ccore->conf)
				g_object_weak_unref (G_OBJECT (combo->priv->ccore->conf),
						     (GWeakNotify) mg_conf_weak_notify, combo);
			
			utility_combo_free_core (combo->priv->ccore);
			combo->priv->ccore = NULL;
		}

		g_free (combo->priv);
		combo->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void 
mg_entry_combo_set_property (GObject              *object,
			     guint                 param_id,
			     const GValue         *value,
			     GParamSpec           *pspec)
{
	MgEntryCombo *combo = MG_ENTRY_COMBO (object);
	if (combo->priv) {
		switch (param_id) {
		case PROP_SET_DEFAULT_IF_INVALID:
			if (combo->priv->set_default_if_invalid != g_value_get_boolean (value)) {
				guint attrs;

				combo->priv->set_default_if_invalid = g_value_get_boolean (value);
				attrs = mg_data_entry_get_attributes (MG_DATA_ENTRY (combo));

				if (combo->priv->set_default_if_invalid && (attrs & MG_DATA_ENTRY_DATA_NON_VALID)) {
					if (combo->priv->ccore->data_model_valid)
						choose_auto_default_value (combo);
				}
			}
			break;
		}
	}
}

static void
mg_entry_combo_get_property (GObject              *object,
			     guint                 param_id,
			     GValue               *value,
			     GParamSpec           *pspec)
{
	MgEntryCombo *combo = MG_ENTRY_COMBO (object);
	if (combo->priv) {
		switch (param_id) {
		case PROP_SET_DEFAULT_IF_INVALID:
			g_value_set_boolean (value, combo->priv->set_default_if_invalid);
			break;
		}
	}
}

#ifdef USE_OLD_GTK_COMBO_WIDGET
static void
combo_contents_changed_cb (GtkWidget *entry, MgEntryCombo *combo)
{
	gint selection;
	
	if (!GTK_LIST (GTK_COMBO (combo->priv->combo_entry)->list)->selection)
		return;

	selection = g_list_index (GTK_LIST (GTK_COMBO (combo->priv->combo_entry)->list)->children,
				  GTK_LIST (GTK_COMBO (combo->priv->combo_entry)->list)->selection->data);
	
	if ((selection == 0) && combo->priv->null_possible) /* Set to NULL? */
		mg_entry_combo_set_values (combo, NULL);
	else {
		if (combo->priv->null_possible)
			selection --; /* remove the "" entry count */

		if (combo->priv->ccore->data_model_valid) {
			combo->priv->null_forced = FALSE;
			combo->priv->default_forced = FALSE;
			combo->priv->data_valid = TRUE;
			
			/* updating the node's values */
			if (combo->priv->ccore->data_model && combo->priv->ccore->data_model_valid &&
			    GTK_LIST (GTK_COMBO (combo->priv->combo_entry)->list)->selection) {
				GSList *list;
				list = combo->priv->ccore->nodes;
				while (list) {
					ComboNode *node = COMBO_NODE (list->data);
					node->value = gda_data_model_get_value_at (combo->priv->ccore->data_model, 
										   node->position, selection);
					/* g_print ("%s(): Set Node value to %s\n", __FUNCTION__, */
/* 						 node->value ? gda_value_stringify (node->value) : "Null"); */
					list = g_slist_next (list);
				}
			}
			
			g_signal_emit_by_name (G_OBJECT (combo), "status_changed");
			mg_entry_combo_emit_signal (combo);
		}
	}
}


/* Forces the selection of an item in the items list, without emitting a single signal */
static void
set_selection_combo_model (MgEntryCombo *combo, gint pos)
{
	real_combo_block_signals (combo);
	gtk_list_select_item (GTK_LIST (GTK_COMBO (combo->priv->combo_entry)->list), pos);
	real_combo_unblock_signals (combo);
}

#endif

/**
 * mg_entry_combo_set_values
 * @combo: a #MgEntryCombo widet
 * @values: a list of #GdaValue values, or %NULL
 *
 * Sets the values of @combo to the specified ones. None of the
 * values provided in the list is modified.
 *
 * An error can occur when there is no corresponding value(s) to be displayed
 * for the provided values.
 *
 * If @values is %NULL, then the entry itself is set to NULL, and no error is returned if the entry
 * can be NULL.
 */
void
mg_entry_combo_set_values (MgEntryCombo *combo, GList *values)
{
	gboolean err = FALSE;
	gint current_index = -1;
	gboolean allnull = TRUE;
	GList *list;

	g_return_if_fail (combo && IS_MG_ENTRY_COMBO (combo));
	g_return_if_fail (combo->priv);

	/* try to determine if all the values are NULL or of type GDA_VALUE_TYPE_NULL */
	list = values;
	while (list && allnull) {
		if (list->data && (gda_value_get_type ((GdaValue *)list->data) != GDA_VALUE_TYPE_NULL))
			allnull = FALSE;
		list = g_list_next (list);
	}

	/* actual values settings */
	if (!allnull) {
		g_return_if_fail (g_list_length (values) == g_slist_length (combo->priv->ccore->nodes));
		
		if (combo->priv->ccore->data_model_valid) {
			gint row = 0, nbrows;
			
			nbrows = gda_data_model_get_n_rows (combo->priv->ccore->data_model);
			while ((row < nbrows) && (current_index == -1)) {
				GSList *nodes = combo->priv->ccore->nodes;
				GList *argptr = values;
				gboolean equal = TRUE;
				ComboNode *node;
				const GdaValue *model_value, *arg_value;
				
				while (argptr && equal) {
					GdaValueType type1=GDA_VALUE_TYPE_NULL, type2=GDA_VALUE_TYPE_NULL;
					
					node = COMBO_NODE (nodes->data);
					model_value = gda_data_model_get_value_at (combo->priv->ccore->data_model, 
										   node->position, row);
					arg_value = (GdaValue*) (argptr->data);
					
					if (arg_value)
						type1 = gda_value_get_type (arg_value);
					if (model_value)
						type2 = gda_value_get_type (model_value);
					if (type1 != type2)
						equal = FALSE;
					else
						if (type1 != GDA_VALUE_TYPE_NULL)
							equal = !gda_value_compare (model_value, arg_value);
					
					nodes = g_slist_next (nodes);
					argptr = g_list_next (argptr);
				}
				
				if (equal) {
					current_index = row;
					nodes = combo->priv->ccore->nodes;
					argptr = values;

					while (nodes) {
						node = COMBO_NODE (nodes->data);
						node->value = gda_data_model_get_value_at (combo->priv->ccore->data_model, 
											   node->position, row);
						
						nodes = g_slist_next (nodes);
						argptr = g_list_next (argptr);
					}
				}
				row++;
			}
		}
		
		if (current_index == -1) 
			err = TRUE;
		else {
			if (combo->priv->null_possible)
				current_index++;

			combo->priv->null_forced = FALSE;
			combo->priv->default_forced = FALSE;
#ifdef USE_OLD_GTK_COMBO_WIDGET
			set_selection_combo_model (combo, current_index);
#else
			TO_IMPLEMENT;
#endif
		}
	}
	else  { /* set to NULL */
		if (combo->priv->null_possible) {
			GSList *list;
		
			/* the 1st entry in the real combo is already a "" entry, so select it. */
#ifdef USE_OLD_GTK_COMBO_WIDGET
			set_selection_combo_model (combo, 0);
#else
			TO_IMPLEMENT;
#endif

			/* adjusting the values */
			list = combo->priv->ccore->nodes;
			while (list) {
				COMBO_NODE (list->data)->value = NULL;
				
				list = g_slist_next (list);
			}
			combo->priv->null_forced = TRUE;
		}
		else 
			err = TRUE;
	}

	combo->priv->data_valid = !err;
	g_signal_emit_by_name (G_OBJECT (combo), "status_changed");

	if (!err) 
		/* notify the status and contents changed */
		mg_entry_combo_emit_signal (combo);
}

/**
 * mg_entry_combo_get_values
 * @combo: a #MgEntryCombo widet
 *
 * Get the values stored within @combo. The returned values are a copy of the ones
 * within @combo, so they must be freed afterwards, the same for the list.
 *
 * Returns: a new list of values
 */
GList *
mg_entry_combo_get_values (MgEntryCombo *combo)
{
	GSList *list;
	GList *retval = NULL;
	g_return_val_if_fail (combo && IS_MG_ENTRY_COMBO (combo), NULL);
	g_return_val_if_fail (combo->priv, NULL);

	list = combo->priv->ccore->nodes;
	while (list) {
		ComboNode *node = COMBO_NODE (list->data);
		
		if (node->value) 
			retval = g_list_append (retval, gda_value_copy (node->value));
		else 
			retval = g_list_append (retval, gda_value_new_null ());

		list = g_slist_next (list);
	}

	return retval;
}

/**
 * mg_entry_combo_set_values_orig
 * @combo: a #MgEntryCombo widet
 * @values: a list of #GdaValue values
 *
 * Sets the original values of @combo to the specified ones. None of the
 * values provided in the list is modified.
 */
void
mg_entry_combo_set_values_orig (MgEntryCombo *combo, GList *values)
{
	GSList *list;

	g_return_if_fail (combo && IS_MG_ENTRY_COMBO (combo));
	g_return_if_fail (combo->priv);

	 mg_entry_combo_set_values (combo, values);

	/* clean all the orig values */
	list = combo->priv->ccore->nodes;
	while (list) {
		ComboNode *node = COMBO_NODE (list->data);
		if (node->value_orig) {
			gda_value_free (node->value_orig);
			node->value_orig = NULL;
		}
		
		list = g_slist_next (list);
	}

	if (values) {
		GSList *nodes;
		GList *argptr;
		const GdaValue *arg_value;
		gboolean equal = TRUE;
		
		g_return_if_fail (g_list_length (values) == g_slist_length (combo->priv->ccore->nodes));
		
		/* 
		 * first make sure the value types are the same as for the data model 
		 */
		nodes = combo->priv->ccore->nodes;
		argptr = values;
		while (argptr && nodes && equal) {
			GdaFieldAttributes *attrs;
			GdaValueType type=GDA_VALUE_TYPE_NULL;
			
			attrs = gda_data_model_describe_column (combo->priv->ccore->data_model, 
								COMBO_NODE (nodes->data)->position);
			arg_value = (GdaValue*) (argptr->data);
			
			if (arg_value)
				type = gda_value_get_type (arg_value);
			equal = (type == attrs->gda_type);
			
			nodes = g_slist_next (nodes);
			argptr = g_list_next (argptr);
		}
		
		/* 
		 * then, actual copy of the values
		 */
		if (equal) {
			nodes = combo->priv->ccore->nodes;
			argptr = values;
			while (argptr && nodes && equal) {
				if (argptr->data)
					COMBO_NODE (nodes->data)->value_orig = gda_value_copy ((GdaValue*) (argptr->data));
				nodes = g_slist_next (nodes);
				argptr = g_list_next (argptr);
			}
		} 
	}
}

/**
 * mg_entry_combo_get_values_orig
 * @combo: a #MgEntryCombo widet
 *
 * Get the original values stored within @combo. The returned values are the ones
 * within @combo, so they must not be freed afterwards; the list has to be freed afterwards.
 *
 * Returns: a new list of values
 */
GList *
mg_entry_combo_get_values_orig (MgEntryCombo *combo)
{
	GSList *list;
	GList *retval = NULL;
	gboolean allnull = TRUE;

	g_return_val_if_fail (combo && IS_MG_ENTRY_COMBO (combo), NULL);
	g_return_val_if_fail (combo->priv, NULL);

	list = combo->priv->ccore->nodes;
	while (list) {
		ComboNode *node = COMBO_NODE (list->data);

		if (node->value_orig && 
		    (gda_value_get_type (node->value_orig) != GDA_VALUE_TYPE_NULL))
			allnull = FALSE;
		
		retval = g_list_append (retval, node->value_orig);
		
		list = g_slist_next (list);
	}

	if (allnull) {
		g_list_free (retval);
		retval = NULL;
	}

	return retval;
}

/**
 * mg_entry_combo_set_values_default
 * @combo: a #MgEntryCombo widet
 * @values: a list of #GdaValue values
 *
 * Sets the default values of @combo to the specified ones. None of the
 * values provided in the list is modified.
 */
void
mg_entry_combo_set_values_default (MgEntryCombo *combo, GList *values)
{
	g_return_if_fail (combo && IS_MG_ENTRY_COMBO (combo));
	g_return_if_fail (combo->priv);
	TO_IMPLEMENT;
}


/* 
 * MgDataEntry Interface implementation 
 */

static void
mg_entry_combo_set_value (MgDataEntry *iface, const GdaValue *value)
{
	MgEntryCombo *combo;
                                                                                                                                  
        g_return_if_fail (iface && IS_MG_ENTRY_COMBO (iface));
        combo = MG_ENTRY_COMBO (iface);
        g_return_if_fail (combo->priv);
        g_return_if_fail (!value ||
                          (value && (gda_value_isa (value, GDA_VALUE_TYPE_LIST) ||
                                     gda_value_isa (value, GDA_VALUE_TYPE_LIST))));

	TO_IMPLEMENT;
}

static GdaValue *
mg_entry_combo_get_value (MgDataEntry *iface)
{
        MgEntryCombo *combo;
                                                                                                                                  
        g_return_val_if_fail (iface && IS_MG_ENTRY_COMBO (iface), NULL);
        combo = MG_ENTRY_COMBO (iface);
        g_return_val_if_fail (combo->priv, NULL);
                                                                                                                                  
	TO_IMPLEMENT;
	
	return NULL;
}

static void
mg_entry_combo_set_value_orig (MgDataEntry *iface, const GdaValue * value)
{
        MgEntryCombo *combo;
                                                                                                                                  
        g_return_if_fail (iface && IS_MG_ENTRY_COMBO (iface));
        combo = MG_ENTRY_COMBO (iface);
        g_return_if_fail (combo->priv);
                                                                                                                                  
	TO_IMPLEMENT;
}
                                                                                                                                  
static const GdaValue *
mg_entry_combo_get_value_orig (MgDataEntry *iface)
{
        MgEntryCombo *combo;
                                                                                                                                  
        g_return_val_if_fail (iface && IS_MG_ENTRY_COMBO (iface), NULL);
        combo = MG_ENTRY_COMBO (iface);
        g_return_val_if_fail (combo->priv, NULL);
                     
	TO_IMPLEMENT;

	return NULL;
}

static void
mg_entry_combo_set_value_default (MgDataEntry *iface, const GdaValue * value)
{
        MgEntryCombo *combo;
                                                                                                                                  
        g_return_if_fail (iface && IS_MG_ENTRY_COMBO (iface));
        combo = MG_ENTRY_COMBO (iface);
        g_return_if_fail (combo->priv);
                                                                                                                                  
	TO_IMPLEMENT;
}


static void
mg_entry_combo_set_attributes (MgDataEntry *iface, guint attrs, guint mask)
{
	MgEntryCombo *combo;

	g_return_if_fail (iface && IS_MG_ENTRY_COMBO (iface));
	combo = MG_ENTRY_COMBO (iface);
	g_return_if_fail (combo->priv);

	/* Setting to NULL */
	if (mask & MG_DATA_ENTRY_IS_NULL) {
		if ((mask & MG_DATA_ENTRY_CAN_BE_NULL) &&
		    !(attrs & MG_DATA_ENTRY_CAN_BE_NULL))
			g_return_if_reached ();
		if (attrs & MG_DATA_ENTRY_IS_NULL) {
			mg_entry_combo_set_values (combo, NULL);
			
			/* if default is set, see if we can keep it that way */
			if (combo->priv->default_forced) {
				GSList *list;
				gboolean allnull = TRUE;
				list = combo->priv->ccore->nodes;
				while (list && allnull) {
					if (COMBO_NODE (list->data)->value_default && 
					    (gda_value_get_type (COMBO_NODE (list->data)->value_default) != 
					     GDA_VALUE_TYPE_NULL))
						allnull = FALSE;
					list = g_slist_next (list);
				}

				if (!allnull)
					combo->priv->default_forced = FALSE;
			}

			mg_entry_combo_emit_signal (combo);
			return;
		}
		else {
			combo->priv->null_forced = FALSE;
			mg_entry_combo_emit_signal (combo);
		}
	}

	/* Can be NULL ? */
	if (mask & MG_DATA_ENTRY_CAN_BE_NULL)
		if (combo->priv->null_possible != (attrs & MG_DATA_ENTRY_CAN_BE_NULL) ? TRUE : FALSE) {
			combo->priv->null_possible = (attrs & MG_DATA_ENTRY_CAN_BE_NULL) ? TRUE : FALSE;
			display_combo_model (combo);
		}


	/* Setting to DEFAULT */
	if (mask & MG_DATA_ENTRY_IS_DEFAULT) {
		if ((mask & MG_DATA_ENTRY_CAN_BE_DEFAULT) &&
		    !(attrs & MG_DATA_ENTRY_CAN_BE_DEFAULT))
			g_return_if_reached ();
		if (attrs & MG_DATA_ENTRY_IS_DEFAULT) {
			GList *tmplist = NULL;
			GSList *list;
			
			list = combo->priv->ccore->nodes;
			while (list) {
				tmplist = g_list_append (tmplist, COMBO_NODE (list->data)->value_default);
				list = g_slist_next (list);
			}
			mg_entry_combo_set_values (combo, tmplist);
			g_list_free (tmplist);

			/* if NULL is set, see if we can keep it that way */
			if (combo->priv->null_forced) {
				GSList *list;
				gboolean allnull = TRUE;
				list = combo->priv->ccore->nodes;
				while (list && allnull) {
					if (COMBO_NODE (list->data)->value_default && 
					    (gda_value_get_type (COMBO_NODE (list->data)->value_default) != 
					     GDA_VALUE_TYPE_NULL))
						allnull = FALSE;
					list = g_slist_next (list);
				}
				
				if (!allnull)
					combo->priv->null_forced = FALSE;
			}

			combo->priv->default_forced = TRUE;
			mg_entry_combo_emit_signal (combo);
			return;
		}
		else {
			combo->priv->default_forced = FALSE;
			mg_entry_combo_emit_signal (combo);
		}
	}

	/* Can be DEFAULT ? */
	if (mask & MG_DATA_ENTRY_CAN_BE_DEFAULT)
		combo->priv->default_possible = (attrs & MG_DATA_ENTRY_CAN_BE_DEFAULT) ? TRUE : FALSE;
	
	/* Modified ? */
	if (mask & MG_DATA_ENTRY_IS_UNCHANGED) {
		if (attrs & MG_DATA_ENTRY_IS_UNCHANGED) {
			GList *tmplist = NULL;
			GSList *list;
			
			list = combo->priv->ccore->nodes;
			while (list) {
				tmplist = g_list_append (tmplist, COMBO_NODE (list->data)->value_orig);
				list = g_slist_next (list);
			}
				
			mg_entry_combo_set_values (combo, tmplist);
			g_list_free (tmplist);
			combo->priv->default_forced = FALSE;
			mg_entry_combo_emit_signal (combo);
		}
	}

	/* Actions buttons ? */
	if (mask & MG_DATA_ENTRY_ACTIONS_SHOWN) {
		GValue *gval;
		combo->priv->show_actions = (attrs & MG_DATA_ENTRY_ACTIONS_SHOWN) ? TRUE : FALSE;
		
		gval = g_new0 (GValue, 1);
		g_value_init (gval, G_TYPE_BOOLEAN);
		g_value_set_boolean (gval, combo->priv->show_actions);
		g_object_set_property (G_OBJECT (combo), "actions", gval);
		g_free (gval);
	}

	/* NON WRITABLE attributes */
	if (mask & MG_DATA_ENTRY_DATA_NON_VALID) 
		g_warning ("Can't force a MgDataEntry to be invalid!");

	if (mask & MG_DATA_ENTRY_HAS_VALUE_ORIG)
		g_warning ("Having an original value is not a write attribute on MgDataEntry!");

	g_signal_emit_by_name (G_OBJECT (combo), "status_changed");
}

static guint
mg_entry_combo_get_attributes (MgDataEntry *iface)
{
	guint retval = 0;
	MgEntryCombo *combo;
	GSList *list;
	GList *list2;
	gboolean isnull = TRUE;
	gboolean isunchanged = TRUE;
	gboolean orig_value_exists = FALSE;

	g_return_val_if_fail (iface && IS_MG_ENTRY_COMBO (iface), 0);
	combo = MG_ENTRY_COMBO (iface);
	g_return_val_if_fail (combo->priv, 0);

	list = combo->priv->ccore->nodes;
	while (list) {
		gboolean changed = FALSE;

		/* NULL? */
		if (COMBO_NODE (list->data)->value &&
		    (gda_value_get_type (COMBO_NODE (list->data)->value) != GDA_VALUE_TYPE_NULL))
			isnull = FALSE;
		
		/* is unchanged */
		if (COMBO_NODE (list->data)->value_orig) {
			orig_value_exists = TRUE;

			if (COMBO_NODE (list->data)->value && 
			    (gda_value_get_type (COMBO_NODE (list->data)->value) == 
			     gda_value_get_type (COMBO_NODE (list->data)->value_orig))) {
				if (gda_value_get_type (COMBO_NODE (list->data)->value) == GDA_VALUE_TYPE_NULL)
					changed = FALSE;
				else {
					if (gda_value_compare (COMBO_NODE (list->data)->value, 
							       COMBO_NODE (list->data)->value_orig))
						changed = TRUE;
				}
			}
			else
				changed = TRUE;
		}

		if (changed || !orig_value_exists)
			isunchanged = FALSE;

		list = g_slist_next (list);
	}

	if (isunchanged)
		retval = retval | MG_DATA_ENTRY_IS_UNCHANGED;

	if (isnull || combo->priv->null_forced)
		retval = retval | MG_DATA_ENTRY_IS_NULL;

	/* can be NULL? */
	if (combo->priv->null_possible) 
		retval = retval | MG_DATA_ENTRY_CAN_BE_NULL;
	
	/* is default */
	if (combo->priv->default_forced)
		retval = retval | MG_DATA_ENTRY_IS_DEFAULT;
	
	/* can be default? */
	if (combo->priv->default_possible)
		retval = retval | MG_DATA_ENTRY_CAN_BE_DEFAULT;
	

	/* actions shown */
	if (combo->priv->show_actions)
		retval = retval | MG_DATA_ENTRY_ACTIONS_SHOWN;

	/* data valid? */
	if (! combo->priv->data_valid)
		retval = retval | MG_DATA_ENTRY_DATA_NON_VALID;
	else {
		GSList *nodes;
		gboolean allnull = TRUE;
		
		nodes = combo->priv->ccore->nodes;
 		while (nodes) {
			ComboNode *node = COMBO_NODE (nodes->data);

			/* all the nodes are NULL ? */
			if (node->value && (gda_value_get_type (node->value) != GDA_VALUE_TYPE_NULL))
				allnull = FALSE;

			nodes = g_slist_next (nodes);
		}

		if ((allnull && !combo->priv->null_possible) ||
		    (combo->priv->null_forced && !combo->priv->null_possible))
			retval = retval | MG_DATA_ENTRY_DATA_NON_VALID;
	}

	/* has original value? */
	list2 = mg_entry_combo_get_values_orig (combo);
	if (list2) {
		retval = retval | MG_DATA_ENTRY_HAS_VALUE_ORIG;
		g_list_free (list2);
	}

	return retval;
}


static gboolean
mg_entry_combo_expand_in_layout (MgDataEntry *iface)
{
	MgEntryCombo *combo;

	g_return_val_if_fail (iface && IS_MG_ENTRY_COMBO (iface), FALSE);
	combo = MG_ENTRY_COMBO (iface);
	g_return_val_if_fail (combo->priv, FALSE);

	return FALSE;
}
