/* query-field.c
 *
 * Copyright (C) 2001 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "query.h"
#include "query-field-private.h"
#include "marshal.h"


static void query_field_class_init (QueryFieldClass * class);
static void query_field_init (QueryField * srv);
static void query_field_dispose (GObject   * object);

static void m_modified (QueryField *field);
/* 
 * Main static functions 
 */
static QueryFieldIface *find_interface (QueryField *qf);
static void init_types_list (QueryFieldClass * class);



/*
 * static variables 
 */
/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* get a pointer on the class of QueryField for static data access */
static QueryFieldClass *query_field_class = NULL;

enum
{
	FIELD_MODIFIED,
	FIELD_TYPE_CHANGED,
	PRINT_CHANGED,
	NAME_CHANGED,
	ALIAS_CHANGED,
	STATUS_CHANGED,
	NULLIFIED,
	LAST_SIGNAL
};

static gint query_field_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0, 0, 0 };


guint
query_field_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (QueryFieldClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) query_field_class_init,
			NULL,
			NULL,
			sizeof (QueryField),
			0,
			(GInstanceInitFunc) query_field_init
		};		

		type = g_type_register_static (G_TYPE_OBJECT, "QueryField", &info, 0);
	}
	return type;
}

static void
query_field_class_init (QueryFieldClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	query_field_class = class;
	parent_class = g_type_class_peek_parent (class);

	query_field_signals[FIELD_MODIFIED] =
		g_signal_new ("field_modified",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryFieldClass, field_modified),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	query_field_signals[FIELD_TYPE_CHANGED] =
		g_signal_new ("field_type_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryFieldClass, field_type_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	query_field_signals[NAME_CHANGED] =
		g_signal_new ("name_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryFieldClass, name_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	query_field_signals[PRINT_CHANGED] =
		g_signal_new ("print_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryFieldClass, print_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	query_field_signals[ALIAS_CHANGED] =
		g_signal_new ("alias_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryFieldClass, alias_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	query_field_signals[STATUS_CHANGED] =
		g_signal_new ("status_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryFieldClass, status_changed),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	query_field_signals[NULLIFIED] =
		g_signal_new ("nullified",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (QueryFieldClass, nullified),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);

	class->field_modified = 
		(void (*)(QueryField * field)) m_modified;;
	class->field_type_changed = NULL;
	class->print_changed = NULL;
	class->name_changed = NULL;
	class->alias_changed = NULL;
	class->status_changed = NULL;
	class->nullified = NULL;

	/* need to initialize the possible object types */
	init_types_list (class);

	object_class->dispose = query_field_dispose;
}

/* this function creates a list of all the possible query object types */
static void init_types_list (QueryFieldClass * class)
{
	class->field_types = NULL;
	class->field_types = g_slist_append (class->field_types, query_field_field_get_iface());
	class->field_types = g_slist_append (class->field_types, query_field_allfields_get_iface());
	class->field_types = g_slist_append (class->field_types, query_field_function_get_iface());
	class->field_types = g_slist_append (class->field_types, query_field_value_get_iface());
	/* ... */
}

static void
query_field_init (QueryField * qf)
{
	qf->priv = g_new0 (QueryFieldPrivate, 1);
	qf->priv->query = NULL;
	qf->priv->name = NULL;
	qf->priv->alias = NULL;
	qf->priv->is_printed = FALSE;
	qf->priv->is_hidden = FALSE;
	qf->priv->activated = FALSE;
	qf->priv->private_data = NULL; /* safe value */
	qf->priv->id = 0;
	qf->priv->ref_counter = 0;
}

GObject   *
query_field_new (Query *q, gchar * name, QueryFieldType field_type)
{
	GObject   *obj;
	QueryField *qf;
	QueryFieldIface *iface;

	g_return_val_if_fail (q && IS_QUERY (q), NULL);

	obj = g_object_new (QUERY_FIELD_TYPE, NULL);
	qf = QUERY_FIELD (obj);

	qf->priv->query = q;

	qf->priv->id = query_get_unique_field_id (q, 0);
	if (name)
		qf->priv->name = g_strdup (name);

	qf->priv->field_type = field_type;

	iface = find_interface (qf);
	g_assert (iface != NULL);
	(* iface->init)(qf);

	return obj;
}

/* detects the kind of object to build and makes a new one. q can't be NULL */
GObject   *
query_field_new_objref (Query *q, gchar * name, gchar *type)
{
	QueryFieldType field_type = QUERY_FIELD_LAST;
	gboolean found = FALSE;
	GObject   *obj = NULL;

	g_return_val_if_fail (q && IS_QUERY (q), NULL);

	if ((*type == 'f') && (*(type+1) == 'i')) {
		field_type = QUERY_FIELD_FIELD;
		found = TRUE;
	}

	if ((!found) && (*type == 'a') && (*(type+1) == 'l')) { 
		field_type = QUERY_FIELD_ALLFIELDS;
		found = TRUE;
	}

	if ((!found) && (*type == 'a') && (*(type+1) == 'g')) {
		field_type = QUERY_FIELD_AGGREGATE;
		found = TRUE;
	}

	if ((!found) && (*type == 'f') && (*(type+1) == 'u')) { 
		field_type = QUERY_FIELD_FUNCTION;
		found = TRUE;
	}

	if ((!found) && (*type == 'v') && (*(type+1) == 'a')) { 
		field_type = QUERY_FIELD_VALUE;
		found = TRUE;
	}

	if ((!found) && (*type == 'q') && (*(type+1) == 'u') && (strlen(type)==5)) { 
		field_type = QUERY_FIELD_QUERY;
		found = TRUE;
	}

	if ((!found) && (*type == 'q') && (*(type+1) == 'u')) { 
		field_type = QUERY_FIELD_QUERY_FIELD;
		found = TRUE;
	}


	if (found) 
		obj = query_field_new (q, name, field_type);
	else
		g_warning ("QueryField'type not found at %s line %d\n", __FILE__, __LINE__);

	return obj;
}

/* creates a QueryField which is a copy of the one given in arg */
GObject   *
query_field_new_copy (QueryField *field)
{
	GObject   *obj;
	QueryField *newfield;
	QueryFieldIface *iface;

	g_return_val_if_fail (field && IS_QUERY_FIELD (field), NULL);
       
	obj = query_field_new (field->priv->query, field->priv->name, field->priv->field_type);
	newfield = QUERY_FIELD (obj);

	iface = find_interface (newfield);
	g_assert (iface != NULL);

	(* iface->destroy)(newfield);
	(* iface->init)(newfield);
	(* iface->copy_other_field)(newfield, field);

	/* object's name and print name */
	if (field->priv->name)
		query_field_set_name (newfield, field->priv->name);
	if (field->priv->alias)
		query_field_set_alias (newfield, field->priv->alias);

	/* set the "copy_of" attribute */
	g_object_set_data (obj, "copy_of", field);

	return obj;
}

static void 
m_modified (QueryField *field)
{
	/* set the "copy_of" attribute */
	g_object_set_data (G_OBJECT (field), "copy_of", NULL);
}

void
query_field_nullify (QueryField *qf)
{
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	if (qf->priv->activated)
		query_field_deactivate (qf);

#ifdef debug_signal
	g_print (">> 'NULLIFIED' from query_field_nullify (%p)\n", qf);
#endif
	g_signal_emit (G_OBJECT (qf), query_field_signals[NULLIFIED], 0);
#ifdef debug_signal
	g_print (">> 'NULLIFIED' from query_field_nullify (%p)\n", qf);
#endif	
}


static GSList *copy_recurs (QueryField *field, QueryField *user, GHashTable *hash, GSList *objects);

GObject   *
query_field_new_copy_all   (QueryField *field, GSList **list)
{
	GSList *nlist, *fields, *last=NULL;
	GHashTable *hash;

	g_assert (list);

	if (*list) {
		nlist = *list;
		last = g_slist_last (*list);
	}
	else
		nlist = NULL;
	hash = g_hash_table_new (NULL, NULL);

	nlist = copy_recurs (field, NULL, hash, nlist);
	g_hash_table_destroy (hash);
	
	*list = nlist;

	fields = nlist;
	while (fields) {
		g_object_set_data (G_OBJECT (fields->data), "qf_list", list);
		fields = g_slist_next (fields);
	}

	if (nlist) {
		if (last) {
			if (g_slist_next (last))
				return G_OBJECT (g_slist_next (last)->data);
			else {
				g_warning ("Copy of QueryField did not create a new QueryField!");
				return G_OBJECT (nlist->data);
			}
		}		   
		else
			return G_OBJECT (nlist->data);
	}
	else
		return NULL;
}

static GSList *
copy_recurs (QueryField *field, QueryField *user, GHashTable *hash, GSList *objects)
{
	QueryField *qf;
	gpointer ptr;
	GSList *list, *objs;
	
	g_print ("COPY RECURS copying %p (user=%p)\n", field, user);
	list = objects;
	/* we check if field has been copied and marked as such in the hash table,
	   or we create a new one, copy of field */
	if ((ptr = g_hash_table_lookup (hash, field))) 
		qf = QUERY_FIELD (ptr);
	else {
		qf = QUERY_FIELD (query_field_new_copy (field));
		list = g_slist_append (list, qf);

		/* if the qf has a name, then we don't want a duplicate and so we
		   put it into the hash table for future lookup */
		if (field->priv->name && *(field->priv->name)) {
			/* insert into hash table */
			g_hash_table_insert (hash, field, qf);
		}
	}

	/* replace the reference in the calling object (user) if any */
	if (user) {
		g_object_set_data (G_OBJECT (user), "qf_list", &list);		
		query_field_replace_ref_ptr (user, G_OBJECT (field), G_OBJECT (qf));
	}
	
	/* apply this function recursively to the monitored objects if they are
	 QueryFields */
	objs = query_field_get_monitored_objects (field);
	while (objs) {
		if (IS_QUERY_FIELD (objs->data)) 
			list = copy_recurs (QUERY_FIELD (objs->data), qf, hash, list);
		objs = g_slist_next (objs);
	}
	g_slist_free (objs);

	return list;
}

void 
query_field_copy_object (QueryField *dest, QueryField *orig)
{
	QueryFieldIface *iface1, *iface2;
	gboolean types = FALSE, name = FALSE, alias = FALSE;

	g_return_if_fail (dest != NULL);
	g_return_if_fail (orig != NULL);

	/* finding interfaces */
	iface1 = find_interface (orig);
	iface2 = find_interface (dest);
	g_assert (iface1 != NULL);
	g_assert (iface2 != NULL);

	/* keep a memory of which changes will occur to emit
	   the right signals */
	if (orig->priv->field_type != dest->priv->field_type)
		types = TRUE;
	if (strcmp (orig->priv->name, dest->priv->name))
		name = TRUE;
	if (strcmp (orig->priv->alias, dest->priv->alias))
		alias = TRUE;

	/* freeing any memory, etc that was used by this object */
	(* iface2->destroy)(dest);
	(* iface2->init)(dest);

	/* copying the other field */
	(* iface1->copy_other_field)(dest, orig);

	/* copy the name and alias */
	if (dest->priv->name) {
		g_free (dest->priv->name);
		dest->priv->name = NULL;
	}

	/* FIXME: use the right methods to do this */
	if (orig->priv->name)
		dest->priv->name = g_strdup (orig->priv->name);
	if (dest->priv->alias) {
		g_free (dest->priv->alias);
		dest->priv->alias = NULL;
		dest->priv->is_printed = FALSE;
	}
	if (orig->priv->alias) {
		dest->priv->alias = g_strdup (orig->priv->alias);
		dest->priv->is_printed = TRUE;
	}

	/* emit necessary signals */
	if (name) {
#ifdef debug_signal
		g_print (">> 'NAME_CHANGED' from query_field_copy_object\n");
#endif
		g_signal_emit (G_OBJECT (dest), query_field_signals[NAME_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'NAME_CHANGED' from query_field_copy_object\n");
#endif	
	}
	if (alias) {
#ifdef debug_signal
		g_print (">> 'ALIAS_CHANGED' from query_field_copy_object\n");
#endif
		g_signal_emit (G_OBJECT (dest), query_field_signals[ALIAS_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'ALIAS_CHANGED' from query_field_copy_object\n");
#endif	
	}
	if (types) {
#ifdef debug_signal
		g_print (">> 'FIELD_TYPE_CHANGED' from query_field_copy_object\n");
#endif
		g_signal_emit (G_OBJECT (dest), query_field_signals[FIELD_TYPE_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'FIELD_TYPE_CHANGED' from query_field_copy_object\n");
#endif	
	}

#ifdef debug_signal
	g_print (">> 'FIELD_MODIFIED' from query_field_copy_object\n");
#endif
	g_signal_emit (G_OBJECT (dest), query_field_signals[FIELD_MODIFIED], 0);
#ifdef debug_signal
	g_print ("<< 'FIELD_MODIFIED' from query_field_copy_object\n");
#endif	
}

gboolean
query_field_is_equal (QueryField *field, QueryField *other)
{
	QueryFieldIface *iface;

	if (other->priv->field_type != field->priv->field_type)
		return FALSE;

	iface = find_interface (field);
	g_assert (iface != NULL);
	return (* iface->is_equal_to)(field, other);
}

static QueryFieldIface *
find_interface (QueryField *qf)
{
	GSList *list;
	QueryFieldIface *retval = NULL;

	g_assert (qf);
	g_assert (IS_QUERY_FIELD (qf));

	g_assert (query_field_class != NULL);
	list = 	QUERY_FIELD_CLASS(query_field_class)->field_types;

	while (list && !retval) {
		if (((QueryFieldIface *)(list->data))->field_type == qf->priv->field_type)
			retval = (QueryFieldIface *)(list->data);
		list = g_slist_next (list);
	}

	return retval;
}


static void 
query_field_dispose (GObject   * object)
{
	QueryField *qf;

	g_return_if_fail (object != NULL);
        g_return_if_fail (IS_QUERY_FIELD (object));

	qf = QUERY_FIELD (object);

	if (qf->priv) {
		QueryFieldIface *iface;
		iface = find_interface (qf);
		g_assert (iface != NULL);
		(* iface->destroy)(qf);
		
		if (qf->priv->name) {
			g_free (qf->priv->name);
			qf->priv->name = NULL;
		}
		
		if (qf->priv->alias) {
			g_free (qf->priv->alias);
			qf->priv->alias = NULL;
			qf->priv->is_printed = FALSE;
		}
		g_free (qf->priv);
		qf->priv = NULL;
	}

	parent_class->dispose (object);
}


void 
query_field_set_name (QueryField * qf, const gchar * name)
{
	gboolean changed = FALSE;
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	if (name && *name) {
		if (qf->priv->name && !strcmp (qf->priv->name, name))
			return;
		else { /* needs replacement */
			gchar *str;
			
			str = g_strdup (name);

			/* look for name duplication */
			if (qf->priv->query && g_slist_find (qf->priv->query->fields, qf)) {
				guint i = 1;
				GSList *list;
				
				list = qf->priv->query->fields;
				while (list) {
					if (QUERY_FIELD (list->data)->priv->name &&
					    !strcmp (str, QUERY_FIELD (list->data)->priv->name)) {
						g_free (str);
						str = g_strdup_printf ("%s_%d", name, i++);
						list = qf->priv->query->fields;
					}
					else
						list = g_slist_next (list);
				}
			}
				
			if (qf->priv->name)
				g_free (qf->priv->name);
			qf->priv->name = str;
			changed = TRUE;
		}
	}
	else {
		if (qf->priv->name) {
			g_free (qf->priv->name);
			qf->priv->name = NULL;
			changed = TRUE;
		}
	}

	if (changed) {
#ifdef debug_signal
		g_print (">> 'NAME_CHANGED' from query_field_set_name\n");
#endif
		g_signal_emit (G_OBJECT (qf), query_field_signals[NAME_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'NAME_CHANGED' from query_field_set_name\n");
#endif
	}
}

void 
query_field_set_alias (QueryField * qf, const gchar * alias)
{
	gboolean changed = FALSE;
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	if (alias && *alias) {
		if (qf->priv->alias && !strcmp (qf->priv->alias, alias))
			return;
		else { /* needs replacement */
			gchar *str;
			
			str = g_strdup (alias);

			/* look for name duplication */
			if (qf->priv->query && g_slist_find (qf->priv->query->fields, qf)) {
				guint i = 1;
				GSList *list;
				
				list = qf->priv->query->fields;
				while (list) {
					if (QUERY_FIELD (list->data)->priv->alias &&
					    !strcmp (str, QUERY_FIELD (list->data)->priv->alias)) {
						g_free (str);
						str = g_strdup_printf ("%s_%d", alias, i++);
						list = qf->priv->query->fields;
					}
					else
						list = g_slist_next (list);
				}
			}
				
			if (qf->priv->alias)
				g_free (qf->priv->alias);
			qf->priv->is_printed = TRUE;
			qf->priv->alias = str;

			changed = TRUE;
		}
	}
	else {
		if (qf->priv->alias) {
			g_free (qf->priv->alias);
			qf->priv->alias = NULL;
			qf->priv->is_printed = FALSE;
			changed = TRUE;
		}
	}

	if (changed) {
#ifdef debug_signal
		g_print (">> 'ALIAS_CHANGED' from query_field_set_alias\n");
#endif
		g_signal_emit (G_OBJECT (qf), query_field_signals[ALIAS_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'ALIAS_CHANGED' from query_field_set_alias\n");
#endif		
	}
}

void
query_field_set_is_printed (QueryField * qf, gboolean is_printed)
{
	gboolean changed = FALSE;
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	if (is_printed != qf->priv->is_printed) {
		changed = TRUE;
		if (qf->priv->is_printed) {
			if (qf->priv->alias) {
				g_free (qf->priv->alias);
				qf->priv->alias = NULL;
			}
			qf->priv->is_printed = FALSE;
		}
		else {
			if (qf->priv->name)
				qf->priv->alias = g_strdup (qf->priv->name);
			else
				qf->priv->alias = g_strdup ("");
			qf->priv->is_printed = TRUE;
		}
	}

	if (changed) {
#ifdef debug_signal
		g_print (">> 'PRINT_CHANGED' from query_field_set_is_printed\n");
#endif
		g_signal_emit (G_OBJECT (qf), query_field_signals[PRINT_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'PRINT_CHANGED' from query_field_set_is_printed\n");
#endif		

#ifdef debug_signal
		g_print (">> 'ALIAS_CHANGED' from query_field_set_is_printed\n");
#endif
		g_signal_emit (G_OBJECT (qf), query_field_signals[ALIAS_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'ALIAS_CHANGED' from query_field_set_is_printed\n");
#endif		
	}
}

void
query_field_set_is_hidden (QueryField * qf, gboolean is_hidden)
{
	gboolean changed = FALSE;
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	if (is_hidden != qf->priv->is_hidden) {
		changed = TRUE;
		qf->priv->is_hidden = is_hidden;
	}

	if (changed) {
#ifdef debug_signal
		g_print (">> 'FIELD_MODIFIED' from query_field_set_is_hidden\n");
#endif
		g_signal_emit (G_OBJECT (qf), query_field_signals[FIELD_MODIFIED], 0);
#ifdef debug_signal
		g_print ("<< 'FIELD_MODIFIED' from query_field_set_is_hidden\n");
#endif		
	}	
}

void       
query_field_set_id (QueryField * qf, guint id)
{
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));
	g_return_if_fail (qf->priv->query);

	if (qf->priv->id != id)
		qf->priv->id = query_get_unique_field_id (qf->priv->query, id);
}

void 
query_field_set_activated (QueryField *qf, gboolean activated)
{
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	if (qf->priv->activated != activated) {
		qf->priv->activated = activated;
#ifdef debug_signal
		g_print (">> 'STATUS_CHANGED' from query_field_set_activated\n");
#endif
		g_signal_emit (G_OBJECT (qf), query_field_signals[STATUS_CHANGED], 0);
#ifdef debug_signal
		g_print ("<< 'STATUS_CHANGED' from query_field_set_activated\n");
#endif
	}
}

void 
query_field_deactivate (QueryField * qf)
{
	QueryFieldIface *iface;

	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	iface = find_interface (qf);
	g_assert (iface != NULL);
	(* iface->deactivate)(qf);
}

void 
query_field_activate (QueryField * qf)
{
	QueryFieldIface *iface;

	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	iface = find_interface (qf);
	g_assert (iface != NULL);
	(* iface->activate)(qf);
}


GtkWidget * 
query_field_get_edit_widget (QueryField *qf)
{
	QueryFieldIface *iface;

	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);

	iface = find_interface (qf);
	g_assert (iface != NULL);
	return (* iface->get_edit_widget)(qf);
}

GtkWidget * 
query_field_get_select_widget (QueryField *qf, QueryField *sel_qf, GCallback callback, gpointer data)
{
	QueryFieldIface *iface;

	g_return_val_if_fail (!qf || (qf && IS_QUERY_FIELD (qf)), NULL);

	if (qf) {
		iface = find_interface (qf);
		g_assert (iface != NULL);
		return (* iface->get_sel_widget)(qf, sel_qf, callback, data);
	}
	else { 
		/* default "???" buton */
		GtkWidget *button;

		button = gtk_button_new_with_label (_("???"));
		g_signal_connect (G_OBJECT (button), "clicked", callback, data);
		g_object_set_data (G_OBJECT (button), "qf", NULL);

		/* Set the "QF_obj_emit_sig" attribute so that we can attach attributes to that button
		   which will be transmitted when the user clicks on it */
		g_object_set_data (G_OBJECT (button), "QF_obj_emit_sig", button);

		return button;
	}
}
	
gchar     * 
query_field_render_as_sql   (QueryField *qf, GSList * missing_values)
{
	QueryFieldIface *iface;
	gchar *str, *retval;

	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);

	iface = find_interface (qf);
	g_assert (iface != NULL);

	str = (* iface->render_as_sql)(qf, missing_values);

	if ((qf->priv->field_type != QUERY_FIELD_ALLFIELDS) &&
	    (qf->priv->alias && *(qf->priv->alias))) {
		gchar *alias = query_field_get_sql_alias (qf);		
		retval = g_strdup_printf ("%s AS %s", str, alias);
		g_free (str);
		g_free (alias);
	}
	else
		retval = str;

	return retval;
}

xmlNodePtr  
query_field_render_as_xml   (QueryField *qf, GSList * missing_values)
{
	QueryFieldIface *iface;

	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);

	iface = find_interface (qf);
	g_assert (iface != NULL);
	return (* iface->render_as_xml)(qf, missing_values);
}

gchar     * 
query_field_render_as_string(QueryField *qf, GSList * missing_values)
{
	QueryFieldIface *iface;

	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);

	iface = find_interface (qf);
	g_assert (iface != NULL);
	return (* iface->render_as_string)(qf, missing_values);
}

xmlNodePtr  
query_field_save_to_xml     (QueryField *qf)
{
	QueryFieldIface *iface;
	xmlNodePtr node;
	gchar *str;

	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);

	iface = find_interface (qf);
	g_assert (iface != NULL);

	/* node creation */
	node = xmlNewNode (NULL, "QueryField");
	if (qf->priv->name)
		xmlSetProp (node, "name", qf->priv->name);
	if (qf->priv->is_printed) {
		xmlSetProp (node, "is_printed", "t");
		if (qf->priv->alias && *(qf->priv->alias))
			xmlSetProp (node, "alias", qf->priv->alias);
	}
	else
		xmlSetProp (node, "is_printed", "f");
	
	xmlSetProp (node, "is_hidden", (qf->priv->is_hidden)? "t" : "f");

	str = query_field_get_xml_id (qf);
	xmlSetProp (node, "id", str);
	g_free (str);

	/* node filling */
	(* iface->save_to_xml)(qf, node);
	/* FIXME */
	return node;
}

gchar * 
query_field_get_xml_id      (QueryField *qf)
{
	gchar *str;
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);
	
	str = g_strdup_printf ("QU%d:QF%d", qf->priv->query->id, qf->priv->id);
	
	return str;
}

void        
query_field_load_from_xml   (QueryField *qf, xmlNodePtr node)
{
	QueryFieldIface *iface;
	gchar *str;
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	iface = find_interface (qf);
	g_assert (iface != NULL);

	str = xmlGetProp (node, "name");
	if (str) {
		query_field_set_name (qf, str);
		g_free (str);
	}
	str = xmlGetProp (node, "is_printed");
	if (str) {
		if (*str == 't') {
			gchar *str2;
			str2 = xmlGetProp (node, "alias");
			if (str2) {
				query_field_set_alias (qf, str2);
				g_free (str2);
			}
			else
				query_field_set_is_printed (qf, TRUE);
		}
		else {
			query_field_set_is_printed (qf, FALSE);
		}
		g_free (str);
	}

	str = xmlGetProp (node, "is_hidden");
	if (str) {
		if (*str == 't') 
			query_field_set_is_hidden (qf, TRUE);
		else
			query_field_set_is_hidden (qf, FALSE);
		g_free (str);
	}


	str = xmlGetProp (node, "id");
	if (str) {
		gchar *ptr;
		ptr = strtok (str, ":");
		ptr = strtok (NULL, ":");
		query_field_set_id (qf, atoi (ptr+2));
		g_free (str);
	}
	
	(* iface->load_from_xml)(qf, node);
}


GSList *
query_field_get_monitored_objects (QueryField *qf)
{
	QueryFieldIface *iface;
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);

	iface = find_interface (qf);
	g_assert (iface != NULL);
	return (* iface->get_monitored_objects)(qf);
}

void 
query_field_replace_ref_ptr (QueryField *qf, GObject *old, GObject *new)
{
	QueryFieldIface *iface;
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	/* is the ptr to be raplaced the one to Query ? */
	if ((G_OBJECT (qf->priv->query) == old) &&
	    (new && IS_QUERY (new))) {
		qf->priv->query = QUERY (new);
		return;
	}

	iface = find_interface (qf);
	g_assert (iface != NULL);
	return (* iface->replace_comp)(qf, -1, old, new);
}

void 
query_field_replace_ref_int (QueryField *qf, gint ref, QueryField *new)
{
	QueryFieldIface *iface;
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	iface = find_interface (qf);
	g_assert (iface != NULL);
	return (* iface->replace_comp)(qf, ref, NULL, G_OBJECT (new));
}

const gchar *
query_field_get_name (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);
	if (qf->priv->name && *(qf->priv->name))
		return qf->priv->name;
	else
		return NULL;
}

const gchar *
query_field_get_alias (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);
	if (qf->priv->alias && *(qf->priv->alias))
		return qf->priv->alias;	
	else
		return NULL;
}

gchar *
query_field_get_sql_alias (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);
	if (qf->priv->alias && *(qf->priv->alias)) 
		return g_strdup_printf ("\"%s\"", qf->priv->alias);
	else
		return NULL;
}

QueryFieldType
query_field_get_ftype (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), QUERY_FIELD_LAST);
	return qf->priv->field_type;
}

const gchar *
query_field_get_ftype_name (QueryField *qf)
{
	QueryFieldIface *iface;
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);

	iface = find_interface (qf);
	g_assert (iface != NULL);
	return iface->pretty_name;
}

gboolean
query_field_get_is_printed (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), FALSE);
	return qf->priv->is_printed;
}

gboolean
query_field_get_is_hidden (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), TRUE);
	return qf->priv->is_hidden;
}

guint
query_field_get_id (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), 0);
	return qf->priv->id;
}

Query *
query_field_get_query (QueryField *qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), NULL);
	return qf->priv->query;
}

gboolean
query_field_get_activated (QueryField * qf)
{
	g_return_val_if_fail (qf && IS_QUERY_FIELD (qf), FALSE);
	return qf->priv->activated;
}
