/* spgs.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libbonobo.h>
#include <gnome-speech/gnome-speech.h>

#include "spgs.h"
#include "spgscb.h"
#include "libsrconf.h"
#include "SRMessages.h"


#undef SPGS_DEBUG
#undef SPGS_DEBUG_CB

typedef enum
{
    GS_ERR_GENERIC,
    GS_ERR_SERVER_ACTIVATION,
    GS_ERR_DRIVER_NAME_INVALID,
    GS_ERR_NO_VOICES,
    GS_ERR_VOICE_CREATE,
    GS_ERR_VOICE_INVALID,
    GS_ERR_BONOBO_DUPLICATE,
    GS_ERR_LAST_DEFINED
} GSErrorType;

#define GS_ERR_NO  GS_ERR_LAST_DEFINED

char *gs_errors [GS_ERR_NO] =
{
    "Generic error",
    "Activation of \"gnome-speech\" server failed.",
    "Failed to get the driver name from server",
    "Current driver has no voices",
    "Could NOT create speaker",
    "Invalid voice",
    "Failed to duplicate CORBA Object and reference it"
};
/*_________________________________< GLOBALS >________________________________*/

typedef struct 
{
    float 	min;
    float	max;
} GSMinMaxType;

typedef enum
{
    GS_PARAM_RATE,
    GS_PARAM_PITCH,
    GS_PARAM_VOLUME
}GSParameter;

typedef struct
{
    GNOME_Speech_SynthesisDriver	driver;
    gint                          	voice_idx;
    GNOME_Speech_Speaker          	speaker;
    BonoboObject        		*callback;
    gboolean				cb_support;
    /*parameters MIN and MAX*/
    GSMinMaxType			rate;
    GSMinMaxType			pitch;
    GSMinMaxType			volume;
} GSSpeaker;

typedef GHashTable GSSpeakers;
/*_________________________________</GLOBALS >________________________________*/
TTSEngine 				*tts_engine;

static GSSpeakers  			*gs_speakers_hash_table;
static GSSpeaker  			*default_speaker;
static GNOME_Speech_Speaker 		current_speaker = CORBA_OBJECT_NIL;

static CORBA_Environment		ev = { 0 };


/*___________________________________< AUX >__________________________________*/
/**
 * gs_ev:
 *
 * Initialize the exception handling mechanism of CORBA.
 *
 * Returns: Exception's data in a #CORBA_Environment structure.
 **/
CORBA_Environment *
gs_ev (void)
{
    CORBA_exception_init (&ev);
    return &ev;
}

/**
 * gs_peek_ev:
 *
 * Returns: Exception's data in a #CORBA_Environment structure.
 **/

CORBA_Environment *
gs_peek_ev (void)
{
    return &ev;
}
/**
 * gs_check_ev:
 *
 * Returns: #FALSE if there was an exception, #TRUE otherwise.
 **/
gboolean
gs_check_ev (GSErrorType	error_number,
	     int 		line)
{
    CORBA_Environment *ev = gs_peek_ev ();
    
    if ( ev && ev->_major != CORBA_NO_EXCEPTION)
    {
	char *error;
	
	error = bonobo_exception_get_text (ev);

	sru_warning ("Exception \"%s\" [\"%s\"] occured at line %d.",
		    gs_errors[error_number], error, line);

	g_free (error);
	
	CORBA_exception_free (ev);
	
	return FALSE;
    }
    return TRUE;
}
/*___________________________________</AUX >__________________________________*/


/*_____________________________< GS CALLBACK >________________________________*/
void
gs_callback (gint 	reason, 
	     gpointer 	client_data)
{
    switch (reason)
    {
	 case GNOME_Speech_speech_callback_speech_started:
#ifdef  SPGS_DEBUG_CB
	    sru_message ("gs_callback - event started"); 
#endif
    	    tts_engine->callback (TTS_EV_SPEECH_STARTED, NULL, client_data);
	    break;

	case GNOME_Speech_speech_callback_speech_ended:
#ifdef  SPGS_DEBUG_CB
	    sru_message ("gs_callback - event ended"); 
#endif
    	    tts_engine->callback (TTS_EV_END_OF_SPEECH, NULL, client_data);
	    break;

	case GNOME_Speech_speech_callback_speech_progress:
#ifdef  SPGS_DEBUG_CB
	    sru_message ("gs_callback - event progress"); 
#endif
    	    tts_engine->callback (TTS_EV_WORD_MARKER, NULL, client_data);
	    break;

	default:
    	    sru_warning ("gs_callback - unknown event\n");
    	    break;
    }
}
/*_____________________________</GS CALLBACK >________________________________*/

/*_____________________________< GS SPEAKER >_________________________________*/
GSSpeaker *
gs_speaker_new0 (GNOME_Speech_SynthesisDriver	driver,
		 gint                          	voice_idx)
{
    GSSpeaker	*gss = NULL;

    gss = g_new0 (GSSpeaker, 1);

    gss->driver 	= CORBA_OBJECT_NIL;
    gss->speaker	= CORBA_OBJECT_NIL;
    gss->callback 	= NULL;
    
    if (driver)
    {
	/* store the driver */
/*        
	GNOME_Speech_SynthesisDriver_ref (driver, gs_ev ());
	gss->driver = driver;
*/	
        gss->driver = bonobo_object_dup_ref (driver, gs_ev () ); 
	gs_check_ev (GS_ERR_GENERIC, __LINE__);
    }
    
     /* store the voice index in driver */
    gss->voice_idx = voice_idx;  

    return gss;
}

void
gs_speaker_free (GSSpeaker *gss)
{
    if (gss)
    {
#ifdef SPGS_DEBUG
	sru_debug ("gs_speaker_free : voice_idx = %d",
			    gss->voice_idx);
#endif
	if (gss->callback != CORBA_OBJECT_NIL)
	    bonobo_object_unref (gss->callback);

	if (gss->driver != CORBA_OBJECT_NIL)
	{
/*
	    GNOME_Speech_SynthesisDriver_unref(gss->driver, NULL);
*/
	    bonobo_object_release_unref (gss->driver, gs_ev () );
	    gs_check_ev (GS_ERR_GENERIC, __LINE__);
	    
	}
	
	if (gss->speaker != CORBA_OBJECT_NIL)
	{
	    GNOME_Speech_Speaker_unref(gss->speaker, gs_ev () );
	    gs_check_ev (GS_ERR_GENERIC, __LINE__);
	}
	g_free (gss);
	gss = NULL;
    }
}


void
gs_speaker_debug (GSSpeaker	*speaker)
{
#ifdef SPGS_DEBUG
    if (speaker)
    {
	sru_debug ("_____________________________");
	sru_debug ("speaker             %p", speaker);
	sru_debug ("speaker->driver     %p", speaker->driver);
	sru_debug ("speaker->voice_idx  %d", speaker->voice_idx);
	sru_debug ("speaker->speaker    %p", speaker->speaker);
	sru_debug ("speaker->callback   %p", speaker->callback);
	sru_debug ("speaker->cb_support %d", speaker->cb_support);

	sru_debug ("default_speaker     %p", default_speaker);
	sru_debug ("current_speaker     %p", current_speaker);
	sru_debug ("_____________________________");
    }
#endif
}


float
gs_speaker_procent_to_units (GSSpeaker	*speaker,
			    GSParameter	parameter,
			    float	procent)
{
    float units = 0.0;

    if (!speaker)
	return 0;
    if (procent > 100)
	procent = 100;
    
    switch (parameter)
    {
	default :
	    units = procent;
	case GS_PARAM_RATE:
	    units = speaker->rate.min + 
		    (speaker->rate.max -
		     speaker->rate.min) *
		     procent / 100;
	    break;
	case GS_PARAM_PITCH:
	    units = speaker->pitch.min + 
		    (speaker->pitch.max -
		     speaker->pitch.min) *
		     procent / 100;
	    break;
	case GS_PARAM_VOLUME:
	    units = speaker->volume.min + 
		    (speaker->volume.max -
		     speaker->volume.min) *
		     procent / 100;
	    break;    
    }
#ifdef SPGS_DEBUG
    sru_debug ("%d. procents %f , units %f",
		    parameter,
		    procent,
		    units);
#endif
    
    return units;
    
}

gchar*
gs_normilize_driver_name (gchar *driver_name)
{
    gchar *new_driver_name;
    
    new_driver_name = g_strdup (driver_name);
    new_driver_name = g_strdelimit (new_driver_name, "|> <", '_');
    
    return new_driver_name;
}

void
gs_speaker_to_gconf (GSList *driver_voice_list, 
		    const gchar *driver_name)
{
    gchar *name = gs_normilize_driver_name ((gchar*)driver_name);
    srconf_set_data (name, 
		     CFGT_LIST, 
		     driver_voice_list, 
		     SPEECH_DRIVERS_SECTION);
    g_free (name);
}
/*_____________________________</GS SPEAKER >_________________________________*/

/*_______________________< GS SPEAKERS CONTAINER >____________________________*/
void
gs_speakers_init (void)
{
    gs_speakers_hash_table = g_hash_table_new (g_str_hash,
				    	       g_str_equal);
}

void
gs_speakers_flush (gpointer 	key,
		   gpointer 	value,
		   gpointer 	user_data)
{
    if (key)
    {
        g_free (key);
        key = NULL;
    }
	
    gs_speaker_free ( (GSSpeaker *) value);
}

void
gs_speakers_terminate (void)
{
	GSList *drivers = NULL;
	GSList *tmp = NULL;
	
	if (!gs_speakers_hash_table) 
	    return;
	
	g_hash_table_foreach (gs_speakers_hash_table,
			      gs_speakers_flush,
			      NULL);
	g_hash_table_destroy (gs_speakers_hash_table);
	gs_speakers_hash_table = NULL;

	srconf_get_data_with_default(SPEECH_ENGINE_DRIVERS, 
				    CFGT_LIST, &drivers,
				    NULL, 
				    SPEECH_DRIVERS_SECTION);
	for (tmp = drivers ; tmp ; tmp = tmp->next)
	{
	    srconf_unset_key ((gchar*)tmp->data, 
		             SPEECH_DRIVERS_SECTION);
	    g_free (tmp->data);
	}
	g_slist_free (drivers);
	
	srconf_unset_key (SPEECH_ENGINE_DRIVERS, 
		    	  SPEECH_DRIVERS_SECTION);
	srconf_unset_key (SPEECH_VOICE_COUNT, 
		    	  SPEECH_DRIVERS_SECTION);
}

GSSpeaker *
gs_speakers_get_speaker (gchar 	*key)
{
    GSSpeaker 	*speaker = NULL;

#ifdef  SPGS_DEBUG
    sru_debug ("gs_speakers_get_speaker key : %s", key);
#endif

    if (key)
    {
	speaker =  (GSSpeaker*) g_hash_table_lookup (gs_speakers_hash_table, 
						     key);
    }
    
    return speaker;
}

void
gs_speakers_add_speaker (GSSpeaker 	*speaker, 
			 gchar 		*speaker_key)
{
    if (speaker && speaker_key)
    {	
        g_hash_table_insert (gs_speakers_hash_table,
	            	    g_strdup (speaker_key),
                    	    speaker);
		
    }
}

GSSpeaker *
gs_speakers_select_speaker (gchar *voice_name)
{
    BonoboObject 		*bo;
    GNOME_Speech_VoiceInfoList 	*voices;
    GNOME_Speech_ParameterList	*parameters;
    
    GSSpeaker 			*gss;
    Callback 			*cb;
    gint			param_n;

    if (!voice_name)
    {
	/* default speaker request */
	gss = default_speaker;
    }
    else
    {
	/* specific speaker requested */
	gss = gs_speakers_get_speaker (voice_name);

	/*a speaker with the given key could not be found, 
	  get the default speaker */
	if (!gss)
	{
    	    sru_warning ("gs_select_speaker - Speaker not found, using the default speaker.");
    	    gss = default_speaker;
	}
    }
    /* Instantiation on demand - 
	do I already instantiated a speaker for this voice */

    if (gss && !gss->speaker)
    {
	/* create a speaker with this voice */
	voices = GNOME_Speech_SynthesisDriver_getAllVoices (gss->driver, 
							    gs_ev () );
	if (!gs_check_ev (GS_ERR_GENERIC, __LINE__))
	    return default_speaker;

	if (voices)
	{
	    gss->speaker = GNOME_Speech_SynthesisDriver_createSpeaker (
			    gss->driver,
        		    &voices->_buffer[gss->voice_idx], 
			    gs_ev () );

		if (BONOBO_EX (&ev) )
		{
		    gs_check_ev (GS_ERR_GENERIC, __LINE__);
		    sru_warning ("gs_select_speaker: Could not create speaker for %s voice name.", voice_name);
		    return  default_speaker;
		}


	    if (gss->speaker)
	    {
		parameters = GNOME_Speech_Speaker_getSupportedParameters (gss->speaker,
									  gs_ev () );
		if (BONOBO_EX (&ev) || parameters->_length == 0)
		{
		    gs_check_ev (GS_ERR_GENERIC, __LINE__);

		    sru_warning ("gs_select_speaker: No parameters available for %s. Speaker is not valid.", voice_name);

		    return  default_speaker;
		}
#ifdef  SPGS_DEBUG		    		
		sru_debug ("gs_select_speaker: Parameters available for %s : %d. ", voice_name, parameters->_length);
#endif
		/* get min and max values for the parameters in wich we are interestead*/
		for ( param_n = 0; param_n < parameters->_length ; param_n ++)
		{
		    GNOME_Speech_Parameter *param = &(parameters->_buffer[param_n]);
		    
		    if (!param)
		    {
			GNOME_Speech_Speaker_unref(gss->speaker, NULL);
			gss->speaker = CORBA_OBJECT_NIL;
			return default_speaker;
		    }
		    if (!strcmp (param->name, "rate") ) 
		    {
			gss->rate.min = param->min;
			gss->rate.max = param->max;
#ifdef SPGS_DEBUG
			sru_debug ("param %s min %.2lf max %.2lf current %.2lf",
					param->name,
					gss->rate.min,
					param->max,
					param->current);  
#endif
		    }
		    if (!strcmp (param->name, "pitch") ) 
		    {
			gss->pitch.min = param->min;
			gss->pitch.max = param->max;
#ifdef SPGS_DEBUG
			sru_debug ("param %s min %.2lf max %.2lf current %.2lf",
					param->name,
					param->min,
					param->max,
					param->current);  
#endif
		    }
		    if (!strcmp (param->name, "volume") ) 
		    {
			gss->volume.min = param->min;
			gss->volume.max = param->max;
#ifdef SPGS_DEBUG		    
			sru_debug ("param %s min %.2lf max %.2lf current %.2lf",
					param->name,
					param->min,
					param->max,
					param->current);  
#endif
		    }
		}
/*FIXME*/		
    		/* register a callback for this speaker */
    		if (!gss->callback)
		{
		    gboolean rv = FALSE;
		    
		    cb = callback_new (gs_callback, NULL);
		    bo = BONOBO_OBJECT (cb);

    		    gss->callback = bo;
	
		    rv = GNOME_Speech_Speaker_registerSpeechCallback (gss->speaker,
							bonobo_object_corba_objref (bo),
							gs_ev () );
		    if (!rv || !gs_check_ev (GS_ERR_GENERIC, __LINE__) )
		    {
			if (voice_name)
			{
			    sru_warning ("Callbacks are NOT supported by \"%s\" voice.", voice_name);
			}
			bonobo_object_unref (gss->callback);
		    
			gss->callback = NULL;
			gss->cb_support = FALSE;
		    }
		    else
			gss->cb_support = TRUE;
		}    
		CORBA_free (parameters);
	    }
	    CORBA_free (voices);
	}
    }
    /* !!! TBR !!! should have some sanity check and error reporting here */
    gs_speaker_debug (gss);
    return gss;
}

void
gs_driver_to_gconf (GSList *speech_driver_list)
{
    srconf_set_data (SPEECH_ENGINE_DRIVERS, 
		     CFGT_LIST, 
		     speech_driver_list, 
		     SPEECH_DRIVERS_SECTION);
}

void
gs_speakers_to_gconf (gint 	voice_cnt)
{
    srconf_set_data (SPEECH_VOICE_COUNT, 
		     CFGT_INT, 
		     &voice_cnt, 
		     SPEECH_DRIVERS_SECTION);
}
/*_______________________</GS SPEAKERS CONTAINER >____________________________*/

/*______________________________< GS API >____________________________________*/
void
gs_speak (SRSVoice 	*voice, 
	  SRSText 	*text )
{
    GSSpeaker *gss;

    /* sel current speaker for voice->TTSVoiceName */
    gss = gs_speakers_select_speaker (voice->tts_voice_name);
    if (gss)
    {
	current_speaker = gss->speaker;
    
	if (!gss->cb_support )
	{
	    voice->cb_support = 0;
	}
	else
	{
	    voice->cb_support = 1;
	}
    }
    if (current_speaker)
    {
	CORBA_Environment 	ev;
	float 			val;
	    

	val = gs_speaker_procent_to_units (gss,
					  GS_PARAM_RATE,
					  (float)voice->rate);
	GNOME_Speech_Speaker_setParameterValue (current_speaker, 
						"rate", 
						val, 
						gs_ev () );
	gs_check_ev (GS_ERR_GENERIC, __LINE__);						

	val = gs_speaker_procent_to_units (gss,
					  GS_PARAM_VOLUME,
					  (float)voice->volume);
	GNOME_Speech_Speaker_setParameterValue (current_speaker, 
						"volume", 
						val, 
						gs_ev () );
	gs_check_ev (GS_ERR_GENERIC, __LINE__);
	
	val = gs_speaker_procent_to_units (gss,
					  GS_PARAM_PITCH,	
					  (float)voice->pitch);
	GNOME_Speech_Speaker_setParameterValue (current_speaker, 
						"pitch", 
						val, 
						gs_ev () );
	gs_check_ev (GS_ERR_GENERIC, __LINE__);

        if (text && text->text) 
	{

#ifdef SPGS_DEBUG
	sru_debug ("(%s %d) : gs_speak \"%s\" | current_speaker %p",
		    __FILE__,
		    __LINE__,
		    text->text,
		    current_speaker);
#endif
    	    /* resume_text = text->Text; */
    	    GNOME_Speech_Speaker_say (current_speaker, 
				      text->text, 
				      gs_ev () );

	    if (BONOBO_EX (&ev) )
	    {
		gs_check_ev (GS_ERR_GENERIC, __LINE__);
	    }
        }
    }	
}

void
gs_shut_up (void)
{
    if (current_speaker)
    {
#ifdef SPGS_DEBUG
	sru_debug ("(%s %d) : gs_shut_up |  current_speaker %p",
		    __FILE__,
		    __LINE__,
		    current_speaker);
#endif

	GNOME_Speech_Speaker_stop (current_speaker, 
				   gs_ev () );
    	gs_check_ev (GS_ERR_GENERIC, __LINE__);
    }
}

void gs_pause ()
{
    /* WAITING FOR SUPPORT FROM GS */
    gs_check_ev (GS_ERR_GENERIC, __LINE__);
}

void gs_resume ()
{
    /* WAITING FOR SUPPORT FROM GS */

    gs_check_ev (GS_ERR_GENERIC, __LINE__);
}


Bonobo_ServerInfoList *
gs_init_get_gs_servers (void)
{
    Bonobo_ServerInfoList 	*servers  = CORBA_OBJECT_NIL;
/*
    Bonobo_ServerInfoList 	*callback_servers =  CORBA_OBJECT_NIL;
*/
    if (!bonobo_init (NULL, NULL) )
    { 
	/*this is failure*/
	sru_warning ("Bonobo initialization failed.");
	return NULL;
    }
    
    servers = bonobo_activation_query (
		"repo_ids.has ('IDL:GNOME/Speech/SynthesisDriver:0.2')",
		NULL, 
		gs_ev () );

    if (!gs_check_ev (GS_ERR_SERVER_ACTIVATION, __LINE__) ) 
    {
	return NULL;
    }
    
    if (!servers) 
    {
	sru_warning ("No \"gnome-speech\" drivers were found.");
	return NULL;
    }
/*
    callback_servers = bonobo_activation_query (
		"repo_ids.has ('IDL:GNOME/Speech/SpeechCallback:0.2')",
		NULL, 
		gs_ev () );
    if (!gs_check_ev (GS_ERR_SERVER_ACTIVATION, __LINE__) ||
	!callback_servers) 
    {
	sru_warning ("No \"gnome-speech\" drivers supports callbacks.");
    }
    else
	sru_message ("Some/all \"gnome-speech\" drivers supports callbacks.");
*/
    return servers;
}



CORBA_Object
gs_init_activate_server (Bonobo_ServerInfo 	*info)
{
	CORBA_Object		obj = CORBA_OBJECT_NIL;

	/* atempt to activate the server */

	/*obj = a CORBAobject reference to the newly activated server*/
	obj = bonobo_activation_activate_from_id (info->iid, 
						  0, 
						  NULL, 
						  gs_ev () );
	
	/*check the environment for failure*/
	if (BONOBO_EX (&ev) || obj == CORBA_OBJECT_NIL) 
	{
#ifdef SPGS_DEBUG	
	    sru_warning ("Activation of server %s failed.", info->iid);
#endif
	}
	else
	{
	    if (!GNOME_Speech_SynthesisDriver_driverInit (obj, &ev))
			return CORBA_OBJECT_NIL;
#ifdef SPGS_DEBUG
	    sru_message ("Activation of server %s succedded.", 
			 info->iid);
#endif
	}
	return  obj;
}

int
gs_init (TTSEngine *engine)
{
    gint 			server_n = 0, 
				voice_n	 = 0,
				voice_cnt = 0;
    gchar 			*voice_name = NULL,
				*default_voice_name = NULL;

    CORBA_string 		driver_name;/*, driver_version;*/
    Bonobo_ServerInfoList 	*servers;
    GSSpeaker			*gss;

    GSList			*speech_driver_list = NULL;
    GSList			*driver_voice_list  = NULL;
    GSList			*tmp;

    default_speaker = NULL;
    current_speaker = CORBA_OBJECT_NIL;
    
    /*create internal hash table*/
    gs_speakers_init();
	
    servers = gs_init_get_gs_servers();

    if (!servers)
	return FALSE;
	     
    voice_cnt 	= 0;
    
    for (server_n = 0; server_n < servers->_length; server_n++)
    {		
	Bonobo_ServerInfo 		*info;
	GNOME_Speech_VoiceInfoList 	*voices;
	GNOME_Speech_SynthesisDriver 	driver;
	GSSpeaker 			*gss;

	info = &servers->_buffer[server_n];
	driver = (GNOME_Speech_SynthesisDriver) gs_init_activate_server (info);
	
	if (!driver)
	    continue; /* try next server*/

	driver_name = GNOME_Speech_SynthesisDriver__get_driverName (driver, 
								    gs_ev () );
	if (!gs_check_ev (GS_ERR_DRIVER_NAME_INVALID, __LINE__) )
	{
	    CORBA_free (driver_name);
	    continue; /* try next server*/
	}

	/* create a list of available voices in gconf */
	voices = GNOME_Speech_SynthesisDriver_getAllVoices (driver, 
							    gs_ev () );
	if (!gs_check_ev (GS_ERR_NO_VOICES, __LINE__) 	|| 
	    !voices 					|| 
	    !voices->_length) 
    	{
	    sru_warning ("Driver named \"%s\" has no available voices.", 
			 driver_name);
	    continue; /* try next server*/
	}

	
        current_speaker =  GNOME_Speech_SynthesisDriver_createSpeaker (
			    driver,
        		    &voices->_buffer[0],
			    gs_ev () );
        if (!gs_check_ev (GS_ERR_VOICE_CREATE, __LINE__))
	{
	    sru_warning ("Could not create speaker for \"V0 %s - %s\" voice name.", voices->_buffer[0].name, driver_name );
	    continue;
	}
	for (voice_n = 0; voice_n < voices->_length; ++voice_n) 
	{
	    /* store the driver together with the voice index to allow a speaker creation later */
	    GNOME_Speech_Speaker_setParameterValue (current_speaker, "voice", voice_n+1, gs_ev ());
	    if (!gs_check_ev (GS_ERR_VOICE_INVALID, __LINE__) )
		break;
	    gss = gs_speaker_new0 (driver, voice_n);
    	    
	    voice_name 	= g_strdup_printf ("V%d %s - %s", 
					    voice_n, 
					    voices->_buffer[voice_n].name, 
					    driver_name );
	    driver_voice_list = g_slist_append (driver_voice_list,
						g_strdup (voice_name));

	    voice_cnt++;

    	    /* select a default voice */
    	    if (!default_speaker) 
	    {
		default_speaker = gss; 
#ifdef SRSP_DEBUG
		sru_message ("gs_init: DEFAULT voice_n %d, voice_cnt %d, voice_name %s ",voice_n, voice_cnt, voice_name);
#endif
		default_voice_name = g_strdup (voice_name);		
	    }
    	    gs_speakers_add_speaker (gss, voice_name);

#ifdef SRSP_DEBUG
	    sru_debug("gs_init: Driver \"%s\" voice_name %s",
			driver_name,
			voice_name);
#endif
    	    g_free (voice_name);
	}/*end of voices*/
	bonobo_object_release_unref (current_speaker, gs_ev () );
        
	if (driver_voice_list)
	{
#ifdef SRSP_DEBUG
	    sru_debug("gs_init: Driver \"%s\" has available voices",driver_name);
#endif
	    gs_speaker_to_gconf (driver_voice_list, driver_name);
	    for (tmp = driver_voice_list; tmp ; tmp = tmp->next)
		g_free (tmp->data);
	    g_slist_free (driver_voice_list);
	    driver_voice_list = NULL;
	    
	    speech_driver_list = g_slist_append (speech_driver_list, 
					     g_strdup (gs_normilize_driver_name (driver_name) ) );

	}
	CORBA_free (voices);
	CORBA_free (driver_name);	
	CORBA_Object_release (driver, NULL);
    } /* end of for server */
    
    gs_speakers_to_gconf (voice_cnt);
    gs_driver_to_gconf (speech_driver_list);    
    
    for (tmp = speech_driver_list; tmp ; tmp = tmp->next)
	g_free (tmp->data);
	
    g_slist_free (speech_driver_list);

    CORBA_free (servers);

    /* select a default speaker */
    gss = gs_speakers_select_speaker (default_voice_name);
    if (gss)
	current_speaker = gss->speaker;
    g_free (default_voice_name);
    tts_engine = engine;
    if (voice_cnt)
    {		
	/* fill the engine structure with the actual functions */
   	tts_engine->shut_up	= gs_shut_up;
	tts_engine->speak	= gs_speak;
        tts_engine->pause	= gs_pause;
        tts_engine->resume	= gs_resume;
	tts_engine->terminate	= gs_terminate;
    }
    
    return (voice_cnt > 0) ? TRUE : FALSE;
}

void
gs_terminate (void)
{
    gs_shut_up ();
    gs_speakers_terminate();
    bonobo_debug_shutdown ();
}
/*______________________________</GS API >____________________________________*/
