/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gnome.h>
#include <glade/glade.h>

#include <libxklavier/xklavier_config.h>

#include "../common/switchcuts.h"
#include "xkb_capplet_private.h"

static GtkTooltips *tooltips;

const char *XkbCappletConfigItemDesc( XklConfigItem * ci )
{
  return ci->description[0] == 0 ? ci->name : ci->description;
}

static void XkbCappletSetTip( GtkWidget * widget, XklConfigItem * ci )
{
  const char *tt = XkbCappletConfigItemDesc( ci );
  char *stt;
  if( tooltips == NULL || tt == NULL || *tt == '\0' )
    return;
  stt = g_strstrip( g_strdup( tt ) );
  if( *stt != '\0' )
    gtk_tooltips_set_tip( tooltips, widget, stt, ci->name );
  g_free( stt );
}

const char *XkbModelName2Id( const char *name )
{
  static char id[14 + XKL_MAX_CI_NAME_LENGTH];
  g_snprintf( id, sizeof( id ), "modelsMenuItem%s", name );
  return id;
}

const char *XkbOptionGroupName2Id( const char *groupName,
                                   const char *ctlType )
{
  static char id[11 + XKL_MAX_CI_NAME_LENGTH + 10];
  g_snprintf( id, sizeof( id ), "optionGroup%s%s", groupName, ctlType );
  return id;
}

const char *XkbOptionName2Id( const char *groupName, const char *optionName )
{
  static char id[10 + 2 * XKL_MAX_CI_NAME_LENGTH];
  g_snprintf( id, sizeof( id ), "optionItem%s%s", groupName, optionName );
  return id;
}

void XkbCappletPrepareWidget( const GSwitchItCapplet * gswic,
                              GtkWidget * widget,
                              const char *ctlName, const char *dataKey )
{
  gtk_widget_show( widget );
  g_object_set_data( G_OBJECT( gswic->capplet ), dataKey, widget );
  if( ctlName != NULL )
    gtk_widget_set_name( widget, ctlName );
}

static void ModelComboFiller( const XklConfigItemPtr configItem,
                              GSwitchItCapplet * gswic )
{
  GtkWidget *menu =
    g_object_get_data( G_OBJECT( gswic->capplet ), "modelsMenu_menu" );
  GtkWidget *newItem =
    gtk_menu_item_new_with_label( XkbCappletConfigItemDesc( configItem ) );

  XkbCappletPrepareWidget( gswic, newItem, configItem->name,
                           XkbModelName2Id( configItem->name ) );
  gtk_menu_shell_append( GTK_MENU_SHELL( menu ), newItem );
}

static void FillModelsCombo( GSwitchItCapplet * gswic )
{
  GtkWidget *optionMenu = CappletGetGladeWidget( gswic, "modelsMenu" );
  GtkWidget *newMenu = gtk_menu_new(  );

  XkbCappletPrepareWidget( gswic, newMenu, "modelsMenu_menu",
                           "modelsMenu_menu" );

  XklConfigEnumModels( ( ConfigItemProcessFunc ) ModelComboFiller, gswic );

  gtk_option_menu_set_menu( GTK_OPTION_MENU( optionMenu ), newMenu );

  g_signal_connect( G_OBJECT( newMenu ), "selection-done",
                    G_CALLBACK( XkbCappletSetStateToChanged ), gswic );
}

static void OptionGroupListFiller( const XklConfigItemPtr configItem,
                                   GSwitchItCapplet * gswic )
{
  GtkWidget *listView = g_object_get_data( G_OBJECT( gswic->capplet ),
                                           CURRENT_LIST_PROPERTY );
  GtkListStore *listModel =
    GTK_LIST_STORE( gtk_tree_view_get_model( GTK_TREE_VIEW( listView ) ) );
  GtkTreeIter iter;

  gtk_list_store_append( listModel, &iter );
  gtk_list_store_set( listModel, &iter, 0,
                      g_strdup( XkbCappletConfigItemDesc( configItem ) ), 1,
                      g_strdup( configItem->name ), -1 );
}

static void OptionMenuFiller( const XklConfigItemPtr configItem,
                              GSwitchItCapplet * gswic )
{
  GtkWidget *menu = g_object_get_data( G_OBJECT( gswic->capplet ),
                                       CURRENT_MENU_PROPERTY );
  const char *groupName = g_object_get_data( G_OBJECT( gswic->capplet ),
                                             CURRENT_OPTION_GROUP_NAME_PROPERTY );
  GtkWidget *newItem =
    gtk_menu_item_new_with_label( XkbCappletConfigItemDesc( configItem ) );

  XkbCappletPrepareWidget( gswic, newItem,
                           configItem->name,
                           XkbOptionName2Id( groupName, configItem->name ) );
  gtk_menu_shell_append( GTK_MENU_SHELL( menu ), newItem );
}

static void OptionGroupFiller( const XklConfigItemPtr configItem,
                               Bool allowMultipleChoice,
                               GSwitchItCapplet * gswic )
{
  GtkWidget *superboxS = CappletGetGladeWidget( gswic, "singleOptionsVBox" );
  GtkWidget *superboxM =
    CappletGetGladeWidget( gswic, "multipleOptionsVBox" );
  GtkWidget *align = gtk_alignment_new( 0.5, 0.5, 1, 1 );
  GtkWidget *vbox = gtk_vbox_new( FALSE, 0 );
  GtkWidget *hbox1 = gtk_hbox_new( FALSE, 6 );
  GtkWidget *hbox2 = gtk_hbox_new( FALSE, 6 );
  GSList *optionsList = g_object_get_data( G_OBJECT( gswic->capplet ),
                                           OPTION_CONTROLS_LIST_PROPERTY );
  GtkWidget *label = gtk_label_new( XkbCappletConfigItemDesc( configItem ) );

  g_object_set_data( G_OBJECT( gswic->capplet ),
                     CURRENT_OPTION_GROUP_NAME_PROPERTY, configItem->name );

  XkbCappletPrepareWidget( gswic, align, configItem->name,
                           XkbOptionGroupName2Id( configItem->name,
                                                  "align" ) );
  XkbCappletPrepareWidget( gswic, vbox, configItem->name,
                           XkbOptionGroupName2Id( configItem->name,
                                                  "vbox" ) );
  XkbCappletPrepareWidget( gswic, hbox1, configItem->name,
                           XkbOptionGroupName2Id( configItem->name,
                                                  "hbox1" ) );
  XkbCappletPrepareWidget( gswic, hbox2, configItem->name,
                           XkbOptionGroupName2Id( configItem->name,
                                                  "hbox2" ) );
  XkbCappletPrepareWidget( gswic, label, configItem->name,
                           XkbOptionGroupName2Id( configItem->name,
                                                  "label" ) );

  gtk_container_set_border_width( GTK_CONTAINER( align ), 6 );

  gtk_container_add( GTK_CONTAINER( align ), vbox );
  gtk_box_pack_start( GTK_BOX( vbox ), hbox1, FALSE, FALSE, 0 );
  gtk_box_pack_start( GTK_BOX( vbox ), hbox2, TRUE, TRUE, 0 );
  gtk_box_pack_start( GTK_BOX( hbox1 ), label, FALSE, FALSE, 0 );

  if( optionsList != NULL )
    g_object_set_data( G_OBJECT( gswic->capplet ),
                       OPTION_CONTROLS_LIST_PROPERTY, NULL );

  if( allowMultipleChoice )
  {
    GtkListStore *listModel =
      gtk_list_store_new( 2, G_TYPE_STRING, G_TYPE_STRING );

    GtkWidget *listView =
      gtk_tree_view_new_with_model( GTK_TREE_MODEL( listModel ) );
    GtkCellRenderer *renderer = gtk_cell_renderer_text_new(  );

    GtkTreeViewColumn *column =
      gtk_tree_view_column_new_with_attributes( XkbCappletConfigItemDesc
                                                ( configItem ),
                                                renderer, "text", 0, NULL );

    GtkTreeSelection *select =
      gtk_tree_view_get_selection( GTK_TREE_VIEW( listView ) );
    GtkWidget *scroll = gtk_scrolled_window_new( NULL, NULL );

    gtk_tree_view_append_column( GTK_TREE_VIEW( listView ), column );

    gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( scroll ),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC );
    gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( scroll ),
                                         GTK_SHADOW_IN );

    gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( listView ), FALSE );

    gtk_tree_view_set_vadjustment( GTK_TREE_VIEW( listView ),
                                   GTK_ADJUSTMENT( gtk_adjustment_new
                                                   ( 0, 0, 0, 0, 0, 0 ) ) );

    gtk_box_pack_start( GTK_BOX( superboxM ), align, TRUE, TRUE, 0 );

    optionsList = g_slist_append( optionsList, listView );

    gtk_box_pack_start( GTK_BOX( hbox2 ), scroll, TRUE, TRUE, 0 );
    gtk_container_add( GTK_CONTAINER( scroll ), listView );

    XkbCappletPrepareWidget( gswic, scroll, configItem->name,
                             XkbOptionGroupName2Id( configItem->name,
                                                    "listScroll" ) );

    XkbCappletPrepareWidget( gswic, listView, configItem->name,
                             XkbOptionGroupName2Id( configItem->name,
                                                    "list" ) );
    g_object_set_data( G_OBJECT( gswic->capplet ), CURRENT_LIST_PROPERTY,
                       listView );

    XklConfigEnumOptions( configItem->name,
                          ( ConfigItemProcessFunc )
                          OptionGroupListFiller, gswic );

    g_object_set_data( G_OBJECT( gswic->capplet ), CURRENT_LIST_PROPERTY,
                       NULL );

    gtk_tree_selection_set_mode( select, GTK_SELECTION_MULTIPLE );

    g_signal_connect( G_OBJECT( select ), "changed",
                      G_CALLBACK( XkbCappletSetStateToChanged ), gswic );

    XkbCappletSetTip( listView, configItem );
  } else
  {
    GtkWidget *omenu = gtk_option_menu_new(  );
    GtkWidget *menu = gtk_menu_new(  );
    GtkWidget *newItem = gtk_menu_item_new_with_label( "" );

    gtk_box_pack_start( GTK_BOX( superboxS ), align, FALSE, FALSE, 0 );

    optionsList = g_slist_append( optionsList, omenu );

    XkbCappletPrepareWidget( gswic, omenu, configItem->name,
                             XkbOptionGroupName2Id( configItem->name,
                                                    "omenu" ) );

    gtk_box_pack_start( GTK_BOX( hbox2 ), omenu, FALSE, FALSE, 0 );

    g_object_set_data( G_OBJECT( gswic->capplet ), CURRENT_MENU_PROPERTY,
                       menu );

    XkbCappletPrepareWidget( gswic, newItem,
                             "", XkbOptionName2Id( configItem->name, "" ) );
    gtk_menu_shell_append( GTK_MENU_SHELL( menu ), newItem );

    XklConfigEnumOptions( configItem->name,
                          ( ConfigItemProcessFunc ) OptionMenuFiller, gswic );

    g_object_set_data( G_OBJECT( gswic->capplet ), CURRENT_MENU_PROPERTY,
                       NULL );
    gtk_option_menu_set_menu( GTK_OPTION_MENU( omenu ), menu );

    g_signal_connect( G_OBJECT( menu ), "selection-done",
                      G_CALLBACK( XkbCappletSetStateToChanged ), gswic );

    XkbCappletSetTip( omenu, configItem );
  }

  g_object_set_data( G_OBJECT( gswic->capplet ),
                     OPTION_CONTROLS_LIST_PROPERTY, optionsList );

  g_object_set_data( G_OBJECT( gswic->capplet ), "__currentName", NULL );
}

static void FillOptionsPanel( GSwitchItCapplet * gswic )
{
  GSList *optionControls;

  XklConfigEnumOptionGroups( ( GroupProcessFunc ) OptionGroupFiller, gswic );

  // This trick just sets the list in a way to clean it on destroy
  optionControls = g_object_get_data( G_OBJECT( gswic->capplet ),
                                      OPTION_CONTROLS_LIST_PROPERTY );

  g_object_set_data_full( G_OBJECT( gswic->capplet ),
                          OPTION_CONTROLS_LIST_PROPERTY,
                          optionControls, ( GtkDestroyNotify ) g_slist_free );
}

static void _XkbCappletRealize( GtkWidget * capplet,
                                GSwitchItCapplet * gswic )
{
  XkbCappletPreviewClicked( NULL, gswic );
}

void XkbCappletSetup( GSwitchItCapplet * gswic )
{
  GladeXML *data;
  GtkWidget *widget, *capplet, *switchcutsOMenu, *switchcutsMenu;
  const Switchcut *sc;
  int i;

  glade_gnome_init(  );

  data = glade_xml_new( GLADE_DIR "/xkb-properties.glade", "xkb_capplet", NULL );       // default domain
  gswic->capplet = capplet = glade_xml_get_widget( data, "xkb_capplet" );
  gtk_widget_set_name( capplet, "capplet" );
  g_object_set_data( G_OBJECT( capplet ), "capplet", capplet );

  widget = glade_xml_get_widget( data, "xkb_capplet_widget" );
  g_object_set_data( G_OBJECT( widget ), "gladeData", data );

  g_signal_connect_swapped( G_OBJECT( widget ), "destroy",
                            G_CALLBACK( g_object_unref ), data );
  g_signal_connect( G_OBJECT( capplet ), "close",
                    G_CALLBACK( gtk_main_quit ), data );

  XkbCappletPrepareWidget( gswic, widget, NULL, "xkb_capplet_widget" );

  g_object_set_data( G_OBJECT( capplet ), "gladeData", data );

  tooltips =
    gtk_tooltips_data_get( CappletGetGladeWidget
                           ( gswic, "tbtnUseCustomXkbSettings" ) )->tooltips;

  FillModelsCombo( gswic );
  FillOptionsPanel( gswic );

  XkbCappletInitLayoutButtons( gswic );

  switchcutsMenu = gtk_menu_new(  );

  gtk_object_set_data( GTK_OBJECT( capplet ), "switchcutsMenu",
                       switchcutsMenu );

  sc = switchcuts;
  for( i = total_switchcuts; --i >= 0; sc++ )
  {
    GtkWidget *menuItem = gtk_menu_item_new_with_label( _( sc->name ) );
    gtk_object_set_data( GTK_OBJECT( menuItem ), "switchcut", sc );
    g_signal_connect( G_OBJECT( menuItem ), "activate",
                      G_CALLBACK( XkbCappletSetStateToChanged ), gswic );
    gtk_menu_shell_append( GTK_MENU_SHELL( switchcutsMenu ), menuItem );
    gtk_widget_show( menuItem );
  }

  switchcutsOMenu = glade_xml_get_widget( data, "switchcutsOMenu" );
  XkbCappletPrepareWidget( gswic, switchcutsOMenu, NULL, "switchcutsOMenu" );

  gtk_option_menu_set_menu( GTK_OPTION_MENU( switchcutsOMenu ),
                            switchcutsMenu );

  // extra switchcuts are not necessary for 4.3.0!
  if( XklMultipleLayoutsSupported(  ) )
  {
    GtkWidget *notebook =
      glade_xml_get_widget( data, "xkb_capplet_notebook" );
    gtk_widget_hide( gtk_notebook_get_nth_page
                     ( GTK_NOTEBOOK( notebook ), 3 ) );
  }

  {
    GtkToggleButton *btnc =
      GTK_TOGGLE_BUTTON( CappletGetGladeWidget
                         ( gswic, "tbtnUseCustomXkbSettings" ) );
    GtkToggleButton *btng =
      GTK_TOGGLE_BUTTON( CappletGetGladeWidget
                         ( gswic, "tbtnUseGlobalXkbSettings" ) );

    gtk_toggle_button_set_active( btnc, gswic->config.xkbOverrideSettings );
    gtk_toggle_button_set_active( btng, !gswic->config.xkbOverrideSettings );

    glade_xml_signal_connect_data( data,
                                   "on_tbtnUseCustomXkbSettings_toggled",
                                   GTK_SIGNAL_FUNC
                                   ( XkbCappletUseCustomXkbSettingsButtonToggled ),
                                   gswic );

    glade_xml_signal_connect_data( data,
                                   "on_tbtnUseGlobalXkbSettings_toggled",
                                   GTK_SIGNAL_FUNC
                                   ( XkbCappletUseGlobalXkbSettingsButtonToggled ),
                                   gswic );

    glade_xml_signal_connect_data( data,
                                   "on_btnPreview_clicked",
                                   GTK_SIGNAL_FUNC
                                   ( XkbCappletPreviewClicked ), gswic );

    glade_xml_signal_connect_data( data,
                                   "on_btnZoomIn_clicked",
                                   GTK_SIGNAL_FUNC
                                   ( XkbCappletPreviewZoomIn ), gswic );

    glade_xml_signal_connect_data( data,
                                   "on_btnZoomOut_clicked",
                                   GTK_SIGNAL_FUNC
                                   ( XkbCappletPreviewZoomOut ), gswic );

    glade_xml_signal_connect_data( data,
                                   "on_btnZoomFit_clicked",
                                   GTK_SIGNAL_FUNC
                                   ( XkbCappletPreviewZoomFit ), gswic );

    glade_xml_signal_connect_data( data,
                                   "on_btnZoomDefault_clicked",
                                   GTK_SIGNAL_FUNC
                                   ( XkbCappletPreviewZoomDefault ), gswic );

    glade_xml_signal_connect_data( data,
                                   "on_btnHelp_clicked",
                                   GTK_SIGNAL_FUNC( XkbCappletHelp ), gswic );

    glade_xml_signal_connect_data( data,
                                   "on_btnOk_clicked",
                                   GTK_SIGNAL_FUNC( XkbCappletOk ), gswic );

    glade_xml_signal_connect_data( data,
                                   "on_btnCancel_clicked",
                                   GTK_SIGNAL_FUNC( XkbCappletCancel ),
                                   gswic );

    glade_xml_signal_connect_data( data,
                                   "on_xkb_capplet_unrealize",
                                   GTK_SIGNAL_FUNC( XkbCappletCancel ),
                                   gswic );

    XkbCappletUseSomeXkbSettingsButtonToggled( NULL, gswic,
                                               gswic->config.
                                               xkbOverrideSettings );
  }

  g_signal_connect( G_OBJECT
                    ( CappletGetGladeWidget( gswic, "framePreview" ) ),
                    "realize", G_CALLBACK( _XkbCappletRealize ), gswic );

  gtk_widget_set_sensitive( CappletGetGladeWidget( gswic, "btnOk" ), FALSE );

  gtk_widget_show( capplet );
}
