/* brlxml.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libxml/parser.h>
#include <string.h>
#include <stdlib.h>
#include "braille.h"
#include "brlxml.h"
#include "brlxmlapi.h"

static xmlSAXHandler  *brl_ctx;

static BRL_PARSER_STATE brl_curr_state = BPS_IDLE;
static BRL_PARSER_STATE brl_prev_state = BPS_IDLE;
static int brl_unknown_depth = 0;

static BRL_DISP	*tbrl_disp = NULL;
static BRL_OUT	*tbrl_out = NULL;

static GHashTable* TranslationTableCache;
static BRL_XML_INPUT_PROC XMLClientCallback = NULL;

static gboolean brl_xml_initialized = FALSE;


/* __ TRANSLATION TABLE CACHE _________________________________________________ */

void ttc_key_destroy (gpointer key)
{
	g_free (key);
}

void ttc_value_destroy (gpointer value)
{
	g_free (value);
}

void ttc_init()
{
	
	/* !!! TBI !!! - use this for GLIB2.0: TranslationTableCache = g_hash_table_new_full (g_str_hash, g_str_equal, ttc_key_destroy, ttc_value_destroy);	 */
	TranslationTableCache = g_hash_table_new (g_str_hash, g_str_equal);
}

void ttc_enum (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	g_free (value);
}

void ttc_terminate()
{
	
	/* !!! TBR !!! for GLIB 2.0 g_hash_table_destroy should be enough if I use g_hash_table_new_full */
	
	/* For GLIB 1.2 I enumerate and destroy them one by one */
	g_hash_table_foreach (TranslationTableCache, ttc_enum, NULL);
	g_hash_table_destroy (TranslationTableCache);
}

guint8* ttc_get_translation_table (const gchar* language)
{
	
	gchar* 	key = NULL;
	guint8*	value = NULL;
	
	gchar* 	fn;
	gchar* dir_fn = NULL;
	FILE* 		fp;
	
	/* search the key in the table first */
	value = (guint8*) g_hash_table_lookup (TranslationTableCache, language);
		
	if (!value)
	{
		/* the table is not in the cache, try to load it from the TT file */
			
		/* derive the TT file name from the language name */
		fn = g_strdup_printf ("%s.A2B", language);
		g_strup(fn);
		/* open the file */
	        if (g_file_test (fn, G_FILE_TEST_EXISTS)) 
		{
			fp = fopen (fn, "rb");
		}
		else
		{
			dir_fn = g_strconcat (BRAILLE_TRANS_TABLES_DIR,fn,NULL);
			fp = fopen (dir_fn,"rb");
			g_free (dir_fn);
			dir_fn = NULL;
		}
		if (fp)
		{	
			/* load translation table in memory */
			key = g_strdup (language);
			value = g_malloc0 (TT_SIZE);			
			fread (value, 1, TT_SIZE, fp);	
			
			g_hash_table_insert (TranslationTableCache, key, value);
			
			fclose (fp);	
		}	
		else
		{
		    fprintf(stderr,"brlxml : opening file error\n");
		}
		g_free (fn);
	
	}
	
	return value;

}

/* __ HELPERS _________________________________________________________________ */

guint8 dotstr_to_bits (gchar* dotstr)
{
	guint8 rv = 0;
	gint i, n, dot;
	
	/* DOT							 ?	      1	     2        3      4        5       6       7       8				 */
	guint8 dot_to_bit [] = {0x00, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80};
	
	n = strlen (dotstr);	
	if (n > 3 && g_strncasecmp(dotstr, "dot", 3) == 0)
	{
		/* DOTxxx format */
		for (i = 3; i < n; ++i)
		{
			dot = dotstr[i] - '0';
			if (dot > 0 && dot <= 8)
			{
				 rv |= dot_to_bit[dot];
			}
		}						
	}
	else
	{
		/* assume hex format */
		sscanf (dotstr, "%2x", &i);
		rv = i & 0xFF;
	}
	
	return rv;
}

/* __ BRL_DISP METHODS _________________________________________________________ */

BRL_DISP* brl_disp_new ()
{
	
	BRL_DISP *brl_disp;
	
	brl_disp = g_malloc0(sizeof(BRL_DISP));
	brl_disp->Dots = g_byte_array_new();
	brl_disp->CursorPosition = -1;	/* no cursor */
	brl_disp->CursorStyle = CS_UNDERLINE;
	brl_disp->CursorMask = 0xC0;
	brl_disp->Attribute = 0x00;
	brl_disp->Role = NULL;
	/* brl_disp->Language = NULL; */
	brl_disp->TranslationTable = NULL;

	return brl_disp;
}

void	brl_disp_free (BRL_DISP *brl_disp)
{
	/* !!! TBR !!! after the TT cache is in place */
	/* g_free (brl_disp->Language); */
	/* g_free (brl_disp->TranslationTable);	 */
	g_free (brl_disp->Role);
	g_byte_array_free (brl_disp->Dots, TRUE);
	g_free (brl_disp);
}

BRL_DISP* brl_disp_copy (BRL_DISP *brl_disp)
{
	BRL_DISP* bd;
		
	bd = g_malloc0(sizeof(BRL_DISP));
	
	/* copy all non pointer fields */
	memcpy (bd, brl_disp, sizeof(BRL_DISP));
	
	/* copy the role string */
	if (brl_disp->Role)
	{
 		bd->Role = g_strdup(brl_disp->Role);
	}
	
	/* !!! TBR !!! cache */

#if 0	
	/* copy the language */
	if (brl_disp->Language)
	{
 		bd->Language = g_strdup(brl_disp->Language);
	}		
	
	/* copy the translation table */
	if (brl_disp->TranslationTable)
	{
		bd->TranslationTable = g_malloc0 (TT_SIZE);
 		memcpy (bd->TranslationTable, brl_disp->TranslationTable, TT_SIZE);
	}
#endif

	/* copy the dots array */
	bd->Dots = g_byte_array_new();
	g_byte_array_append (bd->Dots, brl_disp->Dots->data, brl_disp->Dots->len);
	
	return bd;
}

void brl_disp_set_id (BRL_DISP *brl_disp, gchar *id)
{
	brl_disp->ID = atoi(id);
}

void brl_disp_set_role (BRL_DISP *brl_disp, gchar *role)
{
	brl_disp->Role = g_strdup (role);
}

void brl_disp_set_disp_no (BRL_DISP *brl_disp, gchar *dno)
{
	brl_disp->ID = atoi(dno);	/* !!! TBR !!! don't store disp no in the ID... */
}

void brl_disp_set_clear_mode (BRL_DISP *brl_disp, gchar *clear)
{
	
	if (	(g_strcasecmp(clear, "true") == 0) ||
			(g_strcasecmp(clear, "on") == 0) ||
			(g_strcasecmp(clear, "1") == 0)
		)
	{
		brl_disp->ClearDisplay = TRUE;
	}
	else
	{
		brl_disp->ClearDisplay = FALSE;
	}
}

void brl_disp_set_start (BRL_DISP *brl_disp, gchar *start)
{
	brl_disp->Start = atoi(start);
}

void brl_disp_set_offset (BRL_DISP *brl_disp, gchar *offset)
{
	brl_disp->Offset = atoi(offset);
}

void brl_disp_set_cursor_style (BRL_DISP *brl_disp, gchar* style)
{			
			
	if (g_strcasecmp(style, "underline") == 0)
	{
		brl_disp->CursorStyle = CS_UNDERLINE;			
		brl_disp->CursorMask = 0xC0;
		brl_disp->CursorPattern = 0xC0;
	}
	else if (g_strcasecmp(style, "block") == 0)
	{
		brl_disp->CursorStyle = CS_BLOCK;			
		brl_disp->CursorMask = 0xFF;
		brl_disp->CursorPattern = 0xFF;
	}
	else if (g_strcasecmp(style, "user") == 0)
	{
		/* !!! TBI !!! */
		brl_disp->CursorStyle = CS_USER_DEFINED;			
		/* !!! TBI !!! - parse the mask and the pattern */
		/* tbrl_out->CursorMask = 0xFF; */
		/* tbrl_out->CursorPattern = 0xFF; */
	}
	else
	{
		/* for unknown attr value, we use the default */
		brl_disp->CursorStyle = CS_UNDERLINE;			
		brl_disp->CursorMask = 0xC0;
		brl_disp->CursorPattern = 0xC0;
	}

}
void brl_disp_set_cursor (BRL_DISP *brl_disp, gchar *cursor)
{
	brl_disp->CursorPosition = atoi(cursor);
}

void brl_disp_set_text_attr (BRL_DISP *brl_disp, gchar *attr)
{	
	brl_disp->Attribute = dotstr_to_bits(attr);	
}

void brl_disp_load_translation_table (BRL_DISP *brl_disp, gchar* language)
{
	
	brl_disp->TranslationTable = ttc_get_translation_table (language);
	
#if 0	
	gchar* fn;
	FILE* fp;
	
	/* !!! TBR !!! use a cache (i.e. HashTable?) */
			
	g_free (brl_disp->Language);
	g_free (brl_disp->TranslationTable);									/* !!! TBR !!! */
	brl_disp->TranslationTable = g_malloc0(TT_SIZE);			/* !!! TBR !!! */
	
	/* !!! TBI !!! verify if the correrct TT is not already in palce */
	
	/* copy the language string */
	brl_disp->Language = g_strdup (language);
	
	/* create the actual file name */
	fn = g_strdup_printf ("%s.A2B", language);
	g_strup(fn);
	
	/* open the file */
	fp = fopen (fn, "rb");
	if (fp)
	{		
		/* load translation table in memory */
		fread (brl_disp->TranslationTable, 1, TT_SIZE, fp);		

		fclose (fp);	
	}	
	g_free (fn);
	
#endif	

}


void brl_disp_add_dots (BRL_DISP *brl_disp, guint8* dots, int len)
{

/*	fprintf(stderr, "BRL: add dots: ID %08x, len %d\n", brl_disp, len); */
	g_byte_array_append (brl_disp->Dots, dots, len);
}

/* __ BRL_OUT METHODS _________________________________________________________ */

BRL_OUT* brl_out_new ()
{
	BRL_OUT* brl_out;
	brl_out = g_malloc0 (sizeof(BRL_OUT));
	brl_out->Displays = g_array_new (FALSE, FALSE, sizeof (BRL_DISP*));
	/* brl_out->Displays = g_array_new (FALSE, FALSE, sizeof (BRL_DISP)); */
	/* brl_out->TranslationTable = g_malloc0(TT_SIZE);	 */
	/* brl_out->Language = NULL; */
	
	return brl_out;
}

void brl_out_free (BRL_OUT *brl_out)
{
	int i;
	BRL_DISP*	brl_disp;
		
	/* g_free (brl_out->Language); */
	/* g_free (brl_out->TranslationTable); */
	
	/* free the brl_disp */
	for (i = 0; i <brl_out->Displays->len ; ++i)
	{
		brl_disp = g_array_index (brl_out->Displays, BRL_DISP*,i);
		brl_disp_free(brl_disp);
	}
	
	g_array_free (brl_out->Displays, TRUE);
	g_free (brl_out);
}

void brl_out_add_display (BRL_OUT *brl_out, BRL_DISP *brl_disp)
{
	
	BRL_DISP * bd_copy;
		
	/* make a copy of the brl_disp (incl. role and dots) */
	bd_copy = brl_disp_copy (brl_disp);
		
	/* ... and add it	 */
	g_array_append_vals (brl_out->Displays, &bd_copy, 1);
	
}
	
void brl_out_load_translation_table (BRL_OUT *brl_out, gchar* language)
{
	
	brl_out->TranslationTable = ttc_get_translation_table (language);

#if 0	
	gchar* fn;
	FILE* fp;
	
	/* !!! TBR !!! use a cache (i.e. HashTable?) */
			
	g_free (brl_out->Language);
	brl_out->Language = g_strdup (language);
	
	/* create the actual file name */
	fn = g_strdup_printf ("%s.A2B", language);
	g_strup(fn);
	
	/* open the file */
	fp = fopen (fn, "rb");
	if (fp)
	{		
		/* load translation table in memory */
		fread (brl_out->TranslationTable, 1, TT_SIZE, fp);		

		fclose (fp);	
	}
	
	g_free (fn);
#endif
}

void brl_out_set_brl_style (BRL_OUT *brl_out, gchar* style)
{			
	switch (atoi(style))
	{
		case 0:
			brl_out->BrailleStyle = BS_8_DOTS;
		break;
				
		case 1:
			brl_out->BrailleStyle = BS_6_DOTS;
		break;
				
		case 6:
			brl_out->BrailleStyle = BS_6_DOTS;
		break;
				
		case 8:
			brl_out->BrailleStyle = BS_8_DOTS;
		break;
				
		default:
			brl_out->BrailleStyle = BS_8_DOTS;
		break;
	}
}

void brl_out_set_clear_mode (BRL_OUT *brl_out, gchar* clear)
{			

	if (	(g_strcasecmp(clear, "true") == 0) ||
			(g_strcasecmp(clear, "on") == 0) ||
			(g_strcasecmp(clear, "1") == 0)
		)
	{
		brl_out->ClearAllCells = TRUE;
	}
	else
	{
		brl_out->ClearAllCells = FALSE;
	}
	
}

void brl_out_to_driver (BRL_OUT *brl_out)
{
	
	int i, n;
	BRL_DISP *brl_disp;
	gshort did = 0;
	
	/* fprintf (stderr, "brl_out_to_driver\n"); */
	
	if (brl_out->ClearAllCells)
	{
		brl_clear_all();
	}
	
	for (i = 0; i < brl_out->Displays->len; ++i)
	{
		/* get the BRL_DISP */
		brl_disp = g_array_index(brl_out->Displays, BRL_DISP*, i);
						
		/* find the BRL_ID	 from the Role and the No		 */
		did = brl_get_disp_id (brl_disp->Role, brl_disp->ID);
		if (did >= 0 )
		{
        	
			if (brl_disp->ClearDisplay)
			{
				brl_clear_display (did);
			}
		
			/* consider the cursor */
			
			/* fprintf (stderr, "CP:%d, M:%02X, P:%02X\n", brl_disp->CursorPosition, brl_disp->CursorMask, brl_disp->CursorPattern); */
			
			if (brl_disp->CursorPosition >= 0 &&
				brl_disp->CursorPosition < 1024			/* to avoid ill values eating too much mem ... */
			)	
			{
				/* the display has a potentially visible cursor */
				if (brl_disp->CursorPosition >= brl_disp->Dots->len )
				{
					/* cursor outside the current dots, pad with spaces */
					n = brl_disp->CursorPosition - brl_disp->Dots->len + 1;					
					brl_disp_add_dots (brl_disp, g_malloc0(n), n);
				}
				
				/* now we are sure that the cursor is in range */
				/* we merge it to the existing dots, according to it's mask and patern */
				brl_disp->Dots->data[brl_disp->CursorPosition] &= ~brl_disp->CursorMask;	
				brl_disp->Dots->data[brl_disp->CursorPosition] |= brl_disp->CursorPattern & brl_disp->CursorMask;
				
			}
			
			/* do the out */			
 			/* fprintf(stderr, "BRL: set dots: did %d, start %d, len %d, off %d\n",
							did, brl_disp->Start, brl_disp->Dots->len, brl_disp->Offset); */
			
			brl_set_dots (did, brl_disp->Start, &brl_disp->Dots->data[0], brl_disp->Dots->len, brl_disp->Offset, brl_disp->CursorPosition);
		
		}
		
	}
	
	brl_update_dots(1);	/* update all dots now (change to 0 for non-blocking update) */
	
}

/* __ SAX CALLBACKS ___________________________________________________________ */

void brl_startDocument (void *ctx)
{
	/* fprintf (stderr, "BRL: startDocument\n"); */
}

void brl_endDocument (void *ctx)
{
	/* fprintf (stderr, "BRL: endDocument\n"); */
}

void brl_startElement (void *ctx, const xmlChar* name, const xmlChar ** attrs)
{	

	gchar* attr_val;
	gchar* tattr_val;
		
/*	fprintf (stderr, "BRL: startElement: %s\n", name); */
	
	switch (brl_curr_state)
	{
		case BPS_IDLE:
					
			if (g_strcasecmp((gchar*)name, "BRLOUT") == 0)
			{
				/* create a new BRL_OUT object */
				tbrl_out = brl_out_new();				
				
				/* BRLOUT ATTRIBUTES */
				if (attrs)
				{
					while (*attrs)
					{
						/* fprintf (stderr, "attr_val: %s\n", *attrs); */
											
						if (g_strcasecmp((gchar*)*attrs, "language") == 0)
						{
							++attrs;						
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);

							brl_out_load_translation_table (tbrl_out, tattr_val);						
							
							g_free (attr_val);
							
						}
						else if (g_strcasecmp((gchar*)*attrs, "bstyle") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);
							
							brl_out_set_brl_style (tbrl_out, tattr_val);
							
							g_free (attr_val);
						}					
						else if (g_strcasecmp((gchar*)*attrs, "clear") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);
							
							brl_out_set_clear_mode (tbrl_out, tattr_val);
							
							g_free (attr_val);
						}
						else
						{						
							/* unsupported attribute */
							fprintf (stderr, "Attribute ""%s"" is not supported in the BRLOUT tag.\n", *attrs);
							++attrs;						
						}
						++attrs;					
					}
				}
								
				brl_curr_state = BPS_BRL_OUT;
			}
			
		break;
		
		case BPS_BRL_OUT:
			
			if (g_strcasecmp((gchar*)name, "BRLDISP") == 0)
			{
				/* create a new BRL_DISP object */
				tbrl_disp = brl_disp_new();
								
				/* BRLDISP ATTRIBUTES */
				if (attrs)
				{				
					while (*attrs)
					{
						/* fprintf (stderr, "attr_val: %s\n", *attrs); */
						if (g_strcasecmp((gchar*)*attrs, "id") == 0)
						{
							++attrs;						
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);

							brl_disp_set_id (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}
						else if (g_strcasecmp((gchar*)*attrs, "role") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);

							brl_disp_set_role (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}
						else if (g_strcasecmp((gchar*)*attrs, "dno") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);

							brl_disp_set_disp_no (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}					
						else if (g_strcasecmp((gchar*)*attrs, "clear") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);

							brl_disp_set_clear_mode (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}
						else if (g_strcasecmp((gchar*)*attrs, "start") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);
							
							brl_disp_set_start (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}
						else if (g_strcasecmp((gchar*)*attrs, "offset") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);
							
							brl_disp_set_offset (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}
						else if (g_strcasecmp((gchar*)*attrs, "cstyle") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);

							brl_disp_set_cursor_style (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}
						else if (g_strcasecmp((gchar*)*attrs, "cursor") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);

							brl_disp_set_cursor (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}
						else
						{						
							/* unsupported attribute */
							fprintf (stderr, "Attribute ""%s"" is not supported in the BRLDISP tag.\n", *attrs);
							++attrs;						
						}
						++attrs;
					
					}
				}
				brl_curr_state = BPS_BRL_DISP;
			}			
			
		break;
		
		case BPS_BRL_DISP:
		
			if (g_strcasecmp((gchar*)name, "DOTS") == 0)
			{
				brl_curr_state = BPS_DOTS;
			}
			
			if (g_strcasecmp((gchar*)name, "TEXT") == 0)
			{
								
				/* reset display attributes */
				tbrl_disp->Attribute = 0;
				
				/* reset language to default from BRL_OUT (!!! TBR !!! need something more efficient here) */
				tbrl_disp->TranslationTable = tbrl_out->TranslationTable;
				
				/* TEXT ATTRIBUTES */
				if (attrs)
				{
					while (*attrs)
					{
						/* fprintf (stderr, "attr_val: %s\n", *attrs); */
						if (g_strcasecmp((gchar*)*attrs, "language") == 0)
						{
							++attrs;						
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);
							
							brl_disp_load_translation_table (tbrl_disp, tattr_val);
							
							g_free (attr_val);

						}
						else if (g_strcasecmp((gchar*)*attrs, "attr") == 0)
						{
							++attrs;
							
							attr_val = g_strdup((gchar*)*attrs);
							tattr_val = g_strstrip (attr_val);

							brl_disp_set_text_attr (tbrl_disp, tattr_val);
							
							g_free (attr_val);
						}
						else
						{						
							/* unsupported attribute */
							fprintf (stderr, "Attribute ""%s"" is not supported in the TEXT tag.\n", *attrs);
							++attrs;						
						}
						++attrs;					
					}
				}	
				brl_curr_state = BPS_TEXT;
			}

		break;
		
		case BPS_DOTS: break;
		case BPS_TEXT: break;
		
		case BPS_UNKNOWN:
			brl_prev_state = brl_curr_state;
			++brl_unknown_depth;
		break;
	}
		
}

void brl_endElement (void *ctx, const xmlChar* name)
{
	/* fprintf (stderr, "BRL: endElement: %s\n", name); */
	
	switch (brl_curr_state)
	{
		
		case BPS_IDLE:			
		break;
		
		case BPS_BRL_OUT:
		
			if (g_strcasecmp((gchar*)name, "BRLOUT") == 0)
			{
				/* output packet done, make the driver calls */
				brl_out_to_driver (tbrl_out);
				
				/* free it */
				brl_out_free (tbrl_out);
				brl_curr_state = BPS_IDLE;
			}
						
		break;
		
		
		case BPS_BRL_DISP:
		
			if (g_strcasecmp((gchar*)name, "BRLDISP") == 0)
			{
								
				/* finished with the current BRL_DISP, add it to the BRL_OUT.Displays */
				brl_out_add_display (tbrl_out, tbrl_disp); /* NOTE: it puts a copy on the Display container */
				brl_disp_free (tbrl_disp);
				tbrl_disp = NULL;
								
				brl_curr_state = BPS_BRL_OUT;
			}			
			
		break;
		
		case BPS_DOTS:
			if (g_strcasecmp((gchar*)name, "DOTS") == 0)
			{
				brl_curr_state = BPS_BRL_DISP;
			}
		break;
		
		case BPS_TEXT:
			if (g_strcasecmp((gchar*)name, "TEXT") == 0)
			{
				brl_curr_state = BPS_BRL_DISP;
			}
		break;
		
		case BPS_UNKNOWN:
			--brl_unknown_depth;
			if (brl_unknown_depth <= 0)
			{
				brl_curr_state = brl_prev_state;
			}
		break;
	}

}

void brl_characters (void *ctx, const xmlChar *ch, int len)
{
	
	/* char ts[128]; */

	int 			i;
	gchar*		tch;
	gchar**	tokens;
	guint8		tdots;
	guint8*	dotbuff;
		
	tch = g_strndup((gchar*)ch, len);
	tch = g_strstrip(tch);
	switch (brl_curr_state)
	{
		case BPS_IDLE: break;
						
		case BPS_DOTS:
		
			/* add dots to tbrl_disp			 */
			tokens = g_strsplit (tch, " ", 0);
			for (i = 0; tokens[i] != NULL; ++i)
			{								
				/* fprintf (stderr, "%s, ", tokens[i]); */
				tdots = dotstr_to_bits(tokens[i]);				
				brl_disp_add_dots (tbrl_disp, &tdots, 1);
			}
			
			/* fprintf (stderr, "\n");			 */
			g_strfreev (tokens);
		
		break;
		
		case BPS_TEXT:
		{
			gchar *str_utf = NULL;
			gchar *str_crt = NULL;
			glong  str_len;
			str_utf = g_strndup (ch, len);
			str_len = g_utf8_strlen (str_utf, -1);
			/* convert text to braille dots			 */
			dotbuff = malloc (str_len);			
			str_crt = str_utf;					
			
			for (i = 0; i < str_len; ++i)
			{
				
				/* translate char to dots */
				if (tbrl_disp->TranslationTable)
				{
					if (TT_SIZE > g_utf8_get_char (str_crt))
					{
					    /* use the translation table */
					    dotbuff[i] = tbrl_disp->TranslationTable[g_utf8_get_char (str_crt)];
					}
					else
					{
					    /* use the last simbol from translation table, 
					    when the character code is grant then the 
					    number of simbols in translation table*/
					    dotbuff[i] = tbrl_disp->TranslationTable[TT_SIZE - 1];
					}
				}
				else
				{
					/* no table, no dots */
					dotbuff[i] = 0;
				}
				
				/* consider 6/8 dot Braille style */
				if (tbrl_out->BrailleStyle == BS_6_DOTS)
				{
					dotbuff[i] &= 0x3F;	/* clear dot 78 */
				}
				
				/* merge the Braille attributes				 */
				dotbuff[i] |= tbrl_disp->Attribute;
				str_crt = g_utf8_find_next_char (str_crt, NULL);
			}
							
			brl_disp_add_dots (tbrl_disp, dotbuff, str_len );
			
			free (dotbuff);
			free (str_utf);	
		break;
		}
		case BPS_BRL_OUT: break;
		case BPS_BRL_DISP: break;
		case BPS_UNKNOWN: break;
		
	}
	
	g_free (tch);
	
}

void brl_warning (void *ctx, const char *msg, ...)
{
	va_list args;

	va_start (args, msg);
	g_logv ("BRL XML", G_LOG_LEVEL_WARNING, msg, args);
	va_end (args);
	
	/* !!! TBI !!! brl_curr_state = BPS_IDLE */
}

void brl_error (void *ctx, const char *msg, ...)
{
	va_list args;

	va_start (args, msg);
	g_logv ("BRL XML", G_LOG_LEVEL_CRITICAL, msg, args);
	va_end (args);
	/* !!! TBI !!! brl_curr_state = BPS_IDLE */
}

void brl_fatalError (void *ctx, const char *msg, ...)
{
	va_list args;

	va_start (args, msg);
	g_logv ("BRL XML", G_LOG_LEVEL_ERROR, msg, args);
	va_end (args);
	/* !!! TBI !!! brl_curr_state = BPS_IDLE */
}


void brl_xml_output (char* buffer, int len)
{

	if (brl_xml_initialized)
	{
/*		fprintf (stderr, "brl_xml_output: %s len:%d\n", buffer, len);	 */
		xmlSAXParseMemory (brl_ctx, buffer, len, 0); /* recovery == 0??? */
	}
	else
	{
		fprintf (stderr, "ERROR: brl_xml_output called before brl_xml_init.\n");
	}
	
}

/* __ BRLXML API ______________________________________________________________ */

void BrailleEvents (BRAILLE_EVENT_CODE EventCode, BRAILLE_EVENT_DATA *EventData)
{
	
	int ix;
	char xml_out[1024];

	
	if (!XMLClientCallback) return;
	

	switch (EventCode)
	{
		case bec_raw_byte:
		case bec_key_bits:	/* <KEYSTATE></KEYSTATE> ?		 */
		    return;
		    break;
		case bec_key_codes:
		case bec_switch_pad:
		case bec_sensor:
		default:
		    break;
	}
	
	ix = 0;	
	ix += sprintf (&xml_out[ix], "<BRLIN>\n");	/* BRLIN entry tag */
	switch (EventCode)
	{
	
		case bec_key_codes:
			ix += sprintf (&xml_out[ix], "<KEY>%s</KEY>\n", EventData->KeyCodes);			
		break;
			
		case bec_sensor:
			ix += sprintf (&xml_out[ix], "<SENSOR bank=\"%d\" display=\"%d\" technology=\"%d\">%s</SENSOR>\n",			
				EventData->Sensor.Bank,
				EventData->Sensor.AssociatedDisplay,
				EventData->Sensor.Technology,
				EventData->Sensor.SensorCodes
			);			
		break;
			
		case bec_switch_pad:
			ix += sprintf (&xml_out[ix], "<SWITCH>%s</SWITCH>\n",				
				EventData->SwitchPad.SwitchCodes
			);						
		break;
		
/*		
		case bec_sensor:
			fprintf (stderr, "\nSENSOR: value:%d, bank:%d, disp:%d, tech:%d",
							EventData->Sensor.Value,
							EventData->Sensor.Bank,
							EventData->Sensor.AssociatedDisplay,
							EventData->Sensor.Technology
			);
			break;
*/

		default:
			fprintf (stderr, "brlxml: unsupported input event\n");
		break;
	}
	
	ix += sprintf (&xml_out[ix], "</BRLIN>\n");	/* BRLIN exit tag */
	XMLClientCallback (xml_out, ix);

}


int brl_xml_init (char* DeviceName, int Port, BRL_XML_INPUT_PROC InputCallbackProc)
{

	int rv = 1;
	
	if (!brl_xml_initialized)
	{

		XMLClientCallback = InputCallbackProc;
		xmlInitParser();
	
		/* initialize translation table cache */
		ttc_init();
	
		/* initialize the XML parser */
		brl_ctx = g_malloc0(sizeof(xmlSAXHandler));
		
		brl_ctx->startDocument = brl_startDocument;
		brl_ctx->endDocument = brl_endDocument;
		brl_ctx->startElement = brl_startElement;
		brl_ctx->endElement = brl_endElement;
		brl_ctx->characters = brl_characters;

		brl_ctx->warning =   brl_warning;
		brl_ctx->error = brl_error;
		brl_ctx->fatalError = brl_fatalError;
	
		/* initialize the low level Braille stuff */
	
		brl_init();		
		rv = brl_open_device (DeviceName, Port, BrailleEvents);
	
		brl_xml_initialized = TRUE;
	
	}
	else
	{
		fprintf (stderr, "WARNING: brl_xml_init called more than once.\n");
	}

	return rv;	
}

void brl_xml_terminate ()
{
	/* !!! TBR !!!	 */
	
	if (brl_xml_initialized )
	{
		if (brl_ctx) g_free(brl_ctx);
	
		ttc_terminate();
		brl_terminate();
		
		brl_xml_initialized = FALSE;
	}
	else
	{
		fprintf (stderr, "WARNING: brl_xml_terminate called more than once.\n");
	}
	
}

/* __ RECYCLE BIN ______________________________________________________________ */


