/* gok-settings-dialog.c
*
* Copyright 2002 Sun Microsystems, Inc.,
* Copyright 2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <gnome.h>
#include "gok-settings-dialog.h"
#include "gok-page-keysizespace.h"
#include "gok-page-actions.h"
#include "gok-page-feedbacks.h"
#include "gok-page-accessmethod.h"
#include "gok-page-wordcomplete.h"
#include "gok-page-accessmethod.h"
#include "gok-data.h"
#include "support.h"
#include "gok-log.h"
#include "main.h"

/* pointer to the settings dialog window */
static GtkWidget* m_pWindowSettings;

/* Private functions */

/**
* gok_settingsdialog_open
* 
* Creates the GOK settings dialog.
*
* @bShow: If TRUE the settings dialog will be shown. If FALSE the dialog will be
* created but not shown.
*
* returns: TRUE if the settings dialog could be created, FALSE if not.
**/
gboolean gok_settingsdialog_open (gboolean bShow)
{
	GtkWidget* pButtonTry;
	GtkWidget* pButtonRevert;
	GtkWidget* pButtonOK;
	GtkWidget* pButtonCancel;
	GtkWidget* pButtonHelp;
	GtkWidget* hbuttonbox;
	GtkTooltips *tooltips;

	m_pWindowSettings = create_dialogSettings();
	if (m_pWindowSettings == NULL)
	{
		gok_log_x ("Error: Settings dialog not created in gok_settingsdialog_open!\n");
		return FALSE;
	}

	/* initialize all the pages */
	gok_settings_page_keysizespace_initialize (m_pWindowSettings);
	gok_page_accessmethod_initialize (m_pWindowSettings);
	gok_page_actions_initialize (m_pWindowSettings);
	gok_page_feedbacks_initialize (m_pWindowSettings);
	gok_page_wordcomplete_initialize (m_pWindowSettings);
	
	/* create a backup copy of all the settings */
	gok_data_backup_settings();
	
	tooltips = gtk_tooltips_new ();
	
/* add the buttons (OK, CANCEL, APPLY HELP) at the bottom of the dialog */
/* I'm doing this here because Glade generates only the GTK 1.2 calls and we need GTK 1.3 or greater*/

	/* get the button box that will hold the buttons */
	hbuttonbox = lookup_widget (m_pWindowSettings, "hboxStockButtons");
	if (hbuttonbox == NULL)
	{
		gok_log_x ("Error: Can't find the button box for settings dialog in gok_settingsdialog_open!\n");
		return FALSE;
	}

#if GTK_MAJOR_VERSION < 2
#if GTK_MINOR_VERSION < 3
/* these calls can be used only on GTK 1.2 and below */

#define EARLY_GTK_VERSION 1

	pButtonTry = gnome_stock_button (GNOME_STOCK_BUTTON_TRY);
	pButtonRevert = gnome_stock_button (GNOME_STOCK_BUTTON_REVERT);
	pButtonOK = gnome_stock_button (GNOME_STOCK_BUTTON_OK);
	pButtonCancel = gnome_stock_button (GNOME_STOCK_BUTTON_CANCEL);
	pButtonHelp = gnome_stock_button (GNOME_STOCK_BUTTON_HELP);
 
#endif /* #if GTK_MINOR_VERSION < 3 */
#endif /* #if GTK_MAJOR_VERSION < 2 */

#ifndef EARLY_GTK_VERSION
/* these calls can be used only on GTK 1.3 and above */

	pButtonTry = gtk_button_new_from_stock (GTK_STOCK_JUMP_TO);
	pButtonRevert = gtk_button_new_from_stock (GTK_STOCK_REVERT_TO_SAVED);
	pButtonOK = gtk_button_new_from_stock (GTK_STOCK_OK);
	pButtonCancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	pButtonHelp = gtk_button_new_from_stock (GTK_STOCK_HELP);
#endif

	gtk_button_set_label (GTK_BUTTON(pButtonTry), "Try");
/*	gtk_button_set_use_stock (GTK_BUTTON(pButtonTry), TRUE); */

	gtk_widget_ref (pButtonTry);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonTry", pButtonTry,
	                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonTry);
	gtk_box_pack_start (GTK_BOX (hbuttonbox), pButtonTry, FALSE, FALSE, 10);
	GTK_WIDGET_SET_FLAGS (pButtonTry, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip (tooltips, pButtonTry, _("try these settings"), NULL);

	gtk_signal_connect (GTK_OBJECT (pButtonTry), "clicked",
                      GTK_SIGNAL_FUNC (on_button_try),
                      NULL);

	gtk_widget_ref (pButtonRevert);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonRevert", pButtonRevert,
	                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonRevert);
	gtk_box_pack_start (GTK_BOX (hbuttonbox), pButtonRevert, FALSE, FALSE, 10);
	GTK_WIDGET_SET_FLAGS (pButtonRevert, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip (tooltips, pButtonRevert, _("revert to original settings"), NULL);

	gtk_signal_connect (GTK_OBJECT (pButtonRevert), "clicked",
                      GTK_SIGNAL_FUNC (on_button_revert),
                      NULL);

	gtk_widget_ref (pButtonOK);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonOK", pButtonOK,
	                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonOK);

	gtk_box_pack_start (GTK_BOX (hbuttonbox), pButtonOK, FALSE, FALSE, 10);
	GTK_WIDGET_SET_FLAGS (pButtonOK, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip (tooltips, pButtonOK, _("use these settings"), NULL);

	gtk_signal_connect (GTK_OBJECT (pButtonOK), "clicked",
                      GTK_SIGNAL_FUNC (on_button_ok),
                      NULL);

	gtk_widget_ref (pButtonCancel);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonCancel", pButtonCancel,
	                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonCancel);
	gtk_box_pack_start (GTK_BOX (hbuttonbox), pButtonCancel, FALSE, FALSE, 10);

	GTK_WIDGET_SET_FLAGS (pButtonCancel, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip (tooltips, pButtonCancel, _("cancel"), NULL);

	gtk_signal_connect (GTK_OBJECT (pButtonCancel), "clicked",
                      GTK_SIGNAL_FUNC (on_button_cancel),
                      NULL);
	
	gtk_widget_ref (pButtonHelp);
	gtk_object_set_data_full (GTK_OBJECT (m_pWindowSettings), "pButtonHelp", pButtonHelp,
	                            (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (pButtonHelp);
	gtk_box_pack_start (GTK_BOX (hbuttonbox), pButtonHelp, FALSE, FALSE, 10);

	GTK_WIDGET_SET_FLAGS (pButtonHelp, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip (tooltips, pButtonHelp, _("display user help"), NULL);

	gtk_signal_connect (GTK_OBJECT (pButtonHelp), "clicked",
                      GTK_SIGNAL_FUNC (on_button_help),
                      NULL);
	
	/* show/hide the settings dialog */
	if (bShow == TRUE)
	{
		gtk_widget_show (m_pWindowSettings);
	}

	return TRUE;
}

/**
* gok_settingsdialog_show
* 
* Displays the GOK settings dialog.
*
* returns: TRUE if the settings dialog was shown, FALSE if not.
**/
gboolean gok_settingsdialog_show()
{
	gboolean bFirstShowing;
	
	g_assert (m_pWindowSettings != NULL);

	bFirstShowing = (GDK_IS_DRAWABLE(m_pWindowSettings->window) == FALSE) ? TRUE : FALSE;

	gtk_widget_show (m_pWindowSettings);

	/* if this is the first time we're showing the dialog then redisplay
	  the key size/spacing page so the example keys are centered */
	if (bFirstShowing == TRUE)
	{
		gok_settings_page_keysizespace_refresh();
	}
	
	return TRUE;
}

/**
* gok_settingsdialog_hide
* 
* Hides the GOK settings dialog.
**/
void gok_settingsdialog_hide()
{
	g_assert (m_pWindowSettings != NULL);

 	gtk_widget_hide (m_pWindowSettings);
}

/**
* gok_settingsdialog_get_window
*
* returns: A pointer to the settings dialog window.
**/
GtkWidget* gok_settingsdialog_get_window ()
{
 	return m_pWindowSettings;
}

/**
* gok_settingsdialog_close
* 
* Destroys the GOK settings dialog.
*
* returns: void
**/
void gok_settingsdialog_close()
{
	g_assert (m_pWindowSettings != NULL);
	gtk_widget_destroy (m_pWindowSettings);
	
	gok_page_accessmethod_close();
}

/**
* on_button_ok
* @pButton: Pointer to the button that was clicked.
* @user_data: Any user data associated with the button.
* 
* The OK button has been clicked. Apply any changes from the settings and
* hide the settings dialog.
**/
void on_button_ok (GtkButton* button, gpointer user_data)
{
	/* apply the current settings */
	on_button_try (NULL, NULL);
	
	/* copy the current settings to the backup */
	gok_settingsdialog_backup_settings();
	
	/* hide the settings dialog */
	gok_settingsdialog_hide();
}

/**
* on_button_try
* @pButton: Pointer to the button that was clicked.
* @user_data: Any user data associated with the button.
* 
* The TRY button has been clicked. Apply any changes from the settings but
* don't hide the dialog
**/
void on_button_try (GtkButton* button, gpointer user_data)
{
	gboolean bDataChanged;

	bDataChanged = FALSE;
	
	/* use the current settings */
	if (gok_settings_page_keysizespace_apply() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_page_actions_apply() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_page_feedbacks_apply() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_page_accessmethod_apply() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_page_wordcomplete_apply() == TRUE)
	{
		bDataChanged = TRUE;
	}
	
	/* if any settings have changed then refresh the keyboard */
	if (bDataChanged == TRUE)
	{
		gok_main_display_scan_reset();		
	}
}

/**
* on_button_revert
* @pButton: Pointer to the button that was clicked.
* @user_data: Any user data associated with the button.
* 
* The REVERT button has been clicked. Returns the settings to the way they were
* and update the keyboard.
**/
void on_button_revert (GtkButton* button, gpointer user_data)
{
	gboolean bDataChanged;

	bDataChanged = FALSE;
	
	if (gok_data_restore_settings() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_settings_page_keysizespace_revert() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_page_accessmethod_revert() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_page_wordcomplete_revert() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_page_actions_revert() == TRUE)
	{
		bDataChanged = TRUE;
	}
	if (gok_page_feedbacks_revert() == TRUE)
	{
		bDataChanged = TRUE;
	}

	/* if any setting has changed then refresh the keyboard and dialog controls */
	if (bDataChanged == TRUE)
	{
		gok_main_display_scan_reset();
	}
}

/**
* on_button_cancel
* @pButton: Pointer to the button that was clicked.
* @user_data: Any user data associated with the button.
* 
* The CANCEL button has been clicked. Return settings to the way they were and
* update the keyboard and hide the settings dialog.
**/
void on_button_cancel (GtkButton* pButton, gpointer user_data)
{
	/* revert to the original settings */
	on_button_revert(NULL, NULL);

	/* hide the settings dialog */
	gok_settingsdialog_hide();
}

/**
* on_button_help
* @pButton: Pointer to the button that was clicked.
* @user_data: Any user data associated with the button.
*
* The HELP button has been clicked.
**/
void on_button_help (GtkButton* pButton, gpointer user_data)
{
	GtkWidget* pNotebook;
	
	pNotebook = lookup_widget (m_pWindowSettings, "notebook1");
	if (pNotebook == NULL)
	{
		gok_log_x ("Can't find notebook!\n");
		return;
	}
	
	switch (gtk_notebook_get_current_page (GTK_NOTEBOOK (pNotebook)))
	{
		case 0:
			gnome_help_display ("gok.xml","gok-settings-appearance", NULL);
			break;
			
		case 1:
			gnome_help_display ("gok.xml", "gok-settings-actions", NULL);
			break;
			
		case 2:
 			gnome_help_display ("gok.xml", "gok-settings-feedback", NULL);
			break;

		case 3:
			gnome_help_display ("gok.xml", "gok-settings-accessmethods",
                                            NULL);
			break;
			
		case 4:
			gnome_help_display ("gok.xml", "gok-settings-prediction", NULL);
			break;
			
		default:
			break;
	}
}

/**
* gok_settingsdialog_refresh
* 
* Refreshes the dialog controls. This should be called after the user has resized the keyboard
* (which causes the key size to change) or if the gok data has changed.
**/
void gok_settingsdialog_refresh ()
{
	/* only the keysize/space page needs to get refreshed */
	gok_settings_page_keysizespace_refresh();
}

/**
* gok_settingsdialog_backup_settings
* 
* Copies all the member settings to backup.
**/
void gok_settingsdialog_backup_settings ()
{
	gok_settings_page_keysizespace_backup();
	gok_page_actions_backup();
	gok_page_feedbacks_backup();
	gok_page_accessmethod_backup();
	gok_page_wordcomplete_backup();
	
	gok_data_backup_settings();
}

/**
* gok_settingsdialog_sort
* @pItem1: pointer to item #1.
* @pItem2: pointer to item #2.
*
* returns: 0 if the items are the same, -1 if pItem1 is less than pItem 2.
* Returns 1 if pItem1 is greater than pItem2.
**/
gint gok_settingsdialog_sort (gconstpointer pItem1, gconstpointer pItem2)
{
	return strcmp (pItem1, pItem2);
}

