/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* unit-test-location.c: Test location functionality

   Copyright (C) 2007 Stefan Walter

   The Gnome Keyring Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Keyring Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Author: Stef Walter <stef@memberwebs.com>
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "run-auto-test.h"

#include "common/gkr-location.h"

/* 
 * Each test looks like (on one line):
 *     void unit_test_xxxxx (CuTest* cu)
 * 
 * Each setup looks like (on one line):
 *     void unit_setup_xxxxx (void);
 * 
 * Each teardown looks like (on one line):
 *     void unit_teardown_xxxxx (void);
 * 
 * Tests be run in the order specified here.
 */
 
#define MEDIA_DEVICE "/media/DummyMount"
#define MEDIA_DEVICE2 "/media/DummyMount2"
#define MEDIA_SERIAL "MEDIA-0239482093"


void unit_test_location_simple (CuTest* cu)
{
	gchar *path = g_build_filename (g_get_home_dir (), "blah", NULL);
	gchar *path2;
	GQuark loc, child;
	
	loc = gkr_location_from_path (path);
	CuAssert (cu, "should return a non-zero loc quark", loc != 0);

	g_print ("quark: %s\n", g_quark_to_string (loc));
	
	path2 = gkr_location_to_path (loc);
	CuAssert (cu, "should return non-null path", path2 != NULL);
	CuAssert (cu, "should return the same path", strcmp (path2, path) == 0);

	child = gkr_location_from_child (loc, "3");
	CuAssert (cu, "should return a non-zero loc quark", loc != 0);

	child = gkr_location_from_child (child, "2");
	CuAssert (cu, "should return a non-zero loc quark", loc != 0);

	g_print ("child quark: %s\n", g_quark_to_string (child));

	path2 = gkr_location_to_path (child);
	CuAssert (cu, "should return non-null path", path2 != NULL);
	
	CuAssert (cu, "should be volume", gkr_location_is_volume (GKR_LOCATION_VOLUME_HOME));
	CuAssert (cu, "should not be volume", !gkr_location_is_volume (loc));
	CuAssert (cu, "should not be volume", !gkr_location_is_volume (child));
}

void unit_test_location_trailing (CuTest* cu)
{
	GQuark one, two, ref;
	
	one = gkr_location_from_child (GKR_LOCATION_VOLUME_LOCAL, "/blah/");
	CuAssert (cu, "should return a non-zero quark", one != 0);

	two = gkr_location_from_child (GKR_LOCATION_VOLUME_LOCAL, "blah//");
	CuAssert (cu, "should return a non-zero quark", two != 0);

	ref = gkr_location_from_child (GKR_LOCATION_VOLUME_LOCAL, "blah");
	CuAssert (cu, "should return a non-zero quark", ref != 0);

	/* Should all be identical */
	CuAssert (cu, "stripping of leading and trailing slashes did not work", ref == one);
	CuAssert (cu, "stripping of leading and trailing slashes did not work", ref == two);
}

void unit_test_location_parent (CuTest* cu)
{
	GQuark child, ref, parent;

	ref = gkr_location_from_child (GKR_LOCATION_VOLUME_LOCAL, "first");
	CuAssert (cu, "should return a non-zero quark", ref != 0);
	
	child = gkr_location_from_child (ref, "second");
	CuAssert (cu, "should return a non-zero quark", child != 0);
	
	parent = gkr_location_to_parent (child);
	CuAssert (cu, "should return a non-zero quark", parent != 0);
	CuAssert (cu, "parent location does not equal original", parent == ref);
	
	/* Should return the volume */
	parent = gkr_location_to_parent (parent);
	CuAssert (cu, "should return a non-zero quark", parent != 0);
	CuAssert (cu, "parent of parent location does not equal volume", parent == GKR_LOCATION_VOLUME_LOCAL);
}

void unit_test_location_media (CuTest* cu)
{
	gchar *path = g_build_filename (MEDIA_DEVICE, "testo", NULL);
	gchar *path2;
	GQuark loc;
	
	/* Device is inserted */
	gkr_location_manager_register (gkr_location_manager_get (), 
	                               MEDIA_SERIAL, MEDIA_DEVICE, "Test Media");
	
	loc = gkr_location_from_path (path);
	CuAssert (cu, "should return a non-zero loc quark", loc != 0);
	
	path2 = gkr_location_to_path (loc);
	CuAssert (cu, "should return non-null path", path2 != NULL);
	CuAssert (cu, "should return the same path", strcmp (path2, path) == 0);

	/* Device is removed */
	gkr_location_manager_unregister (gkr_location_manager_get (), MEDIA_SERIAL);
	
	/* Device is inserted at another path */
	gkr_location_manager_register (gkr_location_manager_get (), 
	                               MEDIA_SERIAL, MEDIA_DEVICE2, "Test Media");
	
	path2 = gkr_location_to_path (loc);
	CuAssert (cu, "should return non-null path", path2 != NULL);
	CuAssert (cu, "should return a path at new prefix", strncmp (path2, MEDIA_DEVICE2, strlen (MEDIA_DEVICE2)) == 0);
	
}

void unit_test_location_fileops (CuTest* cu)
{
	const guchar *data = (guchar*)"TEST DATA FOR FILE";
	guchar *result;
	gsize n_result, len;
	gboolean ret;
	GQuark loc;
	
	loc = gkr_location_from_child (GKR_LOCATION_VOLUME_FILE, "/tmp/gkr-test-location-fileops");
	CuAssert (cu, "should return a non-zero quark", loc != 0);
	
	len = strlen ((gchar*)data);
	ret = gkr_location_write_file (loc, data, len, NULL);
	CuAssert (cu, "should be successful writing to temp file", ret == TRUE);

	ret = gkr_location_read_file (loc, &result, &n_result, NULL);
	CuAssert (cu, "should be successful reading from temp file", ret == TRUE);
	CuAssert (cu, "should have read same length as written", n_result == len);
	CuAssert (cu, "should have read same data as written", memcmp (data, result, len) == 0);
	
	ret = gkr_location_delete_file (loc, NULL);
	CuAssert (cu, "should have successfully deleted file", ret == TRUE);

	ret = gkr_location_read_file (loc, &result, &n_result, NULL);
	CuAssert (cu, "shouldn't be able to read from deleted file", ret == FALSE);

	ret = gkr_location_delete_file (loc, NULL);
	CuAssert (cu, "should be able to successfully delete non-existant file", ret == TRUE);
}
